/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils/enum',
    'io.ox/office/editframework/utils/operationerror',
    'io.ox/office/editframework/model/operationcontext'
], function (Enum, OperationError, OperationContext) {

    'use strict';

    // class OperationContext =================================================

    describe('EditFramework class OperationContext', function () {

        it('should exist', function () {
            expect(OperationContext).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // a fake document model
        var docModel = {};
        // the test operation
        var operation = { name: 'test', bool0: false, bool1: true, num: 2.5, nan: Number.NaN, str: 'a', estr: '', enum: 'e1', obj: { i: 1 }, arr: [3.5], earr: [], pos1: [0, 1, 2], pos2: [-1], pos3: ['a'] };
        // the tested operation context
        var context = new OperationContext(docModel, operation, false);
        // a default object value
        var defObj = { a: 42 };
        // a default array value
        var defArr = [42];
        // an enumeration class
        var MyEnum = Enum.create('e1 e2 e3');

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a context', function () {
                expect(context).to.be.an.instanceof(OperationContext);
                expect(context).to.have.a.property('docModel', docModel);
                expect(context).to.have.a.property('operation', operation);
                expect(context).to.have.a.property('external', false);
                expect(new OperationContext(docModel, operation, true)).to.have.a.property('external', true);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "error"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('error');
            });
            it('should throw', function () {
                expect(context.error.bind(context, 'bad')).to.throw(OperationError, 'bad');
            });
        });

        describe('method "ensure"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('ensure');
            });
            function ensure(cond) { return context.ensure.bind(context, cond, 'bad'); }
            it('should do nothing on truthy condition', function () {
                expect(ensure(true)).to.not.throw;
                expect(ensure(1)).to.not.throw;
                expect(ensure('a')).to.not.throw;
                expect(ensure({})).to.not.throw;
                expect(ensure([])).to.not.throw;
            });
            it('should throw on falsy condition', function () {
                expect(ensure(false)).to.throw(OperationError, 'bad');
                expect(ensure(0)).to.throw(OperationError, 'bad');
                expect(ensure('')).to.throw(OperationError, 'bad');
                expect(ensure(null)).to.throw(OperationError, 'bad');
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('has');
            });
            it('should return true for existing properties', function () {
                expect(context.has('bool0')).to.equal(true);
                expect(context.has('bool1')).to.equal(true);
                expect(context.has('num')).to.equal(true);
                expect(context.has('nan')).to.equal(true);
                expect(context.has('str')).to.equal(true);
                expect(context.has('estr')).to.equal(true);
                expect(context.has('obj')).to.equal(true);
                expect(context.has('arr')).to.equal(true);
                expect(context.has('earr')).to.equal(true);
            });
            it('should return false for missing properties', function () {
                expect(context.has('wrong')).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('get');
            });
            function get(name) { return context.get.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.get('bool0')).to.equal(operation.bool0);
                expect(context.get('bool1')).to.equal(operation.bool1);
                expect(context.get('num')).to.equal(operation.num);
                expect(context.get('nan')).to.be.nan;
                expect(context.get('str')).to.equal(operation.str);
                expect(context.get('estr')).to.equal(operation.estr);
                expect(context.get('obj')).to.equal(operation.obj);
                expect(context.get('arr')).to.equal(operation.arr);
                expect(context.get('earr')).to.equal(operation.earr);
            });
            it('should throw on missing properties', function () {
                expect(get('wrong')).to.throw(OperationError);
            });
        });

        describe('method "getOpt"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOpt');
            });
            it('should return existing properties', function () {
                expect(context.getOpt('bool0', true)).to.equal(operation.bool0);
                expect(context.getOpt('bool1', true)).to.equal(operation.bool1);
                expect(context.getOpt('num', 42)).to.equal(operation.num);
                expect(context.getOpt('nan', 42)).to.be.nan;
                expect(context.getOpt('str', '42')).to.equal(operation.str);
                expect(context.getOpt('estr', '42')).to.equal(operation.estr);
                expect(context.getOpt('obj', defObj)).to.equal(operation.obj);
                expect(context.getOpt('arr', defArr)).to.equal(operation.arr);
                expect(context.getOpt('earr', defArr)).to.equal(operation.earr);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOpt('wrong', 42)).to.equal(42);
            });
        });

        describe('method "getBool"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getBool');
            });
            function getBool(name) { return context.getBool.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getBool('bool0')).to.equal(operation.bool0);
                expect(context.getBool('bool1')).to.equal(operation.bool1);
            });
            it('should throw on missing properties', function () {
                expect(getBool('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getBool('num')).to.throw(OperationError);
                expect(getBool('nan')).to.throw(OperationError);
                expect(getBool('str')).to.throw(OperationError);
                expect(getBool('estr')).to.throw(OperationError);
                expect(getBool('obj')).to.throw(OperationError);
                expect(getBool('arr')).to.throw(OperationError);
                expect(getBool('earr')).to.throw(OperationError);
            });
        });

        describe('method "getOptBool"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptBool');
            });
            function getOptBool(name, def) { return context.getOptBool.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptBool('bool0', true)).to.equal(operation.bool0);
                expect(context.getOptBool('bool1', true)).to.equal(operation.bool1);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptBool('wrong', true)).to.equal(true);
                expect(context.getOptBool('wrong', false)).to.equal(false);
                expect(context.getOptBool('wrong')).to.equal(false);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptBool('num')).to.throw(OperationError);
                expect(getOptBool('nan')).to.throw(OperationError);
                expect(getOptBool('str')).to.throw(OperationError);
                expect(getOptBool('estr')).to.throw(OperationError);
                expect(getOptBool('obj')).to.throw(OperationError);
                expect(getOptBool('arr')).to.throw(OperationError);
                expect(getOptBool('earr')).to.throw(OperationError);
            });
        });

        describe('method "getNum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getNum');
            });
            function getNum(name) { return context.getNum.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getNum('num')).to.equal(operation.num);
            });
            it('should throw on missing properties', function () {
                expect(getNum('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getNum('bool0')).to.throw(OperationError);
                expect(getNum('bool1')).to.throw(OperationError);
                expect(getNum('nan')).to.throw(OperationError);
                expect(getNum('str')).to.throw(OperationError);
                expect(getNum('estr')).to.throw(OperationError);
                expect(getNum('obj')).to.throw(OperationError);
                expect(getNum('arr')).to.throw(OperationError);
                expect(getNum('earr')).to.throw(OperationError);
            });
        });

        describe('method "getOptNum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptNum');
            });
            function getOptNum(name, def) { return context.getOptNum.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptNum('num', 42)).to.equal(operation.num);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptNum('wrong', 42)).to.equal(42);
                expect(context.getOptNum('wrong')).to.equal(0);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptNum('bool0')).to.throw(OperationError);
                expect(getOptNum('bool1')).to.throw(OperationError);
                expect(getOptNum('nan')).to.throw(OperationError);
                expect(getOptNum('str')).to.throw(OperationError);
                expect(getOptNum('estr')).to.throw(OperationError);
                expect(getOptNum('obj')).to.throw(OperationError);
                expect(getOptNum('arr')).to.throw(OperationError);
                expect(getOptNum('earr')).to.throw(OperationError);
            });
        });

        describe('method "getInt"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getInt');
            });
            function getInt(name) { return context.getInt.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getInt('num')).to.equal(Math.floor(operation.num));
            });
            it('should throw on missing properties', function () {
                expect(getInt('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getInt('bool0')).to.throw(OperationError);
                expect(getInt('bool1')).to.throw(OperationError);
                expect(getInt('nan')).to.throw(OperationError);
                expect(getInt('str')).to.throw(OperationError);
                expect(getInt('estr')).to.throw(OperationError);
                expect(getInt('obj')).to.throw(OperationError);
                expect(getInt('arr')).to.throw(OperationError);
                expect(getInt('earr')).to.throw(OperationError);
            });
        });

        describe('method "getOptInt"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptInt');
            });
            function getOptInt(name, def) { return context.getOptInt.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptInt('num', 42)).to.equal(Math.floor(operation.num));
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptInt('wrong', 42)).to.equal(42);
                expect(context.getOptInt('wrong', 42.5)).to.equal(42);
                expect(context.getOptInt('wrong')).to.equal(0);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptInt('bool0')).to.throw(OperationError);
                expect(getOptInt('bool1')).to.throw(OperationError);
                expect(getOptInt('nan')).to.throw(OperationError);
                expect(getOptInt('str')).to.throw(OperationError);
                expect(getOptInt('estr')).to.throw(OperationError);
                expect(getOptInt('obj')).to.throw(OperationError);
                expect(getOptInt('arr')).to.throw(OperationError);
                expect(getOptInt('earr')).to.throw(OperationError);
            });
        });

        describe('method "getStr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getStr');
            });
            function getStr(name) { return context.getStr.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getStr('str')).to.equal(operation.str);
                expect(context.getStr('str', true)).to.equal(operation.str);
            });
            it('should throw on missing properties', function () {
                expect(getStr('wrong')).to.throw(OperationError);
            });
            it('should handle empty strings', function () {
                expect(getStr('estr')).to.throw(OperationError);
                expect(context.getStr('estr', true)).to.equal('');
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getStr('bool0')).to.throw(OperationError);
                expect(getStr('bool1')).to.throw(OperationError);
                expect(getStr('num')).to.throw(OperationError);
                expect(getStr('nan')).to.throw(OperationError);
                expect(getStr('obj')).to.throw(OperationError);
                expect(getStr('arr')).to.throw(OperationError);
                expect(getStr('earr')).to.throw(OperationError);
            });
        });

        describe('method "getOptStr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptStr');
            });
            function getOptStr(name, def) { return context.getOptStr.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptStr('str', '42')).to.equal(operation.str);
                expect(context.getOptStr('str', '42', true)).to.equal(operation.str);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptStr('wrong', '42')).to.equal('42');
                expect(context.getOptStr('wrong')).to.equal('');
            });
            it('should handle empty strings', function () {
                expect(getOptStr('estr')).to.throw(OperationError);
                expect(getOptStr('estr', '42')).to.throw(OperationError);
                expect(context.getOptStr('estr', true)).to.equal('');
                expect(context.getOptStr('estr', '42', true)).to.equal('');
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptStr('bool0')).to.throw(OperationError);
                expect(getOptStr('bool1')).to.throw(OperationError);
                expect(getOptStr('num')).to.throw(OperationError);
                expect(getOptStr('nan')).to.throw(OperationError);
                expect(getOptStr('obj')).to.throw(OperationError);
                expect(getOptStr('arr')).to.throw(OperationError);
                expect(getOptStr('earr')).to.throw(OperationError);
            });
        });

        describe('method "getEnum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getEnum');
            });
            function getEnum(name) { return context.getEnum.bind(context, name, MyEnum); }
            it('should return existing properties', function () {
                expect(context.getEnum('enum', MyEnum)).to.equal(MyEnum.E_1);
            });
            it('should throw on missing properties', function () {
                expect(getEnum('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getEnum('bool0')).to.throw(OperationError);
                expect(getEnum('bool1')).to.throw(OperationError);
                expect(getEnum('num')).to.throw(OperationError);
                expect(getEnum('nan')).to.throw(OperationError);
                expect(getEnum('str')).to.throw(OperationError);
                expect(getEnum('estr')).to.throw(OperationError);
                expect(getEnum('obj')).to.throw(OperationError);
                expect(getEnum('arr')).to.throw(OperationError);
                expect(getEnum('earr')).to.throw(OperationError);
            });
        });

        describe('method "getOptEnum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptEnum');
            });
            function getOptEnum(name, def) { return context.getOptEnum.bind(context, name, MyEnum, def); }
            it('should return existing properties', function () {
                expect(context.getOptEnum('enum', MyEnum, MyEnum.E_2)).to.equal(MyEnum.E_1);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptEnum('wrong', MyEnum, MyEnum.E_2)).to.equal(MyEnum.E_2);
                expect(context.getOptEnum('wrong', MyEnum)).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptEnum('bool0')).to.throw(OperationError);
                expect(getOptEnum('bool1')).to.throw(OperationError);
                expect(getOptEnum('num')).to.throw(OperationError);
                expect(getOptEnum('nan')).to.throw(OperationError);
                expect(getOptEnum('str')).to.throw(OperationError);
                expect(getOptEnum('estr')).to.throw(OperationError);
                expect(getOptEnum('obj')).to.throw(OperationError);
                expect(getOptEnum('arr')).to.throw(OperationError);
                expect(getOptEnum('earr')).to.throw(OperationError);
            });
        });

        describe('method "getObj"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getObj');
            });
            function getObj(name) { return context.getObj.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getObj('obj')).to.equal(operation.obj);
            });
            it('should throw on missing properties', function () {
                expect(getObj('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getObj('bool0')).to.throw(OperationError);
                expect(getObj('bool1')).to.throw(OperationError);
                expect(getObj('num')).to.throw(OperationError);
                expect(getObj('nan')).to.throw(OperationError);
                expect(getObj('str')).to.throw(OperationError);
                expect(getObj('estr')).to.throw(OperationError);
                expect(getObj('arr')).to.throw(OperationError);
                expect(getObj('earr')).to.throw(OperationError);
            });
        });

        describe('method "getOptObj"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptObj');
            });
            function getOptObj(name, def) { return context.getOptObj.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptObj('obj', defObj)).to.equal(operation.obj);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptObj('wrong', defObj)).to.equal(defObj);
                expect(context.getOptObj('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptObj('bool0')).to.throw(OperationError);
                expect(getOptObj('bool1')).to.throw(OperationError);
                expect(getOptObj('num')).to.throw(OperationError);
                expect(getOptObj('nan')).to.throw(OperationError);
                expect(getOptObj('str')).to.throw(OperationError);
                expect(getOptObj('estr')).to.throw(OperationError);
                expect(getOptObj('arr')).to.throw(OperationError);
                expect(getOptObj('earr')).to.throw(OperationError);
            });
        });

        describe('method "getArr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getArr');
            });
            function getArr(name) { return context.getArr.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getArr('arr')).to.equal(operation.arr);
                expect(context.getArr('earr')).to.equal(operation.earr);
            });
            it('should throw on missing properties', function () {
                expect(getArr('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getArr('bool0')).to.throw(OperationError);
                expect(getArr('bool1')).to.throw(OperationError);
                expect(getArr('num')).to.throw(OperationError);
                expect(getArr('nan')).to.throw(OperationError);
                expect(getArr('str')).to.throw(OperationError);
                expect(getArr('estr')).to.throw(OperationError);
                expect(getArr('obj')).to.throw(OperationError);
            });
        });

        describe('method "getOptArr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptArr');
            });
            function getOptArr(name, def) { return context.getOptArr.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptArr('arr', defArr)).to.equal(operation.arr);
                expect(context.getOptArr('earr', defArr)).to.equal(operation.earr);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptArr('wrong', defArr)).to.equal(defArr);
                expect(context.getOptArr('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptArr('bool0')).to.throw(OperationError);
                expect(getOptArr('bool1')).to.throw(OperationError);
                expect(getOptArr('num')).to.throw(OperationError);
                expect(getOptArr('nan')).to.throw(OperationError);
                expect(getOptArr('str')).to.throw(OperationError);
                expect(getOptArr('estr')).to.throw(OperationError);
                expect(getOptArr('obj')).to.throw(OperationError);
            });
        });

        describe('method "getPos"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getPos');
            });
            function getPos(name) { return context.getPos.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getPos('pos1')).to.equal(operation.pos1);
            });
            it('should throw on missing properties', function () {
                expect(getPos('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getPos('bool0')).to.throw(OperationError);
                expect(getPos('bool1')).to.throw(OperationError);
                expect(getPos('num')).to.throw(OperationError);
                expect(getPos('nan')).to.throw(OperationError);
                expect(getPos('str')).to.throw(OperationError);
                expect(getPos('estr')).to.throw(OperationError);
                expect(getPos('obj')).to.throw(OperationError);
                expect(getPos('arr')).to.throw(OperationError);
                expect(getPos('earr')).to.throw(OperationError);
                expect(getPos('pos2')).to.throw(OperationError);
                expect(getPos('pos3')).to.throw(OperationError);
            });
        });

        describe('method "getOptPos"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptPos');
            });
            function getOptPos(name) { return context.getOptPos.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getOptPos('pos1')).to.equal(operation.pos1);
            });
            it('should return null for missing properties', function () {
                expect(context.getOptPos('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptPos('bool0')).to.throw(OperationError);
                expect(getOptPos('bool1')).to.throw(OperationError);
                expect(getOptPos('num')).to.throw(OperationError);
                expect(getOptPos('nan')).to.throw(OperationError);
                expect(getOptPos('str')).to.throw(OperationError);
                expect(getOptPos('estr')).to.throw(OperationError);
                expect(getOptPos('obj')).to.throw(OperationError);
                expect(getOptPos('arr')).to.throw(OperationError);
                expect(getOptPos('earr')).to.throw(OperationError);
                expect(getOptPos('pos2')).to.throw(OperationError);
                expect(getOptPos('pos3')).to.throw(OperationError);
            });
        });
    });

    // ========================================================================
});
