/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

require('../lib/define')('intervalset/main', [
    'io.ox/office/tk/container/sortedarray',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/intervalset',
    'perf!intervalset/intervalset1',
    'perf!intervalset/intervalset2'
], function (SortedArray, Interval, IntervalArray, IntervalSet, IntervalSet1, IntervalSet2) {

    'use strict';

    const { LogTable } = require('../lib/utils');

    // total time for a single test (milliseconds)
    const TEST_TIME = 100;

    // size of the boundary interval
    const BOUNDARY_SIZE = 1024 * 1024;

    // maximum interval length for different test runs
    const MAX_LENGTHS = [1e1, 1e2, 1e3, 1e4, 1e5, 1e6];

    // number of intervals to insert into the set per test run
    const SET_SIZES = [1e1, 3e1, 1e2, 3e2, 1e3, 3e3, 1e4, 3e4];

    // column width in output tables
    const WIDTH = 8;

    // ========================================================================

    const ZEROS = '000000000';

    function formatl(index) {
        return (ZEROS + index).slice(-10);
    }

    function firstSorter(interval) {
        return formatl(interval.first) + formatl(interval.last);
    }

    function containsIndexInSortedL(array, index) {
        let found = false;
        array.values().some(interval => (index < interval.first) || (found = (index <= interval.last)));
        return found;
    }

    // ========================================================================

    function createInterval(min, max) {
        const rnd1 = Math.random();
        const len = min + Math.floor((1 - rnd1 * rnd1 * rnd1) * (max - min + 1));
        const rnd2 = Math.random();
        const first = Math.floor((rnd2 * rnd2) * (BOUNDARY_SIZE - len));
        return new Interval(first, first + len - 1);
    }

    function createData(n, l) {
        return _.times(n, () => createInterval(5, l));
    }

    function createArray(data) {
        const arr = new IntervalArray();
        data.forEach(int => arr.push(int));
        return arr;
    }

    function createSorted(data) {
        const arr = new SortedArray(firstSorter);
        data.forEach(int => arr.insert(int));
        return arr;
    }

    function createSet(data) {
        const set = new IntervalSet(new Interval(0, BOUNDARY_SIZE - 1));
        data.forEach(int => set.insert(int));
        return set;
    }

    function createSet1(data) {
        const set = new IntervalSet1(new Interval(0, BOUNDARY_SIZE - 1));
        data.forEach(int => set.insert(int));
        return set;
    }

    function createSet2(data) {
        const set = new IntervalSet2(new Interval(0, BOUNDARY_SIZE - 1));
        data.forEach(int => set.insert(int));
        return set;
    }

    function randIndex() {
        const rnd = Math.random();
        return Math.floor(rnd * rnd * BOUNDARY_SIZE);
    }

    function containsIndex(cont) {
        return cont.containsIndex(randIndex());
    }

    function containsSorted(arr) {
        return containsIndexInSortedL(arr, randIndex());
    }

    function runTest(func, ...args) {
        const t0 = Date.now();
        let dur = 0, cycles = 0;
        while (dur < TEST_TIME) {
            func(...args);
            dur = Date.now() - t0;
            cycles += 1;
        }
        return Math.round(dur / cycles * 1e6);
    }

    const TEST_RUNS = [
        { title: 'IntervalArray', create: createArray,  contains: containsIndex },
        { title: 'SortedArray',   create: createSorted, contains: containsSorted },
        { title: 'IntervalSet',   create: createSet,    contains: containsIndex },
        { title: 'IntervalSet1',  create: createSet1,   contains: containsIndex },
        { title: 'IntervalSet2',  create: createSet2,   contains: containsIndex }
    ];

    console.log('');
    const table = new LogTable().col(8).col(6, true);
    const heads1 = ['', ''];
    const heads2 = ['l', 'n'];
    TEST_RUNS.forEach(e => {
        table.col(WIDTH).col(WIDTH, true);
        heads1.push({ msg: e.title, span: 2 });
        heads2.push('insert', 'find');
    });
    table.head(...heads1).head(...heads2);
    MAX_LENGTHS.forEach(l => {
        table.sep();
        SET_SIZES.forEach(n => {
            const data = createData(n, l);
            const containers = TEST_RUNS.map(e => e.create(data));
            const results = [l, n];
            TEST_RUNS.forEach((e, i) => {
                results.push(Math.round(runTest(e.create, data) / n));
                results.push(runTest(e.contains, containers[i]));
            });
            table.row(...results);
        });
    });
});
