/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

require('../lib/define')('cellmatrix/main', [
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/model/cellmodel',
    'perf!cellmatrix/cellmatrixold',
    'perf!cellmatrix/cellmatrix',
    'perf!cellmatrix/cellmatrix32'
], async (ValueMap, Address, CellModel, CellMatrixOld, CellMatrix, CellMatrix32) => {

    'use strict';

    const { pause, getTime, getDuration, LogTable } = require('../lib/utils');

    // number of cells to insert into the matrix per test run
    const CELL_COUNT = [1e2, 2e2, 5e2, 1e3, 2e3, 5e3, 1e4, 2e4, 5e4, 1e5].reverse();

    // matrix density per test run
    const MATRIX_DENSITY = [0.5, 0.1, 0.01, 0.001];

    // intervals for iterators
    const ITER_INTERVALS = [{ first: 0, last: 1e6 }, { first: 2, last: 1e6 }, { first: 1, last: 9 }];

    // SheetModel mock ========================================================

    const collection = {
        on() {},
        isEntryVisible() { return true; }
    };

    const sheetModel = {
        getColCollection() { return collection; },
        getRowCollection() { return collection; }
    };

    // ========================================================================

    function createModels(count, density) {
        const models = new ValueMap();
        const area = Math.round(count / density);
        const cols = Math.round(Math.sqrt(area / 5));
        const rows = Math.round(area / cols);
        for (let i = 0; i < count; i += 1) {
            while (true) {
                const rnd1 = Math.random();
                const rnd2 = Math.random();
                const a = new Address(Math.floor(rnd1 * rnd1 * cols), Math.floor(rnd2 * rnd2 * rnd2 * rows));
                const key = a.key();
                if (!models.has(key)) {
                    models.insert(key, new CellModel(a));
                    break;
                }
            }
        }
        return models;
    }

    function insertModels(matrix, models, count) {
        models.forEach(matrix.insertModel, matrix);
        return count;
    }

    function forEachModels(matrix) {
        let n = 0;
        ITER_INTERVALS.forEach(interval => {
            matrix.forEachModel(interval, interval, () => { n += 1; });
        });
        return n;
    }

    function iterateModels(matrix) {
        let n = 0;
        ITER_INTERVALS.forEach(interval => {
            const iterator = matrix.createModelIterator(interval, interval);
            while (!iterator.next().done) { n += 1; }
        });
        return n;
    }

    function forEachMap(models) {
        let count = 0;
        models.forEach(() => { count += 1; });
        return count;
    }

    function iterateMap(models) {
        let count = 0;
        const iterator = models.iterator();
        while (!iterator.next().done) { count += 1; }
        return count;
    }

    function removeModels(matrix, models, count) {
        models.forEach(matrix.removeModel, matrix);
        return count;
    }

    function takeTime(func, ...args) {
        const t0 = getTime();
        const n = func(...args);
        return Math.round(getDuration(t0) / n);
    }

    const TEST_SPECS = [
        { title: 'CellModelMatrix(old)',    create: sheetModel => new CellMatrixOld(sheetModel, true), foreach: false },
        { title: 'CellMatrix (Array)',      create: sheetModel => new CellMatrix(sheetModel, 'v', true), foreach: true },
        { title: 'CellMatrix (Int32Array)', create: (sheetModel, models) => new CellMatrix32(sheetModel, models, true), foreach: true }
    ];

    const table = new LogTable().col(6).col(6, true).col(7).col(7, true);
    const heads1 = ['', '', { msg: 'ValueMap', span: 2 }];
    const heads2 = ['', '', 'forEach', 'iterate'];
    const heads3 = ['count', 'dens%', 'ns/cell', 'ns/cell'];
    TEST_SPECS.forEach(test => {
        if (test.foreach) { table.col(7); heads3.push('ns/cell'); }
        table.col(7).col(7).col(7, true);
        heads1.push({ msg: test.title, span: test.foreach ? 4 : 3 });
        heads2.push('insert');
        if (test.foreach) { heads2.push('forEach'); }
        heads2.push('iterate', 'remove');
        heads3.push('ns/cell', 'ns/cell', 'ns/cell');
    });
    table.head(...heads1).head(...heads2).head(...heads3).sep();

    async function runTest(count, density) {

        const results = [];
        const models = createModels(count, density);

        for (const test of TEST_SPECS) {
            const matrix = test.create(sheetModel, models);
            results.push(takeTime(insertModels, matrix, models, count));
            if (test.foreach) { results.push(takeTime(forEachModels, matrix, count)); }
            results.push(takeTime(iterateModels, matrix, count));
            results.push(takeTime(removeModels, matrix, models, count));
            await pause(100);
        }

        return results;
    }

    const short = process.argv.slice(2).includes('--short');

    async function runTests(count) {

        let allResults = null;

        for (const density of MATRIX_DENSITY) {
            const results = await runTest(count, density);
            if (!short) { table.row(count, density * 100, '', '', ...results); }
            if (allResults) {
                allResults = allResults.map((t, i) => t + results[i]);
            } else {
                allResults = results.slice();
            }
        }

        const models = createModels(count, 0.1);
        const t1 = takeTime(forEachMap, models);
        const t2 = takeTime(iterateMap, models);
        table.row(count, 'avg', t1, t2, ...allResults.map(t => Math.round(t / MATRIX_DENSITY.length)));

        if (!short) { table.sep(); }
    }

    // dry-run for memory initialization
    await runTest(1000, 0.1);
    await pause(500);

    // real test runs
    for (const count of CELL_COUNT) {
        await runTests(count);
    }
});
