/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('cellmatrix/indexarray', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/extarray',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Iterator, ExtArray, SheetUtils) {

    'use strict';

    // constants ==============================================================

    // convenience shortcuts
    var floor = Math.floor;
    var SingleIterator = Iterator.SingleIterator;
    var NestedIterator = Iterator.NestedIterator;
    var IntervalArray = SheetUtils.IntervalArray;

    // class IndexArray =======================================================

    /**
     * An instance of this class represents a sorted sparse array of elements
     * of type T having a column/row index.
     *
     * @constructor
     *
     * @extends ExtArray
     *
     * @param {T => number} resolver
     *  Column/row index resolver callback function used for binary search in
     *  this array. Receives an array element, and must return the column/row
     *  index of that element. Will be called in the context of this instance.
     */
    var IndexArray/*<T>*/ = ExtArray.extend(function (resolver) {

        // base constructor
        ExtArray.call(this);

        // resolver callback function, bound to this instance
        this._resolveIndex = resolver;
    });

    // public methods ---------------------------------------------------------

    /**
     * Returns an existing array element with the passed column/row index.
     *
     * @param {number} index
     *  The column/row index of the array element to be returned.
     *
     * @returns {T|null}
     *  The existing array element; or null, if no array element has been
     *  found.
     */
    IndexArray.prototype.get = function (index) {

        // fast binary search for the element at "index"
        var ai = this._findArrayIndex(index, 0, true);

        // return found array element, or null
        return (ai >= 0) ? this[ai] : null;
    };

    /**
     * Creates an iterator that visits all elements contained in this array
     * whose column/row index is covered by the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {boolean} [reverse=false]
     *  If set to true, the passed intervals will be processed in reversed
     *  order, as well as the elements of this array in each interval.
     *
     * @returns {Iterator<T>}
     *  The new iterator. The result values will be array elements.
     */
    IndexArray.prototype.iteratorIn = function (intervals, reverse) {

        // create an array iterator that visits all index intervals
        var iterator = (intervals instanceof Array) ? intervals.iterator({ reverse: reverse }) : new SingleIterator({ value: intervals });

        // create an iterator that visits the indexes in the passed intervals
        return new NestedIterator(iterator, function (interval) {
            return this.iterator(this._getLoopLimits(interval, reverse));
        }, null, this);
    };

    /**
     * Invokes the specified callback function for all elements contained in
     * this array that are covered by the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {T => void} callback
     *  The callback function to be invoked for each array element. Receives
     *  the array element as first parameter. The return value will be ignored.
     *
     * @param {boolean} [reverse=false]
     *  If set to true, the passed intervals will be processed in reversed
     *  order, as well as the array elements in each interval.
     */
    IndexArray.prototype.forEachIn = function (intervals, callback, reverse) {

        // visit each interval in the specified order
        var arrayForEach = reverse ? IntervalArray.forEachReverse : IntervalArray.forEach;
        arrayForEach(intervals, function (interval) {

            // visit the cell models in the current interval
            var limits = this._getLoopLimits(interval, reverse);
            for (var ai = limits.begin, end = limits.end, step = limits.step; ai !== end; ai += step) {
                callback(this[ai]);
            }
        }, this);
    };

    /**
     * Invokes the specified predicate callback function for all elements
     * contained in this array that are covered by the passed index intervals,
     * until it returns a truthy value.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {T => any} predicate
     *  The predicate callback function to be invoked for each array element.
     *  Receives the array element as first parameter. If this function returns
     *  a truthy value, the loop will stop immediately.
     *
     * @param {boolean} [reverse=false]
     *  If set to true, the passed intervals will be processed in reversed
     *  order, as well as the array elements in each interval.
     *
     * @returns {boolean}
     *  Whether the predicate callback function has returned a truthy value for
     *  an array element.
     */
    IndexArray.prototype.someIn = function (intervals, predicate, reverse) {

        // visit each interval in the specified order
        var arraySome = reverse ? IntervalArray.someReverse : IntervalArray.some;
        return arraySome(intervals, function (interval) {

            // visit the cell models in the current interval
            var limits = this._getLoopLimits(interval, reverse);
            for (var ai = limits.begin, end = limits.end, step = limits.step; ai !== end; ai += step) {
                if (predicate(this[ai])) { return true; }
            }

            return false;
        }, this);
    };

    /**
     * Invokes the specified predicate callback function for all elements
     * contained in this array that are covered by the passed index intervals,
     * until it returns a falsy value.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {T => any} predicate
     *  The predicate callback function to be invoked for each array element.
     *  Receives the array element as first parameter. If this function returns
     *  a falsy value, the loop will stop immediately.
     *
     * @param {boolean} [reverse=false]
     *  If set to true, the passed intervals will be processed in reversed
     *  order, as well as the array elements in each interval.
     *
     * @returns {boolean}
     *  Whether the predicate callback function has returned a truthy value for
     *  every array element.
     */
    IndexArray.prototype.everyIn = function (intervals, predicate, reverse) {

        // visit each interval in the specified order
        var arrayEvery = reverse ? IntervalArray.everyReverse : IntervalArray.every;
        return arrayEvery(intervals, function (interval) {

            // visit the cell models in the current interval
            var limits = this._getLoopLimits(interval, reverse);
            for (var ai = limits.begin, end = limits.end, step = limits.step; ai !== end; ai += step) {
                if (!predicate(this[ai])) { return false; }
            }

            return true;
        }, this);
    };

    /**
     * Inserts the passed value at its sorted position into this array. It will
     * not be checked whether this array already contains an element with the
     * same column/row index as the passed value (performance). Callers have to
     * ensure uniqueness of the indexes by themselves.
     *
     * @param {T} value
     *  The value to be inserted into this vector.
     */
    IndexArray.prototype.insert = function (value) {

        // fast binary search for the first element at or after "index"
        var index = this._resolveIndex(value);
        var ai = this._findArrayIndex(index);

        // replace or insert the array element
        if (ai === this.length) {
            this.push(value);
        } else {
            this.splice(ai, 0, value);
        }
    };

    /**
     * Removes an existing value from this array. This method will remove the
     * array element whose resolved column/row index is equal to the column/row
     * index of the passed value.
     *
     * @param {T} value
     *  The value to be removed from this array.
     */
    IndexArray.prototype.remove = function (value) {

        // fast binary search for the element at "index"
        var index = this._resolveIndex(value);
        var ai = this._findArrayIndex(index, 0, true);

        // remove the value from the array
        if (ai >= 0) { this.splice(ai, 1); }
    };

    /**
     * Returns an existing array element with the passed column/row index, or
     * calls the generator callback function to create a new element that will
     * be inserted into the array.
     *
     * @param {number} index
     *  The column/row index of the array element to be returned.
     *
     * @param {number => T} generator
     *  The generator callback function that will be called if this array does
     *  not contain an element with the specified column/row index, and that
     *  must create and return the new array element.
     *
     * @param {object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {T}
     *  The already existing, or the created and inserted array element.
     */
    IndexArray.prototype.getOrCreate = function (index, generator, context) {

        // fast binary search for the first element at or after "index"
        var ai = this._findArrayIndex(index);

        // return existing array element
        var length = this.length;
        if ((ai < length) && (this._resolveIndex(this[ai]) === index)) { return this[ai]; }

        // create and insert new array element
        var value = generator.call(context, index);
        if (ai === length) {
            this.push(value);
        } else {
            this.splice(ai, 0, value);
        }
        return value;
    };

    // private methods --------------------------------------------------------

    /**
     * Fast binary search for the first array element at or after the passed
     * column/row index.
     *
     * @param {number} index
     *  The column/row index of the element to be found.
     *
     * @param {number} [start=0]
     *  The array index of the element where the binary search should start. By
     *  default, the entire array will be searched.
     *
     * @param {boolean} [exact=false]
     *  If set to true, the element at the found array index must match exactly
     *  with the passed column/row index.
     *
     * @returns {number}
     *  The array index of the element equal to or greater than the passed
     *  column/row index.
     */
    IndexArray.prototype._findArrayIndex = function (index, start, exact) {
        var ai0 = start || 0, ai1 = this.length;
        if (ai0 < ai1) {
            var index0 = this._resolveIndex(this[ai0]);
            if (exact ? (index === index0) : (index <= index0)) { return ai0; }
        }
        while (ai0 < ai1) {
            var ai = floor((ai0 + ai1) / 2);
            var cmp = this._resolveIndex(this[ai]) - index;
            if (cmp > 0) { ai1 = ai; } else if (cmp < 0) { ai0 = ai + 1; } else { return ai; }
        }
        return exact ? -1 : ai0;
    };

    /**
     * Returns the settings for array iterators and for-loops needed to visit
     * the array elements in the passed index interval.
     *
     * @param {Interval} interval
     *  The index interval to be iterated.
     *
     * @param {boolean} [reverse=false]
     *  Whether to iterate backwards.
     *
     * @returns {object}
     *  A result object compatible with the options of class ArrayIterator,
     *  with the following properties:
     *  - {number} begin
     *      Array index of the first element to be visited.
     *  - {number} end
     *      Array index of the element after the last element to be visited.
     *  - {number} step
     *      Step width for usage in for-loops (1 or -1 according to direction).
     *  - {boolean} reverse
     *      The direction flag passed in the "reverse" parameter.
     */
    IndexArray.prototype._getLoopLimits = function (interval, reverse) {

        // fast binary search for the first and last array elements
        var ai1 = this._findArrayIndex(interval.first);
        var ai2 = this._findArrayIndex(interval.last + 1, ai1);

        // adjust array indexes for reverse mode
        return {
            begin: reverse ? (ai2 - 1) : ai1,
            end: reverse ? (ai1 - 1) : ai2,
            step: reverse ? -1 : 1,
            reverse: !!reverse
        };
    };

    // exports ================================================================

    return IndexArray;

});
