/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('cellmatrix/cellmatrixold', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/sortedarray',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/subtotalresult',
    'io.ox/office/spreadsheet/model/cellmodel'
], function (Iterator, SortedArray, BaseObject, SheetUtils, SubtotalResult, CellModel) {

    'use strict';

    // convenience shortcuts
    var TransformIterator = Iterator.TransformIterator;
    var NestedIterator = Iterator.NestedIterator;
    var Interval = SheetUtils.Interval;
    var IntervalArray = SheetUtils.IntervalArray;

    // class CellModelVector ==================================================

    /**
     * An instance of this class represents a sorted sparse array of cells in a
     * single column or row of the cell collection.
     *
     * @constructor
     *
     * @extends SortedArray
     *
     * @param {Boolean} columns
     *  Whether this cell vector is a column vector (true; cells will be sorted
     *  by their row index), or a row vector (false; cells will be sorted by
     *  their column index).
     */
    var CellModelVector = SortedArray.extend(function (collection, columns) {

        // private properties -------------------------------------------------

        // the column/row collection in oppsite direction needed to skip hidden columns/rows
        this._collection = collection;
        // cached subtotal results for all cells in this vector
        this._subtotals = null;
        // name of the property in a cell model referring to this instance
        this._vectorProp = columns ? 'cv' : 'rv';

        // base constructor ---------------------------------------------------

        SortedArray.call(this, columns ? CellModel.row : CellModel.col);

    }); // class CellModelVector

    // public methods ---------------------------------------------------------

    /**
     * Inserts a new cell model into this vector, and updates the cached
     * subtotals result if available.
     *
     * @param {CellModel} cellModel
     *  The cell model to be inserted into this vector.
     *
     * @returns {CellModelVector}
     *  A reference to this instance.
     */
    CellModelVector.prototype.insert = function (cellModel) {
        cellModel[this._vectorProp] = this;
        if (this._subtotals) { this._subtotals.addValue(cellModel.v); }
        return SortedArray.prototype.insert.call(this, cellModel);
    };

    /**
     * Removes a cell model from this vector, and invalidates the cached
     * subtotals result.
     *
     * @param {CellModel} cellModel
     *  The cell model to be removed from this vector.
     *
     * @returns {CellModelVector}
     *  A reference to this instance.
     */
    CellModelVector.prototype.remove = function (cellModel) {
        cellModel[this._vectorProp] = null;
        // removing a non-blank cell invalidates the cached subtotals
        if (!cellModel.isBlank()) { this._subtotals = null; }
        return SortedArray.prototype.remove.call(this, cellModel);
    };

    /**
     * Returns the subtotals result of all cells contained in this cell vector.
     * Once calculated, the subtotals result will be cached for subsequent
     * invocations.
     *
     * @returns {SubtotalResult}
     *  The subtotals result of all cells contained in this cell vector.
     */
    CellModelVector.prototype.getSubtotals = function () {

        // return the cached subtotals directly
        if (this._subtotals) { return this._subtotals; }

        // simultaneously iterate through the column/row collection to find hidden columns/rows efficiently
        var collection = this._collection;
        var iterator = collection.createIterator(collection.getFullInterval(), { visible: true, unique: true });
        var iterResult = iterator.next();

        // collect the subtotals directly in synchronous mode
        var subtotals = this._subtotals = new SubtotalResult();
        this.forEach(function (cellModel, index) {

            // find the next visible column/row interval that does not end before the current cell
            while (!iterResult.done && (iterResult.value.uniqueInterval.last < index)) {
                iterResult = iterator.next();
            }

            // skip the cell model, if it is located in a hidden column/row
            // (i.e., next visible column/row interval starts after the current cell model)
            if (!iterResult.done && (iterResult.value.uniqueInterval.first <= index)) {
                subtotals.addValue(cellModel.v);
            }
        });
        return subtotals;
    };

    /**
     * Invalidates the cached subtotals result of this cell vector.
     *
     * @param {Interval} [changedInterval]
     *  If specified, the cached subtotals will be invalidated only if this
     *  vector contains at least one entry in that index interval.
     *
     * @returns {CellModelVector}
     *  A reference to this instance.
     */
    CellModelVector.prototype.clearSubtotals = function (changedInterval) {
        var dirty = this._subtotals;
        if (dirty && changedInterval) {
            var desc = this.find(changedInterval.first);
            dirty = desc && (desc.index <= changedInterval.last);
        }
        if (dirty) { this._subtotals = null; }
        return this;
    };

    // class CellModelMatrix ==================================================

    /**
     * An instance of this class represents a sorted sparse matrix of cells in
     * the cell collection for a specific orientation (cells in column vectors,
     * or in row vectors). The double-sorted data structure allows fast binary
     * lookup, and fast iterator implementations for existing cells in large
     * cell ranges.
     *
     * Example: An instance of this class is used as row matrix. In this case,
     * it will contain several row vectors (instances of the class SortedArray)
     * that contain the cell models of all defined (existing) cells in the
     * respective rows.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {Boolean} columns
     *  Whether this instance contains column vectors (true; vectors will be
     *  sorted by column index), or row vectors (false; vectors will be sorted
     *  by row index).
     */
    var CellModelMatrix = BaseObject.extend({ constructor: function (sheetModel, columns) {

        // sorter callback function for indexes of cell vectors
        var sorter = columns ? CellModel.col : CellModel.row;

        // the sorted array of sorted cell vectors
        var matrix = new SortedArray(function (vector) { return sorter(vector.first()); });

        // the column/row collection in matrix direction needed to skip hidden columns/rows
        var collection1 = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection();

        // the column/row collection in oppsite direction needed to skip hidden columns/rows
        var collection2 = columns ? sheetModel.getRowCollection() : sheetModel.getColCollection();

        // base constructor ---------------------------------------------------

        BaseObject.call(this, sheetModel);

        // private methods ----------------------------------------------------

        /**
         * Creates a new instance of a cell model vector.
         *
         * @returns {CellModelVector}
         *  A new instance of a cell model vector.
         */
        function createVector() {
            return new CellModelVector(collection2, columns);
        }

        /**
         * Creates an iterator that visits all existing cell vectors inside the
         * specified index intervals contained in this matrix.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Boolean} [reverse=false]
         *  If set to true, the passed intervals will be processed in reversed
         *  order, as well as the cell vectors in each interval.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the current cell
         *  vector (class CellModelVector) as value.
         */
        function createVectorIterator(intervals, reverse) {

            // create an array iterator that visits all intervals
            var iterator = IntervalArray.iterator(intervals, { reverse: reverse });

            // create an iterator that visits the single vectors in the passed intervals
            // (the values of the resulting iterator are the vectors)
            return new NestedIterator(iterator, function (interval) {
                return matrix.intervalIterator(interval.first, interval.last, reverse);
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the smallest interval containing all cell vectors in this
         * cell matrix.
         *
         * @returns {Interval}
         *  The smallest interval containing all cell vectors in this cell
         *  matrix; or null, if tis matrix is empty.
         */
        this.getUsedInterval = function () {
            return matrix.empty() ? null : new Interval(matrix.index(matrix.first()), matrix.index(matrix.last()));
        };

        /**
         * Inserts a new cell model into this matrix.
         *
         * @param {CellModel} cellModel
         *  The new cell model to be inserted into this matrix. This matrix
         *  DOES NOT take ownership of the cell models; and it DOES NOT check
         *  whether it already contains a cell model with the same address as
         *  the passed cell model (it is up to the caller to ensure uniqueness
         *  of the cell models).
         *
         * @returns {CellModelMatrix}
         *  A reference to this instance.
         */
        this.insertModel = function (cellModel) {
            matrix.getOrCreate(sorter(cellModel), createVector).insert(cellModel);
            return this;
        };

        /**
         * Removes a cell model from this matrix.
         *
         * @param {CellModel} cellModel
         *  The cell model to be removed from this matrix. If this matrix does
         *  not contain a cell model at the address of the passed cell model,
         *  nothing will happen.
         *
         * @returns {CellModelMatrix}
         *  A reference to this instance.
         */
        this.removeModel = function (cellModel) {
            var desc = matrix.find(sorter(cellModel), 'exact');
            if (!desc) { return this; }
            var vector = desc.value;
            vector.remove(cellModel);
            if (vector.empty()) { desc.remove(); }
            return this;
        };

        /**
         * Creates an index iterator that generates all indexes of all cell
         * vectors inside the specified intervals contained in this matrix.
         *
         * @param {IntervalArray|Interval} vectorIntervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Boolean} [reverse=false]
         *  If set to true, the passed intervals will be processed in reversed
         *  order, as well as the indexes of the cell vectors in each interval.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the index of the
         *  current cell vector as value.
         */
        this.createIndexIterator = function (vectorIntervals, reverse) {

            // create an iterator that visits the single vectors in the passed intervals
            var iterator = createVectorIterator(vectorIntervals, reverse);

            // transform the iterator result to column/row indexes
            return new TransformIterator(iterator, function (vector, result) {
                return { value: result.index };
            });
        };

        /**
         * Creates an iterator that visits all existing cell models in this
         * matrix covered by the ranges formed by the passed index intervals.
         * The cell models will be visited vector-by-vector.
         *
         * @param {IntervalArray|Interval} vectorIntervals
         *  An array of index intervals, or a single index interval, specifying
         *  all cell model vectors to be visited.
         *
         * @param {IntervalArray|Interval} entryIntervals
         *  An array of index intervals, or a single index interval, specifying
         *  all cells to be visited in each cell model vector.
         *
         * @param {Boolean} [reverse=false]
         *  If set to true, the passed intervals will be processed in reversed
         *  order.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the current cell
         *  model (class CellModel) as value.
         */
        this.createModelIterator = function (vectorIntervals, entryIntervals, reverse) {

            // create an iterator that visits the single vectors in the passed intervals
            var iterator = createVectorIterator(vectorIntervals, reverse);

            // create an iterator that visits the inner intervals for a single vector,
            // and forwards the information about the current vector to the result
            iterator = new NestedIterator(iterator, function () {
                return IntervalArray.iterator(entryIntervals, { reverse: reverse });
            }, function (outerResult, innerResult) {
                return { value: innerResult.value, vector: outerResult.value };
            });

            // create an iterator that visits the cell models in a single interval in a vector
            return new NestedIterator(iterator, function (interval, result) {
                return result.vector.intervalIterator(interval.first, interval.last, reverse);
            });
        };

        /**
         * Creates an iterator that visits the subtotal results of all cell
         * model vectors covered by the specified index intervals.
         *
         * @param {IntervalArray|Interval} vectorIntervals
         *  An array of index intervals, or a single index interval, specifying
         *  all cell model vectors to be visited.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the current
         *  subtotal result (class SubtotalResult) as value.
         */
        this.createSubtotalIterator = function (vectorIntervals) {
            var visibleIntervals = IntervalArray.map(vectorIntervals, collection1.getVisibleIntervals, collection1);
            var iterator = createVectorIterator(visibleIntervals);
            return new TransformIterator(iterator, function (vector, result) {
                result.value = vector.getSubtotals();
                return result;
            });
        };

        // initialization -----------------------------------------------------

        // invalidate cached subtotal results if columns/rows will be hidden or shown
        this.listenTo(collection2, 'change:entries', function (event, interval, changeInfo) {
            if (changeInfo.visibilityChanged) {
                matrix.forEach(function (vector) {
                    vector.clearSubtotals(interval);
                });
            }
        });

        this.registerDestructor(function () {
            sorter = matrix = null;
        });

    } }); // class CellModelMatrix

    // exports ================================================================

    return CellModelMatrix;

});
