/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

(function () {

    const { pause, getTime, getDuration, LogTable } = require('../lib/utils');

    const max = Math.max;
    const ceil = Math.ceil;

    class SpliceArray {
        constructor(n) {
            this.a = new Array(n).fill(0);
        }
        insert(i, v) {
            this.a.splice(i, 0, v);
        }
        remove(i) {
            this.a.splice(i, 1);
        }
    }

    function mixExt(BaseClass) {
        return class extends BaseClass {
            _expand() {
                if (this.a.length === this.l) { this._set(); }
            }
            _shrink() {
                if ((this.a.length >= 30) && (this.l * 3 < this.a.length)) { this._set(); }
            }
        };
    }

    function mixCopy(BaseClass) {
        return class extends BaseClass {
            insert(i, v) {
                this._expand();
                this.a.copyWithin(i + 1, i, this.l);
                this.a[i] = v;
                this.l += 1;
            }
            remove(i) {
                this.a.copyWithin(i, i + 1, this.l);
                this.l -= 1;
                this._shrink();
            }
        };
    }

    function mixLoop(BaseClass) {
        return class extends BaseClass {
            insert(i, v) {
                this._expand();
                const a = this.a;
                for (let j = this.l; j > i; j -= 1) { a[j] = a[j - 1]; }
                a[i] = v;
                this.l += 1;
            }
            remove(i) {
                const a = this.a;
                for (let j = i + 1, l = this.l; j < l; j += 1) { a[j - 1] = a[j]; }
                this.l -= 1;
                this._shrink();
            }
        };
    }

    const ExtArray = mixExt(class {
        constructor(n) {
            this.a = new Array(n).fill(0);
            this.l = n;
        }
        _set() {
            this.a.length = ceil(this.l * 1.5);
        }
    });

    const CopyArray = mixCopy(ExtArray);
    const LoopArray = mixLoop(ExtArray);

    const ExtArray32 = mixExt(class {
        constructor(n) {
            this.a = new Int32Array(max(n, 20));
            this.l = n;
        }
        _set() {
            const a = this.a;
            this.a = new Int32Array(ceil(this.l * 1.5));
            this.a.set(a.subarray(0, this.l));
        }
    });

    const CopyArray32 = mixCopy(ExtArray32);
    const LoopArray32 = mixLoop(ExtArray32);

    const TEST_SPECS = [
        { title: 'Array (splice)',  Class: SpliceArray },
        { title: 'Array (copy)',    Class: CopyArray },
        { title: 'Array (loop)',    Class: LoopArray },
        { title: 'I32Array (copy)', Class: CopyArray32 },
        { title: 'I32Array (loop)', Class: LoopArray32 }
    ];

    // array size per test run
    const ARRAY_SIZES = [1e3, 2e3, 5e3, 1e4, 2e4, 5e4, 1e5, 2e5, 5e5, 1e6, 2e6, 5e6, 1e7];

    function takeTime(array, method) {
        let d = 0, n = 0, c = 1;
        const t0 = getTime();
        while (d < 5e7) {
            for (let i = 0; i < c; i += 1) { array[method](42); }
            d = getDuration(t0);
            n += c;
            c += 1;
        }
        return Math.round(d / n / 1000);
    }

    async function runTests(n) {
        const results = [n];
        for (let j = 0; j < TEST_SPECS.length; j += 1) {
            const test = TEST_SPECS[j];
            const array = new test.Class(n);
            results.push(takeTime(array, 'insert'));
            results.push(takeTime(array, 'remove'));
            await pause(50);
        }
        return results;
    }

    exports.run = async function () {

        console.log(' ');
        console.log('All results in microseconds per array operation.');
        console.log(' ');
        const table = new LogTable().col(8, true);
        const heads1 = [''];
        const heads2 = ['n'];
        TEST_SPECS.forEach(e => {
            table.col(8).col(8, true);
            heads1.push({ msg: e.title, span: 2 });
            heads2.push('insert', 'remove');
        });
        table.head(...heads1).head(...heads2).sep();

        // dry-run for memory initialization
        await runTests(ARRAY_SIZES[0]);

        // real test runs
        for (const n of ARRAY_SIZES) {
            table.row(...await runTests(n));
        }
        console.log('Done.');
    };
}());
