/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/forms', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'less!io.ox/office/tk/exports',
    'less!io.ox/office/tk/style'
], function (Config, Utils, KeyCodes, LessExports) {

    'use strict';

    // CSS classes added to localized bitmap icons
    var DOCS_ICON_CLASSES = 'lc-' + Config.LOCALE + ' lc-' + Config.LANGUAGE + (Utils.RETINA ? ' retina' : '');

    // CSS class name for caption elements
    var CAPTION_CLASS = 'caption';

    // CSS selector for caption elements in control nodes
    var CAPTION_SELECTOR = '>.' + CAPTION_CLASS;

    // CSS selector for icon elements in control captions
    var ICON_SELECTOR = '>i';

    // CSS selector for text spans in control captions
    var SPAN_SELECTOR = '>span';

    // CSS class for buttons with ambiguous value
    var AMBIGUOUS_CLASS = 'ambiguous';

    // CSS class name for check marks in check boxes
    var CHECK_MARK_CLASS = 'check-mark';

    // attribute name for selected/checked buttons
    var DATA_CHECKED_ATTR = 'data-checked';

    // CSS selector for selected buttons
    var DATA_CHECKED_SELECTOR = '[' + DATA_CHECKED_ATTR + '="true"]';

    // icon sets for checked/unchecked icons (arrays with 'false', 'true', and 'ambiguous' icons)
    var CHECK_ICON_SETS = {
        check: ['fa-none', 'fa-check', 'fa-minus'],
        boxed: ['not-checked', 'checked', 'ambiguous'],
        radio: ['not-checked', 'checked', 'not-checked']
    };

    // private global functions ===============================================

    /**
     * Converts the passed camel-case name to a lower-case name with dashes as
     * separators.
     */
    function convertName(name) {
        return name.replace(/([A-Z])/g, '-$1').toLowerCase();
    }

    /**
     * Converts the passed string or object to the mark-up of a 'style' element
     * attribute.
     *
     * @param {String|Object} [style]
     *  The value for the 'style' element attribute. Can be a string that will
     *  be inserted into the 'style' attribute as is, or can be an object that
     *  maps CSS properties to their values. The property names can be given in
     *  camel-case notation. If omitted, this method returns an empty string.
     *
     * @returns {String}
     *  The string ' style="[value]"' (with leading space character), where
     *  [value] is the value passed to this method; or an empty string, if no
     *  valid value has been passed.
     */
    function createStyleMarkup(style) {

        var styleMarkup = '';

        if (_.isObject(style)) {
            _.each(style, function (value, name) {
                styleMarkup += convertName(name) + ':' + value + ';';
            });
        } else if (_.isString(style)) {
            styleMarkup += style;
        }

        if (styleMarkup.length > 0) {
            styleMarkup = ' style="' + styleMarkup + '"';
        }

        return styleMarkup;
    }

    /**
     * Adds generic attributes for an <input> element or a <textarea> element
     * to the passed set of DOM element attributes.
     *
     * @param {Object} attributes
     *  A map with DOM element attributes to be extended.
     *
     * @param {Object} [options]
     *  Optional parameters passed to the methods Forms.createInputMarkup() or
     *  Forms.createTextAreaMarkup().
     */
    function setGenericInputAttributes(attributes, options) {

        var placeHolder = Utils.getStringOption(options, 'placeholder', null);
        if (placeHolder) { attributes.placeholder = placeHolder; }

        var maxLength = Utils.getIntegerOption(options, 'maxLength', 0);
        if (maxLength > 0) { attributes.maxlength = maxLength; }

        if (Utils.IOS && !Utils.getBooleanOption(options, 'autoCorrection', false)) {
            attributes.autocorrect = 'off';
            attributes.autocapitalize = 'none';
        }
    }

    // static class Forms =====================================================

    // the exported Forms class
    var Forms = {};

    // constants --------------------------------------------------------------

    /**
     * CSS class for hidden DOM elements.
     */
    Forms.HIDDEN_CLASS = 'hidden';

    /**
     * CSS selector for hidden DOM elements.
     */
    Forms.HIDDEN_SELECTOR = '.' + Forms.HIDDEN_CLASS;

    /**
     * CSS selector for visible DOM elements.
     */
    Forms.VISIBLE_SELECTOR = ':not(' + Forms.HIDDEN_SELECTOR + ')';

    /**
     * CSS class for disabled DOM elements.
     *
     * @constant
     */
    Forms.DISABLED_CLASS = 'disabled';

    /**
     * CSS selector for disabled DOM elements.
     *
     * @constant
     */
    Forms.DISABLED_SELECTOR = '.' + Forms.DISABLED_CLASS;

    /**
     * CSS selector for enabled DOM elements.
     *
     * @constant
     */
    Forms.ENABLED_SELECTOR = ':not(' + Forms.DISABLED_SELECTOR + ')';

    /**
     * CSS class for selected (active) DOM elements.
     *
     * @constant
     */
    Forms.SELECTED_CLASS = 'selected';

    /**
     * CSS selector for selected (active) DOM elements.
     *
     * @constant
     */
    Forms.SELECTED_SELECTOR = '.' + Forms.SELECTED_CLASS;

    /**
     * CSS class for DOM container elements that contain the browser focus.
     *
     * @constant
     */
    Forms.FOCUSED_CLASS = 'focused';

    /**
     * CSS selector for DOM elements that can receive the keyboard focus. By
     * convention, all focusable DOM elements MUST have a non-negative value in
     * their 'tabindex' element attribute (also elements which are natively
     * focusable, such as <input> elements of type 'text').
     *
     * @constant
     */
    Forms.FOCUSABLE_SELECTOR = '[tabindex]:not([tabindex^="-"]):visible';

    /**
     * CSS class for title elements.
     *
     * @constant
     */
    Forms.TITLE_CLASS = 'title';

    /**
     * CSS selector for title elements.
     *
     * @constant
     */
    Forms.TITLE_SELECTOR = '.' + Forms.TITLE_CLASS;

    /**
     * CSS class for button elements.
     *
     * @constant
     */
    Forms.BUTTON_CLASS = 'button';

    /**
     * CSS selector for button elements.
     *
     * @constant
     */
    Forms.BUTTON_SELECTOR = '.' + Forms.BUTTON_CLASS;

    /**
     * CSS class name for option button elements in a button group.
     *
     * @constant
     */
    Forms.OPTION_BUTTON_CLASS = 'option-button';

    /**
     * CSS element selector for option button elements in a button group.
     *
     * @constant
     */
    Forms.OPTION_BUTTON_SELECTOR = Forms.BUTTON_SELECTOR + '.' + Forms.OPTION_BUTTON_CLASS;

    /**
     * CSS class for DOM container elements that contain the browser focus.
     *
     * @constant
     */
    Forms.DEFAULT_CLICK_TYPE = Utils.TOUCHDEVICE ? 'tap' : 'click';

    /**
     * CSS class name for elements that show the global tool tip while hovered.
     *
     * @constant
     */
    Forms.APP_TOOLTIP_CLASS = 'app-tooltip';

    // LESS variables ---------------------------------------------------------

    /**
     * A map with values of some LESS variables exported from the source file
     * "office/tk/exports.less".
     *
     * @type Object<String,String>
     * @constant
     */
    Forms.LessVar = LessExports ? LessExports.match(/#io-ox-office-less-variables\s+\..*?\}/g).reduce(function (map, definition) {
        var matches = /\.(\S+)\s*\{\s*content:\s*(\S+)\s*\}/.exec(definition);
        if (matches) { map[matches[1]] = matches[2]; }
        return map;
    }, {}) : {};

    // generic helpers --------------------------------------------------------

    /**
     * Creates and returns the HTML mark-up for a DOM element.
     *
     * @param {String} nodeName
     *  The tag name of the DOM element to be created.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Object} [options.attributes]
     *      The element attributes to be inserted into the DOM element. The
     *      attribute names can be given in camel-case notation. MUST NOT
     *      contain the attribute 'style' if the option 'style' (see below) is
     *      in use.
     *  - {String|Object} [options.style]
     *      Additional CSS style attributes for the control element. Will be
     *      inserted into the 'style' attribute of the element. Can be a string
     *      that will be inserted into the 'style' attribute as is, or can be
     *      an object that maps CSS properties to their values. The property
     *      names can be given in camel-case notation.
     *  - {String} [options.content='']
     *      The HTML mark-up of the DOM element's contents.
     *
     * @returns {String}
     *  The HTML mark-up of the DOM element.
     */
    Forms.createElementMarkup = function (nodeName, options) {

        // the HTML mark-up
        var markup = '<' + nodeName;

        // add the attributes
        _.each(Utils.getObjectOption(options, 'attributes'), function (value, name) {
            markup += ' ' + convertName(name) + '="' + Utils.escapeHTML(String(value)) + '"';
        });

        // add CSS styles and tool tip
        markup += createStyleMarkup(Utils.getOption(options, 'style'));

        // add inner HTML mark-up and close the element
        markup += '>' + Utils.getStringOption(options, 'content', '') + '</' + nodeName + '>';
        return markup;
    };

    /**
     * Adds device-specific CSS marker classes to the specified element. The
     * following CSS classes will be added:
     * - 'touch', if the device has a touch surface.
     * - 'browser-firefox', if the current browser is Firefox.
     * - 'browser-webkit', if the current browser is based on WebKit.
     * - 'browser-chrome', if the current browser is Chrome.
     * - 'browser-safari', if the current browser is Safari.
     * - 'browser-ie', if the current browser is Internet Explorer.
     *
     * @param {HTMLElement|jQuery} node
     *  A single DOM element, or a jQuery collection with one or more elements.
     */
    Forms.addDeviceMarkers = function (node) {

        // convert to jQuery collection
        node = $(node);

        // marker for touch devices
        node.toggleClass('touch', Utils.TOUCHDEVICE);

        // marker for browser types
        node.toggleClass('browser-firefox', !!_.browser.Firefox)
            .toggleClass('browser-webkit', !!_.browser.WebKit)
            .toggleClass('browser-chrome', !!_.browser.Chrome)
            .toggleClass('browser-safari', !!_.browser.Safari)
            .toggleClass('browser-ie', !!_.browser.IE);
    };

    /**
     * Enables or disables the tooltips of the specified DOM elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean} state
     *  Whether to enable (true) or disable (false) the tooltips.
     */
    Forms.enableToolTip = function (nodes, state) {
        if (!Utils.TOUCHDEVICE) {
            $(nodes).toggleClass(Forms.APP_TOOLTIP_CLASS, state);
        }
    };

    /**
     * Adds a tooltip to the specified DOM elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Object|String} [options]
     *  Optional parameters:
     *  - {String} [options.tooltip=null]
     *      The text for the tooltip text shown when the mouse hovers one of
     *      the nodes. If omitted or set to an empty string, the nodes will not
     *      show a tooltip anymore.
     *  If the parameter is a simple string, it will be set directly as the
     *  tooltip.
     */
    Forms.setToolTip = function (nodes, options) {

        var tooltip = (typeof options === 'string') ? options : Utils.getStringOption(options, 'tooltip', '');

        if (tooltip.length === 0) {
            $(nodes).removeAttr('data-original-title title aria-label');
            Forms.enableToolTip(nodes, false);
        } else if (Utils.TOUCHDEVICE) {
            $(nodes).attr({ title: tooltip });
        } else {
            $(nodes).attr({ 'aria-label': tooltip, 'data-original-title': tooltip });
            Forms.enableToolTip(nodes, true);
        }
    };

    /**
     * Returns whether the specified DOM element is in visible state. If node
     * contains the CSS class Forms.HIDDEN_CLASS, it is considered being
     * hidden.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM node according to its CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the element is visible.
     */
    Forms.isVisibleNode = function (node) {
        return $(node).first().is(Forms.VISIBLE_SELECTOR);
    };

    /**
     * Returns whether the specified DOM element and all its ancestors are in
     * visible state. If there is any ancestor containing the CSS class
     * Forms.HIDDEN_CLASS, or the node is not part of the document DOM, it is
     * considered being hidden.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM node according to its CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.deep=false]
     *      If set to true, all ancestors of the passed element will be checked
     *      for visibility too.
     *
     * @returns {Boolean}
     *  Whether the element is visible.
     */
    Forms.isDeeplyVisibleNode = function (node) {
        node = $(node).first();
        return Utils.containsNode(document.body, node[0]) && (node.closest(Forms.HIDDEN_SELECTOR).length === 0);
    };

    /**
     * Filters the visible elements from the passed jQuery collection.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM nodes according to their CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all visible elements.
     */
    Forms.filterVisibleNodes = function (nodes) {
        return $(nodes).filter(Forms.VISIBLE_SELECTOR);
    };

    /**
     * Filters the hidden elements from the passed jQuery collection.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM nodes according to their CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all hidden elements.
     */
    Forms.filterHiddenNodes = function (nodes) {
        return $(nodes).not(Forms.VISIBLE_SELECTOR);
    };

    /**
     * Shows or hides the specified DOM elements.
     *
     * @attention
     *  Modifies the existence of the CSS class Forms.HIDDEN_CLASS only; does
     *  NOT modify the 'display' or any other CSS properties of the DOM node.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean} state
     *  Whether to show (true) or hide (false) the DOM elements.
     */
    Forms.showNodes = function (nodes, state) {
        nodes = $(nodes);
        nodes.toggleClass(Forms.HIDDEN_CLASS, !state);
        nodes.attr('aria-hidden', state ? null : true);
    };

    /**
     * Returns whether the specified DOM element is in enabled state.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the element is enabled.
     */
    Forms.isEnabledNode = function (node) {
        return $(node).first().is(Forms.ENABLED_SELECTOR);
    };

    /**
     * Filters the enabled elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all enabled elements.
     */
    Forms.filterEnabledNodes = function (nodes) {
        return $(nodes).filter(Forms.ENABLED_SELECTOR);
    };

    /**
     * Filters the disabled elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all disabled elements.
     */
    Forms.filterDisabledNodes = function (nodes) {
        return $(nodes).not(Forms.ENABLED_SELECTOR);
    };

    /**
     * Enables or disables the specified DOM elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean} state
     *  Whether to enable (true) or disable (false) the DOM elements.
     */
    Forms.enableNodes = function (nodes, state) {
        nodes = $(nodes);
        nodes.toggleClass(Forms.DISABLED_CLASS, !state);
        nodes.attr('aria-disabled', state ? null : true);
    };

    /**
     * Returns whether the specified DOM element is in selected state.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the element is selected.
     */
    Forms.isSelectedNode = function (node) {
        return $(node).first().is(Forms.SELECTED_SELECTOR);
    };

    /**
     * Filters the selected elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all selected elements.
     */
    Forms.filterSelectedNodes = function (nodes) {
        return $(nodes).filter(Forms.SELECTED_SELECTOR);
    };

    /**
     * Filters the unselected elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all unselected elements.
     */
    Forms.filterUnselectedNodes = function (nodes) {
        return $(nodes).not(Forms.SELECTED_SELECTOR);
    };

    /**
     * Selects or deselects the specified DOM elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean} state
     *  Whether to select (true) or deselect (false) the DOM elements.
     */
    Forms.selectNodes = function (nodes, state) {
        $(nodes).toggleClass(Forms.SELECTED_CLASS, state);
    };

    /**
     * Triggers a 'click' event at the target node of the passed keyboard
     * event. The new click event will contain additional information about the
     * original keyboard event:
     * - {Event} originalEvent
     *      The original keyboard DOM event.
     * - {Number} keyCode
     *      The key code of the original event.
     * - {Boolean} shiftKey
     *      The state of the SHIFT key.
     * - {Boolean} altKey
     *      The state of the ALT key.
     * - {Boolean} ctrlKey
     *      The state of the CTRL key.
     * - {Boolean} metaKey
     *      The state of the META key.
     *
     * @param {jQuery.Event}
     *  The original event, as jQuery event object.
     */
    Forms.triggerClickForKey = function (event) {
        $(event.target).trigger(new $.Event('click', {
            originalEvent: event.originalEvent,
            keyCode: event.keyCode,
            shiftKey: event.shiftKey,
            altKey: event.altKey,
            ctrlKey: event.ctrlKey,
            metaKey: event.metaKey
        }));
    };

    // browser focus ----------------------------------------------------------

    /**
     * Returns all focusable control elements contained in the specified DOM
     * elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be searched. If this object is a jQuery collection,
     *  uses all nodes it contains.
     *
     * @returns {jQuery}
     *  A jQuery collection with all focusable form controls contained in the
     *  passed elements.
     */
    Forms.findFocusableNodes = function (nodes) {
        return $(nodes).find(Forms.FOCUSABLE_SELECTOR);
    };

    /**
     * Returns whether one of the descendant elements in the passed element
     * contains the active (focused) element.
     *
     * @param {HTMLElement|jQuery} nodes
     *  A single DOM element, or a jQuery collection, whose descendants will be
     *  checked for the active (focused) element.
     *
     * @returns {Boolean}
     *  Whether one of the descendants of the passed elements contains the
     *  active (focused) element.
     */
    Forms.containsFocus = function (nodes) {
        return $(nodes).find(Utils.getActiveElement()).length > 0;
    };

    /**
     * Returns whether one of the descendant elements in the passed element, or
     * contains the active (focused) element; or the element itself is the
     * active element.
     *
     * @param {HTMLElement|jQuery} nodes
     *  A single DOM element, or a jQuery collection, that will be checked for
     *  the active (focused) element.
     *
     * @returns {Boolean}
     *  Whether one of the descendants in the passed elements contains the
     *  active (focused) element, or one of the elements itself is the active
     *  element.
     */
    Forms.hasOrContainsFocus = function (nodes) {
        return ($(nodes).filter(Utils.getActiveElement()).length > 0) || Forms.containsFocus(nodes);
    };

    /**
     * Sets the browser focus to the first focusable control element contained
     * in the specified DOM element.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM element to be focused. If this object is a jQuery collection,
     *  searches all nodes it contains for a focusable form control.
     */
    Forms.grabFocus = function (nodes) {
        Utils.setFocus(Forms.findFocusableNodes(nodes).first());
    };

    /**
     * Enables focus navigation with cursor keys in the specified DOM element.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  A DOM element whose descendant focusable form controls can be focused
     *  with the cursor keys.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.homeEnd=false]
     *      If set to true, the HOME key will focus the first, and the END key
     *      will focus the last available form control.
     */
    Forms.enableCursorFocusNavigation = function (rootNode, options) {

        // whether to use HOME/END keys
        var homeEnd = Utils.getBooleanOption(options, 'homeEnd', false);
        // anchor node for vertical movement
        var anchorVertNode = null;
        // anchor node for horizontal movement
        var anchorHorNode = null;

        function setFocus(focusNode, resetAnchor) {
            Utils.setFocus(focusNode);
            if (resetAnchor || !anchorVertNode) { anchorVertNode = focusNode; }
            if (resetAnchor || !anchorHorNode) { anchorHorNode = focusNode; }
        }

        function moveFocus(focusNode, direction) {

            // all focusable nodes contained in the root node
            var focusableNodes = Forms.findFocusableNodes(rootNode);
            // the index of the focused node
            var focusIndex = focusableNodes.index(focusNode);
            // whether to move forward
            var forward = /^(right|down)$/.test(direction);
            // whether to move up/down
            var vertical = /^(up|down)$/.test(direction);
            // position and size of the focused node
            var focusPosition = null;

            // any other node was focused (e.g. the root node itself): start with first/last available node
            if (focusIndex < 0) {
                setFocus(forward ? focusableNodes.first() : focusableNodes.last(), true);
                return;
            }

            // resolve the positions and sizes of all focus nodes
            focusableNodes = focusableNodes.get().map(function (node) {
                var position = Utils.getNodePositionInPage(node);
                return _.extend(position, {
                    node: node,
                    centerX: position.left + position.width / 2,
                    centerY: position.top + position.height / 2
                });
            });

            // remember position of current focus node
            focusPosition = focusableNodes[focusIndex];

            // filter out all nodes that are not reachable in the specified direction
            focusableNodes = focusableNodes.filter(function (nodePosition) {
                switch (direction) {
                    case 'left':
                        return nodePosition.left + nodePosition.width <= focusPosition.left + 1; // add 1 pixel to avoid browser rounding effects
                    case 'right':
                        return nodePosition.right + nodePosition.width <= focusPosition.right + 1;
                    case 'up':
                        return nodePosition.top + nodePosition.height <= focusPosition.top + 1;
                    case 'down':
                        return nodePosition.bottom + nodePosition.height <= focusPosition.bottom + 1;
                }
            });

            // return early, if no focusable nodes left
            if (focusableNodes.length === 0) { return; }

            // insert the centered anchor position into the focus position, if available
            (function () {
                var anchorNode = vertical ? anchorVertNode : anchorHorNode;
                var anchorPosition = anchorNode ? Utils.getNodePositionInPage(anchorNode) : null;
                if (anchorPosition) {
                    if (vertical) {
                        focusPosition.centerX = anchorPosition.left + anchorPosition.width / 2;
                    } else {
                        focusPosition.centerY = anchorPosition.top + anchorPosition.height / 2;
                    }
                }
            }());

            // sort nodes by distance to current focus node
            focusableNodes = _.sortBy(focusableNodes, function (nodePosition) {
                return Math.pow(nodePosition.centerX - focusPosition.centerX, 2) + Math.pow(nodePosition.centerY - focusPosition.centerY, 2);
            });

            // clear the old anchor node for the opposite direction
            if (vertical) { anchorHorNode = null; } else { anchorVertNode = null; }

            // focus the nearest node
            setFocus(focusableNodes[0].node);
        }

        function keyDownHandler(event) {

            // ignore all keyboard events with any modifier keys (only plain cursor keys supported)
            if (KeyCodes.hasModifierKeys(event)) { return; }

            // LEFT/RIGHT cursor keys (not in text fields)
            if (!$(event.target).is('input,textarea')) {
                switch (event.keyCode) {
                    case KeyCodes.LEFT_ARROW:
                        moveFocus(event.target, 'left');
                        return false;
                    case KeyCodes.RIGHT_ARROW:
                        moveFocus(event.target, 'right');
                        return false;
                }
            }

            // UP/DOWN keys (always)
            switch (event.keyCode) {
                case KeyCodes.UP_ARROW:
                    moveFocus(event.target, 'up');
                    return false;
                case KeyCodes.DOWN_ARROW:
                    moveFocus(event.target, 'down');
                    return false;
            }

            // HOME/END keys (if configured)
            if (homeEnd) {
                switch (event.keyCode) {
                    case KeyCodes.HOME:
                        setFocus(Forms.findFocusableNodes(rootNode).first(), true);
                        return false;
                    case KeyCodes.END:
                        setFocus(Forms.findFocusableNodes(rootNode).last(), true);
                        return false;
                }
            }
        }

        $(rootNode).first().on('keydown', keyDownHandler);
    };

    // icons ------------------------------------------------------------------

    /**
     * Returns the complete CSS class names of the specified icon. For icons
     * from the AweSome font (icon name starting with 'fa-'), the class 'fa'
     * will be added. For OX Document icons (icon name starting with 'docs-'),
     * additional classes for the current locale will be added, and the class
     * 'retina' if the current device has a retina display.
     *
     * @param {String} icon
     *  The CSS class name of the icon.
     *
     * @returns {String}
     *  The complete CSS classes for the icon element.
     */
    Forms.getIconClasses = function (icon) {
        return (/^fa-/).test(icon) ? ('fa ' + icon) : (/^docs-/).test(icon) ? (icon + ' ' + DOCS_ICON_CLASSES) : icon;
    };

    /**
     * Creates the HTML mark-up of an <i> DOM element representing an icon.
     *
     * @param {String} icon
     *  The CSS class name of the icon.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.classes]
     *      The names of all CSS classes to be added to the icon element, as
     *      space separated string.
     *  - {String|Object} [options.style]
     *      Additional CSS style attributes for the icon. Will be inserted into
     *      the 'style' attribute of the <i> element. Can be a string that will
     *      be inserted into the 'style' attribute as is, or can be an object
     *      that maps CSS properties to their values. The property names can be
     *      given in camel-case notation.
     *
     * @returns {String}
     *  The HTML mark-up of the icon element, as string.
     */
    Forms.createIconMarkup = function (icon, options) {

        // the CSS class names to be added to the node
        var classes = Utils.getStringOption(options, 'classes', '');

        if (classes.length) { classes = ' ' + classes; }
        classes = Forms.getIconClasses(icon) + classes;

        return Forms.createElementMarkup('i', {
            attributes: { class: classes, 'aria-hidden': 'true' },
            style: Utils.getOption(options, 'style')
        });
    };

    // control captions -------------------------------------------------------

    /**
     * Creates the HTML mark-up of a <span> DOM element representing the text
     * label for a control.
     *
     * @param {String} text
     *  The label text.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.classes]
     *      The names of all CSS classes to be added to the <span> element, as
     *      space separated string.
     *  - {String|Object} [options.style]
     *      Additional CSS style attributes for the label. Will be inserted
     *      into the 'style' attribute of the <span> element. Can be a string
     *      that will be inserted into the 'style' attribute as is, or can be
     *      an object that maps CSS properties to their values. The property
     *      names can be given in camel-case notation.
     *
     * @returns {String}
     *  The HTML mark-up of the label element, as string.
     */
    Forms.createSpanMarkup = function (text, options) {

        // the CSS class names to be added to the node
        var classes = Utils.getStringOption(options, 'classes');
        // the element attributes
        var attributes = classes ? { class: classes } : null;

        return Forms.createElementMarkup('span', {
            attributes: attributes,
            style: Utils.getOption(options, 'style'),
            content: Utils.escapeHTML(text)
        });
    };

    /**
     * Creates the HTML mark-up of an <img> DOM element.
     *
     * @param {String} source
     *  The source URL of the image.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.classes]
     *      The names of all CSS classes to be added to the <img> element, as
     *      space separated string.
     *  - {String} [options.alt]
     *      Alternate text of the image element.
     *
     * @returns {String}
     *  The HTML mark-up of the image element, as string.
     *
     */
    Forms.createImageMarkup = function (source, options) {

        if (!source) { Utils.error('Forms.createImageMarkup(): image source URL missing'); }

        // the CSS class names to be added to the node
        var classes = Utils.getStringOption(options, 'classes');
        // alternate text for the image element
        var altText = Utils.getStringOption(options, 'alt', null);
        // custom picture height
        var pictureHeight = Utils.getIntegerOption(options, 'pictureHeight', 48);
        // custom picture width
        var pictureWidth = Utils.getIntegerOption(options, 'pictureWidth', 48);
        // the element attributes
        var attributes = { src: source, class: classes ? classes : '', width: pictureWidth, height: pictureHeight, alt: altText };

        return Forms.createElementMarkup('img', { attributes: attributes });
    };

    /**
     * Creates the HTML mark-up for icons and text labels for a form control.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.icon]
     *      The CSS class name of the icon. If omitted, no icon will be shown.
     *  - {String|Object} [options.iconClasses]
     *      The names of all CSS classes to be added to the icon element, as
     *      space separated string.
     *  - {String|Object} [options.iconStyle]
     *      Additional CSS style attributes for the icon. Will be inserted into
     *      the 'style' attribute of the <i> element. Can be a string that will
     *      be inserted into the 'style' attribute as is, or can be an object
     *      that maps CSS properties to their values. The property names can be
     *      given in camel-case notation.
     *  - {String} [options.label]
     *      The text label. If omitted, no text will be shown.
     *  - {String|Object} [options.labelClasses]
     *      The names of all CSS classes to be added to the <span> element, as
     *      space separated string.
     *  - {String|Object} [options.labelStyle]
     *      Additional CSS style attributes for the label. Will be inserted
     *      into the 'style' attribute of the <span> element. Can be a string
     *      that will be inserted into the 'style' attribute as is, or can be
     *      an object that maps CSS properties to their values. The property
     *      names can be given in camel-case notation.
     *  - {String} [options.iconPos='leading']
     *      The position of the icon relative to the text label. Possible
     *      values are:
     *      - 'leading': The icon will be located in front of the label.
     *      - 'trailing': The icon will be located after the label.
     *
     * @returns {String}
     *  The HTML mark-up of the complete caption nodes, as string.
     */
    Forms.createCaptionMarkup = function (options) {

        // the icon mark-up
        var icon = Utils.getStringOption(options, 'icon', '');
        // the label mark-up
        var label = Utils.getStringOption(options, 'label', '');
        // the position of the icon
        var iconPos = Utils.getStringOption(options, 'iconPos', 'leading');

        // create the icon mark-up
        if (icon) {
            icon = Forms.createIconMarkup(icon, {
                classes: Utils.getOption(options, 'iconClasses'),
                style: Utils.getOption(options, 'iconStyle')
            });
        }

        // create the label mark-up
        if (label) {
            label = Forms.createSpanMarkup(label, {
                classes: Utils.getOption(options, 'labelClasses'),
                style: Utils.getOption(options, 'labelStyle')
            });
        }

        // concatenate icon and label mark-up in the correct order
        return '<span class="' + CAPTION_CLASS + '">' + ((iconPos === 'trailing') ? (label + icon) : (icon + label)) + '</span>';
    };

    /**
     * Returns the caption container node of the passed control element.
     *
     * @param {HTMLElement|jQuery} controlNode
     *  The control element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {jQuery}
     *  The caption container node of the control element.
     */
    Forms.getCaptionNode = function (controlNode) {
        return $(controlNode).first().find(CAPTION_SELECTOR);
    };

    /**
     * Sets or changes the icon in the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {String} icon
     *  The CSS class name of the new icon.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createIconMarkup(). Additionally, the following options are
     *  supported:
     *  - {Object} [options.pos='auto']
     *      The position of the icon relative to the text label of the control
     *      element. Possible values are:
     *      - 'auto': Replaces an existing icon at its current position. If no
     *          icon exists, the new icon will be inserted in front of the text
     *          label (if existing).
     *      - 'leading': The new icon will be inserted in front of the label.
     *      - 'trailing': The new icon will be inserted after the label.
     */
    Forms.setCaptionIcon = function (controlNodes, icon, options) {

        var iconMarkup = Forms.createIconMarkup(icon, options);
        var iconPos = Utils.getStringOption(options, 'pos', 'auto');

        $(controlNodes).each(function () {

            var captionNode = $(this).find(CAPTION_SELECTOR);
            var iconNode = captionNode.find(ICON_SELECTOR);

            if ((iconPos === 'auto') && (iconNode.length > 0)) {
                iconNode.after(iconMarkup);
            } else if (iconPos === 'trailing') {
                captionNode.append(iconMarkup);
            } else {
                captionNode.prepend(iconMarkup);
            }

            iconNode.remove();
        });
    };

    /**
     * Removes the icon from the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     */
    Forms.removeCaptionIcon = function (controlNodes) {
        $(controlNodes).find(CAPTION_SELECTOR + ICON_SELECTOR).remove();
    };

    /**
     * Inserts an <img> element for a bitmap icon in front of the captions of
     * the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {String} bitmapUrl
     *  The data URL containing the bitmap data.
     *
     * @param {String} [selectedBitmapUrl]
     *  The data URL containing the bitmap data for the selected state of the
     *  control. If omitted, the same bitmap as specified with the parameter
     *  "bitmapUrl" will be used for all states of the controls.
     */
    Forms.setCaptionBitmapIcon = function (controlNodes, bitmapUrl, selectedBitmapUrl) {

        var iconMarkup = '<img class="bitmap-icon' + (selectedBitmapUrl ? ' deselected-only' : '') + '" src="' + bitmapUrl + '">';
        if (selectedBitmapUrl) { iconMarkup += '<img class="bitmap-icon selected-only" src="' + selectedBitmapUrl + '">'; }

        $(controlNodes).find(CAPTION_SELECTOR).prepend(iconMarkup);
    };

    /**
     * Returns the current label text of the passed control element.
     *
     * @param {HTMLElement|jQuery} controlNode
     *  The control element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {String}
     *  The current text label.
     */
    Forms.getCaptionText = function (controlNode) {
        return $(controlNode).first().find(CAPTION_SELECTOR + SPAN_SELECTOR).text();
    };

    /**
     * Changes the label texts in the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {String} text
     *  The new label text.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createSpanMarkup(). Additionally, the following options are
     *  supported:
     *  - {Object} [options.pos='auto']
     *      The position of the label relative to the icon of the control
     *      element. Possible values are:
     *      - 'auto': Replaces an existing label at its current position. If no
     *          label exists, the new label will be inserted after the icon (if
     *          existing).
     *      - 'leading': The new label will be inserted in front of the icon.
     *      - 'trailing': The new label will be inserted after the icon.
     */
    Forms.setCaptionText = function (controlNodes, text, options) {

        var spanMarkup = Forms.createSpanMarkup(text, options);
        var spanPos = Utils.getStringOption(options, 'pos', 'auto');

        $(controlNodes).each(function () {

            var captionNode = $(this).find(CAPTION_SELECTOR);
            var spanNode = captionNode.find(SPAN_SELECTOR);

            if ((spanPos === 'auto') && (spanNode.length > 0)) {
                spanNode.after(spanMarkup);
            } else if (spanPos === 'leading') {
                captionNode.prepend(spanMarkup);
            } else {
                captionNode.append(spanMarkup);
            }

            spanNode.remove();
        });
    };

    /**
     * Removes the text label from the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     */
    Forms.removeCaptionText = function (controlNodes) {
        $(controlNodes).find(CAPTION_SELECTOR + SPAN_SELECTOR).remove();
    };

    /**
     * Changes the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createCaptionMarkup(). Additionally, the following options are
     *  supported:
     *  - {Object} [options.pos='auto']
     *      The position of the caption relative to the other content nodes of
     *      the control element. Possible values are:
     *      - 'auto': Replaces an existing caption at its current position. If
     *          no caption exists, the new caption will be inserted in front of
     *          the other content nodes.
     *      - 'leading': The new caption will be inserted in front of the other
     *          content nodes.
     *      - 'trailing': The new caption will be inserted after the other
     *          content nodes.
     */
    Forms.setCaption = function (controlNodes, options) {

        var captionMarkup = Forms.createCaptionMarkup(options);
        var captionPos = Utils.getStringOption(options, 'pos', 'auto');

        $(controlNodes).each(function () {

            var captionNode = $(this).find(CAPTION_SELECTOR);

            if ((captionPos === 'auto') && (captionNode.length > 0)) {
                captionNode.after(captionMarkup);
            } else if (captionPos === 'trailing') {
                $(this).append(captionMarkup);
            } else {
                $(this).prepend(captionMarkup);
            }

            captionNode.remove();
        });
    };

    // label elements ---------------------------------------------------------

    /**
     * Creates and returns the HTML mark-up for a label element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the methods
     *  Forms.createElementMarkup() and Forms.createCaptionMarkup(). Additional
     *  HTML content specified with the option 'content' will be appended to
     *  the label caption.
     *
     * @returns {String}
     *  The HTML mark-up of the label element.
     */
    Forms.createLabelMarkup = function (options) {

        var captionMarkup = Forms.createCaptionMarkup(options);
        var contentMarkup = Utils.getStringOption(options, 'content', '');

        // A11Y: use <div> instead of <label> to fulfill WCAG 2.0 Success Criteria - labels must reference an input via the 'for' attribute.
        return Forms.createElementMarkup('div', Utils.extendOptions(options, { content: captionMarkup + contentMarkup, attributes: { class: 'label' } }));
    };

    // button elements --------------------------------------------------------

    /**
     * Creates the HTML mark-up of a DOM element containing a drop-down caret
     * symbol intended to be inserted into a button control element.
     *
     * @param {String} direction
     *  The direction for the caret symbol. Can be one of 'up', 'down', 'left',
     *  or 'right'.
     *
     * @returns {String}
     *  The HTML mark-up of the caret node, as string.
     */
    Forms.createCaretMarkup = function (direction) {
        return '<span class="caret-node">' + Forms.createIconMarkup('fa-caret-' + direction) + '</span>';
    };

    /**
     * Creates the HTML mark-up for the check mark of a check box or option
     * button of a radio group.
     *
     * @param {Boolean|Null} state
     *  The state of the created check mark. The Boolean value true results in
     *  a checked icon. The Boolean value false results in an unchecked icon.
     *  The value null is treated as ambiguous state and results in a specific
     *  icon, if the option 'ambiguous' is set. Otherwise, this will be the
     *  unchecked icon.
     *
     * @param {String} design
     *  Specifies the icon set that will be used to visualize the unchecked and
     *  checked state. MUST be one of the following values:
     *  - 'check': Unchecked state is empty space, and checked state is a
     *      simple check mark.
     *  - 'boxed': Unchecked state is an empty rectangle, and checked state is
     *      a rectangle with a check mark.
     *  - 'radio': Unchecked state is an empty circle, and checked state is a
     *      circle with a dot inside.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.ambiguous=false]
     *      If set to true, the ambiguous state (any value but Booleans) will
     *      be represented by a specific distinct icon. Otherwise, the icon for
     *      the unchecked state will be used.
     *
     * @returns {String}
     *  The HTML mark-up for the check mark of a check box or option button.
     */
    Forms.createCheckMarkMarkup = function (state, design, options) {

        // special icon for ambiguous state
        var ambiguous = Utils.getBooleanOption(options, 'ambiguous', false);
        // the icon class name
        var index = (state === true) ? 1 : (ambiguous && _.isNull(state)) ? 2 : 0;

        return '<span class="' + CHECK_MARK_CLASS + '">' + Forms.createIconMarkup(CHECK_ICON_SETS[design][index]) + '</span>';
    };

    /**
     * Creates and returns the HTML mark-up for a button element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the methods
     *  Forms.createElementMarkup() and Forms.createCaptionMarkup(). Additional
     *  HTML content specified with the option 'content' will be appended to
     *  the button caption. Additionally, the following options are supported:
     *  - {String} [options.nodeName='a']
     *      The name of the HTML element that will represent the button. By
     *      default, an <a> anchor element will be used.
     *  - {Boolean} [options.focusable=true]
     *      If set to false, the 'tabindex' attribute of the button element
     *      will be set to the value -1 in order to remove the button from the
     *      keyboard focus chain.
     *
     * @returns {String}
     *  The HTML mark-up of the button element.
     */
    Forms.createButtonMarkup = function (options) {

        var nodeName = Utils.getStringOption(options, 'nodeName', 'a');
        var attributes = Utils.getObjectOption(options, 'attributes');
        var classes = Utils.getStringOption(attributes, 'class', '');
        var focusable = Utils.getBooleanOption(options, 'focusable', true);
        var captionMarkup = Forms.createCaptionMarkup(options);
        var contentMarkup = Utils.getStringOption(options, 'content', '');
        var href = Utils.getStringOption(options, 'href', '');
        var target = Utils.getStringOption(options, 'target', '_blank');
        var ariaOwns = Utils.getStringOption(options, 'aria-owns');

        // add more element attributes, return the mark-up
        attributes = _.extend({ tabindex: focusable ? 0 : -1, role: 'button' }, attributes);
        if (!focusable) { attributes = _.extend({ 'aria-hidden': true }, attributes); }
        attributes.class = Forms.BUTTON_CLASS;
        if (classes.length > 0) { attributes.class += ' ' + classes; }
        if (ariaOwns) { attributes.ariaOwns = ariaOwns; }

        if (href.length > 0) {
            attributes.href = href;
            attributes.target = target;
            // bug #47916# prevent content spoofing
            // see: https://mathiasbynens.github.io/rel-noopener/
            // and: https://github.com/danielstjules/blankshield
            if (attributes.target === '_blank') {
                attributes.rel = 'noopener';
            }
        }

        return Forms.createElementMarkup(nodeName, Utils.extendOptions(options, { attributes: attributes, content: captionMarkup + contentMarkup }));
    };

    /**
     * Returns the value stored in the 'value' data attribute of the specified
     * button element. If the stored value is a function, calls that function
     * (with the button node as first parameter) and returns its result.
     *
     * @param {HTMLElement|jQuery} buttonNode
     *  The button element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {Any}
     *  The value stored at in the button element.
     */
    Forms.getButtonValue = function (buttonNode) {
        var value = $(buttonNode).first().data('value');
        return _.isFunction(value) ? value($(buttonNode)) : value;
    };

    /**
     * Stores the passed value in the 'value' data attribute of the specified
     * button elements.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {Any} value
     *  A value, object, or function that will be copied to the 'value' data
     *  attribute of the button element (see method Forms.getButtonValue() for
     *  details).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.dataValue]
     *      A string value that will be inserted into the 'data-value'
     *      attribute of the button elements. If omitted, the JSON string
     *      representation of the passed value will be used instead (all
     *      double-quote characters will be removed from the string though),
     *      unless the passed value is a function.
     */
    Forms.setButtonValue = function (buttonNodes, value, options) {

        // set the passed value as jQuery data attribute (may be objects or functions)
        buttonNodes = $(buttonNodes).data('value', value);

        // set the alternative data value as plain element attribute
        var dataValue = Utils.getStringOption(options, 'dataValue');
        if (!_.isString(dataValue) && !_.isFunction(value) && !_.isUndefined(value)) {
            dataValue = JSON.stringify(value).replace(/^"|"$/g, ''); // removing leading and ending quotes introduced by JSON.stringify
        }
        buttonNodes.attr('data-value', dataValue || null);
    };

    /**
     * Filters the passed button elements by comparing to a specific value.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements to be filtered. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Any} value
     *  The button value to be filtered for.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Function} [options.matcher=_.isEqual]
     *      A comparison function that returns whether the actual button values
     *      should be included in the result. Receives the passed value as
     *      first parameter, and the value of the current button element as
     *      second parameter. The function must return the Boolean value true
     *      in order to include the respective button element in the result. If
     *      omitted, uses _.isEqual() which compares arrays and objects deeply.
     *
     * @returns {jQuery}
     *  A jQuery collection with all button nodes whose value is equal to the
     *  passed value.
     */
    Forms.filterButtonNodes = function (buttonNodes, value, options) {
        var matcher = Utils.getFunctionOption(options, 'matcher', _.isEqual);
        return $(buttonNodes).filter(function () {
            return matcher(value, Forms.getButtonValue(this)) === true;
        });
    };

    /**
     * Returns whether the specified button element is in selected/checked
     * state.
     *
     * @param {HTMLElement|jQuery} buttonNode
     *  The button element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {Boolean}
     *  Whether the button node is selected/checked.
     */
    Forms.isCheckedButtonNode = function (buttonNode) {
        return $(buttonNode).first().is(Forms.BUTTON_SELECTOR + DATA_CHECKED_SELECTOR);
    };

    /**
     * Filters the selected/checked button elements from the passed jQuery
     * collection.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements to be filtered for.
     *
     * @returns {jQuery}
     *  A jQuery collection with all selected/checked button elements.
     */
    Forms.filterCheckedButtonNodes = function (buttonNodes) {
        return $(buttonNodes).filter(Forms.BUTTON_SELECTOR + DATA_CHECKED_SELECTOR);
    };

    /**
     * Selects or deselects the passed button elements, or sets them into the
     * mixed state in case the value represented by the buttons is ambiguous.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean|Null} state
     *  The new state of the button nodes. The Boolean value true results in a
     *  selected/checked design. The Boolean value false results in an
     *  unselected/unchecked design. The value null is treated as ambiguous
     *  state and results in a specific icon, if the option 'ambiguous' is set.
     *  Otherwise, this will be the unselected/unchecked design.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.design='select']
     *      Specifies the design that will be used to visualize the selected
     *      state. The following values are supported:
     *      - 'select' (default): Selected state will be visualized by a
     *          colored button background.
     *      - 'check': Space for an icon will be added in front of the button
     *          caption. Unselected state remains empty space, and selected
     *          state is a simple check mark.
     *      - 'boxed': Space for an icon will be added in front of the button
     *          caption. Unselected state is an empty rectangle, and selected
     *          state is a rectangle with a check mark.
     *      - 'radio': Space for an icon will be added in front of the button
     *          caption. Unselected state is an empty circle, and selected
     *          state is a circle with a dot inside.
     *  - {Boolean} [options.ambiguous=false]
     *      If set to true, the ambiguous state (any value but Booleans) will
     *      be represented by a fill pattern (if design is 'select'), otherwise
     *      by a specific distinct icon. Without this option, the design for
     *      the unselected state will be used for the ambiguous state.
     */
    Forms.checkButtonNodes = function (buttonNodes, state, options) {

        // the design mode
        var design = Utils.getStringOption(options, 'design', 'select');
        // whether to use special formatting for the ambiguous state
        var ambiguous = Utils.getBooleanOption(options, 'ambiguous', false);

        var haspopup = Utils.getBooleanOption(options, 'haspopup', false);
        // Boolean selected state (without ambiguous state)
        var selected = state === true;

        // remove old check mark icons (if ever someone switches between icon design and selection design)
        buttonNodes = $(buttonNodes);
        buttonNodes.find('>.' + CHECK_MARK_CLASS).remove();

        // update button contents/formatting according to design mode
        if (design === 'select') {
            Forms.selectNodes(buttonNodes, selected);
            buttonNodes.toggleClass(AMBIGUOUS_CLASS, ambiguous && _.isNull(state)).attr('aria-selected', selected);
        } else {
            buttonNodes.prepend(Forms.createCheckMarkMarkup(state, design, { ambiguous: ambiguous }));
        }

        // update generic attributes
        buttonNodes.attr(DATA_CHECKED_ATTR, selected);
        if (haspopup) {
            buttonNodes.attr({ 'aria-haspopup': true, 'aria-expanded': selected });
        } else {
            buttonNodes.attr({ 'aria-pressed': selected, 'aria-checked': selected });
        }
    };

    /**
     * Selects or deselects the button elements with a specific value.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements to be selected or deselected. If this object is a
     *  jQuery collection, uses all nodes it contains.
     *
     * @param {Any} value
     *  The button value to be filtered for.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Function} [options.matcher=_.isEqual]
     *      A comparison function that returns whether the actual button values
     *      should be selected, deselected, or skipped. Receives the passed
     *      value as first parameter, and the value of the current button
     *      element as second parameter. The function must return the Boolean
     *      value true to select the respective button element, the Boolean
     *      value false to deselect the button element, or any other value to
     *      leave the button element unmodified. If omitted, uses _.isEqual()
     *      which compares arrays and objects deeply, and does not skip any
     *      button element.
     *  - {String} [options.design='select']
     *      Specifies the design that will be used to visualize the selected
     *      state. See method Forms.checkButtonNodes() for details.
     *  - {Boolean} [options.multiSelect=true]
     *      If false it stops if the first node matched the value, otherwise
     *      all nodes will be checked.
     */
    Forms.checkMatchingButtonNodes = function (buttonNodes, value, options) {

        // the matcher predicate function
        var matcher = Utils.getFunctionOption(options, 'matcher', _.isEqual);
        var multiSelect = Utils.getBooleanOption(options, 'multiSelect', true);
        var firstMatched = false;

        $(buttonNodes).each(function () {
            var state = matcher(value, Forms.getButtonValue(this));
            if (_.isBoolean(state)) {
                if (state && !multiSelect) {
                    if (firstMatched) {
                        state = false;
                    }
                    firstMatched = true;
                }
                Forms.checkButtonNodes(this, state, options);
            }
        });
    };

    /**
     * Adds a keyboard event handler to the passed button elements, which
     * listens to the ENTER and SPACE keys, and triggers 'click' events at the
     * target node of the received event. The event object will contain the
     * additional property 'keyCode' set to the correct key code. Bubbling of
     * the original keyboard events ('keydown', 'keypress', and 'keyup') will
     * be suppressed, as well as the default browser action. The 'click' event
     * will be triggered after receiving the 'keyup' event.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The nodes to bind the event handles to. May contain button elements, or
     *  any other container elements with descendant button elements.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.tab=false]
     *      If set to true, 'click' events will also be triggered for the TAB
     *      key and the SHIFT+TAB key combination (no other modifier keys must
     *      be pressed).
     */
    Forms.setButtonKeyHandler = function (nodes, options) {

        // whether to listen to the TAB key
        var tab = Utils.getBooleanOption(options, 'tab', false);
        // target node of the last caught 'keydown' event
        var lastKeyDownTarget = null;

        // Bug 28528: ENTER key must be handled explicitly, <a> elements without
        // 'href' attribute do not trigger click events. The 'href' attribute
        // has been removed from the buttons to prevent useless tooltips with the
        // link address. Additionally, SPACE key must always be handled manually.

        // handler for the 'keyup' events
        function keyHandler(event) {

            // whether the target of a 'keyup' event matches the target of preceding 'keydown'
            var isKeyUpWithMatchingTarget = false;

            // Bug 37870: Remember target node of a 'keydown' event, check that 'keyup' event is for the
            // same target. Needed to prevent triggering a click event when pressing ENTER one one node,
            // the changing the browser focus to one of the nodes covered by this method, then receiving
            // the 'keyup' event. This happens for example when pressing the ENTER key on the OK button
            // of a modal dialog. The 'keydown' event causes to close the dialog which restores the old
            // focus node, e.g. a button in a tool bar. The following 'keyup' event for the ENTER key
            // would immediately trigger that button which will open the modal dialog again.
            if (event.type === 'keydown') {
                lastKeyDownTarget = event.target;
            } else if (event.type === 'keyup') {
                isKeyUpWithMatchingTarget = lastKeyDownTarget === event.target;
                lastKeyDownTarget = null;
            }

            // ENTER and SPACE key: wait for keyup event
            if ((event.keyCode === KeyCodes.ENTER) || (event.keyCode === KeyCodes.SPACE)) {
                if (isKeyUpWithMatchingTarget) {
                    Forms.triggerClickForKey(event);
                }
                return false; // stop propagation of ALL keyboard events
            }

            if (tab && (event.type === 'keydown') && KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                Forms.triggerClickForKey(event);
                // let all TAB key events bubble up for focus navigation
            }
        }

        // directly bind the event handler to all button nodes
        $(nodes).filter(Forms.BUTTON_SELECTOR).on('keydown keypress keyup', keyHandler);
        // bind the event handler to all descendant button nodes
        $(nodes).not(Forms.BUTTON_SELECTOR).on('keydown keypress keyup', Forms.BUTTON_SELECTOR, keyHandler);
    };

    // native (Bootstrap) button elements -------------------------------------

    /**
     * Returns whether the specified simple Bootstrap <button> element is in
     * enabled state.
     *
     * @param {HTMLElement|jQuery} buttonNode
     *  The button element to be checked. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @returns {Boolean}
     *  Whether the button node is enabled.
     */
    Forms.isBSButtonEnabled = function (buttonNode) {
        return $(buttonNode).attr('disabled') !== 'disabled';
    };

    /**
     * Enables or disables a simple Bootstrap <button> element.
     *
     * @param {HTMLElement|jQuery} buttonNode
     *  The button elements to be manipulated.
     *
     * @param {Boolean} state
     *  Whether to enable (true) or disable (false) the button elements.
     */
    Forms.enableBSButton = function (buttonNode, state) {
        buttonNode.attr({
            disabled: state ? null : 'disabled',
            'aria-disabled': state ? null : true
        });
    };

    // text input elements ----------------------------------------------------

    /**
     * Creates and returns the HTML mark-up for an <input> element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createElementMarkup(). Additionally, the following options are
     *  supported:
     *  - {String} [options.placeholder='']
     *      A place holder text that will be shown in an empty text field.
     *  - {String} [options.keyboard='text']
     *      Specifies which virtual keyboard should occur on touch devices.
     *      Supported types are 'text' (default) for a generic text field with
     *      alphanumeric keyboard, 'number' for floating-point numbers with
     *      numeric keyboard, 'url' for an alphanumeric keyboard with additions
     *      for entering URLs, or 'email' for an alphanumeric keyboard with
     *      additions for entering e-mail addresses.
     *  - {Number} [options.maxLength]
     *      If specified and a positive number, sets the 'maxLength' attribute
     *      of the text field, used to restrict the maximum length of the text
     *      that can be inserted into the text field.
     *  - {Boolean} [options.autoCorrection=false]
     *      Only for iOs. This option has been introduced with Bug 43868.
     *      By default the autocorrection is off for input fields
     *      in iOS. When setting this to true, the autocorrection is enabled.
     *
     * @returns {String}
     *  The HTML mark-up for the new text field element.
     */
    Forms.createInputMarkup = function (options) {

        var type = Utils.TOUCHDEVICE ? Utils.getStringOption(options, 'keyboard', 'text') : 'text';
        var attributes = Utils.getObjectOption(options, 'attributes', {});
        var tabIndex = Utils.getIntegerOption(options, 'tabindex', 0);

        // add more element attributes, return the mark-up
        attributes = _.extend({ type: type, tabindex: tabIndex, class: 'form-control' }, attributes);
        setGenericInputAttributes(attributes, options);
        return Forms.createElementMarkup('input', Utils.extendOptions(options, { attributes: attributes, content: null }));
    };

    /**
     * Creates and returns the HTML mark-up for a <textarea> element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createElementMarkup(). Additionally, the following options are
     *  supported:
     *  - {String} [options.placeholder='']
     *      A place holder text that will be shown in an empty text area.
     *  - {Number} [options.maxLength]
     *      If specified and a positive number, sets the 'maxLength' attribute
     *      of the text area, used to restrict the maximum length of the text
     *      that can be inserted into the text area.
     *  - {Boolean} [options.autoCorrection=false]
     *      Only for iOs. This option has been introduced with Bug 43868.
     *      By default the autocorrection is off for input fields
     *      in iOS. When setting this to true, the autocorrection is enabled.
     *
     * @returns {String}
     *  The HTML mark-up for the new text area element.
     */
    Forms.createTextAreaMarkup = function (options) {

        var attributes = Utils.getObjectOption(options, 'attributes', {});
        var tabIndex = Utils.getIntegerOption(options, 'tabindex', 0);

        // add more element attributes, return the mark-up
        attributes = _.extend({ tabindex: tabIndex }, attributes);
        setGenericInputAttributes(attributes, options);
        return Forms.createElementMarkup('textarea', Utils.extendOptions(options, { attributes: attributes, content: null }));
    };

    /**
     * Returns the current selection in the passed text input control.
     *
     * @param {HTMLElement|jQuery} inputNode
     *  A text input element (an HTML <input> or <textarea> element). If this
     *  object is a jQuery collection, used the first DOM node it contains.
     *
     * @returns {Object}
     *  An object with the properties 'start' and 'end' containing the start
     *  and end character offset of the selection in the input control.
     */
    Forms.getInputSelection = function (inputNode) {
        inputNode = Utils.getDomNode(inputNode);
        try {
            return { start: inputNode.selectionStart, end: inputNode.selectionEnd };
        } catch (ex) {
            // Bug 31286: Mobile Chrome: input elements of type 'number' do not support selection
            return { start: inputNode.value.length, end: inputNode.value.length };
        }
    };

    /**
     * Returns whether the current selection in the passed text input control
     * is a simple text cursor (no characters selected).
     *
     * @param {HTMLElement|jQuery} inputNode
     *  A text input element (an HTML <input> or <textarea> element). If this
     *  object is a jQuery collection, used the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the current selection in the passed text input control is a
     *  simple text cursor (no characters selected).
     */
    Forms.isCursorSelection = function (inputNode) {
        var selection = Forms.getInputSelection(inputNode);
        return selection.start === selection.end;
    };

    /**
     * Changes the current selection in the passed text input control.
     *
     * @param {HTMLElement|jQuery} inputNode
     *  A text input element (an HTML <input> or <textarea> element). If this
     *  object is a jQuery collection, used the first DOM node it contains.
     *
     * @param {Number} start
     *  The start character offset of the new selection in the input control.
     *
     * @param {Number} [end=start]
     *  The end character offset of the new selection in the input control. If
     *  omitted, sets a text cursor according to the passed start position.
     */
    Forms.setInputSelection = function (inputNode, start, end) {
        inputNode = Utils.getDomNode(inputNode);
        try {
            inputNode.setSelectionRange(start, _.isNumber(end) ? end : start);
        } catch (ex) {
            // Bug 31286: Mobile Chrome: input elements of type 'number' do not support selection
        }
    };

    /**
     * Replaces the current selection of the text input control with the
     * specified text, and places a simple text cursor behind the new text.
     *
     * @param {HTMLElement|jQuery} inputNode
     *  A text input element (an HTML <input> or <textarea> element). If this
     *  object is a jQuery collection, used the first DOM node it contains.
     *
     * @param {String} [text='']
     *  The text used to replace the selected text in the input control. If
     *  omitted, the selection will simply be deleted.
     */
    Forms.replaceTextInInputSelection = function (inputNode, text) {

        var node = Utils.getDomNode(inputNode);
        var start = node.selectionStart;

        text = _.isString(text) ? text : '';
        node.value = Utils.replaceSubString(node.value, start, node.selectionEnd, text);
        node.setSelectionRange(start + text.length, start + text.length);
    };

    // simple or bootstrap elements -------------------------------------------

    /**
     * Creates a DropDown-Menu from/for the bootstrap-framework
     *
     * * @param {Array} arrEntries
     *  An array with objects. Each
     *  descriptor is an object with the following properties:
     *  - {String} title
     *      The text label for the list item.
     *
     * @param {Object} [options]
     *  Optional parameters. The following options are
     *  supported:
     *  - {String} [options.btnText='']
     *      The button text which is visible on the dropdown-opener-button
     *  - {Boolean} [options.withCaret=true]
     *      Show a caret besides the buttontext
     *  - {String} [options.classes='']
     *      Additionally style classes for the surrounding div.
     *  - {String} [options.dropDirection='down']
     *      The direction in which the caret shows and the popup will open
     *
     * @returns {String}
     *   The HTML mark-up of the complete bootstrap dropdown element.
     */
    Forms.createDropDownMarkup = function (arrEntries, options) {

        var dropDirection = Utils.getStringOption(options, 'dropDirection', 'down'),
            classes = Utils.getStringOption(options, 'classes', '') + ((dropDirection === 'down') ? ' dropdown' : ' dropup'),

            // caret
            caret       = Utils.getBooleanOption(options, 'withCaret', true) ? Forms.createElementMarkup('span', { attributes: { class: 'caret' } }) : '',

            // button
            btnText     = Utils.getStringOption(options, 'btnText', ''),
            btnNode     = Forms.createElementMarkup('button', { attributes: { 'data-toggle': 'dropdown', class: 'dropdown-toggle' }, content: btnText + caret }),

            // list
            listNode    = null,
            listNodes   = null;

        // add list-entries
        listNodes = '';
        _.each(arrEntries, function (entry) {
            var title = Utils.getStringOption(entry, 'title', '[no-title]'),
                link = Forms.createElementMarkup('a', { attributes: { role: 'menuitem', tabindex: '-1' }, content: title });

            listNodes += Forms.createElementMarkup('li', { attributes: { role: 'presentation' }, content: link });
        });

        listNode = Forms.createElementMarkup('ul', { attributes: { role: 'menu', class: 'dropdown-menu' }, content: listNodes });

        return Forms.createElementMarkup('div', { attributes: { class: classes }, content: btnNode + listNode });
    };

    /**
     * Creates a simple <select> list control with an optional leading label.
     *
     * @param {Array} entries
     *  An array with descriptor objects for all list entries. Each object MUST
     *  contain the following properties:
     *  - {Any} value
     *      The value associated to the list item.
     *  - {String} label
     *      The text label for the list item.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.classes='']
     *      Additionally style classes for the surrounding div.
     *  - {String} [options.name='']
     *      The name of the <select> element.
     *  - {String} [options.label='']
     *      The leading label for the <select> element. If empty, no label
     *      element will be created.
     *  - {Number} [options.size=0]
     *      The size of the <select> element (text lines). If omitted or set to
     *      the value 0, the list will appear as drop-down list.
     *  - {Boolean} [options.multipleSelection=false]
     *      Whether to make multiple entries selectable or only one.
     *
     * @returns {String}
     *  The HTML mark-up of the complete <select> element group.
     */
    Forms.createSelectBoxMarkup = function (entries, options) {

        var // classes, which will be added to the surrounding div-element
            classes = Utils.trimString(Utils.getStringOption(options, 'classes', '') + ' select-box'),

            // the label text
            textLabel   = Utils.getStringOption(options, 'label', false),
            // the cleaned/trimed label text for attributes like 'id', 'name', ...
            cleanLabel  = textLabel ? Utils.trimAndCleanString(textLabel.toLowerCase().replace(/\s+/g, '')) : false,

            // flag to decide whether a multiple selection should be allowed
            multiple    = Utils.getBooleanOption(options, 'multipleSelection', false) ? 'multiple' : '',
            // the name of the selectBox
            name        = Utils.getStringOption(options, 'name', false) || cleanLabel,
            // the size of the selectBox
            size        = Utils.getNumberOption(options, 'size', 0),

            // helper
            labelNode   = '',
            listNode    = '',
            listNodes   = null,

            // object which contains all selectBox attributes
            selectAttributes = {
                class: 'form-control',
                tabindex: Utils.getIntegerOption(options, 'tabindex', 0)
            };

        // collect list-elements
        listNodes = '';
        entries.forEach(function (entry) {
            var value = Utils.getStringOption(entry, 'value', false),
                label = Utils.getStringOption(entry, 'label', false);

            if (label && value) {
                listNodes += Forms.createElementMarkup('option', { attributes: { value: value }, content: label });
            }
        });

        // add list-elements to select-node
        if (cleanLabel) { selectAttributes.id = cleanLabel; }
        if (multiple) { selectAttributes.multiple = ''; classes += ' multiple'; }
        if (name) { selectAttributes.name = name.toLowerCase().replace(' ', ''); }
        if (size > 0) { selectAttributes.size = size; }
        listNode = Forms.createElementMarkup('select', { attributes: selectAttributes, content: listNodes });

        // create labelNode
        if (textLabel) { labelNode = Forms.createElementMarkup('label', { attributes: { for: cleanLabel }, content: textLabel }); }

        // add label + list to container and return it
        return Forms.createElementMarkup('div', { attributes: { class: classes }, content: labelNode + listNode });
    };

    /**
     * Creates a simple SpinField
     *   with leading and/or tailing label
     *
     * @param {Object} [options]
     *  Optional parameters. The following options are
     *  supported:
     *  - {Float} options.value
     *      Spin default value
     *  - {String} [options.classes='']
     *      Additionally style classes for the surrounding div.
     *  - {String} [options.label='']
     *      The label for the spinField. If empty, no label element will be
     *      created
     *  - {Boolean} [options.tail=false]
     *      The tailing label
     *  - {Float} [options.min=0.0]
     *      Spin min-value
     *  - {Float} [options.max=10.0]
     *      Spin max-value
     *  - {Float} [options.step=0.1]
     *      Spin step-value
     *  - {Boolean} [options.required=false]
     *      Is infill required
     *  - {String} [options.pattern]
     *      The validation pattern
     *
     * @returns {String}
     *   The HTM Lmark-up of the complete spin field element.
     */
    Forms.createSpinFieldMarkup = function (options) {

        var // classes, which will be added to the surrounding div-element
            classes = Utils.trimString(Utils.getStringOption(options, 'classes', '') + ' spin-field'),

            // the leading label
            labelNode = '',
            // the label text
            label       = Utils.getStringOption(options, 'label', false),
            // the cleaned/trimed label text for attributes like 'id', 'name', ...
            cleanLabel  = label ? Utils.trimAndCleanString(label.toLowerCase().replace(' ', '')) : false,

            // the tailing label
            tailNode = '',
            // the tailing text
            tail = Utils.getStringOption(options, 'tail', false),

            // the spin field node
            spinFieldNode = null,
            spinFieldAttributes = {
                min: Utils.getNumberOption(options, 'min', 0.0),
                max: Utils.getNumberOption(options, 'max', 10.0),
                step: Utils.getNumberOption(options, 'step', 0.1).toFixed(1),
                tabindex: Utils.getIntegerOption(options, 'tabindex', 0),
                type: Utils.getStringOption(options, 'type', 'number'),
                value: Utils.getNumberOption(options, 'value')
            };

        // prepare the spinField
        if (cleanLabel) { spinFieldAttributes.id = cleanLabel; }
        if (Utils.getBooleanOption(options, 'required', false)) { spinFieldAttributes.required = ''; }
        if (Utils.getStringOption(options, 'pattern')) { spinFieldAttributes.pattern = Utils.getStringOption(options, 'pattern'); }
        spinFieldNode = Forms.createElementMarkup('input', { attributes: spinFieldAttributes });

        // prepare leading label
        if (label) { labelNode = Forms.createElementMarkup('label', { attributes: { class: 'spin-field-label', for: cleanLabel }, content: label }); }

        // prepare tailing label
        if (tail) { tailNode = Forms.createElementMarkup('span', { attributes: { class: 'spin-field-tail' }, content: tail }); }

        return Forms.createElementMarkup('div', { attributes: { class: classes }, content: labelNode + spinFieldNode + tailNode });
    };

    /**
     * A special event listener function which works for touch and click events
     * and prevents unwanted behaviour from jQuery invoked events on touch devices.
     * It uses the 'faster' tap event on touch devices and a click otherwise.
     *
     * We are in a jQuery/jQuery mobile environment, so native touch events invoke a lot
     * of unnecessary events due to jQuery. The invoked events can even be different on
     * a longer tap or a short tap. There is also some odd behaviour when the
     * focus is changed while jQuery still invokes it's events.
     * All this could lead to unwanted behaviour (e.g. the famous jQuery 'ghost click',
     * or when a menu is closed on touchend, the following events are 'routed'
     * to the the underlying target).
     *

     * @param {Object} options
     *  - {jQuery} options.node
     *      The 'node' for the event listener.
     *
     *  Optional parameters. The following options are
     *  supported:
     *  - {jQuery} [options.delegate]
     *      The jQuery selector for delegation (see .on() function).
     *
     * @param {function} handler
     * The handler function which is attached to the event listener.
     *
     */
    Forms.touchAwareListener = function (options, handler) {

        // the node to which the eventl istener should be attached
        var node = options.node;
        // a possible jQuery selector for delegation
        var delegate = options.delegate;

        var listener = handler;

        if (Forms.DEFAULT_CLICK_TYPE === 'tap') {
            listener = function (e) {
                handler.call(this, e);
                e.preventDefault();
            };
        }

        if (delegate) {
            node.on(Forms.DEFAULT_CLICK_TYPE, delegate, listener);
        } else {
            node.on(Forms.DEFAULT_CLICK_TYPE, listener);
        }
    };

    // debugging --------------------------------------------------------------

    /**
     * Creates multi-line text mark-up from the passed value that can be set as
     * 'title' attribute at an HTML element.
     *
     * @param {Any} value
     *  Any value to be converted to the tooltip content string. Objects and
     *  arrays will be converted to text recursively.
     *
     * @returns {String}
     *  The tooltip content string, as encoded HTML mark-up.
     */
    Forms.createLoggingTooltipMarkup = function (value) {

        // the resulting mark-up, as array of strings
        var markup = [];
        // the current line indentation
        var indent = 0;

        function isAtomic(val) {
            return _.isNumber(val) || _.isBoolean(val) || _.isNull(val) || _.isUndefined(val);
        }

        function appendLine(text) {
            markup.push(Utils.repeatString('&nbsp; &nbsp; ', indent) + Utils.escapeHTML(text));
        }

        function appendProperties(object) {
            indent += 1;
            var keys = (object instanceof Error) ? Object.getOwnPropertyNames(object) : Object.keys(object);
            keys.sort().forEach(function (key) {
                appendValue(key.replace(/"/g, '\\"') + ': ', object[key]);
            });
            indent -= 1;
        }

        function appendElements(array) {
            indent += 1;
            array.forEach(function (element, index) {
                appendValue(index + ': ', element);
            });
            indent -= 1;
        }

        function appendValue(prefix, val) {
            if (markup.length > 500) { return; } // prevent endless loops in cyclic references
            if (isAtomic(val)) {
                appendLine(prefix + String(val));
            } else if (_.isString(val)) {
                val = val.split(/\n/);
                if (val.length === 1) {
                    appendLine(prefix + '"' + val[0].replace(/"/g, '\\"') + '"');
                } else {
                    appendLine(prefix + 'String [' + val.length + ']');
                    appendElements(val);
                }
            } else if (_.isRegExp(val)) {
                appendLine(prefix + val);
            } else if (_.isArray(val)) {
                if ((val.length <= 8) && val.every(isAtomic)) {
                    appendLine(prefix + '[' + val.map(String).join(', ') + ']');
                } else {
                    appendLine(prefix + 'Array [' + val.length + ']');
                    appendElements(val);
                }
            } else if (_.isObject(val)) {
                appendLine(prefix + (_.isFunction(val) ? ('Function [' + val.length + ']') : 'Object'));
                appendProperties(val);
            }
        }

        appendValue('', value);
        return markup.join('&#10;');
    };

    // exports ================================================================

    return Forms;

});
