/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialog/saveasdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialog/folderpickerdialog',
    'gettext!io.ox/office/tk/main'
], function (Utils, FolderPickerDialog, gt) {

    'use strict';

    // class SaveAsDialog =====================================================

    /**
     * A generic dialog to save a file in a Drive folder. This dialog provides
     * a folder picker and a simple text input.
     *
     * @constructor
     *
     * @extends FolderPickerDialog
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application that is
     *  creating the dialog, or an object with a method getWindowId() that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  FolderPickerDialog. Additionally, the following options are supported:
     *  - {String} [initOptions.value='']
     *      The initial value of the text field.
     *  - {String} [initOptions.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     */
    var SaveAsDialog = FolderPickerDialog.extend(function (windowId, initOptions) {

        // label for the save as input node
        var labelNode = $('<label for="save-as-filename">').text(gt('File name'));

        // the text field control
        var inputNode = $('<input type="text" id="save-as-filename" class="form-control" tabindex="0">').attr({
            placeholder: Utils.getStringOption(initOptions, 'placeholder', ''),
            value: Utils.getStringOption(initOptions, 'value', '')
        });

        // base constructor ---------------------------------------------------

        FolderPickerDialog.call(this, windowId, _.extend({
            okLabel: gt('Save'),
            addClass: 'save-as-folder-list',
            height: 350
        }, initOptions));

        // public methods -----------------------------------------------------

        /**
         * Returns the file name contained in the text input element.
         *
         * @returns {String}
         *  The file name contained in the text input element.
         */
        this.getFileName = function () {
            return $.trim(inputNode.val());
        };

        // initialization -----------------------------------------------------

        this.getFolderPicker().getNode().css({ height: 250 });

        // give this dialog a good class name
        this.getPopup().addClass('save-as-dialog');

        // add the folder tree and text field
        this.append(labelNode, inputNode);

        // focus and select the text in the input field
        var saveAsDialog = this,
            folderPicker = this.getFolderPicker();
        folderPicker.listenTo(folderPicker, 'select:item', function () {
            saveAsDialog.setFocusNode(inputNode);
            saveAsDialog.grabFocus();
        });

        // disable OK button if no file name is available
        this.addOkValidator(function () { return this.getFileName(); });

        // resolve dialog promise with folder identifier and file name
        this.setOkHandler(function () {
            return { folderId: this.getFolderId(), fileName: this.getFileName() };
        });

    }); // class SaveAsDialog

    // exports ================================================================

    return SaveAsDialog;

});
