/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/buttongroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/group'
], function (Utils, Forms, Group) {

    'use strict';

    // class ButtonGroup ======================================================

    /**
     * Creates a control that contains a set of buttons. This class does not
     * add any special behavior to the control. Used as base class for controls
     * such as radio groups, or check lists.
     *
     * Instances of this class trigger the following events:
     * - 'create:button'
     *      After a new option button has been created with the method
     *      ButtonGroup.createOptionButton(). Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} buttonNode
     *          The button element, as jQuery object.
     *      (3) {Any} value
     *          The value of the new button element.
     *      (4) {Object} [options]
     *          The options passed to the method
     *          ButtonGroup.createOptionButton().
     *
     * @constructor
     *
     * @extends Group
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application owning the
     *  button group object, or an object with a method 'getWindowId' that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class Group.
     *  Additionally, the following options are supported:
     *  - {Function} [initOptions.matcher=_.isEqual]
     *      A comparison function that returns whether a button element in this
     *      group should match a specific value, e.g. for selection. Receives
     *      an arbitrary value as first parameter, and the value of a button
     *      element as second parameter. The function must return the Boolean
     *      value true to match the respective button element, the Boolean
     *      value false to not match the button element, or any other value to
     *      skip the button element (e.g. to keep the current selection state
     *      of the button unmodified). If omitted, uses _.isEqual() which
     *      compares arrays and objects deeply, and does not skip any button
     *      element.
     *  - {Boolean|Function} [initOptions.sorted=false]
     *      If set to true or a callback function, the button elements will be
     *      sorted according to the custom sort order specified via the
     *      'sortIndex' options of each inserted button (see method
     *      ButtonGroup.createOptionButton() for details). The value true will
     *      cause to sort the elements by their natural order using the
     *      comparison operators of JavaScript. If a callback function has been
     *      passed instead, it will be used as comparator callback for
     *      JavaScript's Array.sort() method.
     *  - {Object} [initOptions.smallerVersion]
     *      A descriptor to define the behavior of this button group when it is
     *      asked to switch to/from a smaller representation.
     *  - {String} [initOptions.tooltip]
     *      Tool tip text shown when the mouse hovers the drop-down version of
     *      this button group.
     */
    function ButtonGroup(windowId, initOptions) {

        // self reference
        var self = this;

        // comparator for option buttons
        var matcher = Utils.getFunctionOption(initOptions, 'matcher', _.isEqual);

        // whether the buttons will be sorted
        var sorted = Utils.getBooleanOption(initOptions, 'sorted', false);

        // custom sort order
        var sortComparator = Utils.getFunctionOption(initOptions, 'sorted');

        // smaller version of group (to fit in smaller resolutions)
        var smallerVersion = Utils.getObjectOption(initOptions, 'smallerVersion');

        // base constructor ---------------------------------------------------

        Group.call(this, windowId, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Sorts all option buttons in this group debounced.
         */
        var sortOptionButtons = this.createDebouncedMethod('ButtonGroup.sortOptionButtons', null, function () {

            // all option buttons in this group
            var optionButtons = self.getOptionButtons().detach();

            // returns the sorting index of the passed button node
            function getSortIndex(node) {
                return Utils.getOption($(node).data('options'), 'sortIndex', 0);
            }

            // sort all buttons by their sorting index
            if (_.isFunction(sortComparator)) {
                optionButtons = optionButtons.get().sort(function (node1, node2) {
                    return sortComparator(getSortIndex(node1), getSortIndex(node2));
                });
            } else {
                optionButtons = _.sortBy(optionButtons.get(), getSortIndex);
            }

            self.getNode().append($(optionButtons));
        });

        /**
         * Modifies the button group for the drop-down view.
         *
         * @param {jQuery} buttonNode
         *  The drop-down button, as jQuery object.
         *
         * @param {Object} dropDownVersion
         *  The following options are supported:
         *  - {String} [dropDownVersion.label]
         *      If specified, the button group gets a parent label.
         */
        function createDropDownVersion(buttonNode, dropDownVersion) {
            if (_.has(dropDownVersion, 'label')) {
                buttonNode.find('.caption').append(
                    Forms.createSpanMarkup(dropDownVersion.label, {
                        classes: 'drop-down'
                    })
                );
            }
        }

        /**
         * Activates or deactivates a smaller version of the button group.
         *
         * @param {Boolean} state
         *  Whether the button group should be small or wide.
         */
        function toggleSmallVersion(state) {

            // an alternative control that is smaller than this group
            var altGroup = Utils.getOption(smallerVersion, 'pendant');

            if (_.isObject(altGroup)) {
                altGroup.getNode().toggleClass('hidden', !state);
                self.getNode().toggleClass('hidden', state);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns all option buttons as jQuery collection.
         *
         * @returns {jQuery}
         *  The jQuery collection with all existing option buttons.
         */
        this.getOptionButtons = function () {
            return this.getNode().children(Forms.OPTION_BUTTON_SELECTOR);
        };

        /**
         * Returns all option buttons that represent the specified value.
         *
         * @param {Any} value
         *  The value to filter the option buttons for.
         *
         * @returns {jQuery}
         *  The jQuery collection with all existing option buttons equal to the
         *  passed value.
         */
        this.findOptionButtons = function (value) {
            return Forms.filterButtonNodes(this.getOptionButtons(), value, { matcher: matcher });
        };

        /**
         * Removes all option buttons from this button group.
         *
         * @returns {ButtonGroup}
         *  A reference to this instance.
         */
        this.clearOptionButtons = function () {
            this.getOptionButtons().remove();
            this.layout();
            return this;
        };

        /**
         * Adds a new option button to this button group.
         *
         * @param {Any} value
         *  The unique value associated to the button. Must not be null.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all formatting options for button
         *  elements supported by the method Forms.createButtonMarkup().
         *  Additionally, the following options are supported:
         *  - {String} [options.dataValue]
         *      A string that will be inserted into the 'data-value' attribute
         *      of the button node. If omitted, the JSON string representation
         *      of the 'value' parameter will be used instead (all double-quote
         *      characters will be removed from the string though), unless the
         *      value is a function.
         *  - {Number|String} [options.sortIndex=0]
         *      Sorting index for the new button. All buttons will be sorted
         *      relatively to their index. Sorting will be stable, multiple
         *      buttons with the same index will remain in insertion order.
         *
         * @returns {ButtonGroup}
         *  A reference to this instance.
         */
        this.createOptionButton = function (value, options) {

            // the new button
            var buttonNode = $(Forms.createButtonMarkup(options)).addClass(Forms.OPTION_BUTTON_CLASS);
            // special options for a drop-down-version of this button
            var dropDownVersion = Utils.getObjectOption(options, 'dropDownVersion');

            // add the button value, and insert the button node
            Forms.setButtonValue(buttonNode, value, options);
            Forms.setToolTip(buttonNode, options);
            buttonNode.data('options', options);
            this.addChildNodes(buttonNode);

            if (dropDownVersion) {
                createDropDownVersion(buttonNode, dropDownVersion);
            }

            // sort the buttons debounced by their index
            if (sorted || sortComparator) { sortOptionButtons(); }

            // notify listeners
            this.trigger('create:button', buttonNode, value, options);
            this.refresh();
            return this;
        };

        /**
         * Overwrites the method of the base class Group to activate the small
         * version of this button group.
         *
         * @returns {ButtonGroup}
         *  A reference to this instance.
         */
        this.activateSmallVersion = function () {
            toggleSmallVersion(true);
            return this;
        };

        /**
         * Overwrites the method of the base class Group to deactivate the
         * small version of this button group.
         *
         * @returns {ButtonGroup}
         *  A reference to this instance.
         */
        this.deactivateSmallVersion = function () {
            toggleSmallVersion(false);
            return this;
        };

        // initialization -----------------------------------------------------

        // set a tool tip that will be used for the drop-down version of this control
        Forms.setToolTip(this.getNode(), initOptions);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = matcher = sortComparator = null;
        });

    } // class ButtonGroup

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: ButtonGroup });

});
