/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/view/controls', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/textframework/view/control/paragraphstylepicker',
    'io.ox/office/textframework/view/control/liststylepicker',
    'io.ox/office/textframework/view/control/changetrackbadge',
    'io.ox/office/textframework/view/control/drawingpositionpicker',
    'io.ox/office/textframework/view/control/commentfiltergroup',
    'io.ox/office/textframework/view/control/fieldformatlist',
    'gettext!io.ox/office/textframework/main'
], function (Utils, LineHeight, EditControls, Labels, ParagraphStylePicker, ListStylePicker, ChangeTrackBadge, DrawingPositionPicker, CommentFilterGroup, FieldFormatList, gt) {

    'use strict';

    // convenience shortcuts
    var Button = EditControls.Button;
    var RadioGroup = EditControls.RadioGroup;
    var RadioList = EditControls.RadioList;
    var CompoundButton = EditControls.CompoundButton;

    // static class TextControls ==============================================

    /**
     * Additional classes defining specialized GUI controls for the OX Text
     * application.
     *
     * @extends EditControls
     */
    var TextControls = _.extend({}, EditControls, {
        ParagraphStylePicker: ParagraphStylePicker,
        ChangeTrackBadge: ChangeTrackBadge,
        DrawingPositionPicker: DrawingPositionPicker,
        CommentFilterGroup: CommentFilterGroup,
        FieldFormatList: FieldFormatList
    });

    // class ParagraphAlignmentGroup ==========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentGroup = RadioGroup.extend({ constructor: function (docView, initOptions) {

        RadioGroup.call(this, docView, Utils.extendOptions({
            tooltip: Labels.PARA_ALIGNMENT_LABEL
        }, initOptions));

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ 'aria-label': Labels.PARA_ALIGNMENT_LABEL, 'aria-labelledby': 'dummy' });

        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, tooltip: style.label });
        }, this);

    } }); // class ParagraphAlignmentGroup

    // class ParaVertAlignmentGroup ==========================================

    /**
     * A picker control for vertical text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParaVertAlignmentGroup = RadioGroup.extend({ constructor: function (docView, initOptions) {

        RadioGroup.call(this, docView, Utils.extendOptions({
            tooltip: Labels.TEXT_VERT_ALIGNMENT_LABEL
        }, initOptions));

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ 'aria-label': Labels.TEXT_VERT_ALIGNMENT_LABEL, 'aria-labelledby': 'dummy' });

        _.each(Labels.PARA_VERT_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, tooltip: style.label });
        }, this);

    } }); // class ParaVertAlignmentGroup

    // class ParagraphAlignmentPicker =========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentPicker = RadioList.extend({ constructor: function (docView, initOptions) {

        RadioList.call(this, docView, Utils.extendOptions({
            icon: 'docs-para-align-left',
            tooltip: Labels.PARA_ALIGNMENT_LABEL,
            updateCaptionMode: 'icon'
        }, initOptions));

        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, label: style.label });
        }, this);

    } }); // class ParagraphAlignmentPicker

    /**
     * A picker control for vertical text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParaVertAlignmentPicker = RadioList.extend({ constructor: function (docView, initOptions) {

        RadioList.call(this, docView, Utils.extendOptions({
            icon: 'docs-cell-v-align-middle',
            label: Labels.TEXT_VERT_ALIGNMENT_LABEL,
            tooltip: Labels.TEXT_VERT_ALIGNMENT_TOOLTIP,
            updateCaptionMode: 'icon'
        }, initOptions));

        _.each(Labels.PARA_VERT_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, label: style.label });
        }, this);

    } }); // class ParaVertAlignmentPicker

    // class TextBoxAlignmentPicker ===========================================

    /**
     * A picker control for horizontal and vertical alignment of paragraphs in
     * text boxes.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.TextBoxAlignmentPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        CompoundButton.call(this, docView, Utils.extendOptions({
            icon: 'docs-para-align-left',
            tooltip: gt('Paragraph alignment'),
            updateCaptionMode: 'none'
        }, initOptions));

        var horizontalGroup = new RadioGroup(docView);
        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            horizontalGroup.createOptionButton(style.value, { icon: style.icon, label: style.label });
        });
        this.addSectionLabel(gt('Horizontal alignment'));
        this.addGroup('paragraph/alignment', horizontalGroup);

        var verticalGroup = new RadioGroup(docView);
        _.each(Labels.PARA_VERT_ALIGNMENT_STYLES, function (style) {
            verticalGroup.createOptionButton(style.value, { icon: style.icon, label: style.label });
        });
        this.addSectionLabel(gt('Vertical alignment'));
        this.addGroup('drawing/verticalalignment', verticalGroup);

    } }); // class TextBoxAlignmentPicker

    // class TableAlignmentPicker ===========================================

    /**
     * A picker control for horizontal and vertical alignment of paragraphs in
     * text boxes.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.TableAlignmentPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        CompoundButton.call(this, docView, Utils.extendOptions({
            icon: 'docs-para-align-left',
            tooltip: gt('Paragraph alignment'),
            updateCaptionMode: 'none'
        }, initOptions));

        var horizontalGroup = new RadioGroup(docView);
        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            horizontalGroup.createOptionButton(style.value, { icon: style.icon, label: style.label });
        });
        this.addSectionLabel(gt('Horizontal alignment'));
        this.addGroup('paragraph/alignment', horizontalGroup);

        var verticalGroup = new RadioGroup(docView);
        _.each(Labels.PARA_VERT_ALIGNMENT_STYLES, function (style) {
            verticalGroup.createOptionButton(style.value, { icon: style.icon, label: style.label });
        });
        this.addSectionLabel(gt('Vertical alignment'));
        this.addGroup('table/verticalalignment', verticalGroup);

    } }); // class TableAlignmentPicker

    // class ParagraphSpacingPicker ===========================================

    /**
     * A picker control for line height inside paragraphs, and spacing between
     * paragraphs.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.ParagraphSpacingPicker = CompoundButton.extend({ constructor: function (docView) {

        CompoundButton.call(this, docView, {
            icon: 'docs-para-line-spacing-100',
            tooltip: Labels.PARAGRAPH_AND_LINE_SPACING_LABEL,
            dropDownVersion: { label: Labels.PARAGRAPH_AND_LINE_SPACING_LABEL }
        });

        var lineHeightGroup = new RadioGroup(docView, { dropDownVersion: { label: Labels.LINE_SPACING_LABEL } });
        lineHeightGroup.createOptionButton(LineHeight.SINGLE,   { icon: 'docs-para-line-spacing-100', label: /*#. text line spacing in paragraphs */ gt('100%') });
        lineHeightGroup.createOptionButton(LineHeight._115,     { icon: 'docs-para-line-spacing-115', label: /*#. text line spacing in paragraphs */ gt('115%') });
        lineHeightGroup.createOptionButton(LineHeight.ONE_HALF, { icon: 'docs-para-line-spacing-150', label: /*#. text line spacing in paragraphs */ gt('150%') });
        lineHeightGroup.createOptionButton(LineHeight.DOUBLE,   { icon: 'docs-para-line-spacing-200', label: /*#. text line spacing in paragraphs */ gt('200%') });
        this.addSectionLabel(Labels.LINE_SPACING_LABEL);
        this.addGroup('paragraph/lineheight', lineHeightGroup);

        var spacingGroup = new RadioGroup(docView, { dropDownVersion: { label: Labels.PARAGRAPH_SPACING_LABEL } });
        spacingGroup.createOptionButton(0, { label: /*#. no distance between selected paragraphs */       gt.pgettext('paragraph-spacing', 'None') });
        spacingGroup.createOptionButton(1, { label: /*#. a normal distance between selected paragraphs */ gt.pgettext('paragraph-spacing', 'Normal') });
        spacingGroup.createOptionButton(2, { label: /*#. a large distance between selected paragraphs */  gt.pgettext('paragraph-spacing', 'Wide') });
        this.addSectionLabel(Labels.PARAGRAPH_SPACING_LABEL);
        this.addGroup('paragraph/spacing', spacingGroup);

    } }); // class ParagraphSpacingPicker

    // class TextPositionGroup ================================================

    /**
     * A button group control for text position (subscript/superscript).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.TextPositionGroup = RadioGroup.extend({ constructor: function (docView, initOptions) {

        RadioGroup.call(this, docView, _.extend({
            tooltip: Labels.TEXT_POSITION_LABEL,
            toggleValue: 'baseline'
        }, initOptions));

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ 'aria-label': Labels.TEXT_POSITION_LABEL, 'aria-labelledby': 'dummy' });

        this.createOptionButton('sub',   { icon: 'docs-font-subscript',   tooltip: gt('Subscript'),   dropDownVersion: { label: gt('Subscript') } });
        this.createOptionButton('super', { icon: 'docs-font-superscript', tooltip: gt('Superscript'), dropDownVersion: { label: gt('Superscript') } });

    } }); // class TextPositionGroup

    // class BulletListStylePicker ============================================

    TextControls.BulletListStylePicker = ListStylePicker.extend({ constructor: function (docView, initOptions) {

        ListStylePicker.call(this, docView, docView.getDocModel().getPredefinedBulletListStyles(), Utils.extendOptions({
            icon: 'docs-list-bullet',
            tooltip: gt('Bullet list'),
            gridColumns: 4,
            dropDownVersion: { label: gt('Bullet list') }
        }, initOptions));

    } }); // class BulletListStylePicker

    // class NumberedListStylePicker ==========================================

    TextControls.NumberedListStylePicker = ListStylePicker.extend({ constructor: function (docView, initOptions) {

        var predefinedStyles = docView.getDocModel().getPredefinedNumberedListStyles();
        var gridColumns = _.where(predefinedStyles, { format: 'decimal' }).length || 4; // using default value 4 (53055)

        ListStylePicker.call(this, docView, predefinedStyles, Utils.extendOptions({
            icon: 'docs-list-numbered',
            tooltip: gt('Numbered list'),
            gridColumns: gridColumns,
            dropDownVersion: { label: gt('Numbered list') }
        }, initOptions));

    } }); // class NumberedListStylePicker

    // class UserFilterPicker =================================================

    /**
     * Drop-down menu control for setting a user filter for the comments in the
     * comments layer.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.UserFilterPicker = CompoundButton.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, {
            label: gt('Comment authors'),
            tooltip: gt('Show comment authors'),
            anchorBorder: 'right left bottom top',
            updateCaptionMode: 'icon',
            smallerVersion: { hideLabel: true }
        });

        // initialization -----------------------------------------------------

        this.addGroup('comment/authorFilter', new CommentFilterGroup(docView), { sticky: true });

    } }); // class UserFilterPicker

    // class CommentDisplayModePicker ============================================

    /**
     * Drop-down menu control for modifying the highlight mode and visibility
     * of the comments in the comments layer.
     * Additionally a list with all comment authors can be used to filter
     * specific authors.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.CommentDisplayModePicker = CompoundButton.extend({ constructor: function (docView, options) {

        var smallDevice = Utils.getBooleanOption(options, 'smallDevice', false);

        // the group of all display mode buttons
        var displayModeGroup = new RadioGroup(docView);

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, {
            icon: 'docs-comment-show',
            label: gt('Markup'),
            tooltip: gt('Highlight comments mode'),
            updateCaptionMode: 'icon',
            smallerVersion: { hideLabel: true }
        });

        // private methods ----------------------------------------------------

        /**
         * Updates the icon after modifying the display mode.
         */
        function displayModeUpdateHandler(newValue, oldValue) {

            if (newValue === oldValue) {  }

            // TODO: Modify the icon
        }

        // initialization -----------------------------------------------------

        if (!smallDevice) {
            displayModeGroup
                .createOptionButton('all',      { label: gt('Highlight all comments') })
                .createOptionButton('selected', { label: gt('Highlight selected comments') });
        }

        displayModeGroup
            .createOptionButton('bubbles',  { label: gt('Show bubble comments') })
            // .createOptionButton('none',  { label: gt('Highlight no comments') })
            .createOptionButton('hidden',   { label: gt('Show no comments') });

        displayModeGroup.registerUpdateHandler(displayModeUpdateHandler);

        this.addGroup('comment/displayMode', displayModeGroup)
            .addSeparator()
            // adding the button, that contains a list of all users who created comments in the document
            .addGroup(null, new TextControls.UserFilterPicker(docView));

    } }); // class CommentDisplayModePicker

    // class CommentDisplayViewPicker ============================================

    /**
     * Drop-down menu control for modifying the view of the comments layer. This is
     * only planned for testing purposes and can be removed in release version.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.CommentDisplayViewPicker = RadioList.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, docView, {
            icon: 'fa-cogs',
            label: _.noI18n('Comment view'),
            tooltip: _.noI18n('Set different comment view mode'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this.createOptionButton('modern',     { icon: 'fa-indent',  label: _.noI18n('Modern') })
            .createOptionButton('avantgard',  { icon: 'fa-outdent', label: _.noI18n('Avantgard') })
            .createOptionButton('classic',    { icon: 'fa-list-alt',  label: _.noI18n('Classic') });

    } }); // class CommentDisplayViewPicker

    // class AnchoragePicker ============================================

    /**
     *
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.AnchoragePicker = RadioList.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, docView, {
            icon: 'docs-position-anchor',
            label: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Alignment'),
            tooltip: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Drawing alignment'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this.createOptionButton('inline',     { icon: 'docs-position-on-text',         label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with text') })
            .createOptionButton('paragraph',  { icon: 'docs-position-on-paragraph',    label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with paragraph') })
            .createOptionButton('margin',     { icon: 'docs-position-on-page-margin',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with page margin') })
            .createOptionButton('page',       { icon: 'docs-position-on-page',         label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with page') });
    } }); // class AnchoragePicker

    // class InTextStyleContextSubMenu ========================================

    /**
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.InTextStyleContextSubMenu = CompoundButton.extend({ constructor: function (docView, initOptions) {

        var self = this;
        var docModel = docView.getDocModel();
        var changeStyleButton = null;

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions({
            label: Labels.PARAGRAPH_STYLE_LABEL
        }, initOptions));

        // initialization -----------------------------------------------------

        this.addGroup('paragraph/createstylesheet', new Button(docView, { label: Labels.CREATE_STYLE_LABEL, tooltip: Labels.CREATE_STYLE_LABEL_TOOLTIP }));

        this.on('group:focus', function () {
            var styleId = docModel.getAttributes('paragraph').styleId;

            //getTranslatedNameByStyleId call of StyleSheetMixin
            changeStyleButton = new Button(docView, {
                label: gt('Change style "%1$s"', docModel.getTranslatedNameByStyleId(styleId)),
                tooltip: Labels.CHANGE_STYLE_LABEL_TOOLTIP,
                value: styleId
            });

            self.addGroup('paragraph/changestylesheet', changeStyleButton);
        });

        this.on('group:blur', function () {
            if (changeStyleButton) {
                self.removeGroup(changeStyleButton);
                changeStyleButton = null;
            }
        });

    } }); // class InTextStyleContextSubMenu

    // class InsertFieldPicker =================================================

    /**
     * Control to pick field from dropdown
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.InsertFieldPicker = CompoundButton.extend({ constructor: function (docView, options) {

        CompoundButton.call(this, docView, Utils.extendOptions(options, {
            label: /*#. button to insert field in document */ gt('Field'),
            tooltip: gt('Insert field'),
            icon: 'docs-field',
            smallerVersion: { hideLabel: true }
        }));

        this.addGroup('document/insertfield', new Button(docView, { value: 'pagenumber',    label: /*#. insert page number */ gt('Page number') }))
            .addGroup('document/insertfield', new Button(docView, { value: 'numpages',      label: /*#. insert number of pages in document */ gt('Page count') }))
            .addGroup('document/insertfield', new Button(docView, { value: 'date',          label: /*#. insert current date and time */ gt('Date & time') }))
            .addGroup('document/insertfield', new Button(docView, { value: 'filename',      label: /*#. insert document name */ gt('Document name') }))
            .addGroup('document/insertfield', new Button(docView, { value: 'author',        label: /*#. insert author's name */ gt('Author name') }));

    } }); // class InsertFieldPicker

    // class TocPicker =================================================

    /**
     * Control to pick Table of contents template from dropdown
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.TocPicker = RadioList.extend({ constructor: function (docView, initOptions) {

        RadioList.call(this, docView, Utils.extendOptions({
            icon: 'docs-table-of-contents',
            label: /*#. button to insert table of contents in document */ gt('Table of contents'),
            tooltip: gt('Insert table of contents'),
            itemDesign: 'grid',
            gridColumns: 1,
            smallerVersion: {
                hideLabel: true
            }
        }, initOptions));

        this.getMenuNode().addClass('toc-picker');

        this.createOptionButton(1, { icon: 'text-toc1', tooltip: /*#. insert table of contents template with dots and page numbers */   gt('With dots and page numbers') })
            .createOptionButton(2, { icon: 'text-toc2', tooltip: /*#. insert table of contents template with page numbers */            gt('With page numbers') })
            .createOptionButton(3, { icon: 'text-toc3', tooltip: /*#. insert table of contents template with headings text only */      gt('Headings only') });

    } }); // class TocPicker

    // class PresentationDateNumberPicker =====================================

    /**
     * Control to pick field from dropdown
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.PresentationDateNumberPicker = CompoundButton.extend({ constructor: function (docView, options) {

        CompoundButton.call(this, docView, Utils.extendOptions(options, {
            label: /*#. button to insert slide number or date */ gt('Field'),
            tooltip: gt('Insert field'),
            icon: 'docs-field',
            smallerVersion: { hideLabel: true }
        }));

        this.addGroup('document/insertfield', new Button(docView, { value: 'slidenum', label: /*#. insert slide number */ gt('Slide number') }))
            .addGroup('document/insertfield', new Button(docView, { value: 'datetime', label: /*#. insert current date and time */ gt('Date & time') }));

    } }); // class PresentationDateNumberPicker

    // exports ================================================================

    return TextControls;

});
