/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/textframework/utils/textutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom'
], function (Utils, Logger, Rectangle, AttributeUtils, DOM) {

    'use strict';

    // static class TextUtils ================================================

    // copy tk/utils methods
    var TextUtils = _.clone(Utils);

    // hangul for korean inputs
    var HANGUL = new RegExp('[\u1100-\u11FF|\u3130-\u318F|\uA960-\uA97F|\uAC00-\uD7AF|\uD7B0-\uD7FF]');

    // a logger bound to a specific URL option
    var logger = new Logger({ prefix: 'TEXT', enable: 'office:log-textutils' });

    // supported locales with the default format
    TextUtils.DEFAULT_PAPER_FORMAT = {
        en_US: 'Letter',
        en_GB: 'A4',
        de_DE: 'A4',
        fr_FR: 'A4',
        fr_CA: 'Letter',
        es_ES: 'A4',
        es_MX: 'Letter',
        nl_NL: 'A4',
        pl_PL: 'A4',
        ja_JP: 'A4',
        it_IT: 'A4',
        zh_CN: 'A4',
        zh_TW: 'A4',
        hu_HU: 'A4',
        sk_SK: 'A4',
        cs_CZ: 'A4',
        lv_LV: 'A4',
        ro_RO: 'A4',
        pt_BR: 'Letter',
        sv_SE: 'A4',
        da_DK: 'A4',
        ru_RU: 'A4'
    };

    /**
     * Characters which starts a composition on the Mac as they can be used to create composed characters
     */
    TextUtils.MAC_COMPOSE_CHARS = ['^', '~', '\xb4', '`'];

    /**
     * Default background attributes for drawing.
     */
    TextUtils.NO_DRAWING_BACKGROUND = { fill: { type: 'none', color: null, color2: null, gradient: null, bitmap: null, pattern: null } };

    /**
     * Default background attributes for slide.
     */
    TextUtils.NO_SLIDE_BACKGROUND = { fill: { type: null, color: null, color2: null, gradient: null, bitmap: null, pattern: null } };

    // default paperformats of the supported locales
    var PAPERSIZE_BY_FORMAT = {
        Letter: { width: 21590, height: 27940 },
        A4: { width: 21000, height: 29700 }
    };

    // private methods ----------------------------------------------------------------

    /**
     * sort assigned drawing by their 'layer-order' attribute
     */
    function sortDrawings(drawings) {
        function getLayerOrder(el) { return parseInt(el.getAttribute('layer-order'), 10); }
        drawings.sort(function (a, b) { return getLayerOrder(a) - getLayerOrder(b); });
        return drawings;
    }

    /**
     * Calculating a specified margin into a specified rectangle with the
     * properties 'top', 'bottom', 'left', 'right', 'width' and 'height'. The
     * values in both objects 'clientRect' and 'margins' must use the same
     * unit, but the values itself must be numbers without unit.
     *
     * @param {Object} clientRect
     *  The rectangle to be expanded.
     *
     * @param {Object} margins
     *  The margins object with the properties 'marginLeft', 'marginRight',
     *  'marginTop' and 'marginBottom'.
     *
     * @returns {Rectangle}
     *  The expanded rectangle object.
     */
    function expandRectangleWithMargin(clientRect, margins) {
        return new Rectangle(
            clientRect.left - margins.marginLeft,
            clientRect.top - margins.marginTop,
            clientRect.width + margins.marginLeft + margins.marginRight,
            clientRect.height + margins.marginTop + margins.marginBottom
        );
    }

    // methods ----------------------------------------------------------------

    /**
     * Searches inside a specified text string for any character, that is
     * included in a list of characters. So this function is an expanded
     * version of the indexOf() method. Additionally it is possible to define a
     * start index inside the text string and to define the search direction.
     * The search in the text string stops immediately, when the first
     * specified character from the character list is found.
     *
     * TODO: This function can probably be performance improved by using a
     * regular expression.
     *
     * @param {String} text
     *  The text string, in which the characters will be searched.
     *
     * @param {Number} pos
     *  The start position, at which the search will begin.
     *
     * @param {Array<String>} charList
     *  The list of characters, that will be searched inside the string.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.reverse=false]
     *      Whether the search shall happen in reverse order, meaning from the
     *      end of the string to the beginning. Default is, that the string is
     *      searched from beginning to end.
     *
     * @returns {Number}
     *  The index of the first found character specified in the character list
     *  inside the specified text string. If no character from the list was
     *  found, '-1' is returned.
     */
    TextUtils.indexOfValuesInsideTextSpan = function (text, pos, charList, options) {

        var // whether at least one character in the specified char list was found
            found = false,
            // the length of the specified text string
            length = 0,
            // whether the search shall be done in reverse direction
            reverse = Utils.getBooleanOption(options, 'reverse', false);

        if (reverse) {
            while (pos >= 0 && !_.contains(charList, text[pos])) { pos--; }
            found = (pos >= 0);
        } else {
            length = text.length;
            while (pos < length && !_.contains(charList, text[pos])) { pos++; }
            found = (pos < length);
        }

        return found ? pos : -1;
    };

    /**
     * Tries to merge the passed text span with its next or previous sibling
     * text span. To be able to merge two text spans, they must contain equal
     * formatting attributes. If merging was successful, the sibling span will
     * be removed from the DOM.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM node to be merged with its sibling text span. If this object is
     *  a jQuery object, uses the first DOM node it contains.
     *
     * @param {Boolean} next
     *  If set to true, will try to merge with the next span, otherwise with
     *  the previous text span.
     */
    TextUtils.mergeSiblingTextSpans = function (node, next) {

        var // the sibling text span, depending on the passed direction
            sibling = null,
            // text in the passed and in the sibling node
            text = null, siblingText = null;

        // passed node and sibling node, as DOM nodes
        node = Utils.getDomNode(node);
        sibling = node[next ? 'nextSibling' : 'previousSibling'];

        // both nodes must be text spans with the same attributes
        if (sibling && DOM.isTextSpan(node) && DOM.isTextSpan(sibling) && AttributeUtils.hasEqualElementAttributes(node, sibling)) {

            // add text of the sibling text node to the passed text node
            text = node.firstChild.nodeValue;
            siblingText = sibling.firstChild.nodeValue;
            node.firstChild.nodeValue = next ? (text + siblingText) : (siblingText + text);

            // remove the entire sibling span element
            $(sibling).remove();
        }
    };

    /**
     * Returns the absolute CSS position of the passed DOM.Point, relative to a
     * specific root node and zoom factor.
     *
     * Info: Please use 'Position.getPixelPositionFromDomPoint' from the OX Text
     * pixel API.
     *
     * @param {DOM.Point} point
     *  the point object, from which the CSS position should be calculated
     *
     * @param {jQuery| HTMLElement} rootNode
     *  the calculated CSS positions are relative to this root node.
     *
     * @param {Number} zoomFactor
     *  the current active zoom factor of the application view.
     *
     * @returns {Object}
     *  with the following CSS position properties:
     *  - {Number} top
     *  - {Number} left
     */
    TextUtils.getCSSPositionFromPoint = function (point, rootNode, zoomFactor) {

        if (!point) { return null; }

        var caretSpan = $('<span>').text('|'),
            cursorElement = $(point.node.parentNode),
            cursorElementLineHeight = parseFloat(cursorElement.css('line-height')),
            zoom = zoomFactor / 100;

        // break cursor element on the text offset
        if (point.offset === 0) {
            cursorElement.before(caretSpan);
        } else {
            if (DOM.isSpan(cursorElement)) {
                DOM.splitTextSpan(cursorElement, point.offset, { append: true });
            }
            cursorElement.after(caretSpan);
        }

        // create caret overlay and calculate its position
        var caretTop = (caretSpan.offset().top - rootNode.offset().top) / zoom  - cursorElementLineHeight  + caretSpan.outerHeight(),
            caretLeft = (caretSpan.offset().left - rootNode.offset().left) / zoom;

        // restore original state of document
        caretSpan.remove();

        if (point.offset > 0) { TextUtils.mergeSiblingTextSpans(cursorElement, true); }

        return { top: caretTop, left: caretLeft };
    };

    /**
     * Returns the absolute CSS position of the passed DOM.Point, relative to a
     * specific root node and zoom factor.
     *
     * Info: Please use 'Position.getPixelPositionFromDomPoint' from the OX Text
     * pixel API.
     *
     * @param {DOM.Point} point1
     *  the start point object, from which the CSS position should be calculated
     * @param {DOM.Point} point2
     *  the end point object, from which the CSS position should be calculated
     *
     * @param {jQuery| HTMLElement} rootNode
     *  the calculated CSS positions are relative to this root node.
     *
     * @param {Number} zoom
     *  the current active zoom factor of the application view.
     *
     * @param {Boolean} start
     *
     * @returns {Object}
     *  with the following CSS position properties:
     *  - {Number} top
     *  - {Number} left
     */
    TextUtils.getCSSRectangleFromPoints = function (point1, point2, rootNode, zoom, start) {
        var range = document.createRange();
        var rect;

        range.setStart(point1.node, point1.offset);
        if (point2) {
            range.setEnd(point2.node, point2.offset);
        } else {
            if (point1.offset + 1 <= point1.node.length) {
                range.setEnd(point1.node, point1.offset + 1);
            } else if (!start && point1.node.length > 0) {
                range.setStart(point1.node, point1.offset - 1);
                range.setEnd(point1.node, point1.offset);
                rect = range.getBoundingClientRect();
                rect = {
                    top: rect.top,
                    left: rect.right,
                    height: rect.height,
                    width: 0
                };
            } else {
                var nextNode = point1.node.nextSibling;
                if (nextNode) {
                    range.setEnd(nextNode, 0);
                } else {
                    var left = $(point1.node.parentNode).offset().left + $(point1.node.parentNode).width() * zoom;
                    rect = {
                        top: $(point1.node.parentNode).offset().top,
                        left: left,
                        height: $(point1.node.parentNode).height() * zoom,
                        width: 0
                    };
                }
            }
        }

        rect = rect ? rect : range.getBoundingClientRect();

        return { top: (rect.top - rootNode.offset().top) / zoom, left: (rect.left - rootNode.offset().left) / zoom, height: rect.height / zoom, width: rect.width / zoom };
    };

    /**
     * Returns whether the passed intervals overlap with at least one index.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed intervals are overlapping.
     */
    TextUtils.intervalOverlapsInterval = function (interval1, interval2) {
        return (interval1.first <= interval2.last) && (interval2.first <= interval1.last);
    };

    /**
     * Returns an array without 'falsy' elements.
     *
     * @param {Array} array
     *  The array, in that 'falsy' elements will be removed.
     *
     * @returns {Array}
     *  An array without 'falsy' elements.
     */
    TextUtils.removeFalsyItemsInArray = function (array) {
        return _.filter(array, function (item) {
            return item;
        });
    };

    /**
     * Calculating an internal user id, that is saved at the node as attribute
     * and saved in the xml file. The base for this id is the app server id.
     *
     * @param {Number} id
     *  The user id used by the app suite server.
     *
     * @returns {Number}
     *  The user id, that is saved in xml file and saved as node attribute.
     */
    TextUtils.calculateUserId = function (id) {
        return _.isNumber(id) ? (id + 123) * 10 - 123 : null;
    };

    /**
     * Calculating the app suite user id from the saved and transported user id.
     *
     * @param {Number} id
     *  The user id, that is saved in xml file and saved as node attribute.
     *
     * @returns {Number}
     *  The user id used by the app suite server.
     */
    TextUtils.resolveUserId = function (id) {
        return _.isNumber(id) ? ((id + 123) / 10 - 123) : null;
    };

    /**
     * Helper function to get the 'character' attributes from a first empty text
     * span in a paragraph. This is used to restore the character attributes at a
     * paragraph, before the paragraph is removed or merged. Then the correct undo
     * operation can be generated.
     *
     * @param {HTMLElement|jQuery} element
     *
     * @returns {Object|Null}
     *  The character attributes of an first empty text span. Or null if this does not
     *  exist.
     */
    TextUtils.getCharacterAttributesFromEmptyTextSpan = function (element) {

        var // the first text span inside the paragraph
            firstSpan = null,
            // the character attributes
            characterAttributes = null;

        if (DOM.isParagraphNode(element)) {

            // saving character attributes from empty text spans at paragraph
            firstSpan = DOM.findFirstPortionSpan(element);
            if (DOM.isEmptySpan(firstSpan) && ('character' in AttributeUtils.getExplicitAttributes(firstSpan))) {
                characterAttributes = AttributeUtils.getExplicitAttributes(firstSpan).character;
            }
        }

        return characterAttributes;
    };

    /**
     * Checks, whether two directly following insertText operations can be merged. This merging
     * happens for two reasons:
     * 1. Create undo operations, that remove complete words, not only characters.
     * 2. Reduce the number of operations before sending them to the server.
     *
     * The merge is possible under the following circumstances:
     * 1. The next operation has no attributes AND the previous operation is NOT a
     *    change tracked operation
     * OR
     * 2. The next operation has only the 'changes' attribute (no other attributes
     *    allowed) and the previous operation has the same 'changes' attribute.
     *    In this comparison not the complete 'changes' attribute is compared, but
     *    only the 'author' property of the 'inserted' object. The date is ignored.
     *
     * @param {Object} lastOperation
     *  The preceeding operation that will be extended if possible.
     *
     * @param {Object} nextOperation
     *  The following operation that will be tried to merge into the preceeding
     *  operation.
     *
     * @returns {Boolean}
     *  Whether the two (insertText) operations can be merged successfully.
     */
    TextUtils.canMergeNeighboringOperations = (function () {

        // Checking, if a specified operation is a change track operation.
        function isChangeTrackInsertOperation(operation) {
            return _.isObject(operation.attrs) && _.isObject(operation.attrs.changes);
        }

        // returns the author of an 'insert' change action from the 'changes' attribute of the passed operation
        function getInsertAuthor(operation) {
            return operation.attrs && operation.attrs.changes && operation.attrs.changes.inserted && operation.attrs.changes.inserted.author;
        }

        // Checking, if two following operations can be merged, although the next operation
        // contains an attribute. Merging is possible, if the attribute object only contains
        // the 'changes' object added by change tracking. If the previous operation contains
        // the same changes object, a merge is possible.
        function isValidNextChangesAttribute(lastOperation, nextOperation) {

            // extract the change authors from both operations
            var lastAuthor = getInsertAuthor(lastOperation);
            var nextAuthor = getInsertAuthor(nextOperation);

            // if the authors are identical, the operations can be merged (ignoring the time)
            // (the 'changes' attribute family must be the only one in the next operation!)
            return _.isString(lastAuthor) && _.isString(nextAuthor) && (lastAuthor === nextAuthor) && (_.keys(nextOperation.attrs).length === 1);
        }

        // return public method from local scope
        return function (lastOperation, nextOperation) {
            // 1. There are no attributes in the next operation AND the previous operation was not a change track operation OR
            // 2. Both are change track operations and have valid change track attributes
            return ((!('attrs' in nextOperation) && !isChangeTrackInsertOperation(lastOperation)) || isValidNextChangesAttribute(lastOperation, nextOperation));
        };
    }());

    /**
     * Returns an Array of all absolute drawings in assigned rootNode,
     * this included anchored Drawing on the page it self and in pages paragraphs
     *
     * @param {JQuery} rootNode
     *  parent node of the drawings (header or footer)
     *
     * @return {Array} Array of HTMLElements
     *  returns a sorted Array all absolute drawings in assigned rootNode,
     */
    TextUtils.getAllAbsoluteDrawingsOnNode = logger.profileMethod('TextUtils.getAllAbsoluteDrawingsOnNode()', function (targetRootNode) {

        var subSelector = '.drawing.absolute';
        var selector = '>.textdrawinglayer >' + subSelector + ', >.marginalcontent>.p ' + subSelector;

        return sortDrawings(targetRootNode.find(selector));
    });

    /**
     * Returns an Array of all absolute drawings in text-app node,
     * this included anchored Drawing on the page it self and in pages paragraphs
     *
     * @param {JQuery} targetRootNode
     *  Node of the current text-app
     *
     * @param {Number} pageNumber
     *  pageNumbers begin at 1
     *
     * @return {Array} Array of HTMLElements
     *  returns a sorted Array all absolute drawings in textdrawinglayer and textcontent,
     */
    TextUtils.getAllAbsoluteDrawingsOnPage = logger.profileMethod('TextUtils.getAllAbsoluteDrawingsOnPage()', function (targetRootNode, pageNumber) {
        var subSelector = '.drawing.absolute:not(.grouped)';
        var selector = '>.textdrawinglayer >' + subSelector + '[page-number="' + pageNumber + '"]';

        var allPageDrawings = targetRootNode.find(selector);

        var pageBreaks = targetRootNode.find('.page-break');

        var fromPB =  $(pageBreaks[pageNumber - 2]);
        var toPB =  $(pageBreaks[pageNumber - 1]);
        var allParas = null;

        if (!fromPB.length || !toPB.length) {
            allParas = targetRootNode.find('>.pagecontent').children();
        }

        if (!fromPB.length) {
            fromPB = $(allParas[0]);
        } else if (fromPB.parent().is('.p')) {

            var nextAll = fromPB.nextAll().find(subSelector);
            allPageDrawings = allPageDrawings.add(nextAll);

            fromPB = fromPB.parent();
        }

        if (!toPB.length) {
            toPB = $(_.last(allParas));
        } else if (toPB.parent().is('.p')) {

            var prevAll = toPB.prevAll().find(subSelector);
            allPageDrawings = allPageDrawings.add(prevAll);

            toPB = toPB.parent();
        }

        var allBetween = fromPB.nextUntil(toPB).add(fromPB).add(toPB).find(subSelector);
        allPageDrawings = allPageDrawings.add(allBetween);

        return sortDrawings(allPageDrawings);
    });

    /**
     * returns an Array of HTMLElements whose Bounding Rectangle intersect the Bounding Rectangle of assigned compare list
     * exclusive the source Element
     *
     * @param {HTMLElement} source
     *  element to compare with
     *
     * @param {Array} compareList
     *  Array of HTMLElements for comparing their BoundingRectangles
     *
     * @param {Object} options
     *  Optional parameters:
     *  @param {String} [options.marginKey=null]
     *      If this key is defined, an optional margin will be added to
     *      the rectangle calculated by the function 'getBoundingClientRect'.
     *      The information about the margins must be specified at the
     *      elements in a jQuery data object using the 'marginKey' as key.
     *      In this data object the properties 'marginTop', 'marginBottom',
     *      'marginLeft' and 'marginRight' must be specified.
     *
     * @return {Array} Array of HTMLElements
     *  returns all elements which intersect with source element
     *  exclusive the source element
     */
    TextUtils.findAllIntersections = logger.profileMethod('TextUtils.findAllIntersections()', function (source, compareList, options) {

        // the jQuerified source element
        var $source = $(source);
        // the rectangle around the source element
        var sourceRect = source.getBoundingClientRect();
        // the key name in the data object, that contain the margin size in pixel
        var marginKey = Utils.getStringOption(options, 'marginKey', null);
        // the result array
        var result = [];

        // ensure to have an instance of the class Rectangle
        if (marginKey && $source.data(marginKey)) {
            sourceRect = expandRectangleWithMargin(sourceRect, $source.data(marginKey));
        } else {
            sourceRect = Rectangle.from(sourceRect);
        }

        _.each(compareList, function (compare) {
            if (compare === source) { return; }

            // the rectangle around the compare element
            var compareRect = compare.getBoundingClientRect();

            if (marginKey && $(compare).data(marginKey)) {
                compareRect = expandRectangleWithMargin(compareRect, $(compare).data(marginKey));
            }

            if (sourceRect.overlaps(compareRect)) {
                result.push(compare);
            }
        });

        return result;
    });

    /**
     * sets assigned Drawings list z-index by their assigned order
     *
     * @param{Array} drawings
     *  sorted Array of HTMLElements
     *
     * @param{Number} startIndex
     *  defines the minimal z-order
     *  (header & Footer should have 11, normal page should have 41)
     *
     */
    TextUtils.sortDrawingsOrder = function (drawings, startIndex) {
        _.each(drawings, function (dr, index) {
            var newIndex = startIndex + index;
            if (newIndex !== dr.style.zIndex) {
                dr.style.zIndex = newIndex;
            }
        });
    };

    /**
     * return the pageNumber for assigned absolute drawings
     *
     * @param{HTMLElement} drawingEl
     *
     * @param{Model} docModel
     *
     * @return{Number} page number
     */
    TextUtils.getPageNumber = logger.profileMethod('TextUtils.getPageNumber()', function (drawingEl, docModel) {

        var pageNumber = drawingEl.getAttribute('page-number');
        var pageDrawing = drawingEl.parentNode.classList.contains('textdrawinglayer');

        if (pageNumber && !pageDrawing) {
            Utils.error('drawing has attribute page number, but is a paragraph oriented drawing!!!');

            pageNumber = null;
            drawingEl.setAttribute('page-number', null);
        }
        if (!pageNumber) {
            pageNumber = docModel.getPageLayout().getPageNumber(drawingEl);
            if (pageDrawing) {
                drawingEl.setAttribute('page-number', pageNumber);
            }
        } else {
            pageNumber = parseInt(pageNumber, 10);
        }

        return pageNumber;
    });

    /**
     * test the assigned string for hangul signs, used in korean language
     */
    TextUtils.isHangulText = function (text) {
        return HANGUL.test(text);
    };

    /**
     * Get the default paper size by the given locale.
     * @param {String} locale the locale to get the default page size
     * @returns {Object}
     *  {Number} width of the page
     *  {Number} height of the page
     *  If the locale is not supported return the page size of DIN A4
     */
    TextUtils.getPaperSizeByLocale = function (locale) {
        var paperFormat = TextUtils.DEFAULT_PAPER_FORMAT[locale];
        var size = paperFormat ? PAPERSIZE_BY_FORMAT[paperFormat] : null;
        return size || PAPERSIZE_BY_FORMAT.A4;
    };

    /**
     * Receiving the attributes that are necessary to remove a drawing background.
     *
     * @returns {Object}
     *  The attributes required to remove the drawing background.
     */
    TextUtils.getEmptyDrawingBackgroundAttributes = function () {
        return _.copy(TextUtils.NO_DRAWING_BACKGROUND, true);
    };

    /**
     * Receiving the attributes that are necessary to remove a slide background.
     *
     * @returns {Object}
     *  The attributes required to remove the slide background.
     */
    TextUtils.getEmptySlideBackgroundAttributes = function () {
        return _.copy(TextUtils.NO_SLIDE_BACKGROUND, true);
    };

    /**
     * This function converts the properties of a rectangle object ('top', 'left',
     * 'width' and 'height'). Specified is a rectangle that has values in pixel
     * relative to the app-content-root node. This is for example created by the
     * selectionBox. All properties are converted into values relative to the page.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {Object} box
     *  The rectangle with the properties 'top', 'left', 'width' and 'height'.
     *  The values are given in pixel relative to the app-content-root node.
     *
     * @returns {Object}
     *  The rectangle with the properties 'top', 'left', 'width' and 'height'.
     *  The values are given in pixel relative to the page node.
     */
    TextUtils.convertAppContentBoxToPageBox = function (app, box) {

        var // the current zoom factor
            zoomFactor = app.getView().getZoomFactor(),
            // the offset of the active slide inside the content root node
            pageOffset = app.getModel().getNode().offset(),
            // the content root node, the base for the position calculation
            contentRootNode = app.getView().getContentRootNode(),
            // the position of the content root node
            pos = contentRootNode.offset(),
            // the horizontal scroll shift
            scrollLeft = contentRootNode.scrollLeft(),
            // the vertical scroll shift
            scrollTop = contentRootNode.scrollTop(),
            // the new box object
            operationBox = {};

        // the left position of the user defined box relative to the slide (can be negative)
        operationBox.left = Utils.round((box.left - (pageOffset.left - pos.left) - scrollLeft) / zoomFactor, 1);
        // the top position of the user defined box relative to the slide (can be negative)
        operationBox.top = Utils.round((box.top - (pageOffset.top - pos.top) - scrollTop) / zoomFactor, 1);
        // the width of the box
        operationBox.width = Utils.round(box.width / zoomFactor, 1);
        // the height of the box
        operationBox.height = Utils.round(box.height / zoomFactor, 1);

        return operationBox;
    };

    // exports ================================================================

    return TextUtils;

});
