/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/components/field/complexfield', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/snapshot',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/field/basefield',
    'gettext!io.ox/office/textframework/main'
], function (DateUtils, LocaleData, Utils, AttributeUtils, Operations, Snapshot, DOM, Position, BaseField, gt) {

    'use strict';

    // class ComplexField =====================================================

    /**
     * An instance of this class represents the model for all complex fields in the
     * edited document.
     *
     * @constructor
     *
     * @extends Field
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function ComplexField(app) {

        var // self reference
            self = this,
            // an object with target string as key and target node as value, the model
            allFields = {},
            // a counter for the complex field ID
            fieldID = 0,
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // page layout object
            pageLayout = null,
            // number formatter object
            numberFormatter = null,
            // range marker object
            rangeMarker = null,
            // change track object
            changeTrack = null;

        // base constructors --------------------------------------------------

        BaseField.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one complex field in the collection (the model).
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         *
         * @param {String} id
         *  The unique id of the complex field node.
         *
         * @param {String} [target]
         *  The target, where the complex field is located.
         */
        function addIntoComplexFieldCollection(field, id, target) {

            var // the complex field node in jQuery format
                $field = $(field),
                // whether the complex field is inside header or footer
                // -> in this case it is useless to save the node itself, but saving
                //    the target allows to find the range with specific id fast
                isMarginalId = target && pageLayout.isIdOfMarginalNode(target);

            // mark field node in comment
            if (target && model.getCommentLayer().isCommentTarget(target)) {
                $field.addClass(DOM.FIELD_IN_COMMENT_CLASS);
            }

            // adding the node also into the type specific collector
            allFields[id] = isMarginalId ? target : $field;
        }

        /**
         * Removing one complex field with specified id from the collection (the model).
         *
         * @param {String} id
         *  The unique id of the complex field node.
         */
        function removeFromComplexFieldCollection(id) {
            delete allFields[id];
        }

        /**
         * Finding in a header or footer specified by the target the complex field
         * with the specified id.
         *
         * @param {String} id
         *  The id string.
         *
         * @param {String} target
         *  The target string to identify the header or footer node
         *
         * @param {Number} [index]
         *  Cardinal number of target node in the document from top to bottom. If passed,
         *  forces to return node with that index in document, from top to bottom.
         *
         * @returns {jQuery|null}
         *  The range marker node with the specified id and type, or null, if no such
         *  range marker exists.
         */
        function getMarginalComplexField(id, target, index) {

            var // the specified header or footer node
                marginalNode = null,
                // the searched complex field with the specified id
                field = null;

            marginalNode = model.getRootNode(target, index);

            if (marginalNode) {
                field = _.find(marginalNode.find(DOM.COMPLEXFIELDNODE_SELECTOR), function (field) { return DOM.getComplexFieldId(field) === id; });
            }

            return field ? $(field) : null;
        }

        /**
         * Private method to mark all nodes that belong to complex field,
         * or, so to say, are inside range start and range end of a field.
         *
         * @param {HTMLElement|jQuery} marker
         *  One range marker node.
         *
         * @param {String} id
         *  Id of processed field.
         */
        function markComplexFieldForHighlighting(cxField, id) {
            var $node = $(cxField).next(),
                $helperNode;

            if (!rangeMarker.getEndMarker(id)) {
                return; // no range marker inserted yet, return
            }

            while ($node.length && (!DOM.isRangeMarkerEndNode($node) || DOM.getRangeMarkerId($node) !== id)) { // #42084, #42674
                if (!$node.data('fieldId')) {
                    $node.addClass(DOM.COMPLEXFIELDMEMBERNODE_CLASS + ' complex' + id).data('fieldId', id);
                }
                $helperNode = $node.next();
                if (!$helperNode.length) {
                    $helperNode = $node.parent().next().children().first(); // if field is in more than one paragraph, search for range-end in next p
                    if (!$helperNode.length) {
                        Utils.warn('complexField.markComplexFieldForHighlighting(): rangeEnd not found!');
                        return;
                    } else {
                        $node = $helperNode;
                    }
                } else {
                    $node = $helperNode;
                }
            }
        }

        /**
         * Removing empty text spans between the range marker start node and the
         * following complex field node.
         *
         * @param {HTMLElement|jQuery} node
         *  The complex field node.
         */
        function removePreviousEmptyTextSpan(field) {

            var // the node preceding the complex field node
                precedingNode = $(field).prev();

            if (precedingNode.length > 0 && precedingNode.prev().length > 0 && DOM.isEmptySpan(precedingNode) && DOM.isRangeMarkerStartNode(precedingNode.prev())) {
                precedingNode.remove();
            }
        }

        /**
         * Helper function to keep the global fieldID up-to-date. After inserting
         * a new complex field or after loading from local storage, this number needs to be updated.
         * Then a new complex field can be inserted from the client with a valid and unique id.
         *
         * @param {String} id
         *  The id of the complex field node.
         */
        function updateFieldID(id) {

            // does the id end with a number?
            // -> then the fieldID needs to be checked

            var // resulting array of the regular expression
                matches = /(\d+)$/.exec(id),
                // the number value at the end of the id
                number = 0;

            if (_.isArray(matches)) {
                number = parseInt(matches[1], 10);

                if (number >= fieldID) {
                    fieldID = number + 1;
                }
            }
        }

        /**
         * Update of content for complex field: Page number
         *
         * @param {jQuery|Node} node
         *  Complex field node.
         *
         * @param {String} format
         *  Format of the number.
         */
        function updatePageNumberFieldCx(node, format) {
            var number = pageLayout.getPageNumber(node);
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = null;
            var endPos = null;
            var rangeEndNode = rangeMarker.getEndMarker(id);

            if (model.isHeaderFooterEditState()) { // no explicit update of pageNumber field in header/footer!
                return;
            }
            if (!rangeEndNode) {
                Utils.error('complexField.updatePageNumberFieldCx(): missing range end node for the id: ', id);
                return;
            }

            startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            endPos = Position.getOxoPosition(selection.getRootNode(), rangeEndNode);

            number = self.formatPageFieldInstruction(number, format);

            if (startPos && endPos && number) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    // disabling changeTracking temporarily, as page number update is not tracked
                    model.getChangeTrack().setChangeTrackSupported(false);
                    return model.deleteSelected().done(function () {
                        model.insertText(number, startPos);
                        selection.setTextSelection(Position.increaseLastIndex(startPos, number.length + 1));
                        model.getChangeTrack().setChangeTrackSupported(true); // activating change track support back
                    });
                });
            } else {
                Utils.error('complexField.updatePageNumberFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, number);
            }
        }

        /**
        * Update of content for complex field: Number of pages.
        *
        * @param {jQuery|Node} node
        *   Complex field node
        *
        * @param {String} format
        *  Format of the number.
        */
        function updateNumPagesFieldCx(node, format) {
            var numPages = pageLayout.getNumberOfDocumentPages();
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = null;
            var endPos = null;
            var rangeEndNode = rangeMarker.getEndMarker(id);

            if (model.isHeaderFooterEditState()) { // no explicit update of numPages field in header/footer!
                return;
            }
            if (!rangeEndNode) {
                Utils.error('complexField.updateNumPagesFieldCx(): missing range end node for the id: ', id);
                return;
            }

            startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            endPos = Position.getOxoPosition(selection.getRootNode(), rangeEndNode);

            numPages = self.formatPageFieldInstruction(numPages, format, true);

            // TODO: Nested complex fields

            if (startPos && endPos && numPages) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    return model.deleteSelected().done(function () {
                        model.insertText(numPages, startPos);
                        selection.setTextSelection(Position.increaseLastIndex(startPos, numPages.length + 1));
                    });
                });
            } else {
                Utils.error('complexField.updateNumPagesFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, numPages);
            }
        }

        /**
         * Update of content for complex field: Date
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} formatCode
         *  Format of date
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.time=false]
         *      If it's time format only.
         */
        function updateDateTimeFieldCx(node, formatCode, options) {

            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            var time = Utils.getBooleanOption(options, 'time', false);
            var localeFormatCode = (formatCode && formatCode !== 'default') ? formatCode : time ? LocaleData.SHORT_TIME : LocaleData.SHORT_DATE;
            var replacedLclFormat = localeFormatCode.replace(/'/g, '"'); // single quotes are replaced with double
            var parsedFormat = numberFormatter.getParsedFormat(replacedLclFormat);
            var formattedDate = numberFormatter.formatValue(parsedFormat, serial);
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            var rangeEndMarker = rangeMarker.getEndMarker(id);
            if (!rangeEndMarker) {
                Utils.error('complexField.updateDateTimeFieldCx(): missing range end node for the id: ', id);
                return;
            }
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeEndMarker);

            if (startPos && endPos) {
                if ($(node).data('datepickerValue')) {
                    $(node).data('datepickerValue', new Date());
                }
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    return model.deleteSelected().done(function () {
                        model.insertText(formattedDate, startPos);
                        selection.setTextSelection(Position.increaseLastIndex(startPos, formattedDate.length + 1));
                    });
                });
            } else {
                Utils.error('complexField.updateDateTimeFieldCx(): Wrong start and end postitions: ', startPos, endPos);
            }

            // TODO: Nested complex fields
        }

        /**
         * Update the content of complex field: FILENAME
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateFileNameFieldCx(node, format) {
            var fileName = app.getFullFileName();
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            var rangeEndNode = rangeMarker.getEndMarker(id);
            if (!rangeEndNode) {
                Utils.error('complexField.updateFileNameFieldCx(): missing range end node for the id: ', id);
                return;
            }
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeEndNode);

            if (format && fileName) {
                if ((/Lower/).test(format)) {
                    fileName = fileName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    fileName = fileName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    fileName = Utils.capitalize(fileName.toLowerCase());
                } else if ((/Caps/).test(format)) {
                    fileName = Utils.capitalizeWords(fileName.toLowerCase());
                }
            }

            // TODO: Nested complex fields

            if (startPos && endPos && fileName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    return model.deleteSelected().done(function () {
                        model.insertText(fileName, startPos);
                        selection.setTextSelection(Position.increaseLastIndex(startPos, fileName.length + 1));
                    });
                });
            } else {
                Utils.error('complexField.updateFileNameFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, fileName);
            }
        }

        /**
         * Update the content of complex field: AUTHOR
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateAuthorFieldCx(node, format) {
            var authorName = app.getClientOperationName();
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            var rangeEndNode = rangeMarker.getEndMarker(id);
            if (!rangeEndNode) {
                Utils.error('complexField.updateAuthorFieldCx(): missing range end node for the id: ', id);
                return;
            }
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeEndNode);

            if (format && authorName) {
                if ((/Lower/).test(format)) {
                    authorName = authorName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    authorName = authorName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    authorName = Utils.capitalize(authorName.toLowerCase());
                } else if ((/Caps/).test(format)) {
                    authorName = Utils.capitalizeWords(authorName.toLowerCase());
                }
            }

            // TODO: Nested complex fields

            if (startPos && endPos && authorName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    return model.deleteSelected().done(function () {
                        model.insertText(authorName, startPos);
                        selection.setTextSelection(Position.increaseLastIndex(startPos, authorName.length + 1));
                    });
                });
            } else {
                Utils.error('complexField.updateAuthorFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, authorName);
            }
        }

        /**
         * Parses field formats for Table of contents and returns set variables in one object
         *
         * @param {Array} formats
         *
         * @returns {Object}
         */
        function getSwitchesFromTOCformats(formats) {
            var returnObj = {
                setAnchorAttribute: false,
                headingLevels: [1, 2, 3, 4, 5, 6, 7, 8, 9], // use page numbers for all 9 levels by default, if not explicitly overwritten by switch
                customSeparator: null,
                forbidenPageNumLevels: [],
                bookmarkName: null
            };

            _.each(formats, function (format) {
                var patternMatch = null;
                var forbidenPageNumLevels = [];
                var startLevel, endLevel;

                if (_.contains(format, 'h')) { // set anchor char attribute
                    returnObj.setAnchorAttribute = true;
                }
                if ((/^o "/).test(format)) { // use only specified heading levels
                    format = _.isString(format) ? format.replace(/"/g, '') : '';
                    patternMatch = format.match(/\d/g);
                    if (patternMatch && patternMatch.length === 2) {
                        var headingLevels = [];
                        for (var i = parseInt(patternMatch[0], 10); i <= parseInt(patternMatch[1], 10); i++) {
                            headingLevels.push(i);
                        }
                        returnObj.headingLevels = headingLevels;
                    }
                }
                if ((/^p "/).test(format)) { // use custom separators instead of tabs
                    format = (format).match(/"(.*?)"/);
                    returnObj.customSeparator = (_.isArray(format) && format.length > 1) ? format[1] : null;
                }
                if ((/^n /).test(format)) { // whether to display page numbers or not, and which levels to forbid
                    patternMatch = format.match(/\d/g);
                    startLevel = (patternMatch && patternMatch.length === 2) ? parseInt(patternMatch[0], 10) : 1;
                    endLevel = (patternMatch && patternMatch.length === 2) ? parseInt(patternMatch[1], 10) : 10;

                    for (var j = startLevel; j <= endLevel; j++) {
                        forbidenPageNumLevels.push(j);
                    }
                    returnObj.forbidenPageNumLevels = forbidenPageNumLevels;
                }
                if ((/^b /).test(format)) { // create toc from bookmarks areas
                    if (format.length > 2) {
                        returnObj.bookmarkName = format.substr(2);
                    }
                }
            });

            return returnObj;
        }

        /**
         * Returns unique jQuery collection of all paragraphs from top to bottom,
         * inside bookmark range, or if left out, inside whole document.
         *
         * @param {String} [bookmarkName]
         *  Name of the bookmark inside which to fetch paragraphs
         *
         * @returns {jQuery}
         */
        function fetchAllParagraphs(bookmarkName) {
            var docParagraphs = $();
            var pageContentNode = DOM.getPageContentNode(model.getNode());
            var bookmarkStart, bookmarkId, bookmarkEnd, bookmarkStartP, bookmarkEndP;

            if (bookmarkName) {
                bookmarkStart = pageContentNode.find('.bookmark[anchor="' + bookmarkName + '"][bmPos="start"]');
                bookmarkId = DOM.getBookmarkId(bookmarkStart);
                bookmarkEnd = pageContentNode.find('.bookmark[bmId="' + bookmarkId + '"][bmPos="end"]');
                bookmarkStartP = bookmarkStart.closest('.p');
                bookmarkEndP = bookmarkEnd.closest('.p');
                while (bookmarkStartP.length && bookmarkStartP[0] !== bookmarkEndP[0]) {
                    docParagraphs = docParagraphs.add(bookmarkStartP);
                    bookmarkStartP = bookmarkStartP.next();
                }
                if (bookmarkStartP[0] && bookmarkStartP[0] === bookmarkEndP[0]) { docParagraphs = docParagraphs.add(bookmarkStartP); }
            } else {
                // fetching all paragraphs in document with specific heading levels
                docParagraphs = pageContentNode.find('.p').not('.marginal');
            }
            return docParagraphs;
        }

        /**
         * Process passed paragraphs and prepares data for Table of Contents insert/update.
         * This method also generates necessary operations for missing bookmarks and pushes to passed generator.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operation.
         *
         * @param {jQuery} paragraphs
         *  Collection of jQuery paragraphs to process.
         *
         * @param {Array} headingLevels
         *  Range of allowed heading levels to search within
         *
         * @param {Boolean} setAnchorAttribute
         *  Whether or not to set anchor character attributes
         *
         * @returns {Array}
         *  Array of objects containing properties important for generating Table of Contents.
         */
        function fetchTOCdata(generator, paragraphs, headingLevels, setAnchorAttribute) {
            var bmIncrement = 1;
            var dataTOC = [];

            _.each(paragraphs, function (paragraph) {
                var paraStyleAttrSet = model.getParagraphStyles().getElementAttributes(paragraph);
                var paraStyle = paraStyleAttrSet && paraStyleAttrSet.styleId;
                var outlineLevel = paraStyleAttrSet && paraStyleAttrSet.paragraph && paraStyleAttrSet.paragraph.outlineLevel;
                var paraHeadLevel = outlineLevel + 1;
                var titleText = $(paragraph).text();
                var oneData, listLabel;

                if (_.contains(headingLevels, paraHeadLevel) && titleText.length) { // if it's allowed heading level
                    listLabel = DOM.isListLabelNode(paragraph.firstChild) ? $(paragraph.firstChild).text() : null;
                    titleText = listLabel ? titleText.substr(listLabel.length) : titleText;
                    listLabel = listLabel ? listLabel + '    ' : null; // add substitute whitespaces for tab between list number and title text
                    oneData = { text: titleText, paraNode: paragraph, pageNum: pageLayout.getPageNumber(paragraph) + '', headingLevel: paraHeadLevel, styleId: paraStyle, listLabelText: listLabel };
                    if (setAnchorAttribute) {
                        // first fetch positions of existing bookmarks in current paragraph
                        var posCounter = 1;
                        var bmStartNode = $(paragraph).children('.bookmark[anchor*=_]').first();
                        var oldBmId = DOM.getBookmarkId(bmStartNode);
                        var anchorName = DOM.getBookmarkAnchor(bmStartNode);
                        var bmEndNode = $(paragraph).children('.bookmark[bmId="' + oldBmId + '"]').not('.bookmark[anchor*=_]');
                        var paraStartPos = Position.getOxoPosition(model.getNode(), paragraph);
                        var endPos = _.clone(paraStartPos);

                        if (!bmStartNode.length) {
                            var startPos = _.clone(paraStartPos);
                            var bmId = self.getNextBookmarkId(bmIncrement);
                            anchorName = self.getNextAnchorName(bmIncrement);
                            bmIncrement++;
                            startPos.push(0);
                            endPos.push(Position.getParagraphLength(model.getNode(), paraStartPos) + posCounter); // +1 for length of bookmark start node
                            generator.generateOperation(Operations.BOOKMARK_INSERT, { start: startPos, id: bmId, anchorName: anchorName, position: 'start' });
                            generator.generateOperation(Operations.BOOKMARK_INSERT, { start: endPos, id: bmId,  position: 'end' });
                        } else if (!bmEndNode.length) {
                            endPos.push(Position.getParagraphLength(model.getNode(), paraStartPos));
                            generator.generateOperation(Operations.BOOKMARK_INSERT, { start: endPos, id: oldBmId,  position: 'end' });
                        }
                        oneData.anchor = anchorName;
                    }
                    dataTOC.push(oneData);
                }
            });
            return dataTOC;
        }

        /**
         * Returns position of right tab in hmm.
         *
         * @param {jQuery} rootNode
         *  Active root node of the document.
         *
         * @param {Array} startParaPos
         *  Oxo position of the paragraph node.
         *
         * @returns {Number}
         */
        function calcRightTabPos(rootNode, startParaPos) {
            var pageLayout = model.getPageLayout();
            var pageWidth = pageLayout.getPageAttribute('width') - pageLayout.getPageAttribute('marginLeft') - pageLayout.getPageAttribute('marginRight');
            var paraNode = Position.getParagraphElement(rootNode, startParaPos);
            var paraWidth = Utils.convertLengthToHmm($(paraNode).width(), 'px');

            return Math.min(paraWidth, pageWidth);
        }

        /**
         * Helper method to generate operations for Table of Contents, and append them to passed generator.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operation.
         *
         * @param {Array} dataTOC
         *  Array of objects with properties for generating TOC
         *
         * @param {Array} startPos
         *  Starting position for TOC field
         *
         * @param {Object} options
         *  @param {Boolean} options.setAnchorAttribute
         *      Whether to set anchor char attribute or not.
         *  @param {Boolean} options.forbidenPageNumLevels
         *      Range of forbiden page number levels.
         *  @param {String} options.customSeparator
         *      If set, custom separator to be used.
         *  @param {String} options.tabStyle
         *      Tab style to be used.
         *
         * @returns [Array] startPos
         *  Oxo position after last generated operation.
         */
        function generateOperationsForTOC(generator, dataTOC, startPos, options) {
            var startParaPos = startPos.slice(0, startPos.length - 1);
            var tempCreatedStyles = [];
            var startIndent = 388, marginBottomDef = 176; // predefined values for 'TOC' style
            var rightTabPos = calcRightTabPos(model.getNode(), startParaPos);
            var setAnchorAttribute = Utils.getBooleanOption(options, 'setAnchorAttribute', false);
            var forbidenPageNumLevels = Utils.getArrayOption(options, 'forbidenPageNumLevels', []);
            var customSeparator = Utils.getStringOption(options, 'customSeparator', null);
            var tabFillChar = Utils.getStringOption(options, 'tabStyle', null); // can be: none, dot, hyphen, underscore
            var noTOCmessage = gt('No table of contents entries in the document.');
            var defParaStyle = model.getDefaultParagraphStyleDefinition().styleId;
            var nullAttributeSet = null;
            var isActiveChangeTracking = model.getChangeTrack().isActiveChangeTracking();

            if (!tabFillChar) {
                var paragraph = Position.getParagraphElement(model.getNode(), startParaPos);
                var paraAttrs = model.getParagraphStyles().getElementAttributes(paragraph);
                var tabStops = (paraAttrs && paraAttrs.paragraph) ? paraAttrs.paragraph.tabStops : [];
                _.each(tabStops, function (el) { if (el.value === 'right') { tabFillChar = el.fillChar; } });
            }
            generator.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: model.getCharacterStyles().buildNullAttributeSet() }); // #53460
            if (!dataTOC.length) {
                generator.generateOperation(Operations.TEXT_INSERT, { start: startPos, text: noTOCmessage, attrs: { character: { bold: true } } });
                //generator.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: { paragraph: { tabStops: [{ value: 'right', pos: rightTabPos, fillChar: tabFillChar }] } } });
                startPos = Position.increaseLastIndex(startPos, noTOCmessage.length);
                generator.generateOperation(Operations.PARA_SPLIT, { start: startPos });
                if (isActiveChangeTracking) { generator.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: { changes: { inserted: model.getChangeTrack().getChangeTrackInfo() } } }); }
                startParaPos = Position.increaseLastIndex(startParaPos);
                startPos = _.clone(startParaPos);
                startPos.push(0);
            }
            _.each(dataTOC, function (oneDataToc, index) {
                var styleIdExt = (_.isString(tabFillChar) && tabFillChar.length) ? tabFillChar[0] : '';
                var TOCStyleId = gt('Contents') + ' ' + oneDataToc.headingLevel + styleIdExt;
                var leftIndent = startIndent * (oneDataToc.headingLevel - 1); // start from indentLeft 0
                var stylesheetAttrs = { paragraph: { indentLeft: leftIndent, marginBottom: marginBottomDef } };
                var paraAttrs = { styleId: TOCStyleId };
                var charAttrs = { character: { } };

                if (setAnchorAttribute) { charAttrs.character.anchor = oneDataToc.anchor; }
                if (tabFillChar) {  stylesheetAttrs.paragraph.tabStops = [{ value: 'right', pos: rightTabPos, fillChar: tabFillChar }]; }

                if (!model.getParagraphStyles().containsStyleSheet(TOCStyleId) && !_.contains(tempCreatedStyles, TOCStyleId)) { // insert missing style
                    generator.generateOperation(Operations.INSERT_STYLESHEET, { styleId: TOCStyleId, styleName: TOCStyleId, type: 'paragraph', attrs: stylesheetAttrs, nextStyleId: defParaStyle, parent: defParaStyle, uiPriority: 39 });
                    tempCreatedStyles.push(TOCStyleId);
                }
                if (index !== 0) { // for first data there is already paragraph
                    generator.generateOperation(Operations.PARA_SPLIT, { start: startPos });
                    startParaPos = Position.increaseLastIndex(startParaPos);
                    startPos = _.clone(startParaPos);
                    startPos.push(0);
                }
                generator.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: paraAttrs });
                if (oneDataToc.listLabelText && oneDataToc.listLabelText.length) {
                    generator.generateOperation(Operations.TEXT_INSERT, { start: startPos, text: oneDataToc.listLabelText, attrs: { styleId: null, character: { anchor: oneDataToc.anchor } } });
                    startPos = Position.increaseLastIndex(startPos, oneDataToc.listLabelText.length);
                }
                if (oneDataToc.text.length) {
                    generator.generateOperation(Operations.TEXT_INSERT, { start: startPos, text: oneDataToc.text, attrs: charAttrs });
                    startPos = Position.increaseLastIndex(startPos, oneDataToc.text.length);
                }

                if (!_.contains(forbidenPageNumLevels, oneDataToc.headingLevel)) {
                    if (customSeparator) {
                        generator.generateOperation(Operations.TEXT_INSERT, { start: startPos, text: customSeparator, attrs: { styleId: null, character: { anchor: oneDataToc.anchor } } });
                        startPos = Position.increaseLastIndex(startPos, customSeparator.length);
                    } else {
                        generator.generateOperation(Operations.TAB_INSERT, { start: startPos, attrs: { styleId: null, character: { anchor: oneDataToc.anchor } } });
                        startPos = Position.increaseLastIndex(startPos);
                    }
                    // without bookmark - insert text only
                    generator.generateOperation(Operations.TEXT_INSERT, { start: startPos, text: oneDataToc.pageNum, attrs: { styleId: null, character: { anchor: oneDataToc.anchor } } });
                    oneDataToc.pageNumOxo = { oxoPosStart: startPos, oxoPosEnd: Position.increaseLastIndex(startPos, oneDataToc.pageNum.length - 1) };
                    startPos = Position.increaseLastIndex(startPos, oneDataToc.pageNum.length);
                }

                if (index === dataTOC.length - 1 && !isActiveChangeTracking) { // split once more for last data in TOC, but not if changetrack is active (prevent extra paragraph at the end)
                    nullAttributeSet = model.getParagraphStyles().buildNullAttributeSet();
                    startParaPos = Position.increaseLastIndex(startParaPos);
                    generator.generateOperation(Operations.PARA_SPLIT, { start: startPos });
                    generator.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: _.extend({ styleId: defParaStyle }, nullAttributeSet) });
                    startPos = _.clone(startParaPos);
                    startPos.push(0);
                }
            });

            return startPos;
        }

        /**
         * Update the content of complex field: TOC - Table of contents
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String|Array} format
         *  Format of the field with switches. Can be one string, or array of strings.
         *
         *  @param {Boolean} [options.testMode=false]
         *      When set to true, pagination is ignored.
         *      Warning: used for speeding up unit tests only. Do not use this flag for other purposes!
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the insertion of table of contents
         *  ended. It is rejected, if the dialog has been canceled.
         */
        function updateTableOfContents(node, formats, options) {
            if (_.isString(formats)) { formats = [formats]; } // if there is only one switch, it is string. Convert to array for iteration

            var testMode = Utils.getBooleanOption(options, 'testMode', false);
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            var rangeEndNode = rangeMarker.getEndMarker(id);
            if (!rangeEndNode) {
                Utils.error('complexField.updateTableOfContents(): missing range end node for the id: ', id);
                return;
            }
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeEndNode);
            var generator = model.createOperationGenerator();
            var secGenerator = model.createOperationGenerator();
            var stopedPageNumUpdate = false;
            var fieldSwitches = getSwitchesFromTOCformats(formats);
            var setAnchorAttribute = fieldSwitches.setAnchorAttribute;
            var headingLevels = fieldSwitches.headingLevels;
            var forbidenPageNumLevels = fieldSwitches.forbidenPageNumLevels;
            var customSeparator = fieldSwitches.customSeparator;
            var bookmarkName = fieldSwitches.bookmarkName;
            var docParagraphs = fetchAllParagraphs(bookmarkName);
            var dataTOC = [];
            var changeTracking = model.getChangeTrack();
            var isActiveChangeTracking = changeTracking.isActiveChangeTracking();
            var tocPromise = null;
            var target = model.getActiveTarget();

            // blocking keyboard input during applying of operations
            model.setBlockKeyboardEvent(true);

            if (startPos && endPos) {
                tocPromise = model.getUndoManager().enterUndoGroup(function () {
                    // the deferred to keep the undo group open until it is resolved or rejected
                    var undoPromise = null;
                    var snapshot = new Snapshot(app);

                    selection.setTextSelection(startPos, endPos);
                    undoPromise = model.deleteSelected({ snapshot: snapshot, warningLabel: gt('Preparing Table of contents.') }).then(function () {
                        model.setBlockKeyboardEvent(true);

                        dataTOC = fetchTOCdata(generator, docParagraphs, headingLevels, setAnchorAttribute);
                        generateOperationsForTOC(generator, dataTOC, startPos, { setAnchorAttribute: setAnchorAttribute, forbidenPageNumLevels: forbidenPageNumLevels, customSeparator: customSeparator });

                        if (isActiveChangeTracking) { changeTracking.handleChangeTrackingDuringPaste(generator.getOperations()); }
                        // if toc in part of the document with different target, extend operations, but not bookmarks that are inserted always in main document node
                        if (target) {
                            _.each(generator.getOperations(), function (operation) {
                                if (operation.name !== Operations.BOOKMARK_INSERT) { model.extendPropertiesWithTarget(operation, target); }
                            });
                        }

                        // fire apply operations asynchronously
                        var operationsDef = model.applyOperations(generator, { async: true });

                        app.getView().enterBusy({
                            cancelHandler: function () {
                                if (operationsDef && operationsDef.abort) {
                                    snapshot.apply();  // restoring the old state
                                    stopedPageNumUpdate = true;
                                    app.enterBlockOperationsMode(function () { operationsDef.abort(); }); // block sending of operations
                                }
                            },
                            immediate: true,
                            warningLabel: /*#. shown while applying selected operations to update Table of contents */ gt('Updating Table of contents will take some time, please wait...')
                        });

                        // handle the result of  operations
                        return operationsDef
                            .progress(function (progress) {
                                // update the progress bar according to progress of the operations promise
                                app.getView().updateBusyProgress(progress);
                            });
                    });

                    // it might happen that pagination is affected by creating TOC, run second pass to update page numbers if necessary
                    undoPromise = undoPromise.then(function () {
                        if (testMode || target) { return $.when(); } // in test mode, pagination is ignored for speeding up the tests, also in header/footer mode is pagination disabled
                        return model.waitForEvent(model, 'pagination:finished').done(function () {
                            if (!stopedPageNumUpdate) {
                                model.getUndoManager().enterUndoGroup(function () {
                                    _.each(dataTOC, function (oneDataToc) {
                                        var newPageNum = pageLayout.getPageNumber(oneDataToc.paraNode) + ''; // impl conversion from num to string to compare with oneDataToc.pageNum
                                        var pageNumOxo = oneDataToc.pageNumOxo;
                                        if (pageNumOxo && newPageNum !== oneDataToc.pageNum) {
                                            secGenerator.generateOperation(Operations.DELETE, { start: pageNumOxo.oxoPosStart, end: pageNumOxo.oxoPosEnd });
                                            secGenerator.generateOperation(Operations.TEXT_INSERT, { start: pageNumOxo.oxoPosStart, text: newPageNum });
                                        }
                                    });
                                    if (isActiveChangeTracking) { changeTracking.handleChangeTrackingDuringPaste(secGenerator.getOperations()); }
                                    model.setBlockKeyboardEvent(true);
                                    model.applyOperations(secGenerator);
                                    model.setBlockKeyboardEvent(false);
                                });
                            }
                        });
                    });

                    undoPromise.always(function () {
                        app.getView().leaveBusy();
                        // deleting the snapshot
                        if (snapshot) { snapshot.destroy(); }
                        // allowing keyboard events again
                        model.setBlockKeyboardEvent(false);
                        selection.setTextSelection(startPos);
                    });

                    return undoPromise;

                }); // enterUndoGroup()
            } else {
                Utils.error('complexField.updateTableOfContents(): Wrong start end postitions: ', startPos, endPos);
            }
            return tocPromise;
        }

        /**
         * Generates and applies all the necessary operations for creating Table of Contents,
         * defined in format switches.
         *
         * @param {String} format
         *  Field format switches.
         * @param {Object} options
         *  @param {String} options.tabStyle
         *      Parameter for style of the tab stop.
         *  @param {Boolean} [options.testMode=false]
         *      When set to true, pagination is ignored.
         *      Warning: used for speeding up unit tests only. Do not use this flag for other purposes!
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the insertion of table of contents
         *  ended. It is rejected, if the dialog has been canceled.
         */
        function insertTableOfContents(format, options) {
            var extrObj = self.cleanUpAndExtractType(format);
            // if there is only one switch, it is string. Convert to array for iteration
            var instruction = extrObj ? (_.isString(extrObj.instruction) ? [extrObj.instruction] : extrObj.instruction) : null;
            var startPos = selection.getStartPosition();
            var endPos = selection.getEndPosition();
            var fieldStartPos = null;
            var generator1 = model.createOperationGenerator();
            var generator2 = model.createOperationGenerator();
            var stopedPageNumUpdate = false;
            var tabStyle = Utils.getStringOption(options, 'tabStyle', 'dot');
            var testMode = Utils.getBooleanOption(options, 'testMode', false);
            var fieldSwitches = getSwitchesFromTOCformats(instruction);
            var setAnchorAttribute = fieldSwitches.setAnchorAttribute;
            var headingLevels = fieldSwitches.headingLevels;
            var forbidenPageNumLevels = fieldSwitches.forbidenPageNumLevels;
            var customSeparator = fieldSwitches.customSeparator;
            var bookmarkName = fieldSwitches.bookmarkName;
            var docParagraphs = null;
            var dataTOC = [];
            var fieldManager = model.getFieldManager();
            var changeTracking = model.getChangeTrack();
            var selectedFieldFormat = fieldManager.isHighlightState() ? fieldManager.getSelectedFieldFormat() : null;
            var insertHeadline = true;
            var tocPromise = null;
            var nullAttributeSet = null;
            var isActiveChangeTracking = changeTracking.isActiveChangeTracking();
            var target = model.getActiveTarget();

            function increaseSecondLastIndex(position, increment) {
                if (_.isArray(position) && position.length > 1) {
                    position = _.clone(position);
                    position[position.length - 2] += (_.isNumber(increment) ? increment : 1);
                }
                return position;
            }

            function isSelectedFieldTypeTOC() {
                var selFieldType = fieldManager.getSelectedFieldType();
                return _.isString(selFieldType) && selFieldType.indexOf('TOC') > -1;
            }

            if (selectedFieldFormat) { // if cursor is already in some complex field
                var typeToCheck = _.isArray(selectedFieldFormat) ? selectedFieldFormat[0] : (_.isString(selectedFieldFormat) ? selectedFieldFormat : '');
                if (typeToCheck.indexOf('TOC') > -1 || isSelectedFieldTypeTOC()) { // if cursor is already in TOC field, don't insert field in field, but delete and create new one
                    var node = fieldManager.getSelectedFieldNode();
                    var id = DOM.getComplexFieldMemberId(node);
                    startPos = Position.decreaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
                    var rangeEndNode = rangeMarker.getEndMarker(id);
                    if (!rangeEndNode) {
                        Utils.error('complexField.insertTableOfContents(): missing range end node for the id: ', id);
                        return;
                    }
                    endPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), rangeEndNode));
                    fieldStartPos = Position.increaseLastIndex(startPos);
                    insertHeadline = false;
                }
            }

            // blocking keyboard input during applying of operations
            model.setBlockKeyboardEvent(true);

            if (startPos && endPos) {
                tocPromise = model.getUndoManager().enterUndoGroup(function () {
                    // the deferred to keep the undo group open until it is resolved or rejected
                    var undoPromise = null;
                    var snapshot = new Snapshot(app);

                    selection.setTextSelection(startPos, endPos);
                    undoPromise = model.deleteSelected({ warningLabel: gt('Preparing Table of contents.') }).then(function () {
                        var tocHeadlineId = gt('Contents heading');
                        var headlineForTOC = gt('Table of contents');
                        var defParaStyle = model.getDefaultParagraphStyleDefinition().styleId;
                        var startParaPos;

                        model.doCheckImplicitParagraph(startPos);
                        model.setBlockKeyboardEvent(true);

                        startParaPos = startPos.slice(0, startPos.length - 1);
                        if (Position.getParagraphLength(model.getNode(), startParaPos) !== 0) {
                            model.splitParagraph(startPos);
                            if (isActiveChangeTracking) { generator1.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: { changes: { inserted: changeTracking.getChangeTrackInfo() } } }); }
                            startParaPos = Position.increaseLastIndex(startParaPos);
                            if (isActiveChangeTracking) { generator1.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: { changes: { inserted: changeTracking.getChangeTrackInfo() } } }); }
                            startPos = _.clone(startParaPos);
                            startPos.push(0);
                            model.splitParagraph(startPos);
                        }

                        generator1.generateOperation(Operations.RANGE_INSERT, { start: startPos, type: 'field', position: 'start' });
                        fieldStartPos = Position.increaseLastIndex(startPos);
                        generator1.generateOperation(Operations.COMPLEXFIELD_INSERT, { start: fieldStartPos, instruction: format });
                        fieldStartPos = Position.increaseLastIndex(fieldStartPos);

                        docParagraphs = fetchAllParagraphs(bookmarkName);
                        dataTOC = fetchTOCdata(generator1, docParagraphs, headingLevels, setAnchorAttribute);
                        fieldStartPos = generateOperationsForTOC(generator1, dataTOC, fieldStartPos, { setAnchorAttribute: setAnchorAttribute, forbidenPageNumLevels: forbidenPageNumLevels, customSeparator: customSeparator, tabStyle: tabStyle });

                        generator1.generateOperation(Operations.RANGE_INSERT, { start: fieldStartPos, type: 'field', position: 'end' });
                        if (changeTrack.isActiveChangeTracking()) {
                            generator1.generateOperation(Operations.SET_ATTRIBUTES, { start: fieldStartPos, attrs: { character: { anchor: null } } });
                        }

                        // setting change track information at range marker nodes
                        if (changeTrack.isActiveChangeTracking()) {
                            generator1.generateOperation(Operations.SET_ATTRIBUTES, { start: startPos, attrs: { changes: { inserted: changeTracking.getChangeTrackInfo() } } });
                            generator1.generateOperation(Operations.SET_ATTRIBUTES, { start: fieldStartPos, attrs: { changes: { inserted: changeTracking.getChangeTrackInfo() } } });
                        }

                        if (insertHeadline) {
                            nullAttributeSet = model.getParagraphStyles().buildNullAttributeSet();
                            if (!model.getParagraphStyles().containsStyleSheet(tocHeadlineId)) {
                                generator1.generateOperation(Operations.INSERT_STYLESHEET, { styleId: tocHeadlineId, styleName: tocHeadlineId, type: 'paragraph', attrs: { paragraph: { nextStyleId: defParaStyle, outlineLevel: 9 } }, parent: 'Heading1', uiPriority: 39 });
                            }
                            generator1.generateOperation(Operations.TEXT_INSERT, { start: startPos, text: headlineForTOC });
                            startPos = Position.increaseLastIndex(startPos, headlineForTOC.length);
                            startParaPos = startPos.slice(0, startPos.length - 1);
                            generator1.generateOperation(Operations.PARA_SPLIT, { start: startPos });
                            generator1.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: _.extend({ styleId: tocHeadlineId }, nullAttributeSet) });
                            if (isActiveChangeTracking) {
                                startParaPos = Position.increaseLastIndex(startParaPos);
                                generator1.generateOperation(Operations.SET_ATTRIBUTES, { start: startParaPos, attrs: { changes: { inserted: changeTracking.getChangeTrackInfo() } } });
                            }
                        }

                        if (isActiveChangeTracking) { changeTracking.handleChangeTrackingDuringPaste(generator1.getOperations()); }

                        // fire apply operations asynchronously
                        var operationsDef = model.applyOperations(generator1, { async: true });

                        app.getView().enterBusy({
                            cancelHandler: function () {
                                if (operationsDef && operationsDef.abort) {
                                    snapshot.apply();  // restoring the old state
                                    stopedPageNumUpdate = true;
                                    app.enterBlockOperationsMode(function () { operationsDef.abort(); }); // block sending of operations
                                }
                            },
                            immediate: true,
                            warningLabel: /*#. shown while applying selected operations to insert Table of contents */ gt('Inserting Table of contents will take some time, please wait...')
                        });

                        // handle the result of  operations
                        return operationsDef
                            .progress(function (progress) {
                                // update the progress bar according to progress of the operations promise
                                app.getView().updateBusyProgress(progress);
                            });
                    });

                    // it might happen that pagination is affected by creating TOC, run second pass to update page numbers if necessary
                    undoPromise = undoPromise.then(function () {
                        if (testMode || target) { return $.when(); } // in test mode, pagination is ignored for speeding up the tests, also in header/footer mode is pagination disabled
                        return model.waitForEvent(model, 'pagination:finished').done(function () {
                            if (!stopedPageNumUpdate) {
                                _.each(dataTOC, function (oneDataToc) {
                                    var newPageNum = pageLayout.getPageNumber(oneDataToc.paraNode) + ''; // impl conversion from num to string to compare with oneDataToc.pageNum
                                    var pageNumOxo = oneDataToc.pageNumOxo;
                                    if (pageNumOxo && newPageNum !== oneDataToc.pageNum) {
                                        var pageNumOxoPos = pageNumOxo.oxoPosStart;
                                        var endPageNumPos = pageNumOxo.oxoPosEnd;

                                        if (insertHeadline) {
                                            pageNumOxoPos = increaseSecondLastIndex(pageNumOxoPos);
                                            endPageNumPos = increaseSecondLastIndex(endPageNumPos);
                                        }
                                        if (pageNumOxoPos && endPageNumPos) {
                                            generator2.generateOperation(Operations.DELETE, { start: pageNumOxoPos, end: endPageNumPos });
                                            generator2.generateOperation(Operations.TEXT_INSERT, { start: pageNumOxoPos, text: newPageNum });
                                        }
                                    }
                                });
                                if (isActiveChangeTracking) { model.getChangeTrack().handleChangeTrackingDuringPaste(generator2.getOperations()); }
                                model.setBlockKeyboardEvent(true);
                                model.applyOperations(generator2);
                                model.setBlockKeyboardEvent(false);
                            }
                        });
                    });

                    undoPromise.always(function () {
                        app.getView().leaveBusy();
                        // deleting the snapshot
                        if (snapshot) { snapshot.destroy(); }
                        // allowing keyboard events again
                        model.setBlockKeyboardEvent(false);
                        selection.setTextSelection(startPos);
                    });

                    return undoPromise;

                }); // enterUndoGroup()
            } else {
                Utils.error('complexField.insertTableOfContents(): Wrong start end postitions: ', startPos, endPos);
            }
            return tocPromise;
        }

        /**
         * Local helper function that reverts conversion from special field to normal,
         * which is necessary before creating undo operations.
         *
         * @param {jQuery|Node} specialField
         */
        function restoreSpecialField(specialField) {
            var fieldTxtLen = $(specialField).data('length');
            fieldTxtLen = (_.isNumber(fieldTxtLen) && fieldTxtLen > 0) ? (fieldTxtLen - 1) : 1;

            var allChildren = $(specialField).children();
            var firstChild = null;

            // using the first span as representative for all children. Changing the content, so that
            // its length is valid. Keeping data object, so that attributes stay valid (56461).
            // -> is there a case, in which the attributes of the following spans are also required?
            _.each(allChildren, function (child, index) {
                if (index === 0) {
                    firstChild = child;
                } else {
                    $(child).remove();
                }
            });

            $(firstChild).text(Utils.repeatString('1', fieldTxtLen)); // generating child with valid length
            $(specialField).data('length', 1).removeClass(DOM.SPECIAL_FIELD_NODE_CLASSNAME).after(firstChild); // append span behind special complex field
        }

        /**
         * Local helper used in case of undo of changetracked fields, where range start and range end are not changetracked.
         * Method getComplexFieldIdFromStack needs to have insertRange, insertComplexField, insertRange order.
         * Since ranges are not changetracked, on undo of changetrack operation, only insertComplexField operation comes,
         * and id is not found. This method tries to get id from rangeStart node, that is already in dom.
         *
         * @param  {Array} start
         *  Oxo position where field node should be inserted
         * @param  {String} target
         *  Target id of content node for position.
         * @return {String|Null} Found id, or null.
         */
        function tryGetIDfromPrevRangeNode(start, target) {
            var id = null;
            var prevRangeMarkerPos = Position.decreaseLastIndex(start);
            var startMarkerPoint = Position.getDOMPosition(model.getRootNode(target), prevRangeMarkerPos, true);
            var startMarkerNode = startMarkerPoint && startMarkerPoint.node;

            if (DOM.isRangeMarkerStartNode(startMarkerNode) && DOM.getRangeMarkerType(startMarkerNode) === 'field') {
                id = DOM.getRangeMarkerId(startMarkerNode);
                if (self.getComplexField(id)) { // if field with this id is already in a model, it is not correct id
                    id = null;
                }
            }

            return id;
        }

        // methods ------------------------------------------------------------

        /**
         * Whether the document contains complex fields in the document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one complex field.
         */
        this.isEmpty = function () {
            return _.isEmpty(allFields);
        };

        /**
         * Public method to get all complex fields from model.
         *
         * @returns {Object}
         */
        this.getAllFields = function () {
            return allFields;
        };

        /**
         * Public method to get number of all complex fields from model.
         *
         * @returns {Number}
         */
        this.getAllFieldsCount = function () {
            return _.size(allFields);
        };

        /**
         * Provides a jQuerified range start marker node.
         *
         * @returns {jQuery|null}
         *  The start marker with the specified id, or null, if no such start marker exists.
         */
        this.getComplexField = function (id) {

            var // the value saved under the specified id
                field = allFields[id] || null;

            return _.isString(field) ? getMarginalComplexField(id, field) : field;
        };

        /**
         * Getter for target container id of marginal complex field node.
         *
         * @param{String} id
         * Id of the field in collection model.
         *
         * @returns {jQuery|null}
         *  Target container id of marginal complex field node.
         */
        this.getMarginalComplexFieldsTarget = function (id) {

            var // the value saved under the specified id
                field = allFields[id] || null;

            return _.isString(field) ? field : null;
        };

        /**
         * Check, whether the specified target string is a valid target for a complex field.
         *
         * @param {String} target
         *  The target string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a complex field node.
         */
        this.isComplexFieldId = function (id) {
            return self.getComplexField(id) !== null;
        };

        /**
         * Provides a unique ID for the complex field.
         *
         * @returns {String}
         *  A unique id.
         */
        this.getNextComplexFieldID = function () {
            var // the next free complex field id
                fieldIdString = 'fld' + fieldID++,
                // the range marker object
                rangeMarker = model.getRangeMarker();

            // checking, if there also exist no range markers with this id
            while (rangeMarker.isUsedRangeMarkerId(fieldIdString)) {
                fieldIdString = 'fld' + fieldID++;
            }

            return fieldIdString;
        };

        /**
         * Provides a unique ID for the bookmark.
         *
         * @param {Number} [increment]
         *  Optional increment value for generation of ID.
         *
         * @returns {String}
         *  A unique id.
         */
        this.getNextBookmarkId = function (increment) {
            var bookmarkIdString = 0;
            var allIDs = [];
            var allBookmarks = DOM.getPageContentNode(model.getNode()).find('.bookmark[bmPos="start"]');
            var locInc = increment || 1;

            _.each(allBookmarks, function (bmNode) {
                var id = DOM.getBookmarkId(bmNode);
                if (id && id.length > 2) {
                    id = parseInt(id.substr(2), 10);
                    allIDs.push(id);
                }
            });
            allIDs.sort(function (a, b) { return b - a; });
            if (!allIDs.length) { allIDs.push(bookmarkIdString); } // if there were no achors found, add default value
            if (_.isNumber(allIDs[0])) {
                bookmarkIdString = allIDs[0] + locInc;
            }

            return 'bm' + bookmarkIdString;
        };

        /**
         * Provides a unique ID for the anchor.
         *
         * @param {Number} [increment]
         *  Optional increment value for generation of ID.
         *
         * @returns {String}
         *  A unique id.
         */
        this.getNextAnchorName = function (increment) {
            var defAnchorValue = 500000000;
            var allNames = [];
            var allBookmarks = DOM.getPageContentNode(model.getNode()).find('.bookmark[bmPos="start"]');
            var locInc = increment || 1;

            _.each(allBookmarks, function (bmNode) {
                var anchorName = DOM.getBookmarkAnchor(bmNode);
                if (anchorName) {
                    anchorName = anchorName.match(/\d+/g);
                    if (anchorName) {
                        if (_.isArray(anchorName)) {
                            anchorName = anchorName[0];
                        }
                        anchorName = parseInt(anchorName, 10);
                        allNames.push(anchorName);
                    }
                }
            });
            allNames.sort(function (a, b) { return b - a; });
            if (!allNames.length) { allNames.push(defAnchorValue); } // if there were no achors found, add default value
            if (_.isNumber(allNames[0])) { // highest value is sorted at first position
                defAnchorValue = allNames[0] + locInc;
            }

            return '_Toc' + defAnchorValue;
        };

        /**
         * Public method to remove complex field from model.
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         *
         */
        this.removeFromComplexFieldCollection = function (node) {
            removeFromComplexFieldCollection(DOM.getComplexFieldId(node));
        };

        /**
         * Handler for insertComplexField operations. Inserts a complex field node into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.
         *
         * @param {Object} attrs
         *  The attributes of the complex field.
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        this.insertComplexFieldHandler = function (start, instruction, attrs, target) {

            var // the range marker element
                complexFieldNode = $('<div>', { contenteditable: false }).addClass('inline ' + DOM.COMPLEXFIELDNODE_CLASS),
                // the text span needed for inserting the range marker node
                textSpan = model.prepareTextSpanForInsertion(start, null, target),
                // id of the field, to connect with range markers
                id = model.getFieldManager().getComplexFieldIdFromStack(),
                // the start marker node, that belongs to this complex field
                startMarker = null;

            if (!textSpan) { return false; }

            if (!id) {
                // On undo change-tracked complex field,
                // range start and range end are not handled, and it is possible not to fetch id from stack.
                // Try to fetch range marker and id from previous dom node
                id = tryGetIDfromPrevRangeNode(start, target);

                if (!id) {
                    Utils.error('ComplexField.insertComplexFieldHandler: no id!');
                    return false;
                }
            }
            startMarker = model.getRangeMarker().getStartMarker(id);

            if (attrs && attrs.character &&  !_.isUndefined(attrs.character.autoDateField)) {
                complexFieldNode.attr('data-auto-date', false);
            }

            // saving also the id at the complex field node
            complexFieldNode.attr('data-field-id', id);
            complexFieldNode.data('fieldInstruction', instruction);

            // inserting the complex field node into the DOM (always using the text span at the valid position)
            complexFieldNode.insertAfter(textSpan);

            if (startMarker && Utils.getDomNode($(textSpan).prev()) === Utils.getDomNode(startMarker)) {
                $(textSpan).remove(); // no text span between start marker node and complex field
            } else {
                Utils.log('ComplexField.insertComplexFieldHandler: critical rangemarker start position.');
                // -> TODO for fixing task 53286: This needs to be handled as error, after filter made changes for 53286
                // Utils.error('ComplexField.insertComplexFieldHandler: invalid position of text span, rangemarker not valid!');
                // return false;
            }

            // setting the (change track) attributes
            if (_.isObject(attrs)) { model.getCharacterStyles().setElementAttributes(complexFieldNode, attrs); }

            // adding the complex field node in the collection of all complex fields
            // -> this also needs to be done after loading document from fast load or using local storage
            addIntoComplexFieldCollection(complexFieldNode, id, target);

            markComplexFieldForHighlighting(complexFieldNode, id);

            // updating the paragraph
            if (model.isImportFinished()) { model.implParagraphChangedSync(complexFieldNode.parent()); }

            // handling the global counter
            updateFieldID(id);

            return true;
        };

        /**
         * Handler for updateComplexField operation.
         * Replaces original complex field DOM node with new one, with new instruction.
         *
         * @param {Number[]} start
         *  The logical start position for the complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.
         *
         * @param {Object} attrs
         *  The attributes of the complex field.
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been updated successfully.
         */
        this.updateComplexFieldHandler = function (start, instruction, attrs, target) {

            var undoManager = model.getUndoManager(),
                undoOperation,
                redoOperation,
                complexFieldNode = Position.getDOMPosition(model.getRootNode(target), start, true),
                oldInstruction,
                id;

            if (!complexFieldNode || !complexFieldNode.node) {
                Utils.error('complexfield.updateComplexFieldHandler(): unable to fetch complex field node!');
                return false;
            }
            complexFieldNode = complexFieldNode.node;

            id = DOM.getComplexFieldId(complexFieldNode);
            oldInstruction = DOM.getComplexFieldInstruction(complexFieldNode);

            if (!id || !oldInstruction) {
                Utils.error('complexfield.updateComplexFieldHandler(): unable to fetch id or instruction!');
                return false;
            }
            // if there is data from datepicker, reset with new value of current date
            if ($(complexFieldNode).data('datepickerValue')) {
                $(complexFieldNode).data('datepickerValue', new Date());
            }
            // saving also the id at the complex field node
            $(complexFieldNode).attr('data-field-id', id);

            if (attrs && attrs.character && !_.isUndefined(attrs.character.autoDateField)) {
                $(complexFieldNode).attr('data-auto-date', attrs.character.autoDateField); // special case: for performance reasons, this is not handled in character styles.
                model.getCharacterStyles().setElementAttributes(complexFieldNode, attrs); // This is needed to remove property set in SetAttributes operation, in data of the element!

                // create the undo action
                if (undoManager.isUndoEnabled()) {
                    undoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: { character: { autoDateField: !attrs.character.autoDateField } }, instruction: oldInstruction };
                    redoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: { character: { autoDateField: attrs.character.autoDateField } }, instruction: oldInstruction };
                    if (target) {
                        undoOperation.target = target;
                        redoOperation.target = target;
                    }
                    undoManager.addUndo(undoOperation, redoOperation);
                }
            }

            if (instruction) {
                $(complexFieldNode).data('fieldInstruction', instruction);

                // create the undo action
                if (undoManager.isUndoEnabled()) {
                    undoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: {}, instruction: oldInstruction };
                    redoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: {}, instruction: instruction };
                    if (target) {
                        undoOperation.target = target;
                        redoOperation.target = target;
                    }
                    undoManager.addUndo(undoOperation, redoOperation);
                }
            }

            //markComplexFieldForHighlighting(complexFieldNode, id);

            return true;
        };

        /**
         * Inserting a complex field. This function generates the operations for the ranges and
         * the complex field node. It does not modify the DOM. This function is only executed in
         * the client with edit privileges, not in remote clients.
         *
         * @param {String} fieldType
         *
         * @param {String} fieldFormat
         *
         */
        this.insertComplexField = function (fieldType, fieldFormat) {

            return model.getUndoManager().enterUndoGroup(function () {

                function doInsertComplexField() {
                    var start = selection.getStartPosition(),
                        fieldStart = Position.increaseLastIndex(start),
                        textStart = Position.increaseLastIndex(fieldStart),
                        rangeEndStart = null,
                        generator = model.createOperationGenerator(),
                        operation = {},
                        insertTextOperation = null,
                        // target for operation - if exists, it's for ex. header or footer
                        target = model.getActiveTarget(),
                        representation = null,
                        instruction = null;

                    model.doCheckImplicitParagraph(start);

                    if (fieldType) {
                        operation.start = fieldStart;

                        if (self.isCurrentDate(fieldType) || self.isCurrentTime(fieldType)) {
                            if (!fieldFormat || fieldFormat === 'default') {
                                fieldFormat = LocaleData.SHORT_DATE;
                            }
                            representation = self.getDateTimeRepresentation(fieldFormat);
                            instruction = fieldType.toUpperCase() + (fieldFormat.length ? ' \\@ "' + fieldFormat + '"' : '');
                            operation.attrs = operation.attrs || {};
                            operation.attrs.character = operation.attrs.character || {};
                            operation.attrs.character = { autoDateField: false };
                        } else if (self.isFileName(fieldType) || self.isAuthor(fieldType)) {
                            representation = self.isFileName(fieldType) ? app.getFullFileName() : app.getClientOperationName();
                            if ((/Lower/i).test(fieldFormat)) {
                                representation = representation.toLowerCase();
                            } else if ((/Upper/i).test(fieldFormat)) {
                                representation = representation.toUpperCase();
                            } else if ((/FirstCap/i).test(fieldFormat)) {
                                representation = Utils.capitalize(representation.toLowerCase());
                            } else if ((/Caps/i).test(fieldFormat)) {
                                representation = Utils.capitalizeWords(representation.toLowerCase());
                            }
                            instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                        } else if (self.isNumPages(fieldType)) {
                            representation = self.formatPageFieldInstruction(pageLayout.getNumberOfDocumentPages(), fieldFormat, true);
                            instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                        } else if (self.isPageNumber(fieldType)) {
                            representation = self.formatPageFieldInstruction(pageLayout.getPageNumber(), fieldFormat);
                            instruction = 'PAGE' + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                        }

                        if (representation.length) {
                            operation.instruction = instruction;
                            model.extendPropertiesWithTarget(operation, target);

                            insertTextOperation = { start: textStart, text: representation };

                            // modifying the attributes, if changeTracking is activated
                            if (changeTrack.isActiveChangeTracking()) {
                                operation.attrs = operation.attrs || {};
                                operation.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                                insertTextOperation.attrs = { changes: { inserted: changeTrack.getChangeTrackInfo(), removed: null } };

                            }
                            generator.generateOperation(Operations.RANGE_INSERT, { start: start, type: 'field', position: 'start' });
                            generator.generateOperation(Operations.COMPLEXFIELD_INSERT, operation);
                            generator.generateOperation(Operations.TEXT_INSERT, insertTextOperation);
                            rangeEndStart = Position.increaseLastIndex(textStart, representation.length);
                            generator.generateOperation(Operations.RANGE_INSERT, { start: rangeEndStart, type: 'field', position: 'end' });

                            // setting change track information at range marker nodes
                            if (changeTrack.isActiveChangeTracking()) {
                                generator.generateOperation(Operations.SET_ATTRIBUTES, { start: start, attrs: { changes: _.copy(operation.attrs.changes, true) } });
                                generator.generateOperation(Operations.SET_ATTRIBUTES, { start: rangeEndStart, attrs: { changes: _.copy(operation.attrs.changes, true) } });
                            }

                            model.applyOperations(generator.getOperations());

                            selection.setTextSelection(Position.increaseLastIndex(rangeEndStart));
                        } else {
                            Utils.warn('complexField.insertComplexField(): representation missing!');
                        }
                    }
                }

                if (selection.hasRange()) {
                    return model.deleteSelected()
                        .done(function () {
                            doInsertComplexField();
                        });
                }

                doInsertComplexField();
                return $.when();

            });
        };

        /**
         * Creates and applies operations for generating Table of Contents as complex field.
         *
         * @param {String} fieldFormat
         *
         * @param {Object} options
         *  @param {String} options.tabStyle
         *      Style of the tabstops in table of contents
         *  @param {Boolean} [options.testMode=false]
         *      When set to true, pagination is ignored.
         *      Warning: used for speeding up unit tests only. Do not use this flag for other purposes!
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the insertion of table of contents
         *  ended. It is rejected, if the dialog has been canceled.
         */
        this.insertTOCField = function (fieldFormat, options) {
            return insertTableOfContents(fieldFormat, options);
        };

        /**
         * Creates and applies operations for updating Table of Contents as complex field.
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String|Array} format
         *  Format of the field with switches. Can be one string, or array of strings.
         *
         * @param {Object} options
         *  @param {String} options.tabStyle
         *      Style of the tabstops in table of contents
         *  @param {Boolean} [options.testMode=false]
         *      When set to true, pagination is ignored.
         *      Warning: used for speeding up unit tests only. Do not use this flag for other purposes!
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the insertion of table of contents
         *  ended. It is rejected, if the dialog has been canceled.
         */
        this.updateTOCField = function (node, instruction, options) {
            var dividedData = self.cleanUpAndExtractType(instruction);
            instruction = dividedData.instruction;

            return updateTableOfContents(node, instruction, options);
        };

        /**
         * Public method that updates format of selected complex field.
         *
         * @param {String} fieldType
         *  Type of the field.
         * @param {String} fieldFormat
         *  New format that field will have.
         * @param {Array} fieldNodePos
         *  Oxo position of complex field node itself, between range markers.
         * @param {String} fieldId
         *  Id of updated field
         *
         * @returns {jQuery.Promise}
         */
        this.updateComplexFieldFormat = function (fieldType, fieldFormat, fieldNodePos, fieldId) {
            var generator = model.createOperationGenerator();
            var operation = {};
            var target = model.getActiveTarget();
            var representation = null;
            var instruction = null;
            var insertTextOperation = null;
            var textStart = Position.increaseLastIndex(fieldNodePos);
            var startPos = fieldNodePos;
            var rangeEndNode = rangeMarker.getEndMarker(fieldId);
            if (!rangeEndNode) {
                Utils.error('complexField.updateComplexFieldFormat(): missing range end node for the id: ', fieldId);
                return $.when();
            }
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeEndNode);

            if (fieldType && fieldNodePos.length) {
                operation.start = fieldNodePos;

                if (self.isCurrentDate(fieldType) || self.isCurrentTime(fieldType)) {
                    if (!fieldFormat || fieldFormat === 'default') {
                        fieldFormat = LocaleData.SHORT_DATE;
                    }
                    representation = self.getDateTimeRepresentation(fieldFormat);
                    instruction = fieldType.toUpperCase() + (fieldFormat.length ? ' \\@ "' + fieldFormat + '"' : '');
                } else if (self.isFileName(fieldType) || self.isAuthor(fieldType)) {
                    representation = self.isFileName(fieldType) ? app.getFullFileName() : app.getClientOperationName();
                    if ((/Lower/i).test(fieldFormat)) {
                        representation = representation.toLowerCase();
                    } else if ((/Upper/i).test(fieldFormat)) {
                        representation = representation.toUpperCase();
                    } else if ((/FirstCap/i).test(fieldFormat)) {
                        representation = Utils.capitalize(representation.toLowerCase());
                    } else if ((/Caps/i).test(fieldFormat)) {
                        representation = Utils.capitalizeWords(representation.toLowerCase());
                    }
                    instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                } else if (self.isNumPages(fieldType)) {
                    representation = self.formatPageFieldInstruction(pageLayout.getNumberOfDocumentPages(), fieldFormat, true);
                    instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                } else if (self.isPageNumber(fieldType)) {
                    representation = self.formatPageFieldInstruction(pageLayout.getPageNumber(), fieldFormat);
                    instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                }

                if (representation.length) {
                    operation.instruction = instruction;
                    //operation.id = fieldId;
                    model.extendPropertiesWithTarget(operation, target);

                    insertTextOperation = { start: textStart, text: representation };

                    if (startPos && endPos) {
                        return model.getUndoManager().enterUndoGroup(function () {

                            if (changeTrack.isActiveChangeTracking()) { // insert new field
                                var changeTrackInfo = changeTrack.getChangeTrackInfo();
                                var rangeStartNode = rangeMarker.getStartMarker(fieldId);
                                var startRangePos = Position.getOxoPosition(selection.getRootNode(), rangeStartNode);
                                var startPosNew = _.clone(startRangePos);

                                generator.generateOperation(Operations.RANGE_INSERT, { start: startPosNew, type: 'field', position: 'start', attrs: { changes: { inserted: changeTrackInfo } } });
                                startPosNew = Position.increaseLastIndex(startPosNew);
                                generator.generateOperation(Operations.COMPLEXFIELD_INSERT, { start: startPosNew, instruction: operation.instruction, attrs: { changes: { inserted: changeTrackInfo } } });
                                startPosNew = Position.increaseLastIndex(startPosNew);
                                generator.generateOperation(Operations.TEXT_INSERT, { start: startPosNew, text: representation, attrs: { changes: { inserted: changeTrackInfo } } });
                                startPosNew = Position.increaseLastIndex(startPosNew, representation.length);
                                generator.generateOperation(Operations.RANGE_INSERT, { start: startPosNew, type: 'field', position: 'end', attrs: { changes: { inserted: changeTrackInfo } } });

                                //  mark old field as removed
                                var lenOffset = 3 + representation.length;
                                generator.generateOperation(Operations.SET_ATTRIBUTES, { start: Position.increaseLastIndex(startRangePos, lenOffset), end: Position.increaseLastIndex(endPos, lenOffset), attrs: { changes: { removed: changeTrackInfo } } });

                                model.applyOperations(generator.getOperations());
                                selection.setTextSelection(Position.increaseLastIndex(startPosNew));

                                return $.when();
                            } else {
                                selection.setTextSelection(textStart, endPos);
                                return model.deleteSelected().done(function () {

                                    // create update operation, and update content
                                    generator.generateOperation(Operations.COMPLEXFIELD_UPDATE, operation);
                                    generator.generateOperation(Operations.TEXT_INSERT, insertTextOperation);
                                    model.applyOperations(generator.getOperations());
                                    selection.setTextSelection(Position.increaseLastIndex(textStart, representation.length + 1));
                                });
                            }
                        });
                    } else {
                        Utils.error('complexField.updateDateTimeFieldCx(): Wrong start and end postitions: ', startPos, endPos);
                    }
                } else {
                    Utils.warn('complexField.updateComplexField(): representation missing!');
                }
            }

            return $.when();
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateComplexFieldCollector = function (para) {

            var // whether each single node needs to be checked
                checkMarginal = false,
                // all complex fields inside the paragraph
                nodeFields = null;

            // not necessary for paragraphs in header or footer -> only the target are stored, not the nodes
            if (DOM.isMarginalNode(para)) { return; }

            // if this is not a paragraph, each single node need to be checked
            checkMarginal = !DOM.isParagraphNode(para);

            nodeFields = $(para).find(DOM.COMPLEXFIELDNODE_SELECTOR);

            // update  the complex field nodes in the collection objects
            _.each(nodeFields, function (oneField) {
                if (!checkMarginal || !DOM.isMarginalNode(oneField)) {
                    // simply overwriting the old complex fields with the new values
                    addIntoComplexFieldCollection(oneField, DOM.getComplexFieldId(oneField));
                }
            });
        };

        /**
         * After load from local storage or fast load the collectors for the complex fields need to be filled.
         *
         * @param {Boolean} usedLocalStorage
         *  Whether the document was loaded from the local storage.
         *
         * @param {Boolean} usedFastLoad
         *  Whether the document was loaded with the fast load process.
         */
        this.refreshComplexFields = function (usedLocalStorage, usedFastLoad) {

            var // the page content node of the document
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // all range marker nodes in the document
                allComplexFieldNodes = pageContentNode.find(DOM.COMPLEXFIELDNODE_SELECTOR),
                // a collector for all marginal template nodes
                allMargins = pageLayout.getHeaderFooterPlaceHolder().children(),
                // collector for all complex nodes in comment layer
                commentFieldNodes = $(DOM.getCommentLayerNode(model.getNode())).find(DOM.COMPLEXFIELDNODE_SELECTOR),
                // the target string node
                target = '',
                // whether an update of header and footer is required
                updateMarginalNodes = false;

            // helper function to add a collection of range marker nodes into the model
            function addMarginalNodesIntoCollection(collection) {
                _.each(collection, function (oneComplexField) {
                    addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField), target);
                    updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                    if (usedFastLoad) {
                        removePreviousEmptyTextSpan(oneComplexField);
                        updateMarginalNodes = true;
                    }
                });
                pageLayout.replaceAllTypesOfHeaderFooters();
            }

            // reset model
            allFields = {};

            // 1. Search in page content node
            // 2. Search in header/footer template node

            // adding all range marker nodes into the collector (not those from header or footer)
            _.each(allComplexFieldNodes, function (oneComplexField) {
                if (!DOM.isMarginalNode($(oneComplexField).parent())) {
                    addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField));
                    updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                    if (usedFastLoad) { removePreviousEmptyTextSpan(oneComplexField); }
                }
            });

            // adding the range markers that are located inside header or footer (using the template node)
            _.each(allMargins, function (margin) {

                var // a collector for all range marker nodes inside one margin
                    allMarginComplexFieldNodes = $(margin).find(DOM.COMPLEXFIELDNODE_SELECTOR);

                if (allMarginComplexFieldNodes.length > 0) {
                    target = DOM.getTargetContainerId(margin);
                    addMarginalNodesIntoCollection(allMarginComplexFieldNodes);
                }
            });

            // adding fields that are in comment layer
            _.each(commentFieldNodes, function (oneComplexField) {
                addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField));
                updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                if (usedFastLoad) {
                    removePreviousEmptyTextSpan(oneComplexField);
                }
                // mark field node
                $(oneComplexField).addClass(DOM.FIELD_IN_COMMENT_CLASS);
            });

            // if empty text spans were removed in header or footer node, an update of the header and
            // footer nodes in the document is required
            if (updateMarginalNodes) {
                // TODO:
                // Refreshing the marginal nodes in the document with the nodes from the template node (not available yet)
            }

            _.each(pageContentNode.find(DOM.RANGEMARKER_STARTTYPE_SELECTOR).filter('[data-range-type="field"]'), function (rangeStartNode) {
                var attributes = $(rangeStartNode).data('attributes');
                if (attributes && attributes.character && attributes.character.field && attributes.character.field.formFieldType === 'checkBox') {
                    $(rangeStartNode).text(attributes.character.field.checked ? '☒' : '☐');
                }
            });
        };

        /**
         * After deleting a complete paragraph, it is necessary, that all complex field nodes
         * in the deleted paragraph are also removed from the model collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.removeAllInsertedComplexFields = function (para) {

            var // all range markers inside the paragraph
                nodeFields = $(para).find(DOM.COMPLEXFIELDNODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(nodeFields, function (field) {

                var // the id of the marker node
                    fieldId = DOM.getComplexFieldId(field);

                if (_.isString(fieldId)) {
                    removeFromComplexFieldCollection(fieldId);
                }
            });
        };

        /**
         * When pasting content, that contains complex fields, it is necessary, that the
         * complex fields get a new unique id.
         *
         * @param {Object[]} operations
         *  The collection with all paste operations.
         */
        this.handlePasteOperationTarget = function (operations) {

            var // a container for all complex field ids
                allIDs = [],
                // a container for all range marker ids of complex fields
                allRangeMarkerIDs = {};

            // replacing the complex field ids
            function replaceID(ops, id1, id2) {

                _.each(ops, function (op) {
                    if (op) {
                        if (op.id === id1) { op.id = id2; }
                        if (op.target === id1) { op.target = id2; }
                    }
                });
            }

            // collecting all 'id's of insertComplexField operations and assign new target values
            _.each(operations, function (operation) {
                if (operation && operation.name === Operations.COMPLEX_FIELD_INSERT) {
                    allIDs.push(operation.id);
                } else if (operation && operation.name === Operations.RANGE_INSERT && operation.type === 'field') {
                    allRangeMarkerIDs[operation.id] = 1;
                }
            });

            // iterating over all registered complex field ids
            if (allIDs.length > 0) {
                _.each(allIDs, function (oldID) {
                    var newID = self.getNextComplexFieldID();
                    // iterating over all operations, so that also the range markers will be updated
                    replaceID(operations, oldID, newID);
                    delete allRangeMarkerIDs[oldID];
                });
            }

            // are there still unmodified range marker targets (maybe only the start marker was pasted)
            _.each(_.keys(allRangeMarkerIDs), function (oldID) {
                var newID = self.getNextComplexFieldID();
                // iterating over all operations
                replaceID(operations, oldID, newID);
            });

        };

        /**
         * Checks instruction, and depending of extracted field type,
         * updates passed complex field node.
         *
         * @param {jQuery|Node} node
         *  Node that is going to be updated.
         *
         * @param {String} instruction
         *  Field instruction.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after updateTableOfContents is resolved,
         *  or immediately in case of any other syncronosly executed field.
         */
        this.updateByInstruction = function (node, instruction) {
            var dividedData = self.cleanUpAndExtractType(instruction);
            var type = dividedData.type;
            var promise = $.when();
            instruction = dividedData.instruction;

            if (type) {
                if (self.isNumPages(type)) {
                    updateNumPagesFieldCx(node, instruction);
                } else if (self.isPageNumber(type)) {
                    updatePageNumberFieldCx(node, instruction);
                } else if (self.isCurrentDate(type)) {
                    updateDateTimeFieldCx(node, instruction);
                } else if (self.isCurrentTime(type)) {
                    updateDateTimeFieldCx(node, instruction, { time: true });
                } else if (self.isFileName(type)) {
                    updateFileNameFieldCx(node, instruction);
                } else if (self.isAuthor(type)) {
                    updateAuthorFieldCx(node, instruction);
                } else if (self.isTableOfContents(type)) {
                    // this is async and returns promise, next field update needs to run after this promise is resolved
                    promise = updateTableOfContents(node, instruction);
                }
            }
            return promise;
        };

        /**
         * Update date and time fields before running download or print action.
         *
         * @param {jQuery|Node} field
         *  Field to be updated.
         *
         * @param {String} instruction
         *  Instruction for updating.
         */
        this.updateDateTimeFieldOnDownload = function (field, instruction) {
            var dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type;
            instruction = dividedData.instruction;

            if (type && (self.isCurrentDate(type) || self.isCurrentTime(type)) && (!DOM.isMarginalNode(field) || !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), field))) { // #42093
                updateDateTimeFieldCx(field, instruction, { time: self.isCurrentTime(type) });
            }
        };

        /**
         * Converts normal complex field to special complex field, when field is inside header/footer and has type PAGE.
         * Content of field is detached and inserted inside complex field.
         * Inverse function from this function is restoreSpecialFields.
         *
         * @param {String} fieldId
         *  Id of the field
         *
         * @param {String} marginalTarget
         *  Id of the marginal target node
         *
         * @param {String} type
         *  Type of field: num pages or page number
         */
        this.convertToSpecialField = function (fieldId, marginalTarget, type) {
            var innerContent = $();
            var rootNode = model.getRootNode(marginalTarget);
            var rangeStart = rootNode.find('[data-range-id="' + fieldId + '"]').filter(DOM.RANGEMARKER_STARTTYPE_SELECTOR);
            var rangeEnd = rootNode.find('[data-range-id="' + fieldId + '"]').filter(DOM.RANGEMARKER_ENDTYPE_SELECTOR);
            if (!rangeStart.length || !rangeEnd.length) {
                Utils.error('complexfield.convertToSpecialField(): missing rangeStart or rangeEnd for the field with id: ', fieldId);
                return;
            }
            var fieldNode = null;
            var startPos = Position.getOxoPosition(rootNode, rangeStart);
            var endPos = Position.getOxoPosition(rootNode, rangeEnd);
            var innerLen = _.last(endPos) - _.last(startPos);
            var isNumPages = (/NUMPAGES/i).test(type);
            var className = isNumPages ? 'cx-numpages' : 'cx-page';
            var replacementNode;
            var helperNode;

            if (innerLen > 0) {
                innerLen -= 1;
            } else {
                Utils.warn('complexField.convertToSpecialField(): inner length is < 1');
            }

            if (rangeStart.length) {
                fieldNode = rangeStart.next();
                if (!DOM.isComplexFieldNode(fieldNode)) {
                    Utils.warn('complexField.convertToSpecialField(): cx field node not found after range start!');
                    return;
                }
            }

            helperNode = rangeEnd.prev();
            while (helperNode.length && (!DOM.isComplexFieldNode(helperNode) || DOM.getComplexFieldId(helperNode) !== fieldId)) {
                innerContent = innerContent.add(helperNode);
                helperNode = helperNode.prev();
            }
            innerContent.detach().empty();
            fieldNode.append(innerContent).data('length', innerLen).addClass(DOM.SPECIAL_FIELD_NODE_CLASSNAME).addClass(className);

            // the child spans must be valid text spans with text node to enable iteration with 'DOM.iterateTextSpans' (56996)
            _.each(innerContent, function (node) {
                DOM.ensureExistingTextNodeInSpecialField(node);
            });

            if (!DOM.isMarginalPlaceHolderNode(rootNode.parent())) {
                replacementNode = rootNode.children().clone(true);
                replacementNode.find('[contenteditable="true"]').attr('contenteditable', false);
                pageLayout.getHeaderFooterPlaceHolder().children('[data-container-id="' + marginalTarget + '"]').empty().append(replacementNode);
            }
        };

        /**
         * Restoring special field to normal is required before delete operation.
         *
         *@param {jQuery} specialField
         */
        this.restoreSpecialField = function (specialField) {
            restoreSpecialField(specialField);
        };

        /**
         * Check if there are special fields of type page in header, inside given searchNode,
         * and if yes, restore them before deleting them.
         *
         * @param {Node} searchNode
         *  Node which we search for special fields.
         *
         * @param {Number} startOffset
         *  Start offset of given node.
         *
         * @param {Number} endOffset
         *  End offset of given node.
         */
        this.checkRestoringSpecialFields = function (searchNode, startOffset, endOffset) {

            if (DOM.isMarginalNode(searchNode) && $(searchNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR).length) {
                _.each($(searchNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR), function (specialField) {
                    var fieldPos = Position.getOxoPosition(selection.getRootNode(), specialField, 0);
                    if (Position.isNodePositionInsideRange(fieldPos, startOffset, endOffset)) {
                        restoreSpecialField(specialField);
                    }
                });
            }
        };

        /**
         * Check if there are special fields in container node, and revert conversion from special field to normal,
         * which is necessary before creating undo operations.
         *
         * @param {jQuery|Node} containerNode
         *  Can be whole header or footer, not restricted to paragraph parts.
         *  For that, see function: checkRestoringSpecialFields
         */
        this.checkRestoringSpecialFieldsInContainer = function (containerNode) {
            _.each($(containerNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR), function (specialField) {
                restoreSpecialField(specialField);
            });
        };

        /**
         * After cloning headers&footers special page fields have to be updated with proper page number.
         */
        this.updateCxPageFieldsInMarginals = function () {
            var $pageContentNode = null,
                firstHeadNum = 1,
                lastFootNum = null,
                // first header in doc
                $firstHeaderFields = null,
                // last footer in doc
                $lastFooterFields = null,
                numPagesNodes = null;

            // no special fields in the document, don't make unnecessary loops
            if (self.isEmpty() || !pageLayout.getHeaderFooterPlaceHolder().find(DOM.SPECIAL_FIELD_NODE_SELECTOR).length) { return; }

            // performance: assign variables only if really needed
            $pageContentNode = DOM.getPageContentNode(model.getNode());
            lastFootNum = pageLayout.getNumberOfDocumentPages();
            // first header in doc - page number always 1
            $firstHeaderFields = pageLayout.getFirstHeaderWrapperNode().children(DOM.HEADER_SELECTOR).find('.cx-page');
            // last footer in doc gets total page count as page number
            $lastFooterFields = pageLayout.getLastFooterWrapperNode().children(DOM.FOOTER_SELECTOR).find('.cx-page');

            _.each($firstHeaderFields, function (headerField) {
                var $headerField = $(headerField),
                    headerFormat = DOM.getComplexFieldInstruction($headerField);

                // nesting of fields in special fields (page fields in header/footer) is not allowed;
                // first span should not be deleted, to preserve formatting
                $headerField.children().not('span:first').remove();

                $headerField.children().empty().html(self.formatPageFieldInstruction(firstHeadNum, headerFormat));
            });

            _.each($lastFooterFields, function (footerField) {
                var $footerField = $(footerField),
                    footerFormat = DOM.getComplexFieldInstruction($footerField);

                $footerField.children().not('span:first').remove();
                $footerField.children().empty().html(self.formatPageFieldInstruction(lastFootNum, footerFormat));
            });

            _.each(pageLayout.getPageBreaksCollection(), function (pageBreak) {
                var $pageBreak = $(pageBreak),
                    headerNumber = $pageBreak.data('page-num') + 1,
                    footerNumber = $pageBreak.data('page-num'),
                    $headerFields = $pageBreak.children(DOM.HEADER_SELECTOR).find('.cx-page'),
                    $footerFields = $pageBreak.children(DOM.FOOTER_SELECTOR).find('.cx-page');

                _.each($headerFields, function (headerField) {
                    var $headerField = $(headerField),
                        headerFormat = DOM.getComplexFieldInstruction($headerField);

                    $headerField.children().not('span:first').remove();

                    $headerField.children().empty().html(self.formatPageFieldInstruction(headerNumber, headerFormat));
                });

                _.each($footerFields, function (footerField) {
                    var $footerField = $(footerField),
                        footerFormat = DOM.getComplexFieldInstruction($footerField);

                    $footerField.children().not('span:first').remove();
                    $footerField.children().empty().html(self.formatPageFieldInstruction(footerNumber, footerFormat));
                });
            });

            // update number of pages fields, which all have same value
            if (pageLayout.getHeaderFooterPlaceHolder().find('.cx-numpages').length) {
                numPagesNodes = $pageContentNode.find(DOM.PAGE_BREAK_SELECTOR).add(pageLayout.getFirstHeaderWrapperNode()).add(pageLayout.getLastFooterWrapperNode()).find('.cx-numpages');
                _.each(numPagesNodes, function (fieldNode) {
                    var $fieldNode = $(fieldNode),
                        fieldFormat = DOM.getComplexFieldInstruction($fieldNode);

                    // nesting of fields in special fields (page fields in header/footer) is not allowed;
                    // first span should not be deleted, to preserve formatting
                    $fieldNode.children().not('span:first').remove();

                    $fieldNode.children().empty().html(self.formatPageFieldInstruction(lastFootNum, fieldFormat, true));
                });
            }
        };

        /**
         * Function called after importing document.
         * It loops all fields in collection and:
         *  - marks them for highlighting,
         *  - updates date and time fields
         *  - converts page fields in header&footer to special fields
         */
        this.updateDateTimeAndSpecialFields = function () {
            var updatePageFields = false,
                returnOriginalCursorPos = false,
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition();

            _.each(allFields, function (entry, fieldId) {
                var field = self.getComplexField(fieldId),
                    instruction = DOM.getComplexFieldInstruction(field),
                    dividedData = self.cleanUpAndExtractType(instruction),
                    type = dividedData.type,
                    format = null;

                instruction = dividedData.instruction;

                markComplexFieldForHighlighting(field, fieldId);

                if (type) {
                    if ((self.isCurrentDate(type) || self.isCurrentTime(type)) && app.isEditable()) {
                        format = instruction;

                        if (DOM.isAutomaticDateField(field) && (!DOM.isMarginalNode(field) || !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), field))) { // update only if not fixed field, #42093
                            // if previously updated field is marginal, and this is not, leave header edit state
                            selection.swichToValidRootNode(field);
                            updateDateTimeFieldCx(field, format, { time: self.isCurrentTime(type) });
                            if (pageLayout.isIdOfMarginalNode(entry)) {
                                markComplexFieldForHighlighting(self.getComplexField(fieldId), fieldId);
                                pageLayout.getHeaderFooterPlaceHolder().children('.' + entry).empty().append(selection.getRootNode().children().clone(true));
                            }
                            returnOriginalCursorPos = true;
                        }
                    } else if (_.isString(entry) && pageLayout.isIdOfMarginalNode(entry) && self.isPageNumber(type)) {
                        if (!DOM.isSpecialField(field)) {
                            self.convertToSpecialField(fieldId, entry, type);
                        }
                        updatePageFields = true;
                    }
                } else {
                    Utils.warn('complexField.updateDateTimeAndSpecialFields: invalid type: ', type);
                }
            });
            if (updatePageFields) {
                // update reference in collection because of replaced node
                pageLayout.headerFooterCollectionUpdate();
                pageLayout.replaceAllTypesOfHeaderFooters(); // complex fields in header/footers are refreshed in this method call, if any
            }
            if (returnOriginalCursorPos) {
                // if there was update of date fields, return cursor to first document position
                if (model.isHeaderFooterEditState()) {
                    pageLayout.leaveHeaderFooterEditMode();
                }
                model.setActiveTarget();
                selection.setNewRootNode(model.getNode());
                selection.setTextSelection(startPos, endPos);
            }
        };

        /**
         * Public method for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNodeCx = function ($node, isHeader, pageCount) {
            if (isHeader) {
                if (DOM.isHeaderWrapper($node.parent())) {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader, 1);
                } else {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader);
                }
            } else {
                if (DOM.isFooterWrapper($node.parent())) {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader, pageCount);

                } else {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader);
                }
            }
        };

        /**
         *
         *
         * @param $node
         * @param pageNum
         */
        this.updatePageNumInCurrentMarginalCx = function ($node, isHeader, pageNum) {
            _.each($node.find('.cx-page'), function (field) {
                var $field = $(field),
                    format = DOM.getComplexFieldInstruction($field),
                    pageNumber = pageNum || ($(field).parentsUntil('.pagecontent', '.page-break').data('page-num') + (isHeader ? 1 : 0));

                $field.children().empty().html(self.formatPageFieldInstruction(pageNumber, format));
            });
        };

        /**
         * Public method for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNodeCx = function ($node, pageCount) {
            _.each($node.find('.cx-numpages'), function (fieldNode) {
                var $fieldNode = $(fieldNode),
                    fieldFormat = DOM.getComplexFieldInstruction($fieldNode);

                $fieldNode.children().empty().html(self.formatPageFieldInstruction(pageCount, fieldFormat, true));
            });
        };

        /**
        * Update date field with value from popup, datepicker or input field.
        * @param {jQuery} node
        *   Field node that is going to be updated
        * @param {String} fieldValue
        *   Value with which will field be updated.
        */
        this.updateDateFromPopupValue = function (node, fieldValue) {
            var formattedDate = fieldValue;
            var id = DOM.getComplexFieldMemberId(node);
            var isMarginal = DOM.isMarginalNode(node);
            var rootNode = isMarginal ? DOM.getClosestMarginalTargetNode(node) : selection.getRootNode();
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(rootNode, node));
            var rangeEndNode = rangeMarker.getEndMarker(id);
            if (!rangeEndNode) {
                Utils.error('complexField.updateDateFromPopupValue(): missing range end node for the id: ', id);
                return;
            }
            var endPos = Position.getOxoPosition(rootNode, rangeEndNode);

            if (startPos && endPos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    return model.deleteSelected().done(function () {
                        model.insertText(formattedDate, startPos);
                        selection.setTextSelection(Position.increaseLastIndex(startPos, formattedDate.length + 1));
                    });
                });
            } else {
                Utils.error('complexField.updateDateTimeFieldCx(): Wrong start and end postitions: ', startPos, endPos);
            }
        };

        /**
         * Public method to mark all nodes that belong to complex field,
         * and are inside range start and range end of a field.
         *
         * @param {HTMLElement|jQuery} field
         *  Complex field node.
         *
         * @param {String} id
         *  Id of processed field.
         */
        this.markComplexFieldForHighlighting = function (field, id) {
            markComplexFieldForHighlighting(field, id);
        };

        /**
         * Public method to mark all complex fields contents after undo/redo.
         */
        this.markAllFieldsAfterUndo = function () {
            _.each(allFields, function (node, fieldId) {
                markComplexFieldForHighlighting(node, fieldId);
            });
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            selection = model.getSelection();
            pageLayout = model.getPageLayout();
            numberFormatter = model.getNumberFormatter();
            rangeMarker = model.getRangeMarker();
            changeTrack = model.getChangeTrack();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = model = selection = pageLayout = numberFormatter = rangeMarker = changeTrack = allFields = null;
        });

    } // class ComplexField

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class BaseField
    return BaseField.extend({ constructor: ComplexField });
});
