/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/components/field/basefield', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/textframework/utils/textutils'
], function (DateUtils, Formatter, TriggerObject, Utils) {

    'use strict';

    // class BaseField ============================================

    /**
     * An instance of this class represents the base class for all fields in the
     * edited document.
     *
     * @constructor
     *
     */
    function BaseField(app) {

        var
            self = this,
            // instance of the model
            model = null,
            // instance of the number formatter
            numberFormatter = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this, app);

        // public methods -----------------------------------------------------

        /**
         * Helper functions to determine type of field from instruction
         */
        this.isNumPages = function (type) {
            return (/NUMPAGES/i).test(type);
        };

        this.isPageNumber = function (type) {
            return (/PAGE/i).test(type) && !(/PAGEREF/i).test(type);
        };

        this.isCurrentDate = function (type) {
            return (/^DATE/i).test(type);
        };

        this.isCurrentTime = function (type) {
            return (/^TIME/i).test(type);
        };

        this.isTableOfContents = function (type) {
            return (/^TOC/i).test(type);
        };

        /*
        function isPrintDate(type) {
            return (/PRINTDATE/i).test(type);
        }

        function isSaveDate(type) {
            return (/SAVEDATE/i).test(type);
        }
        */

        this.isFileName = function (type) {
            return (/FILE-?NAME/i).test(type);
        };

        this.isAuthor = function (type) {
            return (/^AUTHOR/i).test(type) || type === 'author-name' || type === 'creator';
        };

        /**
         * Parse and trim instruction string. Remove escape chars,
         * format specific @ and * switches, and trims whitespaces.
         *
         * @param {String} instruction
         */
        this.trimFormatInstruction = function (instruction) {
            var tempFormat = instruction.replace(/"/g, '');

            if (!tempFormat) { return null; }

            tempFormat = tempFormat.replace(/[@*]/g, '');

            return Utils.trimString(tempFormat);
        };

        /**
         * From raw passed instruction string, extract type,
         * and clean up remaining instruction for future formatting.
         *
         * @param {String} instruction
         *  Instruction to be cleaned up.
         *
         * @returns {Object}
         */
        this.cleanUpAndExtractType = function (instruction) {
            var type = null,
                isDateTime = null;

            instruction = instruction.replace(/\\\* MERGEFORMAT/g, ''); // ignoring merge format for now
            instruction = instruction.replace(/\\\./g, '.'); // strip escape characters
            instruction = instruction.split('\\');
            if (_.isArray(instruction) && instruction.length > 0) {
                type =  Utils.trimString(instruction[0]);

                if ((self.isCurrentDate(type) || self.isCurrentTime(type))) {
                    isDateTime = true;
                }
                if (instruction.length > 1) {
                    if (instruction.length > 2) { // can happen than field can have combined more than one format
                        // first format always wins
                        if (isDateTime) {
                            if (instruction[1].indexOf('@') === 0) {
                                instruction = self.trimFormatInstruction(instruction[1]);
                            }
                        } else {
                            if (instruction[1].indexOf('*') === 0) {
                                instruction = self.trimFormatInstruction(instruction[1]);
                            }
                        }
                    } else {
                        instruction = self.trimFormatInstruction(instruction[1]);
                    }

                } else {
                    instruction = null;
                }
                instruction = instruction || 'default'; // convert null to default format
            }
            return { type: type, instruction: instruction };
        };

        /**
         * Converts number for page and numPages fields into proper format code.
         *
         * @param {Number} number
         *  Number to be inserted into field.
         *
         * @param {String} format
         *  Format of the number inserted into field.
         *
         * @param {Boolean} [numPages]
         *  NumPages field have some formats different from PageNum.
         *
         * @returns {String}
         *  Formatted number.
         */
        this.formatPageFieldInstruction = function (number, format, numPages) {

            if (!_.isNumber(number)) {
                Utils.warn('basefield.formatPageFieldInstruction(): invalid number argument!');
                return '';
            }

            // ODF format codes
            if (app.isODF()) {
                switch (format) {
                    case 'A': return Formatter.formatAlphabetic(number, 'repeat');
                    case 'a': return Formatter.formatAlphabetic(number, 'repeat', { lower: true });
                    case 'I': return Formatter.formatRoman(number, 4);
                    case 'i': return Formatter.formatRoman(number, 4, { lower: true });
                    default: return number.toString();
                }
            }

            // OOXML format codes
            if (_.isString(format)) {
                if (/ALPHABETIC/.test(format)) { return Formatter.formatAlphabetic(number, 'repeat'); }
                if (/alphabetic/.test(format)) { return Formatter.formatAlphabetic(number, 'repeat', { lower: true }); }
                if (/ROMAN/.test(format)) { return Formatter.formatRoman(number, 4); }
                if (/roman/.test(format)) { return Formatter.formatRoman(number, 4, { lower: true }); }
                if (/ArabicDash/.test(format) && !numPages) { return '- ' + number + ' -'; }
                if (/Hex/.test(format) && numPages) { return number.toString(16); }
                if (/Ordinal/.test(format) && numPages) { return number + '.'; }
            }
            return number.toString();
        };

        /**
         * Create current date(time) value string with passed format type.
         *
         * @param {String} formatCode
         * @return {String}
         */
        this.getDateTimeRepresentation = function (formatCode) {
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            return numberFormatter.formatValue(parsedFormat, serial);
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            numberFormatter = model.getNumberFormatter();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = numberFormatter = null;
        });

    } // class BaseField

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BaseField });
});
