/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/iconrenderer', [
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/canvas'
], function (ValueMap, Canvas) {

    'use strict';

    // mathematical constants
    var PI = Math.PI;
    var PI16 = PI / 6;
    var PI14 = PI / 4;
    var PI12 = PI / 2;
    var PI34 = 3 * PI14;
    var PI54 = 5 * PI14;
    var PI56 = 5 * PI16;
    var PI74 = 7 * PI14;

    // class IconRenderer =====================================================

    /**
     * Renderer for the single icons of an icon set.
     *
     * @constructor
     *
     * @param {Number} size
     *  The effective icon size (width and height) that will be used to render
     *  the icons, in pixels. See static method IconRenderer.getEffectiveSize()
     *  for more details.
     */
    function IconRenderer(size) {

        // the effective icon size
        this.size = size;

        // line width for outlines, according to icon size
        this.LW = this.lineWidth(1);
        // line width for thick lines used for signs with shadow effect
        this.LW2 = this.lineWidth(2.5);
        // line offset needed to keep the outlines on entire pixels
        this.LO = this.LW / 2;

        // inner size (maximum dimension of the path)
        this.SI = this.size - this.LW;

        // leading position for outlines (top or left lines)
        this.C0 = this.coord(0);
        // center position for outlines (centered horizontal or vertical lines)
        this.CC = this.size / 2;
        // trailing position for outlines (for bottom or right lines)
        this.CE = this.mirror(this.C0);

        // radius of a full-size circle
        this.R = this.CC - this.C0;

        // the rendering context to be drawn into (will be set while creating an icon)
        this.context = null;
        // the cache for pre-rendered icons, mapped by icon identifiers
        this.cache = new ValueMap();

    } // class IconRenderer

    // constants --------------------------------------------------------------

    /**
     * A map with names and RGB values of all colors used in icon sets.
     *
     * @constant
     * @type Object<String>
     */
    IconRenderer.COLORS = {
        BLACK: '#555',
        BLUE: '#46f',
        GRAY: '#bbb',
        GREEN: '#2a4',
        PINK: '#faa',
        RED: '#e42',
        WHITE: '#fff',
        YELLOW: '#dc2'
    };

    // static methods ---------------------------------------------------------

    /**
     * Returns the effective line width for the specified icon size, and base
     * line width, to be used to render the outlines of icons.
     *
     * @param {Number} size
     *  The icon size, in pixels.
     *
     * @param {Number} width
     *  The base width of the lines to be drawn, in pixels.
     *
     * @returns {Number}
     *  The effective line width to be used to render the outlines of icons.
     */
    IconRenderer.getLineWidth = function (size, width) {
        return Math.ceil(size * width / 25);
    };

    /**
     * Returns the effective size used to render the icons for the passed size.
     * The passed size will be restricted to a minimum of 7 pixels, and may be
     * reduced by one pixel to keep the horizontal/vertical center lines on
     * entire pixels (this prevents blurry lines in the icons).
     *
     * @param {Number} size
     *  The maximum size available for rendering the icons, in pixels.
     *
     * @returns {Number}
     *  The effective icon size used to render the icons.
     */
    IconRenderer.getEffectiveSize = function (size) {

        // use a minimum size of 7 pixels
        var effSize = Math.max(size, 7);

        // get the base line width for outlines
        var lineWidth = IconRenderer.getLineWidth(effSize, 1);

        // reduce icon size to keep the horizontal/vertical center lines on entire pixels
        // (odd icon size for odd line widths; and even size for even line widths)
        var pad = (lineWidth & 1) ? (1 - (effSize & 1)) : (effSize & 1);
        return effSize - pad;
    };

    /**
     * Creates an icon renderer for the passed size in pixels, and stores the
     * renderer instances per effective icon size in an internal cache.
     *
     * @param {Number} size
     *  The maximum size available for rendering the icons, in pixels.
     *
     * @returns {IconRenderer}
     *  The icon renderer for the passed size. The effective icon size used by
     *  the renderer is contained in its property 'size'.
     */
    IconRenderer.create = (function () {

        // creates and caches an icon renderer per effective icon size
        var createRenderer = _.memoize(function (size) {
            return new IconRenderer(size);
        });

        // return the implementation of the method from local scope
        return function (size) {
            return createRenderer(IconRenderer.getEffectiveSize(size));
        };
    }());

    // private methods --------------------------------------------------------

    /**
     * Returns the effective line width for the specified base width, according
     * to the size of the icons to be rendered.
     *
     * @param {Number} width
     *  The base width of the lines to be drawn, in pixels.
     *
     * @returns {Number}
     *  The effective line width to be used to render the outlines of icons.
     */
    IconRenderer.prototype.lineWidth = function (width) {
        return IconRenderer.getLineWidth(this.size, width);
    };

    /**
     * Returns the effective value of a path coordinate. The resulting position
     * will be adjusted so that horizontal/vertical outlines drawn at these
     * coordinate will be kept on entire pixels to prevent blurry lines.
     *
     * @param {Number} pos
     *  The relative position inside the icon, in the interval [0;1]. The value
     *  0 represents the leading border (same value as the property 'C0'). The
     *  value 1 represents the trailing border (same value as the property
     *  'CE').
     *
     * @returns {Number}
     *  The effective value of a path coordinate.
     */
    IconRenderer.prototype.coord = function (pos) {
        return this.LO + Math.round(this.SI * pos);
    };

    /**
     * Returns the effective value of a path coordinate. Works similarly as the
     * method IconRenderer.coord(), but does not adjust the effective position
     * to entire pixels.
     *
     * @param {Number} pos
     *  The relative position inside the icon, in the interval [0;1]. The value
     *  0 represents the leading border (same value as the property 'C0'). The
     *  value 1 represents the trailing border (same value as the property
     *  'CE').
     *
     * @returns {Number}
     *  The effective value of a path coordinate.
     */
    IconRenderer.prototype.exact = function (pos) {
        return this.LO + this.SI * pos;
    };

    /**
     * Mirrors the passed effective coordinate which helps to create symmetric
     * icons. Calling the method IconRenderer.coord() with mirrored relative
     * positions may not result in the desired symmetric effective coordinates
     * due to internal rounding effects.
     *
     * @param {Number} coord
     *  The effective corrdinate, as returned for example by the methods
     *  IconRenderer.coord() or IconRenderer.exact().
     *
     * @returns {Number}
     *  The opposite coordinate.
     */
    IconRenderer.prototype.mirror = function (coord) {
        return this.size - coord;
    };

    /**
     * Returns a new empty path instance.
     *
     * @returns {Path}
     *  A new empty path instance.
     */
    IconRenderer.prototype.path = function () {
        return this.context.createPath();
    };

    /**
     * Draws (fills and/or strokes) the specified path into the internal
     * rendering context.
     *
     * @param {Path} path
     *  The canvas path to be drawn.
     *
     * @param {String|Null} color
     *  The name of a color, as defined in the color map IconRenderer.COLORS;
     *  or null to retain the current fill color.
     *
     * @param {String} [mode='all']
     *  The rendering mode for the path (one of 'all', 'stroke', or 'fill').
     *  See method Canvas.drawPath() for details.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawPath = function (path, color, mode) {
        if (color) { this.context.setFillStyle(IconRenderer.COLORS[color]); }
        this.context.setLineStyle({ width: this.LW, style: 'black' }).drawPath(path, mode);
        return this;
    };

    /**
     * Draws the specified label into the internal rendering context, as thick
     * lines with a black shadow effect.
     *
     * @param {Path} path
     *  The canvas path to be drawn.
     *
     * @param {String} color
     *  The name of a color to be used to draw the lines, as defined in the
     *  color map IconRenderer.COLORS.
     *
     * @param {Boolean} [shadow=false]
     *  Whether to draw the shadow effect.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawLabel = function (path, color, shadow) {
        this.context.setLineStyle({ width: this.LW2, style: IconRenderer.COLORS[color] });
        if (shadow) { this.context.setShadowStyle({ color: 'black', blur: this.LW2 }); }
        this.context.drawPath(path, 'stroke');
        return this;
    };

    /**
     * Draws the specified symbol into the internal rendering context.
     *
     * @param {Object} symbolId
     *  The identifier of a symbol.
     *
     * @param {Any} symbolArg
     *  Additional arguments needed to draw the symbol. The data type depends
     *  on the specified symbol. Usually, this argument consists of one or more
     *  color names used to fill the areas of the symbol.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawSymbol = (function () {

        // all symbol renderer implementations, expecting an initialized rendering context
        var SYMBOL_MAP = {

            arrowDown: function (color) {
                var c0 = this.C0;
                var c3 = this.coord(0.3);
                var c5 = this.CC;
                var c7 = this.mirror(c3);
                var ce = this.CE;
                this.drawPath(this.path().pushPolygon(c5, ce, c0, c5, c3, c5, c3, c0, c7, c0, c7, c5, ce, c5), color);
            },

            arrowRight: function (color) {
                this.drawSymbolRot270('arrowDown', color);
            },

            arrowUp: function (color) {
                this.drawSymbolRot180('arrowDown', color);
            },

            arrowDiagDown: function (color) {
                var c0 = this.C0;
                var c15 = this.coord(0.15);
                var c28 = this.coord(0.28);
                var c85 = this.mirror(c15);
                var d = (c85 - c28 - c15 + c0) / 2;
                this.drawPath(this.path().pushPolygon(c15, c85, c85, c85, c85, c15, c85 - d, c15 + d, c28, c0, c0, c28, c15 + d, c85 - d), color);
            },

            arrowDiagUp: function (color) {
                this.drawSymbolRot270('arrowDiagDown', color);
            },

            flag: function (color) {
                var c0 = this.C0;
                var c1 = this.coord(0.1);
                var c2 = this.coord(0.2);
                var c3 = this.coord(0.3);
                var c6 = 2 * c3 - c0;
                var c9 = this.mirror(c1);
                var ce = this.CE;
                this.drawPath(this.path().pushLineChain(c2, c0, c9, c3, c2, c6), color);
                this.drawPath(this.path().pushRect(c1, c0, c2 - c1, ce - c0), 'BLACK');
            },

            star: function (colors) {
                // uppermost tip
                var c1x = this.CC, c1y = this.C0;
                // upper left/right tips
                var c2xl = this.C0, c2xr = this.CE, c2y = this.coord(0.363);
                // lower left/right tips
                var c3xl = this.coord(0.155), c3xr = this.mirror(c3xl), c3y = this.coord(0.951);
                // upper left/right inner angles
                var c4xl = this.exact(0.345), c4xr = this.mirror(c4xl), c4y = this.exact(0.313);
                // lower left/right inner angles
                var c5xl = this.exact(0.25), c5xr = this.mirror(c5xl), c5y = this.exact(0.607);
                // lowermost inner angle
                var c6x = this.CC, c6y = this.exact(0.789);

                // path for right background area
                var pathR = this.path().pushPolygon(c1x, c1y, c4xr, c4y, c2xr, c2y, c5xr, c5y, c3xr, c3y, c6x, c6y);
                // path of the full star
                var pathO = this.path().pushPolygon(c1x, c1y, c4xr, c4y, c2xr, c2y, c5xr, c5y, c3xr, c3y, c6x, c6y, c3xl, c3y, c5xl, c5y, c2xl, c2y, c4xl, c4y);

                this.drawPath(pathO, colors[0], 'fill');
                this.drawPath(pathR, colors[1], 'fill');
                this.drawPath(pathO, null, 'stroke');
            },

            circle: function (color) {
                this.drawPath(this.path().pushCircle(this.CC, this.CC, this.R), color);
            },

            quarter: function (args) {
                var c0 = this.C0;
                var c5 = this.CC;
                var ce = this.CE;
                var fillPath = this.path().moveTo(c5, c0);
                var strokePath = this.path().pushCircle(c5, c5, this.R).moveTo(c5, c0);
                switch (args.quarters) {
                    case 1:
                        fillPath.arc(c5, c5, this.R, -PI12, 0).lineTo(c5, c5);
                        strokePath.lineTo(c5, c5).lineTo(ce, c5);
                        break;
                    case 2:
                        fillPath.arc(c5, c5, this.R, -PI12, PI12);
                        strokePath.lineTo(c5, ce);
                        break;
                    case 3:
                        fillPath.arc(c5, c5, this.R, -PI12, PI).lineTo(c5, c5);
                        strokePath.lineTo(c5, c5).lineTo(c0, c5);
                        break;
                }
                this.drawPath(this.path().pushCircle(c5, c5, this.R), args.color1, 'fill');
                this.drawPath(fillPath.close(), args.color2, 'fill');
                this.drawPath(strokePath, null, 'stroke');
            },

            trafficLight: function (color) {
                var c15 = this.coord(0.15);
                var c85 = this.mirror(c15);
                var r1 = c15 - this.C0;
                var r2 = this.CC - this.coord(0.1);
                this.drawPath(this.path().arc(c15, c15, r1, PI, -PI12).arc(c85, c15, r1, -PI12, 0).arc(c85, c85, r1, 0, PI12).arc(c15, c85, r1, PI12, PI).close(), 'BLACK');
                this.drawPath(this.path().pushCircle(this.CC, this.CC, r2), color);
            },

            rating: function (colors) {
                var c5 = this.CC;
                var w = Math.floor((c5 - this.C0) / 2);
                var c0 = c5 - 2 * w;
                var c25 = c5 - w;
                var c75 = c5 + w;
                var ce = c5 + 2 * w;
                var h = ce - c0;
                this.drawPath(this.path().pushRect(c0, c0 + 3 * w, w, h - 3 * w), colors[0]);
                this.drawPath(this.path().pushRect(c25, c0 + 2 * w, w, h - 2 * w), colors[1]);
                this.drawPath(this.path().pushRect(c5, c0 + w, w, h - w), colors[2]);
                this.drawPath(this.path().pushRect(c75, c0, w, h), colors[3]);
            },

            boxes: function (colors) {
                var c0 = this.C0;
                var c15 = this.coord(0.15);
                var c5 = this.CC;
                var c85 = this.mirror(c15);
                var ce = this.CE;
                var r = c15 - this.C0;
                this.drawPath(this.path().moveTo(c5, c5).lineTo(c5, ce).arc(c15, c85, r, PI12, PI).lineTo(c0, c5).close(), colors[0]);
                this.drawPath(this.path().moveTo(c5, c5).lineTo(ce, c5).arc(c85, c85, r, 0, PI12).lineTo(c5, ce).close(), colors[1]);
                this.drawPath(this.path().moveTo(c5, c5).lineTo(c0, c5).arc(c15, c15, r, PI, -PI12).lineTo(c5, c0).close(), colors[2]);
                this.drawPath(this.path().moveTo(c5, c5).lineTo(c5, c0).arc(c85, c15, r, -PI12, 0).lineTo(ce, c5).close(), colors[3]);
            },

            squareSign: function (color) {
                var c15 = this.coord(0.15);
                var c5 = this.CC;
                var c85 = this.mirror(c15);
                var r = c15 - this.C0;
                this.drawPath(this.path().arc(c5, c15, r, -PI34, -PI14).arc(c85, c5, r, -PI14, PI14).arc(c5, c85, r, PI14, PI34).arc(c15, c5, r, PI34, -PI34).close(), color);
            },

            triangleSign: function (color) {
                var c15 = this.coord(0.15);
                var c5 = this.CC;
                var c75 = this.coord(0.75);
                var c85 = this.mirror(c15);
                var r = c15 - this.C0;
                this.drawPath(this.path().arc(c5, c15, r, -PI56, -PI16).arc(c85, c75, r, -PI16, PI12).arc(c15, c75, r, PI12, -PI56).close(), color);
            },

            cross: function (color) {
                var c0 = this.C0;
                var c2 = this.coord(0.2);
                var c5 = this.CC;
                var c8 = this.mirror(c2);
                var ce = this.CE;
                var d = c2 - c0;
                var c3 = c5 - d;
                var c7 = c5 + d;
                this.drawPath(this.path().pushPolygon(c2, c0, c0, c2, c3, c5, c0, c8, c2, ce, c5, c7, c8, ce, ce, c8, c7, c5, ce, c2, c8, c0, c5, c3), color);
            },

            exclamation: function (color) {
                var c0 = this.C0;
                var c35 = this.coord(0.35);
                var c5 = this.CC;
                var c6 = this.coord(0.6);
                var ce = this.CE;
                var r = c5 - c35;
                this.drawPath(this.path().pushRect(c35, c0, 2 * r, c6 - c0).pushCircle(c5, ce - r, r), color);
            },

            check: function (color) {
                var c0 = this.C0;
                var c4 = this.coord(0.4);
                var c6 = this.mirror(c4);
                var ce = this.CE;
                var d = this.coord(0.2) - c0;
                var o = Math.floor(d / 2);
                this.drawPath(this.path().pushPolygon(c0, c6 - o, c4, ce - o, ce, c4 - o, ce - d, c4 - d - o, c4, ce - 2 * d - o, c0 + d, c6 - d - o), color);
            },

            minus: function (color) {
                var c0 = this.C0;
                var c4 = this.coord(0.38);
                var c6 = this.mirror(c4);
                this.drawPath(this.path().pushRect(c0, c4, this.SI, c6 - c4), color);
            },

            triangleDown: function (color) {
                var c0 = this.C0;
                var c25 = this.coord(0.25);
                var c5 = this.CC;
                var c75 = c25 + c5 - c0;
                var ce = this.CE;
                this.drawPath(this.path().pushPolygon(c0, c25, c5, c75, ce, c25), color);
            },

            triangleUp: function (color) {
                this.drawSymbolRot180('triangleDown', color);
            },

            crossLabel: function () {
                var c3 = this.coord(0.3);
                var c7 = this.mirror(c3);
                this.drawLabel(this.path().pushLine(c3, c3, c7, c7).pushLine(c3, c7, c7, c3), 'WHITE', true);
            },

            exclamationLabel: function () {
                var c25 = this.coord(0.25);
                var c5 = this.CC;
                var c55 = this.coord(0.55);
                var c75 = this.mirror(c25);
                this.drawLabel(this.path().pushLine(c5, c25, c5, c55).pushLine(c5, c75, c5, c75 + 1), 'WHITE', true);
            },

            checkLabel: function () {
                var c25 = this.coord(0.25);
                var c75 = this.mirror(c25);
                var c5 = (c75 + c25) / 2;
                var d = (c75 - c25) / 3;
                this.drawLabel(this.path().pushLineChain(c25, c5, c25 + d, c5 + d, c75, c5 - d), 'WHITE', true);
            },

            smileyFace: function (type) {

                // eyes
                var cxl = this.coord(0.35);
                var cxr = this.mirror(cxl);
                var cy = this.coord(0.4);
                this.drawPath(this.path().pushCircle(cxl, cy, 1).pushCircle(cxr, cy, 1), 'BLACK');

                // mouth
                var r = this.coord(0.35);
                switch (type) {
                    case 'neutral':
                        cxl = this.coord(0.35);
                        cxr = this.mirror(cxl);
                        cy = this.coord(0.65);
                        this.drawPath(this.path().pushLine(cxl, cy, cxr, cy), null, 'stroke');
                        break;
                    case 'good':
                        cy = this.coord(0.4);
                        this.drawPath(this.path().pushArc(this.CC, cy, r, PI14, PI34), null, 'stroke');
                        break;
                    case 'bad':
                        cy = this.CE;
                        this.drawPath(this.path().pushArc(this.CC, cy, r, PI54, PI74), null, 'stroke');
                        break;
                }
            }
        };

        // return the implementation of the method from local scope
        return function (symbolId, symbolArg) {
            SYMBOL_MAP[symbolId].call(this, symbolArg);
            return this;
        };
    }());

    /**
     * Rotates the rendering context by 180 degrees, and draws the specified
     * symbol.
     *
     * @param {Object} symbolId
     *  The identifier of a symbol to be drawn rotated.
     *
     * @param {Any} symbolArg
     *  Additional arguments needed to draw the symbol. See description of the
     *  method IconRenderer.drawSymbol() for details.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawSymbolRot180 = function (symbolId, symbolArg) {
        this.context.render(function () {
            this.context.translate(0, this.size).scale(1, -1);
            this.drawSymbol(symbolId, symbolArg);
        }, this);
        return this;
    };

    /**
     * Rotates the rendering context by 270 degrees clockwise, and draws the
     * specified symbol.
     *
     * @param {Object} symbolId
     *  The identifier of a symbol to be drawn rotated.
     *
     * @param {Any} symbolArg
     *  Additional arguments needed to draw the symbol. See description of the
     *  method IconRenderer.drawSymbol() for details.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawSymbolRot270 = function (symbolId, symbolArg) {
        this.context.render(function () {
            this.context.translate(0, this.size).rotate(-PI12);
            this.drawSymbol(symbolId, symbolArg);
        }, this);
        return this;
    };

    // public methods ---------------------------------------------------------

    /**
     * Draws the specified icon from an icon set.
     *
     * @param {ContextWrapper} context
     *  The rendering context to draw the icon into.
     *
     * @param {Number} x
     *  The horizontal position of the icon in the passed context.
     *
     * @param {Number} y
     *  The vertical position of the icon in the passed context.
     *
     * @param {String} iconSetId
     *  The identifier of an icon set, as used in document operations (ignoring
     *  the character case).
     *
     * @param {Number} iconIndex
     *  The zero-based index of the icon from the icon set to be drawn.
     *
     * @returns {IconRenderer}
     *  A reference to this instance.
     */
    IconRenderer.prototype.drawIcon = (function () {

        // maps identifiers of all supported icons to arrays of symbol descriptors
        var ICON_MAP = {

            // arrows
            redArrowDown:        [{ id: 'arrowDown',     arg: 'RED'    }],
            yellowArrowDiagDown: [{ id: 'arrowDiagDown', arg: 'YELLOW' }],
            yellowArrowRight:    [{ id: 'arrowRight',    arg: 'YELLOW' }],
            yellowArrowDiagUp:   [{ id: 'arrowDiagUp',   arg: 'YELLOW' }],
            greenArrowUp:        [{ id: 'arrowUp',       arg: 'GREEN'  }],
            grayArrowDown:       [{ id: 'arrowDown',     arg: 'GRAY'   }],
            grayArrowDiagDown:   [{ id: 'arrowDiagDown', arg: 'GRAY'   }],
            grayArrowRight:      [{ id: 'arrowRight',    arg: 'GRAY'   }],
            grayArrowDiagUp:     [{ id: 'arrowDiagUp',   arg: 'GRAY'   }],
            grayArrowUp:         [{ id: 'arrowUp',       arg: 'GRAY'   }],

            // flags
            redFlag:    [{ id: 'flag', arg: 'RED'    }],
            yellowFlag: [{ id: 'flag', arg: 'YELLOW' }],
            greenFlag:  [{ id: 'flag', arg: 'GREEN'  }],

            // circles
            redCircle:          [{ id: 'circle',  arg: 'RED'    }],
            yellowCircle:       [{ id: 'circle',  arg: 'YELLOW' }],
            greenCircle:        [{ id: 'circle',  arg: 'GREEN'  }],
            grayCircle:         [{ id: 'circle',  arg: 'GRAY'   }],
            blackCircle:        [{ id: 'circle',  arg: 'BLACK'  }],
            pinkCircle:         [{ id: 'circle',  arg: 'PINK'   }],
            redCrossCircle:     [{ id: 'circle',  arg: 'RED'    }, { id: 'crossLabel'       }],
            yellowExclamCircle: [{ id: 'circle',  arg: 'YELLOW' }, { id: 'exclamationLabel' }],
            greenCheckCircle:   [{ id: 'circle',  arg: 'GREEN'  }, { id: 'checkLabel'       }],
            grayQuarter0:       [{ id: 'circle',  arg: 'WHITE'  }],
            grayQuarter1:       [{ id: 'quarter', arg: { quarters: 1, color1: 'WHITE', color2: 'GRAY' } }],
            grayQuarter2:       [{ id: 'quarter', arg: { quarters: 2, color1: 'WHITE', color2: 'GRAY' } }],
            grayQuarter3:       [{ id: 'quarter', arg: { quarters: 3, color1: 'WHITE', color2: 'GRAY' } }],
            grayQuarter4:       [{ id: 'circle',  arg: 'GRAY'   }],

            // smileys
            greenSmileyGood:     [{ id: 'circle',  arg: 'GREEN'  }, { id: 'smileyFace', arg: 'good'    }],
            yellowSmileyGood:    [{ id: 'circle',  arg: 'YELLOW' }, { id: 'smileyFace', arg: 'good'    }],
            yellowSmileyNeutral: [{ id: 'circle',  arg: 'YELLOW' }, { id: 'smileyFace', arg: 'neutral' }],
            yellowSmileyBad:     [{ id: 'circle',  arg: 'YELLOW' }, { id: 'smileyFace', arg: 'bad'     }],
            redSmileyBad:        [{ id: 'circle',  arg: 'RED'    }, { id: 'smileyFace', arg: 'bad'     }],

            // traffic lights
            redTrafficLight:    [{ id: 'trafficLight', arg: 'RED'    }],
            yellowTrafficLight: [{ id: 'trafficLight', arg: 'YELLOW' }],
            greenTrafficLight:  [{ id: 'trafficLight', arg: 'GREEN'  }],

            // ratings
            whiteStar:      [{ id: 'star',   arg: ['WHITE',  'WHITE']  }],
            halfYellowStar: [{ id: 'star',   arg: ['YELLOW', 'WHITE']  }],
            yellowStar:     [{ id: 'star',   arg: ['YELLOW', 'YELLOW'] }],
            blueRating0:    [{ id: 'rating', arg: ['GRAY', 'GRAY', 'GRAY', 'GRAY'] }],
            blueRating1:    [{ id: 'rating', arg: ['BLUE', 'GRAY', 'GRAY', 'GRAY'] }],
            blueRating2:    [{ id: 'rating', arg: ['BLUE', 'BLUE', 'GRAY', 'GRAY'] }],
            blueRating3:    [{ id: 'rating', arg: ['BLUE', 'BLUE', 'BLUE', 'GRAY'] }],
            blueRating4:    [{ id: 'rating', arg: ['BLUE', 'BLUE', 'BLUE', 'BLUE'] }],
            blueBoxes0:     [{ id: 'boxes',  arg: ['GRAY', 'GRAY', 'GRAY', 'GRAY'] }],
            blueBoxes1:     [{ id: 'boxes',  arg: ['BLUE', 'GRAY', 'GRAY', 'GRAY'] }],
            blueBoxes2:     [{ id: 'boxes',  arg: ['BLUE', 'BLUE', 'GRAY', 'GRAY'] }],
            blueBoxes3:     [{ id: 'boxes',  arg: ['BLUE', 'BLUE', 'BLUE', 'GRAY'] }],
            blueBoxes4:     [{ id: 'boxes',  arg: ['BLUE', 'BLUE', 'BLUE', 'BLUE'] }],

            // signs
            redSquareSign:      [{ id: 'squareSign',   arg: 'RED'    }],
            yellowTriangleSign: [{ id: 'triangleSign', arg: 'YELLOW' }],
            redCross:           [{ id: 'cross',        arg: 'RED'    }],
            yellowExclam:       [{ id: 'exclamation',  arg: 'YELLOW' }],
            greenCheck:         [{ id: 'check',        arg: 'GREEN'  }],
            yellowMinus:        [{ id: 'minus',        arg: 'YELLOW' }],
            redTriangleDown:    [{ id: 'triangleDown', arg: 'RED'    }],
            greenTriangleUp:    [{ id: 'triangleUp',   arg: 'GREEN'  }]
        };

        // maps identifiers of all supported icon sets (lower-case!) to arrays of icon identifiers (see ICON_MAP)
        var ICON_SET_MAP = {

            // 3 icons
            '3arrows':         ['redArrowDown', 'yellowArrowRight', 'greenArrowUp'],
            '3arrowsgray':     ['grayArrowDown', 'grayArrowRight', 'grayArrowUp'],
            '3flags':          ['redFlag', 'yellowFlag', 'greenFlag'],
            '3trafficlights1': ['redCircle', 'yellowCircle', 'greenCircle'],
            '3trafficlights2': ['redTrafficLight', 'yellowTrafficLight', 'greenTrafficLight'],
            '3signs':          ['redSquareSign', 'yellowTriangleSign', 'greenCircle'],
            '3symbols':        ['redCrossCircle', 'yellowExclamCircle', 'greenCheckCircle'],
            '3symbols2':       ['redCross', 'yellowExclam', 'greenCheck'],
            '3stars':          ['whiteStar', 'halfYellowStar', 'yellowStar'],
            '3triangles':      ['redTriangleDown', 'yellowMinus', 'greenTriangleUp'],
            '3smilies':        ['yellowSmileyGood', 'yellowSmileyNeutral', 'yellowSmileyBad'], // ODF only, with wrong spelling 'Smilies'
            '3colorsmilies':   ['greenSmileyGood', 'yellowSmileyNeutral', 'redSmileyBad'], // ODF only, with wrong spelling 'Smilies'

            // 4 icons
            '4arrows':         ['redArrowDown', 'yellowArrowDiagDown', 'yellowArrowDiagUp', 'greenArrowUp'],
            '4arrowsgray':     ['grayArrowDown', 'grayArrowDiagDown', 'grayArrowDiagUp', 'grayArrowUp'],
            '4redtoblack':     ['blackCircle', 'grayCircle', 'pinkCircle', 'redCircle'],
            '4rating':         ['blueRating1', 'blueRating2', 'blueRating3', 'blueRating4'],
            '4trafficlights':  ['blackCircle', 'redCircle', 'yellowCircle', 'greenCircle'],

            // 5 icons
            '5arrows':         ['redArrowDown', 'yellowArrowDiagDown', 'yellowArrowRight', 'yellowArrowDiagUp', 'greenArrowUp'],
            '5arrowsgray':     ['grayArrowDown', 'grayArrowDiagDown', 'grayArrowRight', 'grayArrowDiagUp', 'grayArrowUp'],
            '5rating':         ['blueRating0', 'blueRating1', 'blueRating2', 'blueRating3', 'blueRating4'],
            '5quarters':       ['grayQuarter0', 'grayQuarter1', 'grayQuarter2', 'grayQuarter3', 'grayQuarter4'],
            '5boxes':          ['blueBoxes0', 'blueBoxes1', 'blueBoxes2', 'blueBoxes3', 'blueBoxes4']
        };

        // return the implementation of the method from local scope
        return function (context, x, y, iconSetId, iconIndex) {

            // the icon set (array of icon identifiers)
            var iconSet = ICON_SET_MAP[iconSetId.toLowerCase()];
            if (!iconSet) { return; }

            // the icon identifier
            var iconId = iconSet[iconIndex];
            if (!iconId) { return; }

            // resolve an existing icon from the internal cache, or create a new canvas element
            var iconCanvas = this.cache.getOrCreate(iconId, function () {

                // internal helper canvas needed to render the icons
                var iconCanvas = new Canvas(Canvas.SINGLETON).initialize({ width: this.size, height: this.size });

                // render the icon into the canvas
                iconCanvas.render(function (iconContext) {

                    // initialize the rendering context
                    iconContext.setLineStyle({ cap: 'round', join: 'round' });

                    // draw all symbols into the internal rendering context
                    this.context = iconContext;
                    ICON_MAP[iconId].forEach(function (symbolDesc) {
                        this.drawSymbol(symbolDesc.id, symbolDesc.arg);
                    }, this);
                    this.context = null;
                }, this);

                return iconCanvas;
            }, this);

            // copy the icon to the specified rendering context, either from canvas, or from image
            context.drawImage(iconCanvas, x, y);
            return this;
        };
    }());

    // exports ================================================================

    return IconRenderer;

});
