/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/drawingrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/model/drawing/commentmodel',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendererbase',
    'io.ox/office/spreadsheet/view/render/renderframemanager',
    'io.ox/office/spreadsheet/view/popup/drawingcontextmenu'
], function (Utils, Iterator, ValueSet, DrawingFrame, AttributeUtils, Config, SheetCommentModel, TextFrameUtils, RenderUtils, RendererBase, RenderFrameManager, DrawingContextMenu) {

    'use strict';

    // convenience shortcuts
    var SerialIterator = Iterator.SerialIterator;

    // private global functions ===============================================

    /**
     * Returns whether the DOM drawing frame of the passed drawing model needs
     * to be refreshed after it has been created or changed.
     *
     * @param {DrawingModel} drawingModel
     *  The model of a drawing object.
     *
     * @returns {Boolean}
     *  Whether the DOM drawing frame of the passed drawing model needs to be
     *  refreshed.
     */
    function supportsFormatting(drawingModel) {
        return drawingModel.getType() !== 'group';
    }

    /**
     * Creates an SVG element instance.
     *
     * @param {String} tag
     *  The TAG name of the SVG element.
     *
     * @returns {SVGElement}
     *  The new SVG element instance.
     */
    function createSvgElement(tag) {
        return document.createElementNS('http://www.w3.org/2000/svg', tag);
    }

    /**
     * Returns the client nodes for the passed DOM drawing frame that have been
     * registered with the function registerClientNodes().
     *
     * @param {jQuery} drawingFrame
     *  The DOM drawing frame.
     *
     * @returns {jQuery|Null}
     *  The DOM client nodes registered for the passed DOM drawing frame if
     *  existing, as jQuery collection; otherwise null.
     */
    function getClientNodes(drawingFrame) {
        return drawingFrame.data('clientNodes') || null;
    }

    /**
     * Registers additional DOM nodes for the passed DOM drawing frame that are
     * not descendants of its root node but share its visibility state etc.
     *
     * @param {jQuery} drawingFrame
     *  The DOM drawing frame to be updated.
     *
     * @param {jQuery|HTMLElement} clientNodes
     *  The DOM nodes to be associated with the drawing frame.
     */
    function registerClientNodes(drawingFrame, clientNodes) {
        var oldClientNodes = drawingFrame.data('clientNodes');
        var newClientNodes = oldClientNodes ? oldClientNodes.add(clientNodes) : $(clientNodes);
        drawingFrame.data('clientNodes', newClientNodes);
    }

    /**
     * Toggles the CSS class of the passed drawing frame, and all of its client
     * nodes.
     *
     * @param {jQuery} drawingFrame
     *  The DOM drawing frame to be updated.
     *
     * @param {String} className
     *  The name of the CSS class to be toggled.
     *
     * @param {Boolean} [state]
     *  Whether to add (true), or to remove (false) the CSS class. If omitted,
     *  the current state of the CSS class will be toggled.
     */
    function toggleFrameClass(drawingFrame, className, state) {

        // toggle the CSS class at the drawing frame
        drawingFrame.toggleClass(className, state);

        // toggle additional client nodes associated to the drawing frame
        var clientNodes = drawingFrame.data('clientNodes');
        if (clientNodes) { clientNodes.toggleClass(className, state); }
    }

    // class DrawingRenderer ==================================================

    /**
     * Renders the drawing objects and cell comments of the active sheet into
     * the DOM drawing layers shown in a single grid pane.
     *
     * @constructor
     *
     * @extends RendererBase
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this drawing layer renderer.
     */
    var DrawingRenderer = RendererBase.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the spreadsheet application, model, and view
        var docView = gridPane.getDocView();
        var docModel = docView.getDocModel();
        var app = docModel.getApp();

        // the container node containing all cell overlay nodes with indicators (below drawing objects and form controls)
        var overlayLayerNode = gridPane.createLayerNode('comment-overlay-layer', { before: 'form-layer' });

        // the root layer node containing the page node
        var drawingLayerNode = gridPane.createLayerNode('drawing-layer');

        // the container node containing all connector lines between comments and anchor cells
        // (above drawing obejsts, but below all comment frames)
        var connectorLayerNode = gridPane.createLayerNode('comment-connector-layer');

        // the root layer node containing the page node
        var commentLayerNode = gridPane.createLayerNode('drawing-layer comment-layer');

        // the manager for the DOM drawing frames of the drawing objects to be rendered
        var drawingFrameManager = new RenderFrameManager(docView, drawingLayerNode);

        // the manager for the DOM drawing frames of the cell comments to be rendered
        var commentFrameManager = new RenderFrameManager(docView, commentLayerNode);

        // the context menu for drawing objects
        var contextMenu = new DrawingContextMenu(gridPane, drawingLayerNode);

        // current layer rectangle covered by this renderer
        var layerRectangle = null;

        // all drawing models whose drawing frames need to be rendered
        var pendingModelSet = new ValueSet('getUid()');

        // the drawing models that have been made visible temporarily
        var tempShowModelSet = new ValueSet('getUid()');

        // the model of the cell comment whose overlay node is currently hovered
        var hoverCommentModel = null;

        // delay timer to show the cell comment whose overlay node is currently hovered
        var hoverCommentTimer = null;

        // all drawing frames selected by a remote user
        var remoteDrawingFrames = [];

        // base constructor ---------------------------------------------------

        RendererBase.call(this, gridPane);

        // private methods ----------------------------------------------------

        /**
         * Returns the frame node manager that is responsible for the passed
         * drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object.
         *
         * @returns {RenderFrameManager}
         *  The drawing frame manager responsible for the passed drawing model.
         */
        function getFrameManager(drawingModel) {
            return (drawingModel.getType() === 'comment') ? commentFrameManager : drawingFrameManager;
        }

        /**
         * Renders a selection frame into the passed drawing frame.
         *
         * @param {jQuery} drawingFrame
         *  The drawing frame to be selected.
         *
         * @param {Boolean} [force=false]
         *  If set to true, an existing selection frame will be removed.
         */
        function updateSelectionFrame(drawingFrame, force) {

            var editMode = docView.isEditable() && !docView.isSheetLocked();
            var drawingModel = DrawingFrame.getModel(drawingFrame);

            if (force) { DrawingFrame.clearSelection(drawingFrame); }

            DrawingFrame.drawSelection(drawingFrame, {
                movable: editMode,
                resizable: editMode,
                rotatable: editMode && !app.isODF() && drawingModel.isDeepRotatable(), // bug 52930: disable rotation in ODS
                rotation: drawingModel.getRotationDeg(),
                adjustable: editMode && docView.hasSingleDrawingSelection(),
                isMultiSelection: !docView.hasSingleDrawingSelection()
            });
        }

        /**
         * Updates the dirty flags of the passed drawing frame according to the
         * passed options.
         */
        function updateDirtyFlags(drawingFrame, options) {
            var dirtyFormat = !!drawingFrame.data('dirtyFormat') || Utils.getBooleanOption(options, 'format', false);
            var dirtyText = !!drawingFrame.data('dirtyText') || Utils.getBooleanOption(options, 'text', false);
            drawingFrame.data({ dirtyFormat: dirtyFormat, dirtyText: dirtyText });
        }

        /**
         * Updates the CSS position of the passed text frame node.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object.
         *
         * @param {jQuery} textFrame
         *  The DOM text frame of the passed drawing model.
         *
         * @param {Object} textPosition
         *  The original position of the text frame at 100% zoom level without
         *  flipping or rotation, as received from the rendering cache.
         */
        function updateTextFramePosition(drawingModel, textFrame/*, textPosition*/) {

            // the current zoom factor of the active sheet
            var zoom = docView.getZoomFactor();
            // text needs to be flipped horizontally, if horizontal/vertical flipping does not match
            var flipText = drawingModel.getEffectiveFlipping().reverseRot;
            // text direction property of drawing shape
            var vert = drawingModel.getMergedAttributeSet(true).shape.vert;
            // text direction angle
            var dirAngle = vert === 'vert270' ? 270 : (vert === 'vert' || vert === 'eaVert' ? 90 : 0);
            // reverse horizontal scaling to flip the text back
            var scaleX = flipText ? -1 : 1;

            textFrame.css({
                fontSize: Math.round(16 * zoom) + 'px',
                transform: 'scaleX(' + scaleX + ') rotate(' + dirAngle + 'deg)'
            });
        }

        /**
         * Updates the formatting and contents of cell comment frames.
         *
         * @param {CommentModel} commentModel
         *  The model of a cell comment whose drawing frame will be updated.
         *
         * @param {jQuery} drawingFrame
         *  The drawing frame of a cell comment.
         */
        function updateCommentFrame(commentModel, drawingFrame) {

            // TODO: import and evaluate font settings
            //var charAttrs = commentModel.getMergedAttributeSet(true).character;
            //var fontSize = Math.round(charAttrs.fontSize * docView.getZoomFactor());
            var fontSize = Math.round(9 * docView.getZoomFactor());

            // the container element for all text lines
            var paddingX = Math.round(3 * docView.getZoomFactor());
            var paddingY = Math.round(docView.getZoomFactor());
            var textFrame = $('<div class="textframe" style="padding:' + paddingY + 'px ' + paddingX + 'px;">');

            // append all text lines as separate paragraph elements
            commentModel.getText().split('\n').forEach(function (textLine) {
                // TODO: import and evaluate font settings
                var paragraphNode = $('<div class="p" style="font-family:Tahoma,sans-serif;font-size:' + fontSize + 'pt;">').text(textLine);
                textFrame.append(paragraphNode);
            });

            // insert the text contents into the drawing frame
            DrawingFrame.getAndClearContentNode(drawingFrame).append(textFrame);
        }

        /**
         * Updates the position and visibility of the overlay node for the
         * specified cell comment.
         *
         * @param {SheetCommentModel} commentModel
         *  The model of a cell comment.
         *
         * @returns {Boolean}
         *  Whether the comment anchor is located in visible columns and rows.
         */
        function updateCommentOverlayNode(commentModel) {

            // the drawing frame for the cell comment
            var drawingFrame = commentFrameManager.getDrawingFrame(commentModel);
            // the cell overlay node with the indicator corner
            var overlayNode = drawingFrame.data('overlayNode');
            // the location of the anchor cell (with merged range), in pixels
            var anchorRect = commentModel.getAnchorRectangle({ pixel: true, expandMerged: true });

            // hide overlay nodes of comments in hidden columns/rows
            var hidden = anchorRect.area() === 0;
            overlayNode.toggleClass('hidden', hidden);

            // comment in hidden cell: immediately hide the drawing frame
            if (hidden) {
                toggleFrameClass(drawingFrame, 'hidden', true);
                return false;
            }

            // update the position and size of the overlay node
            var cellRect = gridPane.convertToLayerRectangle(anchorRect);
            overlayNode.css(cellRect.expandSelf(0, 0, -1, -1).toCSS());
            return true;
        }

        /**
         * Updates the position and visibility of the SVG connector node for
         * the specified cell comment.
         *
         * @param {SheetCommentModel} commentModel
         *  The model of a cell comment.
         */
        function updateCommentConnectorNode(commentModel) {

            // the drawing frame for the cell comment
            var drawingFrame = commentFrameManager.getDrawingFrame(commentModel);
            // the cell overlay node with the indicator corner
            var overlayNode = drawingFrame.data('overlayNode');
            // the drawing frame for the cell comment
            var connectorNode = getClientNodes(drawingFrame);
            // the child SVG line node
            var lineNode = connectorNode.children().first();

            // the pixel position of the connector end point (top-right cell corner)
            var pos2 = overlayNode.position();
            pos2.left += overlayNode.width() - 1;

            // the pixel position of the connector start point (top-left or top-right comment frame corner)
            var pos1 = drawingFrame.position();
            // select top-right corner, if left border is left of the cell corner
            if (pos1.left < pos2.left) { pos1.left += drawingFrame.width() - 1; }

            // the effective bounding rectangle of the connector line
            var x = Math.min(pos1.left, pos2.left);
            var y = Math.min(pos1.top, pos2.top);
            var w = Math.abs(pos1.left - pos2.left);
            var h = Math.abs(pos1.top - pos2.top);

            // set the position of the bounding box for the line
            connectorNode.attr({ width: w + 1, height: h + 1, viewBox: '0 0 ' + w + ' ' + h }).css({ left: x, top: y });
            // set the connector line formatting
            lineNode.attr({ x1: pos1.left - x, y1: pos1.top - y, x2: pos2.left - x, y2: pos2.top - y, stroke: 'black', 'stroke-width': 0.75 });
        }

        /**
         * Returns the drawing frame associated to the passed drawing model, if
         * it needs to be rendered, i.e. it covers the current layer rectangle
         * of this renderer, or it is selected (the latter is needed for move
         * tracking to be able to see the move box independent of the source
         * position of the drawing object).
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model to be checked.
         *
         * @returns {DrawingFrame|Null}
         *  The drawing frame associated to the passed drawing model, if it
         *  needs to be rendered.
         */
        function getRenderDrawingFrame(drawingModel) {

            // the drawing frame manager responsible for the passed model
            var frameManager = getFrameManager(drawingModel);

            // get drawing frame from collection
            var drawingFrame = frameManager.getDrawingFrame(drawingModel);
            if (!drawingFrame) { return null; }

            // do not render hidden drawing objects (but drawing frames that have been made visible temporarily)
            if ((drawingModel === hoverCommentModel) || tempShowModelSet.has(drawingModel) || drawingModel.isEffectivelyVisible()) {

                // resolve absolute location of the drawing object (do not render hidden drawing objects, e.g. in hidden columns/rows)
                var rectangle = drawingModel.getRectangle({ absolute: true });

                // check that the drawing frame is in the visible area of the grid pane, or selected
                if (rectangle && (layerRectangle.overlaps(rectangle) || DrawingFrame.isSelected(drawingFrame))) { return drawingFrame; }
            }

            // hide drawing frame in the DOM
            toggleFrameClass(drawingFrame, 'hidden', true);
            return null;
        }

        /**
         * Refreshes the contents and formatting of the DOM drawing frame
         * associated to the specified drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing object to be rendered.
         */
        var formatDrawingFrame = RenderUtils.profileMethod('DrawingRenderer.formatDrawingFrame()', function (drawingModel) {

            RenderUtils.withLogging(function () {
                RenderUtils.log('uid=' + drawingModel.getUid() + ' type=' + drawingModel.getType() + ' range=' + drawingModel.getRange());
            });

            // immediately remove drawing model from the set of pending models
            pendingModelSet.remove(drawingModel);

            // do not render the drawing frame if it has been scrolled/moved outside in the meantime
            var drawingFrame = getRenderDrawingFrame(drawingModel);
            if (!drawingFrame) { return; }

            // the model frame (drawing frame used as source for the text contents)
            var frameSettings = drawingModel.refreshModelFrame();
            if (!frameSettings) { return; }
            var modelFrame = frameSettings.drawingFrame;

            // whether to refresh the formatting or text contents of the drawing frame
            var dirtyFormat = drawingFrame.data('dirtyFormat');
            var dirtyText = drawingFrame.data('dirtyText');

            // remove the special display for uninitialized drawing frames, and the dirty markers
            drawingFrame.removeClass('uninitialized').data({ dirtyFormat: false, dirtyText: false });

            // text contents need to be copied for new drawing frames and for changed text contents
            var textFrame = dirtyText ? TextFrameUtils.copyTextFrame(modelFrame, drawingFrame) : dirtyFormat ? TextFrameUtils.getTextFrame(drawingFrame) : null;

            // reset all explicit CSS formatting of the textbox before formatting the drawing frame
            if (textFrame) {
                textFrame.css('transform', '');
            }

            // special handling for cell comments (TODO: treat them as regular text shapes)
            if (drawingModel instanceof SheetCommentModel) {
                updateCommentFrame(drawingModel, drawingFrame);
            }

            // update formatting of the drawing frame
            if (dirtyFormat) {
                RenderUtils.takeTime('update formatting', function () {

                    var attrSet = drawingModel.getMergedAttributeSet(true);
                    DrawingFrame.updateFormatting(app, drawingFrame, attrSet, { skipFlipping: true });

                    // if selected drawing contains adjustment values, reposition adjustment points
                    if (DrawingFrame.isSelected(drawingFrame) && attrSet.geometry && attrSet.geometry.avList) {
                        updateSelectionFrame(drawingFrame, true);
                    }
                });
            }

            // update paragraph/character formatting of the text contents
            if (textFrame && frameSettings.textPosition) {
                RenderUtils.takeTime('update text', function () {
                    updateTextFramePosition(drawingModel, textFrame, frameSettings.textPosition);
                    getFrameManager(drawingModel).updateTextFormatting(drawingModel);
                });
            }

            // notify all listeners
            self.trigger('render:drawingframe', drawingFrame, drawingModel);
        });

        /**
         * Registers the passed drawing model for deferred rendering of its
         * formatting, and starts a background task on demand that refreshes
         * the formatting of all registered pending drawing frames.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing object to be rendered.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.immediate=false]
         *      If set to true, the formatting of the drawing frame will be
         *      updated immediately instead of using the background task.
         */
        var formatDrawingFrameDebounced = (function () {

            // the current background task for refreshing drawing frames
            var refreshTimer = null;

            // direct callback: insert the passed drawing model into the set
            function registerDrawingModel(drawingModel, options) {
                if (Utils.getBooleanOption(options, 'immediate', false)) {
                    formatDrawingFrame(drawingModel);
                } else {
                    pendingModelSet.insert(drawingModel);
                }
            }

            // deferred callback: start the background loop if needed
            var startBackgroundLoop = RenderUtils.profileMethod('DrawingRenderer.startBackgroundLoop()', function () {

                // nothing to do, if the timer is running, or if there are no dirty drawings
                if (refreshTimer || pendingModelSet.empty()) { return; }

                // start a background loop that will process all drawing models in 'pendingModelSet'
                refreshTimer = self.repeatSliced(function () {

                    // shift a drawing model from the pending set (exit loop when map is empty)
                    var drawingModel = pendingModelSet.getAny();
                    if (!drawingModel) { return Utils.BREAK; }

                    // refresh contents and formatting
                    formatDrawingFrame(drawingModel);

                }, 'DrawingRenderer.startBackgroundLoop', { delay: 200, slice: 200, interval: 200 });

                // clean up after the timer has finished
                refreshTimer.always(function () { refreshTimer = null; });
            });

            return app.createDebouncedActionsMethodFor(self, 'DrawingRenderer.formatDrawingFrameDebounced', registerDrawingModel, startBackgroundLoop);
        }());

        /**
         * Renders the DOM drawing frame that represents the passed drawing
         * model. The position and visibility of the drawing frame will be
         * updated immediately, the (expensive) formatting will be updated in a
         * background loop.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance whose drawing frame will be updated.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.hidden=false]
         *      Whether the drawing frame or any of its ancestors were hidden,
         *      and therefore need to be fully updated.
         *  - {Boolean} [options.zoom=false]
         *      Whether the zoom factor has been changed, i.e. whether all
         *      drawing frames need to be updated completely regardless of
         *      their current state.
         *  - {Boolean} [options.format=false]
         *      If set to true, the formatting of the drawing frame will always
         *      be updated. By default, the formatting will only be updated, if
         *      the visibility or size of the drawing frame has been changed.
         *  - {Boolean} [options.text=false]
         *      If set to true, the text contents of the drawing frame will
         *      always be updated. By default, the text contents will only be
         *      updated after updating the formatting of the drawing frame.
         *  - {Boolean} [options.immediate=false]
         *      If set to true, the entire formatting of the drawing frame will
         *      be updated immediately instead of debounced with a background
         *      task.
         */
        function renderDrawingFrame(drawingModel, options) {

            // update cell overlay node of comments (also for invisible comments!)
            if (drawingModel instanceof SheetCommentModel) {
                var visibleAnchor = updateCommentOverlayNode(drawingModel);
                if (!visibleAnchor) { return; }
            }

            // the DOM drawing frame (skip drawing objects that are not inside the viewport)
            var drawingFrame = getRenderDrawingFrame(drawingModel);
            if (!drawingFrame) { return; }

            // settings for the model frame (drawing frame used as source for the text contents)
            var frameSettings = drawingModel.refreshModelFrame();
            if (!frameSettings) { return; }

            // copy explicit formatting attributes (expected by text framework, and drawing framework)
            AttributeUtils.setExplicitAttributes(drawingFrame, frameSettings.explicitAttrSet);

            var canvasNode = DrawingFrame.getCanvasNode(drawingFrame);
            var oldCanvasWidth = (canvasNode.length === 1) ? canvasNode.width() : null;
            var oldCanvasHeight = (canvasNode.length === 1) ? canvasNode.height() : null;

            // old location of the drawing frame in pixels (unless it was hidden)
            var oldHidden = Utils.getBooleanOption(options, 'hidden', false) || drawingFrame.hasClass('hidden');
            var oldRectangle = oldHidden ? null : drawingFrame.data('rectangle');
            toggleFrameClass(drawingFrame, 'hidden', false);

            // new location of the drawing frame in pixels (cannot be null due to preparations above)
            var newRectangle = drawingModel.getRectangle();
            drawingFrame.data('rectangle', newRectangle);

            // special display for new drawing frames that have not been formatted yet
            var formatSupported = supportsFormatting(drawingModel);
            if (!oldRectangle && formatSupported) { drawingFrame.addClass('uninitialized'); }

            // formatting and text contents need to be updated if specified, or if shown initially
            updateDirtyFlags(drawingFrame, oldRectangle ? options : { format: true, text: true });

            // set current position and size of the drawing frame as CSS properties
            var cssRectangle = drawingModel.isEmbedded() ? newRectangle : gridPane.convertToLayerRectangle(newRectangle);
            cssRectangle.left = Utils.minMax(cssRectangle.left, -Utils.MAX_NODE_SIZE, Utils.MAX_NODE_SIZE);
            cssRectangle.top = Utils.minMax(cssRectangle.top, -Utils.MAX_NODE_SIZE, Utils.MAX_NODE_SIZE);
            drawingFrame.css(cssRectangle.toCSS());

            // update flipping attributes manually
            var oldFlipH = DrawingFrame.isFlippedHorz(drawingFrame);
            var oldFlipV = DrawingFrame.isFlippedVert(drawingFrame);
            var newFlipH = drawingModel.isFlippedH();
            var newFlipV = drawingModel.isFlippedV();
            var isRotatable = drawingModel.isDeepRotatable();
            var flipData = drawingModel.getEffectiveFlipping();
            drawingFrame.toggleClass(DrawingFrame.FLIPPED_HORIZONTALLY_CLASSNAME, isRotatable ? newFlipH : flipData.flipH);
            drawingFrame.toggleClass(DrawingFrame.FLIPPED_VERTICALLY_CLASSNAME, isRotatable ? newFlipV : flipData.flipV);

            // apply rotation transformation
            var degrees = drawingModel.getRotationDeg();
            DrawingFrame.setCssTransform(drawingFrame, degrees, newFlipH, newFlipV);
            drawingFrame.toggleClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME, degrees !== 0);

            // the selection needs to be redrawn, if the flip state of a connector drawing has changed
            if (((oldFlipH !== newFlipH) || (oldFlipV !== newFlipV)) && DrawingFrame.isSelected(drawingFrame) && DrawingFrame.isConnectorDrawingFrame(drawingFrame)) {
                updateSelectionFrame(drawingFrame, true);
            }

            // the resulting dirty flags
            var zoomChanged = Utils.getBooleanOption(options, 'zoom', false);
            var dirtyFormat = drawingFrame.data('dirtyFormat');
            var dirtyText = drawingFrame.data('dirtyText');

            // fast refresh of DOM positions and size while modifying or zooming visible drawings
            if (formatSupported && oldRectangle && (zoomChanged || dirtyFormat)) {

                // update position of the embedded canvas node for shapes
                if (canvasNode.length === 1) {
                    var widthScale = newRectangle.width / oldRectangle.width;
                    var heightScale = newRectangle.height / oldRectangle.height;
                    canvasNode.css({
                        left: Math.round(Utils.convertCssLength(canvasNode.css('left'), 'px') * widthScale),
                        top: Math.round(Utils.convertCssLength(canvasNode.css('top'), 'px') * heightScale),
                        width: Math.round(oldCanvasWidth * widthScale),
                        height: Math.round(oldCanvasHeight * heightScale)
                    });
                }

                // set correct padding of the text frame according to current zoom factor
                if (frameSettings.textPosition) {
                    TextFrameUtils.withTextFrame(drawingFrame, function (textFrame) {
                        updateTextFramePosition(drawingModel, textFrame, frameSettings.textPosition);
                    });
                }
            }

            // update additional nodes for comments
            if (drawingModel instanceof SheetCommentModel) {
                updateCommentConnectorNode(drawingModel);
            }

            // immediately notify all listeners after the position or size has been changed
            self.trigger('render:drawingframe', drawingFrame, drawingModel);

            // start deferred update of formatting and text contents
            if (formatSupported && (dirtyFormat || dirtyText)) {
                formatDrawingFrameDebounced(drawingModel, options);
            } else {
                pendingModelSet.remove(drawingModel);
            }

            // propagate dirty formatting to all embedded frames
            if (oldHidden) { options = _.extend({}, options, { hidden: true }); }
            if (dirtyFormat) { options = _.extend({}, options, { format: true }); }

            // update existing embedded drawing frames too
            drawingModel.forEachChildModel(function (childModel) {
                renderDrawingFrame(childModel, options);
            });
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model. The position and visibility of the drawing
         * frame will be initialized immediately, the (expensive) formatting
         * will be updated in a background loop.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to create a DOM drawing frame for.
         */
        function createDrawingFrame(drawingModel) {

            // the drawing frame manager responsible for the passed model
            var frameManager = getFrameManager(drawingModel);
            // the new drawing representation, as jQuery object
            var drawingFrame = frameManager.createDrawingFrame(drawingModel);
            if (!drawingFrame) { return; }

            // create additional nodes for comments
            if (drawingModel instanceof SheetCommentModel) {

                // create the cell overlay node
                var overlayNode = $('<div>').appendTo(overlayLayerNode);
                // store the comment model in the jQuery data object (needed to resolve hovering comment)
                overlayNode.data('commentModel', drawingModel);
                // store the overlay node in the drawing frame jQuery data object for fast lookup without DOM access
                drawingFrame.data('overlayNode', overlayNode);

                // create the SVG root node for the connector line
                var connectorNode = createSvgElement('svg');
                connectorLayerNode.append(connectorNode);
                // create the SVG connector line node
                connectorNode.appendChild(createSvgElement('line'));
                // register connector node as client of the comment frame to synchronize visibility
                registerClientNodes(drawingFrame, connectorNode);
            }

            // initially render the drawing frame
            renderDrawingFrame(drawingModel, { format: true, text: true });
        }

        /**
         * Removes the drawing frame that represents the passed drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to remove the DOM drawing frame for.
         */
        function removeDrawingFrame(drawingModel) {

            // the drawing frame manager responsible for the passed model
            var frameManager = getFrameManager(drawingModel);

            // remove the drawing frame and all embedded child frames from the renderer
            (function removeDrawingFrameFromMaps(parentModel) {

                // remove the drawing model form the pending map (do not try to render debounced)
                pendingModelSet.remove(parentModel);

                // do not try to access temporarily shown drawing frames after deletion
                tempShowModelSet.remove(parentModel);
                if (parentModel === hoverCommentModel) { hoverCommentModel = null; }

                // the drawing frame to be removed
                var drawingFrame = frameManager.getDrawingFrame(parentModel);
                if (!drawingFrame) { return; }

                // remove the client nodes from the DOM
                var clientNodes = getClientNodes(drawingFrame);
                if (clientNodes) { clientNodes.remove(); }

                // remove the cell overlay node for comments
                var overlayNode = drawingFrame.data('overlayNode');
                if (overlayNode) { overlayNode.remove(); }

                // remove the drawing frames of all embedded drawing models
                parentModel.forEachChildModel(removeDrawingFrameFromMaps);

            }(drawingModel));

            // remove the drawing frame from the manager (processes child objects internally)
            frameManager.removeDrawingFrame(drawingModel);
        }

        /**
         * Updates the selection of all drawing frames, and triggers the event
         * 'render:drawingselection', if the drawing selection has changed.
         */
        var renderDrawingSelection = this.createDebouncedMethod('DrawingRenderer.renderDrawingSelection', null, function () {

            // all drawing frames currently selected in the DOM
            var oldSelectedFrames = drawingFrameManager.getSelectedDrawingFrames();
            // all drawing frames to be selected in the DOM, according to current selection state
            var newSelectedFrames = self.getDrawingFrames(docView.getSelectedDrawings());

            // create or update the selection frame for all selected drawings
            newSelectedFrames.get().forEach(updateSelectionFrame);

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.not(newSelectedFrames).get().forEach(DrawingFrame.clearSelection);

            // notify listeners
            if (!Utils.equalArrays(oldSelectedFrames.get(), newSelectedFrames.get())) {
                self.trigger('render:drawingselection', newSelectedFrames);
            }
        });

        /**
         * Updates the remote selection of all drawing frames.
         */
        var renderRemoteSelection = Config.SHOW_REMOTE_SELECTIONS ? this.createDebouncedMethod('DrawingRenderer.renderRemoteSelection', null, function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) { return; }

            // the drawing collection of the active sheet
            var drawingCollection = docView.getDrawingCollection();

            // remove the old remote selection nodes from the drawing frames
            remoteDrawingFrames.forEach(function (drawingFrame) {
                drawingFrame.removeAttr('data-remote-user data-remote-color').children('.remote-selection').remove();
            });
            remoteDrawingFrames = [];

            // generate selections of all remote users on the same sheet
            docView.iterateRemoteSelections(function (userName, colorIndex, ranges, drawings) {
                if (!drawings) { return; }

                // add a remote selection node to all selected drawings of the user
                drawings.forEach(function (position) {
                    drawingFrameManager.withDrawingFrame(drawingCollection.getModel(position), function (drawingFrame) {

                        // do not insert multiple selection nodes into a drawing frame
                        if (drawingFrame.children('.remote-selection').length > 0) { return; }

                        // create the child node to display the border around the frame, and the data attributes for the user badge
                        drawingFrame.prepend('<div class="remote-selection">').attr({
                            'data-remote-user': _.noI18n(userName),
                            'data-remote-color': colorIndex
                        });
                        remoteDrawingFrames.push(drawingFrame);
                    });
                });
            });
        }, { delay: 100 }) : _.noop;

        /**
         * Shows the cell comment frame associated to the passed comment model
         * temporarily while its anchor cell is hovered with the mouse pointer.
         *
         * @param {CommentModel} commentModel
         *  The comment model to be shown temporarily.
         */
        function showHoverComment(commentModel) {

            // nothing to be done, if the cell comment is permanently visible
            if (commentModel.isVisible()) { return; }

            // show the cell comment with a specific delay
            hoverCommentTimer = self.executeDelayed(function () {
                hoverCommentModel = commentModel;
                renderDrawingFrame(commentModel, { immediate: true });
            }, 'DrawingRenderer.showHoverComment', 500);
        }

        /**
         * Hides the cell comment frame that has been made visible with the
         * method DrawingRenderer.showHoverComment().
         */
        function hideHoverComment() {

            // abort the delay timer
            if (hoverCommentTimer) {
                hoverCommentTimer.abort();
                hoverCommentTimer = null;
            }

            // hide the cell comment
            if (hoverCommentModel) {
                var commentModel = hoverCommentModel;
                hoverCommentModel = null;
                renderDrawingFrame(commentModel);
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the drawing layer according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('DrawingRenderer.setLayerRange()', function (layerSettings) {

            // whether existing drawing frames need to be updated
            var isVisible = layerRectangle !== null;

            // store new layer rectangle in internal member
            layerRectangle = layerSettings.rectangle;

            // repaint all visible drawing frames and comments; or create all drawing frames
            // if not done yet (e.g. active sheet changed, enabled split in current sheet)
            var drawingIterator = docView.getDrawingCollection().createModelIterator();
            var commentIterator = docView.getCommentCollection().createModelIterator();

            // create a combined iterator for drawing objects and cell comments
            var iterator = new SerialIterator(drawingIterator, commentIterator);

            // render all existing drawing frames, if the layer range has changed
            if (isVisible) {
                var options = (layerSettings.zoomScale !== 1) ? { format: true, zoom: true } : null;
                Iterator.forEach(iterator, function (drawingModel) {
                    renderDrawingFrame(drawingModel, options);
                });
                return;
            }

            // prepare the page containers
            drawingFrameManager.initializePageNode();
            commentFrameManager.initializePageNode();

            // create all DOM drawing frames in this sheet, schedule for deferred rendering
            Iterator.forEach(iterator, createDrawingFrame);

            // paint the entire drawing selection
            renderDrawingSelection();
        });

        /**
         * Resets this renderer, and clears the DOM layer node (called e.g.
         * before switching the active sheet).
         */
        this.hideLayerRange = function () {

            // clear the cache of pending drawing frames
            pendingModelSet.clear();

            // hide the cell comment hovered with the mouse
            hideHoverComment();

            // reset internal settings
            layerRectangle = null;
            remoteDrawingFrames = [];

            // hide all drawing frames
            drawingFrameManager.removeAllDrawingFrames();
            commentFrameManager.removeAllDrawingFrames();

            // remove the cell overlay nodes and connector lines of all cell comments
            overlayLayerNode.empty();
            connectorLayerNode.empty();
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM drawing frame node at the passed document position.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame, as jQuery object; or null, if the drawing
         *  frame does not exist.
         */
        this.getDrawingFrame = function (position) {
            var drawingModel = docView.getDrawingCollection().getModel(position);
            return drawingModel ? drawingFrameManager.getDrawingFrame(drawingModel) : null;
        };

        /**
         * Returns the DOM drawing frame nodes for all specified document
         * positions.
         *
         * @param {Array<Array<Number>>} positions
         *  The document positions of the drawing models.
         *
         * @returns {jQuery}
         *  The DOM nodes of the drawing frames, as jQuery object.
         */
        this.getDrawingFrames = function (positions) {
            return _.reduce(positions, function (drawingFrames, position) {
                return drawingFrames.add(this.getDrawingFrame(position));
            }, $(), this);
        };

        /**
         * Returns the DOM drawing frame that represents the passed drawing
         * model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to return the DOM drawing frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model; or null, if no
         *  drawing frame could be found.
         */
        this.getDrawingFrameForModel = function (drawingModel) {
            return drawingFrameManager.getDrawingFrame(drawingModel);
        };

        /**
         * Temporarily shows the DOM drawing frame associated to the specified
         * drawing model. This method counts its invocations internally. The
         * method DrawingRenderer.tempHideDrawingFrame() will hide the drawing
         * frame after the respective number of invocations.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to show temporarily.
         *
         * @returns {DrawingRenderer}
         *  A reference to this instance.
         */
        this.tempShowDrawingFrame = function (drawingModel) {

            // update the set also in invisible grid panes (drawing frame will become visible after splitting the view)
            tempShowModelSet.insert(drawingModel);

            // render the drawing frame, if the grid pane is visible
            if (gridPane.isVisible()) {
                renderDrawingFrame(drawingModel, { immediate: true });
            }

            return this;
        };

        /**
         * Hides the DOM drawing frame associated to the specified drawing
         * model, after it has been made visible temporarily. See description
         * of the method DrawingRenderer.tempShowDrawingFrame() for more
         * details.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to hide.
         *
         * @returns {DrawingRenderer}
         *  A reference to this instance.
         */
        this.tempHideDrawingFrame = function (drawingModel) {

            // update the set also in invisible grid panes
            tempShowModelSet.remove(drawingModel);

            // hide the drawing frame, if the grid pane is visible
            if (gridPane.isVisible()) {
                renderDrawingFrame(drawingModel);
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // repaint drawing selection (also when document edit mode changes, e.g. existence of resize handles)
        this.listenToWhenVisible(app, 'docs:editmode', renderDrawingSelection);
        this.listenToWhenVisible(docView, 'change:selection change:sheet:attributes', renderDrawingSelection);

        // trigger a 'render:drawingselection' event if active pane changes
        // (no actual rendering of the selection needed, border colors will change via CSS)
        this.listenToWhenVisible(docView, 'change:sheet:viewattributes', function (event, attributes) {
            if ('activePane' in attributes) {
                self.trigger('render:drawingselection', drawingFrameManager.getSelectedDrawingFrames());
            }
        });

        // repaint remote drawing selections
        this.listenToWhenVisible(docView, 'change:viewattributes', function (event, attributes) {
            if ('remoteClients' in attributes) {
                renderRemoteSelection();
            }
        });

        // process new drawing objects inserted into the drawing collection
        this.listenToWhenVisible(docView, 'insert:drawing', function (event, drawingModel) {
            createDrawingFrame(drawingModel);
            renderDrawingSelection();
            renderRemoteSelection();
        });

        // process drawing objects removed from the drawing collection
        this.listenToWhenVisible(docView, 'delete:drawing', function (event, drawingModel) {
            removeDrawingFrame(drawingModel);
            renderDrawingSelection();
            renderRemoteSelection();
        });

        // process drawing objects that have been changed in any way (position, size, attributes)
        this.listenToWhenVisible(docView, 'change:drawing', function (event, drawingModel) {
            renderDrawingFrame(drawingModel, { format: true, text: true });
        });

        // process drawing objects whose text contents have been changed
        this.listenToWhenVisible(docView, 'change:drawing:text', function (event, drawingModel) {
            renderDrawingFrame(drawingModel, { text: true });
        });

        // process drawing objects that have been moved to a new document position (Z order)
        this.listenToWhenVisible(docView, 'move:drawing', function (event, drawingModel) {
            drawingFrameManager.moveDrawingFrame(drawingModel);
            renderDrawingSelection();
            renderRemoteSelection();
        });

        // process new comments inserted into the drawing collection
        this.listenToWhenVisible(docView, 'insert:comment', function (event, commentModel) {
            createDrawingFrame(commentModel);
        });

        // process comments removed from the drawing collection
        this.listenToWhenVisible(docView, 'delete:comment', function (event, commentModel) {
            removeDrawingFrame(commentModel);
        });

        // process comments that have been moved to a new anchor cell
        this.listenToWhenVisible(docView, 'move:comment', function (event, commentModel) {
            renderDrawingFrame(commentModel);
        });

        // process comments that have been changed in any way (position, size, attributes)
        this.listenToWhenVisible(docView, 'change:comment', function (event, commentModel) {
            renderDrawingFrame(commentModel, { format: true, text: true });
        });

        // process comments whose text contents have been changed
        this.listenToWhenVisible(docView, 'change:comment:text', function (event, commentModel) {
            renderDrawingFrame(commentModel, { text: true });
        });

        // repaint comment notifications when merging/unmerging cell ranges
        this.listenToWhenVisible(docView, 'insert:merged delete:merged', function (event, ranges) {
            var commentCollection = docView.getCommentCollection();
            var iterator = commentCollection.createModelIterator({ ranges: ranges });
            Iterator.forEach(iterator, function (commentModel) {
                renderDrawingFrame(commentModel);
            });
        });

        // show comments temporarily when hovering their anchor cell
        overlayLayerNode.on('mouseenter', '>div', function () {
            hideHoverComment();
            var commentModel = $(this).data('commentModel');
            if (commentModel) { showHoverComment(commentModel); }
        });

        // hide comments when leaving their anchor cell
        overlayLayerNode.on('mouseleave', '>div', hideHoverComment);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            commentFrameManager.destroy();
            drawingFrameManager.destroy();
            pendingModelSet.clear();
            tempShowModelSet.clear();
            self = gridPane = docModel = docView = app = null;
            overlayLayerNode = drawingLayerNode = connectorLayerNode = commentLayerNode = null;
            drawingFrameManager = commentFrameManager = null;
            contextMenu = pendingModelSet = tempShowModelSet = null;
            hoverCommentModel = hoverCommentTimer = remoteDrawingFrames = null;
        });

    } }); // class DrawingRenderer

    // exports ================================================================

    return DrawingRenderer;

});
