/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/edit/celltexteditor', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/popup/tooltip',
    'io.ox/office/tk/popup/listmenu',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/sheetselection',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/popup/signaturetooltip',
    'io.ox/office/spreadsheet/view/edit/texteditorbase',
    'io.ox/office/spreadsheet/view/edit/cellundomanager',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, KeyCodes, Forms, Iterator, ToolTip, ListMenu, Color, AttributeUtils, SheetUtils, SheetSelection, PaneUtils, SignatureToolTip, TextEditorBase, CellUndoManager, gt) {

    'use strict';

    // convenience shortcuts
    var Range = SheetUtils.Range;

    // key codes of all keys used to navigate in text area, or in the sheet (quick edit mode)
    var NAVIGATION_KEYCODES = Utils.makeSet([
        KeyCodes.LEFT_ARROW,
        KeyCodes.RIGHT_ARROW,
        KeyCodes.UP_ARROW,
        KeyCodes.DOWN_ARROW,
        KeyCodes.PAGE_UP,
        KeyCodes.PAGE_DOWN,
        KeyCodes.HOME,
        KeyCodes.END
    ]);

    // class CellTextEditor ===================================================

    /**
     * Implementation of the text edit mode in cells.
     *
     * @constructor
     *
     * @extends TextEditorBase
     *
     * @param {SpreadsheetView} docView
     *  The document view that has created this instance.
     */
    var CellTextEditor = TextEditorBase.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the spreadsheet application, model, and other model objects
        var app = docView.getApp();
        var docModel = docView.getDocModel();
        var attributePool = docModel.getCellAttributePool();
        var formulaResource = docModel.getFormulaResource();
        var formulaGrammar = docModel.getFormulaGrammar('ui');
        var formulaParser = docModel.getFormulaParser();

        // the underlay node containing highlighting for formulas
        var cellUnderlayNode = $('<div class="underlay">');
        // the text area control
        var cellInputNode = $(Forms.createTextAreaMarkup());
        // the container element for the text underlay node and the text area control
        var cellInputContainer = $('<div class="textarea-container" data-focus-role="textarea">').append(cellUnderlayNode, cellInputNode);

        var formulaBarMode = false;
        // the underlay node containing highlighting for formulas in the formula tool pane
        var paneUnderlayNode = null;
        // the text area control in the formula tool pane
        var paneInputNode = null;

        // the effective DOM node for text input, also used as anchor node for pop-up nodes
        var textInputNode = cellInputNode;
        // the effective DOM node for text output (mirrored from text input node)
        var textOutputNode = cellInputNode;

        // tooltip attached to the text area for function/parameter help
        var signatureToolTip = new SignatureToolTip(docView, getPopupAnchor);
        // drop-down list attached to the text area (for function names)
        var textAreaListMenu = new ListMenu(docView, { anchor: getPopupAnchor, autoFocus: false, focusableNodes: cellInputNode });
        // tooltip attached to the text area drop-down list (for the selected function)
        var textAreaListItemToolTip = new ToolTip(docView, { anchor: textAreaListMenu.getNode(), anchorBorder: 'right left', anchorAlign: 'center', anchorAlignBox: getTextAreaListButtonNode });
        // unfiltered function entries for the auto-completion list menu
        var functionEntries = null;

        // the model of the sheet containing the edited cell
        var sheetModel = null;
        // the cell collection of the sheet containing the edited cell
        var cellCollection = null;
        // the cell address of the edited cell
        var editAddress = null;
        // the cell attribute set of the edited cell
        var cellAttributeSet = null;
        // the effective font settings for the edited cell
        var fontDesc = null;
        // the effective line height of the edited cell, in pixels
        var lineHeight = null;
        // the token descriptors received form the formula parser for the current formula in the text area
        var tokenDescs = null;
        // the token array representing the current formula in the text area
        var tokenArray = null;

        // local undo/redo manager while cell edit mode is active
        var undoManager = new CellUndoManager(this);
        // additional attributes applied locally at the text area
        var editAttributeSet = null;

        // initial text at the beginning of the edit mode
        var originalText = null;
        // the text at the time the text area has been updated the last time
        var prevEditText = null;
        // collected entries for type-ahead suggestions
        var typeAheadEntries = null;

        // quick edit mode (true), or text cursor mode (false)
        var quickEditMode = null;
        // the formula expression, if the input string starts with specific characters
        var formulaString = null;
        // selection offset into the formula string (1, if formula starts with equality sign; or 0 for plus/minus)
        var formulaOffset = 0;
        // unique identifier of the current range highlighting mode for formulas
        var highlightUid = null;
        // whether to show auto-completion list for a function or defined name
        var funcAutoComplete = false;
        // token descriptor for function/name auto-completion
        var funcAutoTokenDesc = null;
        // whether to keep auto-completion list open although focus is not in text area
        var stickyAutoComplete = false;
        // how many closing parenthesis entered manually will be ignored
        var funcAutoCloseData = [];
        // the text position of the current range address while in range selection mode
        var rangeSelection = null;
        // the state of the mobile-soft-keyboard (open or not)
        var keyboardOpen = false;

        // base constructor ---------------------------------------------------

        TextEditorBase.call(this, docView, 'cell', {
            enterHandler: enterHandler,
            leaveHandler: leaveHandler,
            cancelHandler: cancelHandler,
            formatResolver: formatResolver,
            formatHandler: formatHandler,
            focusResolver: function () { return textInputNode[0]; }
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the current text input element. Used as dynamic resolver
         * function for the anchor node of pop-up menus and tooltips.
         *
         * @returns {jQuery}
         *  The current text input element.
         */
        function getPopupAnchor() {
            return textInputNode;
        }

        /**
         * Applies a transformation to the passed color, and returns the CSS
         * color value.
         */
        function getModifiedCssColor(jsonColor, colorType) {

            // the passed color, as instance of the class Color
            var color = Color.parseJSON(jsonColor);
            // the resolved color, including the effective luma
            var colorDesc = docModel.resolveColor(color, colorType);

            if ((colorType === 'text') && (colorDesc.y > 0.25)) {
                color.transform('shade', 25000 / colorDesc.y);
            } else if ((colorType === 'fill') && (colorDesc.y < 0.75)) {
                color.transform('tint', 25000 / (1 - colorDesc.y));
            }
            return docModel.resolveColor(color, colorType).css;
        }

        /**
         * Updates the CSS formatting of the text area control used for cell
         * edit mode according to the current formatting of the active cell.
         */
        function updateTextAreaStyle() {

            // get cell attributes, and overwrite them with current edit attributes
            cellAttributeSet = attributePool.extendAttributeSet(cellCollection.getAttributeSet(editAddress), editAttributeSet, { clone: true });

            // receive font settings, text line height, and text orientation
            var charAttrs = _.clone(cellAttributeSet.character);
            var cellAttrs = cellAttributeSet.cell;
            fontDesc = docModel.getRenderFont(charAttrs, null, sheetModel.getEffectiveZoom());
            charAttrs.fontSize = fontDesc.size = Utils.minMax(fontDesc.size, 6, 100);
            lineHeight = docModel.getRowHeight(charAttrs, 1);
            var orientation = cellCollection.getTextOrientation(editAddress);

            // set CSS attributes according to formatting of active cell
            cellInputContainer.children().css({
                padding: '0 ' + docModel.getTextPadding(charAttrs, 1) + 'px',
                lineHeight: lineHeight + 'px',
                fontFamily: fontDesc.family,
                fontSize: fontDesc.size + 'pt',
                fontWeight: fontDesc.bold ? 'bold' : 'normal',
                fontStyle: fontDesc.italic ? 'italic' : 'normal',
                textDecoration: AttributeUtils.getCssTextDecoration(charAttrs),
                textAlign: (cellCollection.isFormulaCell(editAddress) && (cellAttrs.alignHor === 'auto')) ? 'left' : orientation.cssTextAlign
            }).attr('dir', orientation.textDir);

            // set cell text color and fill color to the appropriate nodes
            cellInputNode.css('color', getModifiedCssColor(charAttrs.color, 'text'));
            cellUnderlayNode.css('background-color', getModifiedCssColor(cellAttrs.fillColor, 'fill'));
        }

        /**
         * Updates the position and size of the text area control for the cell
         * edit mode, according to the current length of the text in the text
         * area control, and the scroll position of the grid pane.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.direct=false]
         *      If set to true, the position of the text area will be updated
         *      immediately. By default, updating the text area will be done
         *      debounced.
         *  - {Boolean} [options.size=true]
         *      If set to false, the size of the text area will not be changed,
         *      only its position. Used in situations where the content of the
         *      text area does not change (e.g. while scrolling).
         */
        var updateTextAreaPosition = (function () {

            // whether updating the position is pending
            var updatePosition = false;
            // whether updating the size is pending
            var updateSize = false;

            // direct callback, invoked immediately when calling updateTokenArray()
            function directCallback(options) {

                // set the pending update flags for the deferred callback
                updatePosition = true;
                updateSize = updateSize || Utils.getBooleanOption(options, 'size', true);

                // invoke deferred callback immediately if specified, or if the text has caused a new line break
                if (Utils.getBooleanOption(options, 'direct', false) || (cellInputNode.outerHeight() < cellInputNode[0].scrollHeight)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // do nothing if cell edit mode is not active anymore
                if (!self.isActive() || (!updatePosition && !updateSize)) { return; }

                // the position of the edited cell
                var cellRect = null;
                // the visible area of the grid pane
                var visibleRect = null;
                // whether the text in the cell wraps automatically
                var isWrapped = _.isString(cellCollection.getValue(editAddress)) && SheetUtils.hasWrappingAttributes(cellAttributeSet);
                // the effective horizontal CSS text alignment
                var textAlign = cellInputNode.css('text-align');
                // round up width to multiples of the line height
                var blockWidth = Math.max(20, lineHeight);
                // the new width and height of the text area
                var targetWidth = 0, targetHeight = 0;

                // returns required width needed to show all text lines in a single line
                function getMinimumWidth() {

                    // required width for all text lines
                    var requiredWidth = cellInputNode.val().split(/\n/).reduce(function (memo, textLine) {
                        return Math.max(memo, fontDesc.getTextWidth(textLine));
                    }, 0);

                    // round up to block width, enlarge to cell width
                    return Math.max(cellRect.width, Utils.roundUp(requiredWidth, blockWidth) + blockWidth + Utils.SCROLLBAR_WIDTH);
                }

                // returns maximum width according to horizontal alignment and width of visible area
                function getMaximumWidth() {

                    // the distance from left visible border to right cell border (for right alignment)
                    var leftWidth = Utils.minMax(cellRect.right() - visibleRect.left, 0, visibleRect.width);
                    // the distance from left cell border to right visible border (for left alignment)
                    var rightWidth = Utils.minMax(visibleRect.right() - cellRect.left, 0, visibleRect.width);
                    // effective available width
                    var availWidth = 0;

                    // get available width according to text alignment
                    switch (textAlign) {
                        case 'right':
                            availWidth = leftWidth;
                            break;
                        case 'center':
                            availWidth = Math.min(leftWidth, rightWidth) * 2 - cellRect.width;
                            break;
                        default:
                            availWidth = rightWidth;
                    }

                    // enlarge to at least half of the width of the visible area
                    return Math.max(availWidth, Math.round(visibleRect.width / 2));
                }

                // get horizontal and vertical cell position and size
                cellRect = docView.getCellRectangle(editAddress, { expandMerged: true });
                // reduce width by one pixel to exclude the grid line
                cellRect.width = Math.max(blockWidth, cellRect.width - 1);
                // reduce height by one pixel to exclude the grid line
                cellRect.height -= 1;

                // get position and size of the visible area in the grid pane
                visibleRect = self.getEditGridPane().getVisibleRectangle();

                // on small Android devices, soft keyboard can overlay complete grid pane,
                // then the automatic height calculation has zero height,
                // after closing the soft keyboard it still keeps this height and that looks really wrong
                if (Utils.CHROME_ON_ANDROID) {
                    visibleRect.height = Math.max(visibleRect.height, lineHeight);
                }

                // calculate and set the new size of the text area
                if (updateSize) {

                    // find best size, stick to cell width in auto-wrapping mode
                    var minWidth = isWrapped ? cellRect.width : getMinimumWidth();
                    var maxWidth = isWrapped ? visibleRect.width : getMaximumWidth();

                    // calculate width and height of the text area
                    targetWidth = Math.min(minWidth, maxWidth);
                    fontDesc.getCustomFontMetrics(function (node) {

                        // the text whose height will be determined
                        var text = cellInputNode.val();

                        // use NBSP instead of empty text; bug 35143: add NBSP to empty last line
                        if (/(^|\n)$/.test(text)) { text += '\xa0'; }

                        // prepare the helper node for multi-line text
                        node.style.width = targetWidth + 'px';
                        node.style.padding = '0 ' + sheetModel.getTextPadding(cellAttributeSet.character) + 'px';
                        node.style.lineHeight = lineHeight + 'px';
                        node.style.whiteSpace = 'pre-wrap';
                        node.style.wordWrap = 'break-word';

                        // add a text node
                        node.appendChild(document.createTextNode(text));

                        targetHeight = Math.min(Math.max(node.offsetHeight, cellRect.height), visibleRect.height);
                    });

                    cellInputNode.css({ width: targetWidth, height: targetHeight });

                } else {
                    targetWidth = cellInputNode.outerWidth();
                    targetHeight = cellInputNode.outerHeight();
                }

                // calculate and set the new position of the text area
                if (updatePosition) {

                    // left and top position of the text area
                    var leftOffset = cellRect.left;
                    var topOffset = cellRect.top;

                    // calculate horizontal position according to alignment
                    switch (textAlign) {
                        case 'right':
                            leftOffset += cellRect.width - targetWidth;
                            break;
                        case 'center':
                            leftOffset += Math.round((cellRect.width - targetWidth) / 2);
                            break;
                    }

                    // move text area into visible area, set effective position
                    leftOffset = Utils.minMax(leftOffset, visibleRect.left, visibleRect.right() - targetWidth);
                    topOffset = Utils.minMax(topOffset, visibleRect.top, visibleRect.bottom() - targetHeight);
                    cellInputContainer.css({ left: leftOffset - visibleRect.left, top: topOffset - visibleRect.top });
                }

                updatePosition = updateSize = false;
            }

            // return the debounced method updateTextAreaPosition()
            return self.createDebouncedMethod('CellTextEditor.updateTextAreaPosition', directCallback, deferredCallback, { delay: 250, maxDelay: 500 });
        }());

        /**
         * Returns the selected list item node in the drop-down menu element
         * for the text area.
         *
         * @returns {jQuery}
         *  The selected menu item element in the text area drop-down menu.
         */
        function getTextAreaListButtonNode() {
            return textAreaListMenu.getSelectedItemNodes().first();
        }

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionDescriptionToolTip() {

            // the selected button element in the auto-completion menu
            var buttonNode = getTextAreaListButtonNode();
            // the name and type of the selected list item
            var itemValue = Forms.getButtonValue(buttonNode);
            // the resulting tooltip text
            var toolTipLabel = null;

            // create the tooltip text according to the type of the list entry
            switch (Utils.getStringOption(itemValue, 'type')) {

                case 'name':
                    // the model of the selected defined name
                    var nameModel = docModel.getNameCollection().getName(itemValue.key);
                    if (nameModel) {
                        toolTipLabel = nameModel.getFormula('ui', editAddress);
                        if (toolTipLabel) { toolTipLabel = '=' + toolTipLabel; }
                    }
                    break;

                case 'table':
                    // the model of the selected table range
                    var tableModel = docModel.getTable(itemValue.key);
                    if (tableModel) {
                        var sheetName = tableModel.getSheetModel().getName();
                        toolTipLabel = '=' + formulaParser.formatRangeList('ui', tableModel.getRange(), { sheetName: sheetName });
                    }
                    break;

                case 'func':
                    // the help descriptor for the selected function (nothing for defined names or tables)
                    var functionHelp = formulaResource.getFunctionHelp(itemValue.key);
                    if (functionHelp) { toolTipLabel = functionHelp.description; }
                    break;
            }

            // initialize the tooltip
            if (toolTipLabel) {
                textAreaListItemToolTip.setText(_.noI18n(toolTipLabel)).show();
            } else {
                textAreaListItemToolTip.hide();
            }
        }

        /**
         * Creates entries in the text area pop-up list menu.
         *
         * @param {Array<Object>} entries
         *  The unfiltered entries to be inserted into the list menu, as
         *  objects with the properties 'key' (inserted as property 'key' into
         *  the object values of the list entries) and 'name' (inserted as
         *  property 'name' into the object values of the list entries, used as
         *  visible labels of the list entries).
         *
         * @param {String} filter
         *  The filter string. Only entries with a 'name' property starting
         *  with this string (ignoring character case) will be inserted into
         *  the list menu.
         *
         * @param {String} type
         *  The type of the list entries. Will be used as name for a section in
         *  the list menu, and will be inserted as property 'type' into the
         *  object values of the list entries.
         *
         * @param {String} title
         *  Th UI label for the section title for the specified entry type.
         */
        function createListMenuEntries(entries, filter, type, title) {

            // create a section for the list entries
            textAreaListMenu.createSectionNode(type, { label: title });

            // filter for entries starting with the passed filter string
            filter = filter.toUpperCase();
            entries = entries.filter(function (entry) {
                return entry.name.substr(0, filter.length).toUpperCase() === filter;
            });

            // create the list entries
            entries.forEach(function (entry) {
                textAreaListMenu.createItemNode({ key: entry.key, name: entry.name, type: type }, { section: type, label: _.noI18n(entry.name) });
            });
        }

        /**
         * Returns information about the token containing or preceding the
         * passed text cursor position in the current formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the current formula string.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.ignoreWs=false]
         *      If set to true, the method will ignore the white-space token
         *      currently containing the text cursor, and will return the token
         *      preceding the white-space token.
         *  - {Boolean} [options.preferOperand=false]
         *      Specifies how to resolve the situation that the text cursor
         *      position is located between two tokens. If set to true, and one
         *      of the tokens is an operand token (constant value tokens, array
         *      literal tokens, reference tokens, name tokens, or function name
         *      tokens), this token will be returned. Otherwise, always returns
         *      the token preceding the text position.
         *
         * @returns {TokenDescriptor|Null}
         *  A descriptor for the token whose text representation contains the
         *  specified text cursor position. If the text cursor position is
         *  located outside of the formula text, or at its beginning (without
         *  the option 'preferOperand'), this method will return the value
         *  null.
         */
        function findTokenDescriptor(pos, options) {

            // returns whether the specified token is an operand token
            function isOperand(arrayIndex) {
                var tokenDesc = tokenDescs[arrayIndex];
                return tokenDesc && tokenDesc.token.matchesType(/^(value|array|ref|name|func)$/);
            }

            // whether to skip white-space token
            var ignoreWs = Utils.getBooleanOption(options, 'ignoreWs', false);
            // whether to prefer operand tokens
            var preferOperand = Utils.getBooleanOption(options, 'preferOperand', false);

            // cursor is located at the beginning of the formula, and operands are preferred:
            // return the first token (following the text cursor position)
            if (preferOperand && (pos === 0) && isOperand(0)) {
                return tokenDescs[0];
            }

            // invalid text position passed
            if (pos <= 0) { return null; }

            // find the token containing or preceding the passed text position
            var index = Utils.findFirstIndex(tokenDescs, function (tokenDesc) { return pos <= tokenDesc.end; }, { sorted: true });
            var currTokenDesc = tokenDescs[index];
            if (!currTokenDesc) { return null; }

            // check if the cursor is located between two tokens, and the trailing is the preferred operand
            if (preferOperand && (pos === currTokenDesc.end) && !isOperand(index) && isOperand(index + 1)) {
                return tokenDescs[index + 1];
            }

            // skip white-space token if specified
            if (ignoreWs && currTokenDesc.token.isType('ws')) {
                return (index > 0) ? tokenDescs[index - 1] : null;
            }

            // return the descriptor of the found token
            return currTokenDesc;
        }

        /**
         * Returns information about the innermost function containing the
         * passed text cursor position in the current formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the current formula string.
         *
         * @returns {Object|Null}
         *  A descriptor for the innermost function containing the specified
         *  text cursor position, in the following properties:
         *  - {String} funcKey
         *      The unique resource key of the function.
         *  - {Number} paramIndex
         *      The zero-based index of the function parameter covering the
         *      specified text cursor position.
         *  - {Number} paramCount
         *      The total number of parameters found in the function parameter
         *      list. If the formula is incomplete or contains errors, this
         *      number may be wrong, but it will always be greater than the
         *      value returned in the property 'paramIndex'.
         *  If the text cursor position is outside of any function, this method
         *  will return the value null.
         */
        function findFunctionDescriptor(pos) {

            // information about the token at the passed position
            var tokenDesc = findTokenDescriptor(pos, { preferOperand: true });
            // current token index, modified in seeking functions
            var index = 0;
            // the parameter index and count
            var param = 0, count = 0;

            // seek to the preceding opening parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToOpeningParenthesis() {
                var seps = 0;
                while (index >= 0) {
                    switch (tokenDescs[index].token.getType()) {
                        case 'open':
                            return seps;
                        case 'close':
                            index -= 1;
                            seekToOpeningParenthesis();
                            break;
                        case 'sep':
                            seps += 1;
                            break;
                    }
                    index -= 1;
                }
                return 0;
            }

            // seek to the following closing parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToClosingParenthesis() {
                var seps = 0;
                while (index < tokenDescs.length) {
                    switch (tokenDescs[index].token.getType()) {
                        case 'open':
                            index += 1;
                            seekToClosingParenthesis();
                            break;
                        case 'close':
                            return seps;
                        case 'sep':
                            seps += 1;
                            break;
                    }
                    index += 1;
                }
                return seps;
            }

            // invalid text cursor position passed
            if (!tokenDesc) { return null; }

            // try as long as a function has been found, or the beginning of the formula is reached
            index = tokenDesc.index;
            while (index >= 0) {

                // seek to the preceding opening parenthesis
                param = seekToOpeningParenthesis();

                // check if a function name precedes the parenthesis
                var funcTokenDesc = tokenDescs[index - 1];
                if (funcTokenDesc && funcTokenDesc.token.isType('func')) {
                    index += 1;
                    count = seekToClosingParenthesis() + 1;
                    return { funcKey: funcTokenDesc.token.getValue(), paramIndex: param, paramCount: count };
                }

                // continue seeking to preceding opening parenthesis
                index -= 1;
            }

            // cursor is not located inside a function, but it may point to a top-level
            // function name, in that case return this function with invalid parameter index
            if (tokenDesc.token.isType('func') && tokenDescs[tokenDesc.index + 1].token.isType('open')) {
                index = tokenDesc.index + 2;
                count = seekToClosingParenthesis() + 1;
                return { funcKey: tokenDesc.token.getValue(), paramIndex: -1, paramCount: count };
            }

            return null;
        }

        /**
         * Hides the function signature tooltip and the function
         * auto-completion pop-up menu.
         */
        function hideTextAreaPopups() {
            signatureToolTip.hide();
            textAreaListMenu.hide();
        }

        /**
         * Shows the function signature tooltip or the function auto-completion
         * pop-up menu if possible.
         *
         * @param {Boolean} [suppressAutoComplete=false]
         *  If set to true, the pop-up menu for auto-completion of function
         *  names will be suppressed regardless of the current formula text and
         *  cursor position.
         */
        function updateTextAreaPopups(suppressAutoComplete) {

            // show pop-ups only in formula mode, do not show them while dragging
            // a highlighted range, or when selecting new ranges for the formula
            if ((formulaString === null) || docView.hasTrackedHighlightedRange() || sheetModel.getViewAttribute('activeSelection')) {
                hideTextAreaPopups();
                return;
            }

            // the cursor position in the text area (exclude the equality sign)
            var textPos = textInputNode[0].selectionStart - formulaOffset;
            // the text to be used for function auto-completion
            var autoText = null;

            // prefer function auto-completion over function tooltip
            funcAutoTokenDesc = null;

            // auto completion of names: cursor must be located exactly at the end of a function or name token
            var tokenDesc = (funcAutoComplete && !suppressAutoComplete) ? findTokenDescriptor(textPos) : null;
            if (tokenDesc && (tokenDesc.end === textPos)) {

                // get the text entered before the text cursor (may be a name token, a table token, or a function token)
                // TODO: process sheet-local names too
                if ((tokenDesc.token.isType('name') && !tokenDesc.token.hasSheetRef()) || tokenDesc.token.isType('table') || tokenDesc.token.isType('func')) {
                    // Special case: do not show a pop-up menu when a literal number precedes the name.
                    // Otherwise, e.g. entering scientific numbers will show a pop-up for functions starting with 'E'.
                    if ((tokenDesc.index === 0) || !tokenDescs[tokenDesc.index - 1].token.isType('lit')) {
                        autoText = tokenDesc.text;
                    }
                }

                // store token info (needed to replace the entered text with the full function name)
                funcAutoTokenDesc = tokenDesc;
            }

            // fill pop-up menu for auto-completion
            if (_.isString(autoText)) {

                // insert all matching defined names, and all matching function names
                textAreaListMenu.clearContents();

                // insert defined names (the method createListMenuEntries() expects an array of objects with 'key' and 'name' properties)
                // TODO: sheet-local names
                var nameEntries = docModel.getAllNames({ skipHidden: true, skipLocal: true });
                nameEntries = nameEntries.map(function (nameModel) { return nameModel.getLabel(); });
                nameEntries.sort(function (label1, label2) { return label1.localeCompare(label2); });
                nameEntries = nameEntries.map(function (label) { return { key: label, name: label }; });
                createListMenuEntries(nameEntries, autoText, 'name',  gt.pgettext('named ranges', 'Named ranges'));

                // insert table ranges (the method createListMenuEntries() expects an array of objects with 'key' and 'name' properties)
                var tableEntries = docModel.getAllTables().map(function (tableModel) { return tableModel.getName(); });
                tableEntries.sort(function (name1, name2) { return name1.localeCompare(name2); });
                tableEntries = tableEntries.map(function (name) { return { key: name, name: name }; });
                createListMenuEntries(tableEntries, autoText, 'table', /*#. special cell ranges in a spreadsheet for filtering etc. */ gt('Table ranges'));

                // insert the matching function names
                createListMenuEntries(functionEntries, autoText, 'func', /*#. functions in spreadsheet formulas */ gt('Functions'));

                // show the menu, if it contains at least one entry
                if (textAreaListMenu.getItemNodes().length > 0) {
                    signatureToolTip.hide();
                    textAreaListMenu.show().selectItem(0, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return;
                }
            }

            // try to show a tooltip for the current function in a formula
            hideTextAreaPopups();
            var functionInfo = findFunctionDescriptor(textPos);
            if (functionInfo) {
                signatureToolTip.updateSignature(functionInfo.funcKey, functionInfo.paramIndex);
            }
        }

        /**
         * Updates the function tooltip, and other settings according to the
         * text selection in the text area control.
         */
        function updateTextAreaAfterSelection() {

            // do not open the auto-completion menu even if the new cursor position is valid
            funcAutoComplete = false;

            // do not ignore closing parentheses entered manually anymore (function auto-complete)
            funcAutoCloseData = [];

            // defer execution to let the browser process pending key events
            self.executeDelayed(function () {
                // fail-safe check that cell edit mode is still active
                if (self.isActive()) { updateTextAreaPopups(); }
            }, 'CellTextEditor.updateTextAreaAfterSelection');
        }

        /**
         * Updates the highlighting of cell references in the current formula
         * while cell edit mode is active.
         */
        function updateTextAreaHighlighting() {

            // descriptor objects for all resolved cell ranges in the formula
            var rangeDescs = tokenArray.extractRanges({
                refAddress: editAddress,
                targetAddress: editAddress,
                resolveNames: true
            });

            // show the cell ranges located in the active sheet only
            var activeSheet = docView.getActiveSheet();
            rangeDescs = rangeDescs.filter(function (rangeInfo) {
                return rangeInfo.range.containsSheet(activeSheet);
            });

            // nothing to do without any tokens to be highlighted
            if (rangeDescs.length === 0) {
                cellUnderlayNode.empty();
                paneUnderlayNode.empty();
                return;
            }

            // collect the scheme color index and the size (token count) of the range descriptors (map by token index)
            var rangeDataMap = {};
            var styleIndex = 0;
            rangeDescs.forEach(function (rangeDesc) {

                // skip multiple ranges based on the same name token
                if (rangeDesc.index in rangeDataMap) { return; }

                rangeDataMap[rangeDesc.index] = {
                    color: Utils.getSchemeColor(styleIndex),
                    size: rangeDesc.size
                };
                styleIndex += 1;
            });

            // start with the equality sign (unless the edit text does not contain it, e.g. in '+A1+A2')
            var markup = (formulaOffset === 0) ? '' : ('<span>' + Utils.repeatString('=', formulaOffset) + '</span>');

            // process the token descriptor array
            tokenDescs.forEach(function (tokenDesc, index) {

                // special handling for whitespace (line breaks via <br> elements)
                if (tokenDesc.token.isType('ws')) {
                    markup += tokenDesc.text.split(/\n/).map(function (spaces) {
                        return '<span>' + spaces + '</span>';
                    }).join('<br>');
                    return;
                }

                // create a span element with highlight color, if a corresponding range exists
                markup += '<span';
                var rangeData = rangeDataMap[index];
                if (rangeData) { markup += ' data-style="' + rangeData.color + '"';  }
                markup += '>';

                // add the token text representations of the token
                markup += Utils.escapeHTML(tokenDesc.text);
                markup += '</span>';
            });

            cellUnderlayNode[0].innerHTML = markup;
            paneUnderlayNode[0].innerHTML = markup;
        }

        /**
         * Updates the formula token array according to the current contents of
         * the text area control used for cell edit mode. The token array will
         * invoke the changle callback handler that will cause to redraw the
         * highlighted ranges in all grid panes.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  - {Boolean} [options.direct=false]
         *      If set to true, the token array and all other dependent
         *      settings will be updated immediately. By default, updating the
         *      token array will be debounced.
         *  - {Boolean} [options.suppressAutoComplete=false]
         *      If set to true, the pop-up menu for auto-completion of function
         *      names will be suppressed regardless of the current formula text
         *      and cursor position.
         */
        var updateTokenArray = (function () {

            // whether to suppress the auto-completion pop-up menu
            var suppressAutoComplete = false;

            // direct callback, invoked immediately when calling updateTokenArray()
            function directCallback(options) {

                // the current input string in the text area
                var value = textInputNode.val();

                // update the formula expression immediately
                var textOnlyMode = cellCollection.isTextFormat(editAddress);
                formulaString = (!textOnlyMode && /^[-+=]/.test(value)) ? value.replace(/^=/, '') : null;
                formulaOffset = (value === formulaString) ? 0 : 1;

                // update the 'suppress auto-completion' flag
                suppressAutoComplete = suppressAutoComplete || Utils.getBooleanOption(options, 'suppressAutoComplete', false);

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // the cell edit mode may have been finished already
                if (!self.isActive()) { return; }

                // parse the formula string
                if (formulaString !== null) {
                    tokenDescs = tokenArray.parseFormula('ui', formulaString, { refAddress: editAddress });
                } else {
                    tokenArray.clearTokens();
                    tokenDescs = [];
                }

                // insert highlighting mark-up for the underlay node
                updateTextAreaHighlighting();

                // update pop-up nodes and other settings dependent on the text selection
                updateTextAreaPopups(suppressAutoComplete);
                suppressAutoComplete = false;
            }

            // return the debounced method updateTokenArray()
            return self.createDebouncedMethod('CellTextEditor.updateTokenArray', directCallback, deferredCallback, { delay: 100, maxDelay: 300 });
        }());

        /**
         * Callback handler for the token array used while cell edit mode is
         * active. Updates the edited formula expression while the token array
         * will be manipulated externally, e.g. when dragging reference ranges
         * in the sheet area.
         */
        function changeTokenHandler(token, index) {

            // return global token array changes (reparsed formula etc.)
            if (!token) { return; }

            // override the text representation of the changed token
            tokenDescs[index].text = token.getText(formulaGrammar);

            // update the text positions of the following tokens
            Utils.iterateArray(tokenDescs, function (tokenDesc, i) {
                tokenDesc.start = (i === 0) ? 0 : tokenDescs[i - 1].end;
                tokenDesc.end = tokenDesc.start + tokenDesc.text.length;
            }, { begin: index });

            // calculate the new formula string
            var formulaString = tokenDescs.reduce(function (formula, tokenDesc) {
                return formula + tokenDesc.text;
            }, '');

            // set the new formula, add an undo action
            setTextAreaContents(textInputNode.val()[0] + formulaString, { start: tokenDescs[index].end + 1 });
            saveUndoState();

            // immediately update the highlighted cell references
            updateTextAreaHighlighting();
            resetActiveCellSelection();
        }

        /**
         * Changes the contents and selection of the text area, and updates its
         * position and size.
         *
         * @param {String} text
         *  The new contents of the text area.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Number} [options.start]
         *      The new start position of the selection. If omitted, the text
         *      cursor will be moved to the end of the new text.
         *  - {Number} [options.end=options.start]
         *      The new end position of the selection. If omitted, and a start
         *      position has been specified with the option 'start', a simple
         *      text cursor will be shown at the start position.
         *  - {Boolean} [options.suppressAutoComplete=false]
         *      If set to true, the pop-up menu for auto-completion of function
         *      names will be suppressed.
         */
        function setTextAreaContents(text, options) {

            var start = Utils.getIntegerOption(options, 'start', text.length, 0, text.length);
            var end = Utils.getIntegerOption(options, 'end', start, start, text.length);

            cellInputNode.val(text);
            paneInputNode.val(text);

            prevEditText = text;
            Forms.setInputSelection(textInputNode, start, end);
            updateTextAreaPosition({ direct: true });
            updateTokenArray(options);
        }

        /**
         * Replaces the token before the text cursor with the selected entry
         * from the auto-complete menu.
         */
        function applyAutoCompleteText(itemValue) {

            // always hide the pop-up menu
            textAreaListMenu.hide();

            // fail-safety check
            if (!funcAutoTokenDesc || !_.isObject(itemValue)) { return; }

            // get new text and new cursor position
            var newText = itemValue.name;
            var newStart = funcAutoTokenDesc.start + newText.length + formulaOffset;

            // insert parentheses after function name (unless the existing token is
            // a function token which means it already has an opening parenthesis)
            if (itemValue.type === 'func') {
                if (funcAutoTokenDesc.token.isType('name')) {
                    newText += '()';
                    // ignore the next closing parenthesis entered manually
                    funcAutoCloseData.push(true);
                }
                // move cursor behind the opening parenthesis
                newStart += 1;
            }

            // insert the new text into the formula string, prevent showing the
            // auto-completion pop-up list immediately again (important for defined names)
            newText = Utils.replaceSubString(textInputNode.val(), funcAutoTokenDesc.start + formulaOffset, funcAutoTokenDesc.end + formulaOffset, newText);
            setTextAreaContents(newText, { start: newStart, suppressAutoComplete: true });
            saveUndoState();
        }

        /**
         * Changes the quick edit mode while cell edit mode is active, and
         * shows a status label at the bottom of the application pane.
         *
         * @param {Boolean} newQuickEdit
         *  The new value of the quick edit mode.
         */
        function setQuickEditMode(newQuickEditMode) {

            // change the quick edit state
            if (quickEditMode === newQuickEditMode) { return; }
            quickEditMode = newQuickEditMode;

            // update the status label
            docView.setStatusLabel(quickEditMode ?
                //#. Special edit mode in spreadsheet document: quickly type text into cell, use cursor keys to move to next cell
                gt('Insert mode') :
                //#. Standard edit mode in spreadsheet document: edit existing cell text, cursor keys move text cursor inside cell
                gt('Edit mode')
            );
        }

        /**
         * Saves the current state of the cell edit mode on top of the undo
         * stack. If the current undo action is not on top of the stack, all
         * following actions will be deleted first.
         */
        function saveUndoState(fixed) {
            undoManager.saveState(textInputNode.val(), Forms.getInputSelection(textInputNode), editAttributeSet, fixed);
            self.implTriggerEvents('change');
        }

        /**
         * Restores the current state in the undo stack.
         */
        function restoreUndoState(action) {
            editAttributeSet = _.copy(action.attrs, true);
            updateTextAreaStyle();
            setTextAreaContents(action.text, action.sel);
            self.implTriggerEvents('change');
        }

        /**
         * Removes the activated ranges from the view that have been inserted
         * into the formula while cell edit mode is active.
         */
        function resetActiveCellSelection() {
            docView.getSheetModel().setViewAttribute('activeSelection', null);
        }

        /**
         * Handles 'keydown' events in the text input control, while cell edit
         * mode is active.
         */
        function textAreaKeyDownHandler(event) {

            // the list item currently selected in the auto-completion pop-up menu
            var itemNode = textAreaListMenu.getSelectedItemNodes().first();
            // the value of the item node
            var itemValue = Forms.getButtonValue(itemNode);

            // short-cuts in visible pop-up menu for auto-completion
            if (textAreaListMenu.isVisible()) {

                // close pop-up menu on ESCAPE key
                if (event.keyCode === KeyCodes.ESCAPE) {
                    textAreaListMenu.hide();
                    return false;
                }

                // insert selected list item into the formula
                if (KeyCodes.matchKeyCode(event, 'TAB') || KeyCodes.matchKeyCode(event, 'ENTER')) {

                    // Bug 31999: For performance reasons, the token array and the auto-completion pop-up
                    // menu will be updated debounced. To prevent that ENTER/TAB keys insert a function
                    // name unintentionally, the pop-up menu must be updated before applying a list item.
                    updateTokenArray({ direct: true });

                    // list menu may have been hidden by the updateTextAreaPopups() call,
                    // do not apply any list item in this case, but leave cell edit mode
                    if (textAreaListMenu.isVisible()) {
                        applyAutoCompleteText(itemValue);
                        return false;
                    }
                }

                // otherwise, try to move the selected list item
                itemNode = textAreaListMenu.getItemNodeForKeyEvent(event, itemNode);
                if (itemNode.length > 0) {
                    textAreaListMenu.selectItemNode(itemNode, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return false;
                }
            }

            // quit cell edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                // signature-Tooltip is visible
                if (signatureToolTip.isVisible()) {
                    signatureToolTip.hide();
                } else {
                    self.cancelEditMode();
                }
                return false;
            }

            // toggle quick edit mode
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                setQuickEditMode(!quickEditMode);
                // forget last cell range selected in the sheet
                resetActiveCellSelection();
                return false;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to up/down
            var moveDir = PaneUtils.getCellMoveDirection(event);
            var prevMode = formulaBarMode;
            if (moveDir) {
                self.leaveEditMode().done(function () {
                    docView.moveActiveCell(moveDir);
                    // keep edit mode active on small devices
                    if (Utils.SMALL_DEVICE) {
                        keyboardOpen = true;
                        if (prevMode) {
                            paneInputNode.focus();
                        } else {
                            self.enterEditMode();
                        }
                    }
                });
                return false;
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert matrix formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                self.leaveEditMode({ commitMode: event.shiftKey ? 'matrix' : 'fill' });
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Forms.replaceTextInInputSelection(textInputNode, '\n');
                textAreaInputHandler();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // any cursor navigation key
            if (event.keyCode in NAVIGATION_KEYCODES) {
                if (quickEditMode) {
                    // quick edit mode: do not let the text area process the key events
                    event.preventDefault();
                } else {
                    // standard edit mode: do not process the key events (no change of cell selection)
                    event.stopPropagation();
                    // update pop-up nodes and other settings dependent on the new text selection
                    updateTextAreaAfterSelection();
                    // forget last cell range selected in the sheet
                    resetActiveCellSelection();
                }
                // let key event bubble up to move the cell cursor, or text cursor in the text area
                return;
            }

            // all other events will be processed by the text field
        }

        /**
         * Handles 'keypress' events in the text input control, while cell edit
         * mode is active.
         */
        function textAreaKeyPressHandler(event) {

            // the entered Unicode character
            switch (String.fromCharCode(event.charCode)) {

                // Register an opening parenthesis entered manually. The related closing parenthesis
                // will be inserted even if it occurs among other closing parentheses inserted via
                // function auto-completion (see below).
                case '(':
                    funcAutoCloseData.push(false);
                    break;

                // When a function name has been inserted from the auto-completion pop-up menu, skip
                // a closing parenthesis because it has been inserted already by the auto-completion).
                case ')':
                    if (funcAutoCloseData.pop() === true) {
                        // the text cursor needs to be moved one character forward
                        Forms.setInputSelection(textInputNode, Forms.getInputSelection(textInputNode).start + 1);
                        updateTextAreaPopups();
                        return false;
                    }
                    break;
            }
        }

        /**
         * Handles any changes in the text area.
         */
        function textAreaInputHandler() {

            // the current value of the text area
            var currEditText = textInputNode.val();
            // the text selection in the text area control
            var textSelection = Forms.getInputSelection(textInputNode);
            // whether the selection in the text area is a simple cursor
            var isCursor = textSelection.start === textSelection.end;

            // do nothing if the edit text did not change at all
            if (prevEditText === currEditText) { return; }

            // try to show drop-down list with function names and defined names
            funcAutoComplete = isCursor;

            // update text area settings (position, formula range highlighting)
            updateTextAreaPosition();
            updateTokenArray();
            saveUndoState();

            // deferred processing after the event has been processed by the browser
            self.executeDelayed(function () {

                // do nothing if edit mode has been left before this callback will be triggered
                if (!self.isActive()) { return; }

                // synchronize text areas in grid pane and formula pane
                textOutputNode.val(textInputNode.val());

                // Mobile devices may add auto-correction text which modifies the selection after a text change event has
                // been processed. Recalculate the funcAutoComplete flag after the browser has set the final selection.
                funcAutoComplete = Forms.isCursorSelection(textInputNode);

            }, 'CellTextEditor.textAreaInputHandler');

            // forget last active cell range selected in the sheet (formulas with cell references)
            resetActiveCellSelection();

            // show type-ahead suggestion in the text area (not in formula edit mode)
            if (_.isString(prevEditText) && (prevEditText.length < currEditText.length) && isCursor && (textSelection.end === currEditText.length) && (formulaString === null)) {

                // first cycle: collect the contents of all text cells and formula cells resulting in text above and below the edited cell
                if (!typeAheadEntries) {
                    typeAheadEntries = [];

                    // collect the values of all text cells, and the string results of formula cells
                    var boundRange = Range.create(editAddress[0], Math.max(0, editAddress[1] - 256), editAddress[0], Math.min(docModel.getMaxRow(), editAddress[1] + 256));
                    var cellIt = cellCollection.createLinearAddressIterator(editAddress, 'up down', { type: 'value', boundRange: boundRange, skipStart: true });
                    Iterator.forEach(cellIt, function (address) {
                        var value = cellCollection.getValue(address);
                        if ((typeof value === 'string') && (value.length > 0)) {
                            typeAheadEntries.push({ text: value, dist: Math.abs(editAddress[1] - address[1]) });
                        }
                    });

                    // sort the entries in the array by distance to the edited cell
                    typeAheadEntries.sort(function (entry1, entry2) { return entry1.dist - entry2.dist; });

                    // reduce the entries to a simple string array, that is still sorted by distance
                    typeAheadEntries = _.pluck(typeAheadEntries, 'text');
                }

                // find the first matching type-ahead entry (the array is sorted by distance to edited cell, thus it will find the nearest matching entry)
                var suggestion = _.find(typeAheadEntries, function (entry) {
                    return entry.substr(0, currEditText.length).toUpperCase() === currEditText.toUpperCase();
                });

                // insert the remaining text of a matching entry, keep character case of the typed text
                if (suggestion) {
                    suggestion = currEditText + suggestion.substr(currEditText.length);
                    setTextAreaContents(suggestion, { start: currEditText.length, end: suggestion.length });
                    saveUndoState();
                }
            }

            // store current text for next iteration
            prevEditText = currEditText;
        }

        /**
         * Handles mousedown or touchstart events in the text area.
         */
        function textAreaMouseTouchHandler() {
            // leave quick-edit mode (cursor keys will move the text cursor)
            setQuickEditMode(false);
            // update pop-up nodes and other settings dependent on the new text selection
            updateTextAreaAfterSelection();
            // forget last cell range selected in the sheet
            resetActiveCellSelection();
        }

        /**
         * Hides the pop-up nodes if the focus leaves the text area control,
         * unless the auto-completion menu is being clicked.
         */
        var textAreaFocusHandler = (function () {

            // whether to ignore focus event in the text area (no update of pop-up nodes)
            var ignoreFocusEvent = false;

            return function (event) {
                switch (event.type) {

                    // text area focused: rebuild all active pop-up nodes
                    case 'focus':
                        cellInputContainer.addClass(Forms.FOCUSED_CLASS);
                        if (!ignoreFocusEvent) {
                            updateTextAreaPopups();
                        }

                        // switch input/output text area variables
                        if (event.target !== textInputNode[0]) {
                            var tmpNode = textInputNode;
                            textInputNode = textOutputNode;
                            textOutputNode = tmpNode;
                        }

                        break;

                    // text area lost focus: hide pop-up nodes (unless the auto-complete menu
                    // has been clicked, in this case, move focus back to text area immediately)
                    case 'blur':

                        // switch between grid pane and formula pane
                        if (event.relatedTarget === textOutputNode[0]) {
                            self.enterEditMode({ formulaBarMode: !formulaBarMode, text: textInputNode.val(), restart: true });
                            break;
                        }

                        self.executeDelayed(function () {
                            if (self.isActive() && textAreaListMenu.hasFocus()) {
                                ignoreFocusEvent = true;
                                Utils.setFocus(textInputNode);
                                // IE sends focus event deferred
                                self.executeDelayed(function () { ignoreFocusEvent = false; }, 'CellTextEditor.textAreaFocusHandler');
                            } else {
                                cellInputContainer.removeClass(Forms.FOCUSED_CLASS);
                                if (!stickyAutoComplete) { hideTextAreaPopups(); }
                            }
                        }, 'CellTextEditor.textAreaFocusHandler');
                        break;
                }
            };
        }());

        /**
         * Returns whether the current text selection in the text area control
         * is valid to start selecting new ranges for the current formula. The
         * text area must contain a formula string (starting with an equality
         * sign), and the start position of the text selection must be located
         * behind the leading equality sign, an operator, a list separator, or
         * an opening parenthesis).
         */
        function isValidRangeSelectionPosition() {

            // the cursor position in the token array (exclude leading equality sign)
            var cursorPos = textInputNode[0].selectionStart;
            var textPos = cursorPos - formulaOffset;

            // invalid if not in cell edit mode with a formula
            if (!self.isActive() || (formulaString === null)) { return false; }

            // existing view attribute, or cursor at the beginning of the formula is always valid
            if ((textPos === 0) || sheetModel.getViewAttribute('activeSelection')) { return true; }

            // check if cursor is located after a space character that may be interpreted as intersection operator
            if (formulaGrammar.isSpaceIntersection() && (textInputNode.val()[cursorPos - 1] === ' ')) { return true; }

            // preceding token may be an operator, separator, or opening parenthesis (bug 48475: ignore white-space before cursor)
            var tokenDesc = findTokenDescriptor(textPos, { ignoreWs: true });
            return !!tokenDesc && (tokenDesc.end <= textPos) && tokenDesc.token.matchesType(/^(op|sep|open)$/);
        }

        /**
         * When in cell edit mode with a formula, returns the active selection
         * which will be used for keyboard navigation and mouse/touch tracking.
         */
        function getCellSelectionHandler() {

            // the current value of the 'activeSelection' view attribute
            var activeSelection = sheetModel.getViewAttribute('activeSelection');

            // always return existing view attribute
            if (activeSelection) { return activeSelection; }

            // check the text selection in the text area control
            if (!isValidRangeSelectionPosition()) { return null; }

            // start with an empty active selection
            return new SheetSelection();
        }

        /**
         * When in cell edit mode with a formula, selects a single range and
         * inserts the range address into the formula string.
         */
        function setCellSelectionHandler(selection) {

            // do nothing if ranges cannot be selected with the current text selection
            if (!isValidRangeSelectionPosition()) { return false; }

            // store current text selection, if no range are activated yet
            // (selected text will be replaced with initial range addresses)
            if (!sheetModel.getViewAttribute('activeSelection')) {
                rangeSelection = Forms.getInputSelection(textInputNode);
            }

            // build the new formula term to be inserted into the text
            var tmpTokenArray = sheetModel.createCellTokenArray();
            tmpTokenArray.appendRangeList(selection.ranges);
            var references = tmpTokenArray.getFormula('ui');
            var formulaText = Utils.replaceSubString(textInputNode.val(), rangeSelection.start, rangeSelection.end, references);

            // insert the new formula text, set cursor behind the range (no real selection)
            rangeSelection.end = rangeSelection.start + references.length;
            setTextAreaContents(formulaText, { start: rangeSelection.end });
            saveUndoState();

            // update the view attribute
            sheetModel.setViewAttribute('activeSelection', selection);

            // do not perform standard sheet selection
            return true;
        }

        function androidKeyboardHandler() {
            if (!keyboardOpen) {
                keyboardOpen = true;
            } else {
                docView.leaveTextEditMode('cell');
                keyboardOpen = false;
            }
        }

        function iosKeyboardHandler() {
            if (!keyboardOpen) {
                keyboardOpen = true;
            } else if (window.getSelection().anchorNode === null) {
                docView.leaveTextEditMode('cell');
                keyboardOpen = false;
            }
        }

        /**
         * Callback handler for starting the cell edit mode.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.text]
         *      The initial text to insert into the edit area. If omitted, the
         *      current value of the active cell will be inserted.
         *  - {Number} [options.pos]
         *      The initial position of the text cursor. If omitted, the text
         *      cursor will be placed after the entire text. If negative, the
         *      text cursor will be placed from the end of the text.
         *  - {Boolean} [options.quick=false]
         *      If set to true, the quick edit mode will be initiated. All
         *      cursor navigation keys will not move the text cursor in the
         *      text area control but will behave as in the normal sheet
         *      selection mode.
         *  - {Boolean} [options.autoComplete=false]
         *      If set to true, the text passed with the option 'text' will be
         *      used to show an auto-completion suggestion. By default, text
         *      edit mode will start without auto-completion.
         *  - {Boolean} [options.stickyList=false]
         *      If set to true, the auto-completion drop-down list will be kept
         *      open when the text edit control is not focused.
         */
        function enterHandler(gridPane, options) {

            // the active cell must not be locked
            var promise = docView.ensureUnlockedActiveCell();

            // finally, start the cell edit mode
            promise.done(function () {

                // store location of edited cell (sheet may change during range selection mode for formulas)
                editAddress = docView.getActiveCell();
                sheetModel = docView.getSheetModel();
                cellCollection = sheetModel.getCellCollection();
                editAttributeSet = {};

                // initialze the correct DOM node for pop-up menus and DOM events (force within the formula pane on small devices)
                formulaBarMode = Utils.getBooleanOption(options, 'formulaBarMode', false);
                textInputNode = formulaBarMode ? paneInputNode : cellInputNode;
                textOutputNode = formulaBarMode ? cellInputNode : paneInputNode;

                // initialize CSS formatting of the text area (this initializes most class members)
                updateTextAreaStyle();

                // scroll to the active cell (before checking edit mode); on touch devices,
                // scroll the cell to the top border of the grid pane (due to virtual keyboard)
                // don't scroll on small devices because there we put the textarea on top of the screen anyway
                gridPane.scrollToCell(editAddress, { forceTop: Utils.TOUCHDEVICE });

                // Bug 34366: Touch devices scroll the window out of visible area, so that neither the bars
                // (tool and tab) are visible, nor the editable cell. Root cause is the soft-keyboard.
                // However, scroll back to 0,0 and everything is fine. (Delay of 1000 necessary for iOS)
                if (Utils.TOUCHDEVICE) {
                    self.executeDelayed(function () {
                        window.scrollTo(0, 0);
                    }, 'CellTextEditor.enterEditMode', 1000);

                    if (Utils.CHROME_ON_ANDROID) {
                        self.listenTo($(window), 'resize', androidKeyboardHandler);
                    } else if (Utils.IOS) {
                        self.listenTo($(document), 'selectionchange', iosKeyboardHandler);
                    }
                }

                // initial state of quick edit mode
                setQuickEditMode(Utils.getBooleanOption(options, 'quick', false));

                // process events of the text area element
                var eventMap = {
                    keydown: textAreaKeyDownHandler,
                    keypress: textAreaKeyPressHandler,
                    input: textAreaInputHandler,
                    'mousedown mouseup touchstart touchend touchcancel': textAreaMouseTouchHandler,
                    'focus blur': textAreaFocusHandler
                };
                self.listenTo(textInputNode, eventMap);
                self.listenTo(textOutputNode, eventMap);

                // type-ahead suggestions will be collected on first access
                typeAheadEntries = null;

                // the token array stores the parsed formula for range highlighting
                tokenDescs = [];
                tokenArray = sheetModel.createCellTokenArray();
                tokenArray.registerChangeHandler(changeTokenHandler);

                // highlight the ranges in the active sheet contained in the formula expression
                highlightUid = docView.startRangeHighlighting(tokenArray, {
                    priority: true,
                    draggable: true,
                    refAddress: editAddress,
                    targetAddress: editAddress,
                    resolveNames: true
                });

                // insert the text area control into the DOM of the active grid pane, and activate it
                cellInputContainer.show();

                // DOCS-735: touch devices: move cell text input node to top of grid pane
                if (Utils.TOUCHDEVICE) {
                    cellInputContainer.addClass('ontop');
                    app.getWindowNode().append(cellInputContainer);
                } else {
                    gridPane.getNode().append(cellInputContainer);
                }

                // update position of the text area when scrolling the grid pane
                self.listenTo(gridPane, 'change:scrollpos', function () {
                    updateTextAreaPosition({ direct: true, size: false });
                });

                // update zoom settings during edit mode
                self.listenTo(docView, 'change:sheet:viewattributes', function (event, attributes) {
                    if ('zoom' in attributes) {
                        updateTextAreaStyle();
                        updateTextAreaPosition({ direct: true });
                    }
                });

                // get the formatted edit string
                originalText = cellCollection.getEditString(editAddress);
                // the initial text for the text area
                var initialText = Utils.getStringOption(options, 'text', null);
                // set efefctive text contents to be shown in the text area
                var editText = (initialText === null) ? originalText : initialText;

                // the initial cursor position for the text area (negative: position from end of text)
                var cursorPos = Utils.getIntegerOption(options, 'pos', editText.length, -editText.length, editText.length);
                if (cursorPos < 0) { cursorPos = Math.max(0, editText.length + cursorPos); }
                var initSelection = { start: cursorPos, end: cursorPos };

                // set initial contents, position, size, and selection of the text area
                setTextAreaContents(editText, initSelection);

                // selection handler that changes cell ranges in formulas in cell edit mode
                docView.registerCellSelectionHandlers(getCellSelectionHandler, setCellSelectionHandler);

                // register the undo manager at application controller to redirect the GUI commands
                app.getController().registerUndoManager(undoManager);
                // apply the undo actions when undoing/redoing during cell edit mode
                self.listenTo(undoManager, 'restore:state', function (event, action) { restoreUndoState(action); });

                // if an explicit initial text has been passed, create an initial undo state with the original text
                if (initialText !== null) {
                    undoManager.saveState(originalText, { start: 0, end: originalText.length }, {}, true);
                }

                // invoke input handler to handle the initial text (e.g. pop-ups, additional undo state for custom text)
                prevEditText = (Utils.getBooleanOption(options, 'autoComplete', false) && (editText.length > 0)) ? '' : null;
                textAreaInputHandler();

                // special mode to keep auto-completion list menu open without focus in text area
                stickyAutoComplete = Utils.getBooleanOption(options, 'stickyList', false);
                textAreaListMenu.setAutoCloseMode(!stickyAutoComplete);

                // synchronize scroll position of text area and underlay node
                self.listenTo(cellInputNode, 'scroll', function () {
                    cellUnderlayNode.scrollTop(cellInputNode.scrollTop());
                });

                // touch devices: trigger scroll event on textarea to force the correct position of the underlay node
                if (Utils.TOUCHDEVICE) {
                    self.executeDelayed(function () {
                        cellInputNode.trigger('scroll');
                    }, 'CellTextEditor.enterHandler', 200);
                }
            });

            // return the promise to indicate success or failure
            return promise;
        }

        /**
         * Callback handler for leaving the cell edit mode.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.commitMode='cell']
         *      Specifies how to leave the cell edit mode. The following values
         *      are supported:
         *      - 'cell': The current value will be committed to the active
         *          cell of the selection.
         *      - 'fill': The current value will be filled to all cells of the
         *          selection.
         *      - 'matrix': The current value will be inserted as matrix
         *          formula into the selected cell range.
         *      - 'force': Similar to 'cell', but does not remain in cell edit
         *          mode if the formula expression is invalid. Instead, the
         *          edit text will be ignored, and nothing will be changed in
         *          the edit cell. The promise returned by this method will
         *          always be resolved, and never rejected.
         */
        function leaveHandler(options) {

            // shows an alert notification, and keeps the cell edit mode activated
            function keepEditMode(msgCode) {
                docView.yellMessage(msgCode);
                setQuickEditMode(false);
                return self.createRejectedPromise();
            }

            // default commit mode is 'cell' (change the active cell only)
            var commitMode = Utils.getStringOption(options, 'commitMode', 'cell');

            // bounding range of an existing matrix formula covering the active cell
            var matrixRange = cellCollection.getMatrixRange(editAddress);
            // whether to update an existing matrix (prevent auto-expansion of 1x1 matrixes)
            var updateMatrix = matrixRange !== null;
            // fill the active range of the selection with a new matrix formula (ignore multi-selection)
            if (!matrixRange) { matrixRange = docView.getActiveRange(); }

            // the new cell text
            var editText = textInputNode.val();
            // parse the entered text: convert to other data types, or interpret the formula
            var contents = cellCollection.parseCellValue(editText, (commitMode === 'matrix') ? 'mat' : 'val', editAddress, matrixRange);
            // whether to create a formula cell
            var createFormula = contents.f && contents.result;
            // whether the formula expression is invalid (do not create document operations)
            var syntaxError = createFormula && (contents.result.type === 'error');

            // keep edit mode active, if an error has been found in the formula structure
            // (unless leaving the edit mode has been forced by caller)
            if (syntaxError && (commitMode !== 'force')) {
                return keepEditMode('formula:invalid');
            }

            // if matrix edit mode has been left without a formula expression, behave like normal fill mode
            if (!createFormula && (commitMode === 'matrix')) { commitMode = 'fill'; }

            // special checks to keep the edit mode active, for different commit modes
            switch (commitMode) {

                // fail, if single-cell edit wants to change a matrix formula
                case 'cell':
                    if (updateMatrix && !matrixRange.single()) {
                        return keepEditMode('formula:matrix:change');
                    }
                    break;

                // fail, if fill mode wants to change parts of a matrix formula
                case 'fill':
                    if (cellCollection.coversAnyMatrixRange(docView.getSelectedRanges(), 'partial')) {
                        return keepEditMode('formula:matrix:change');
                    }
                    break;

                // fail, if another matrix formula overlapping with the active cell would be changed
                case 'matrix':
                    if (!updateMatrix && cellCollection.coversAnyMatrixRange(matrixRange, 'partial')) {
                        return keepEditMode('formula:matrix:change');
                    }
                    break;
            }

            // deinitialize cell edit mode already here to not interfere with following operation generators
            self.cancelEditMode();

            // nothing more to do with formula syntax error or when editing a matrix, if leaving edit mode is forced
            if ((commitMode === 'force') && (syntaxError || updateMatrix)) {
                return self.createResolvedPromise();
            }

            // add auto-wrapping attribute, if the text contains an explicit line break (except for formulas)
            if (!contents.f && (editText.indexOf('\n') >= 0)) {
                AttributeUtils.insertAttribute(editAttributeSet, 'cell', 'wrapText', true);
            }

            // extract the hyperlink URL (will be generated separately to get an own undo action)
            var url = contents.url;
            delete contents.url;

            // performance: do not pass empty attributes object (prevent useless object processing)
            if (!_.isEmpty(editAttributeSet)) {
                contents.a = contents.a ? attributePool.extendAttributeSet(contents.a, editAttributeSet) : editAttributeSet;
            }

            // send current value to document model after returning to selection mode
            var promise = null;
            switch (commitMode) {
                case 'fill':
                    promise = docView.fillRangeContents(contents);
                    break;
                case 'matrix':
                    promise = docView.setMatrixContents(contents);
                    break;
                default: // 'cell' and 'force'
                    // single cell mode: do not send operations if neither text nor formatting changes
                    // (especially: do not replace shared formula cell with simple formula cell)
                    // (but allow to change a single-cell matrix formula to a simple cell formula)
                    if ((originalText !== editText) || contents.a || updateMatrix) {
                        commitMode = 'cell';
                        promise = docView.setCellContents(contents);
                    }
            }

            // create the hyperlink in a separate step, in order to get an own undo action
            if (url && promise) {
                promise = promise.then(function () {
                    return docView.insertHyperlink(url, { activeCell: commitMode === 'cell', createStyles: true });
                });
            }

            // nothing to do (cell not changed): hide previously shown alert box (bug 47936)
            if (!promise) { docView.hideYell(); }

            return promise || true;
        }

        /**
         * Cleanup when canceling the cell edit mode.
         */
        function cancelHandler() {

            // unregister and destroy the undo manager
            app.getController().restoreUndoManager();
            undoManager.clearStates();

            // deinitialize range highlighting
            docView.endRangeHighlighting(highlightUid);
            docView.unregisterCellSelectionHandlers();
            resetActiveCellSelection();
            sheetModel = cellCollection = null;
            tokenArray = tokenDescs = highlightUid = null;

            // back to cell selection mode
            quickEditMode = formulaString = typeAheadEntries = null;
            hideTextAreaPopups();

            // hide the highlighting mark-up in the underlay nodes
            cellUnderlayNode.empty();
            paneUnderlayNode.empty();

            // hide the text area control (bug 40321: DO NOT detach from DOM, Egde may freeze)
            cellInputContainer.hide();

            // stop listening to view events
            self.stopListeningTo(docView);

            if (Utils.TOUCHDEVICE && Utils.CHROME_ON_ANDROID) {
                $(window).off('resize', androidKeyboardHandler);
            } else if (Utils.TOUCHDEVICE && Utils.IOS) {
                $(document).off('selectionchange', iosKeyboardHandler);
            }

            // deinitialze the DOM anchor node for pop-up menus and DOM events
            self.stopListeningTo(textInputNode);
            self.stopListeningTo(textOutputNode);
            formulaBarMode = false;
            keyboardOpen = false;
            textInputNode = textOutputNode = null;
        }

        /**
         * Returns the merged formatting attributes of the edited cell, merged
         * with the pending formatting attributes applied during cell edit
         * mode.
         *
         * @returns {Object}
         *  The merged attribute set of the edited cell.
         */
        function formatResolver() {
            // merge the cell attributes with the current edit attributes
            return attributePool.extendAttributeSet(cellAttributeSet, editAttributeSet, { clone: true });
        }

        /**
         * Applies the passed formatting attributes to the text area control.
         *
         * @param {Object} attributeSet
         *  The attributes to be applied to the text area control.
         */
        function formatHandler(attributeSet) {

            // store attributes, but do not generate any document operations
            attributePool.extendAttributeSet(editAttributeSet, attributeSet);
            // update the formatting of the text area
            updateTextAreaStyle();
            updateTextAreaPosition({ direct: true });
            // create a fixed undo action that cannot be updated with text typed afterwards
            saveUndoState(true);
            // nothing to wait for
            return true;
        }

        // initialization -----------------------------------------------------

        // get localized names of all visible functions, as sorted array of objects
        // with 'key' and 'name' properties (used for the auto-completion list menu)
        functionEntries = [];
        formulaResource.getFunctionCollection().forEach(function (descriptor, key) {
            if (!descriptor.hidden) {
                functionEntries.push({ key: key, name: descriptor.localName });
            }
        });
        functionEntries = _.sortBy(functionEntries, 'name');

        // additional initialization after import
        docView.waitForImport(function () {
            var formulaPane = docView.getFormulaPane();
            paneInputNode = formulaPane.getTextInputNode();
            paneUnderlayNode = formulaPane.getTextUnderlayNode();
            // bug 53420: user cannot choose function from the list in formula pane
            textAreaListMenu.registerFocusableNodes(paneInputNode);
        });

        // handle click events in the auto-completion pop-up menu
        textAreaListMenu.getNode().on('click', Forms.BUTTON_SELECTOR, function (event) {
            var itemValue = Forms.getButtonValue(event.currentTarget);
            applyAutoCompleteText(itemValue);
        });

        // hide function description tooltip when closing the auto-completion menu
        this.listenTo(textAreaListMenu, 'popup:hide', function () {
            textAreaListItemToolTip.hide();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            cellInputContainer.remove();
            signatureToolTip.destroy();
            textAreaListMenu.destroy();
            textAreaListItemToolTip.destroy();
            undoManager.destroy();

            self = app = docModel = attributePool = undoManager = null;
            formulaResource = formulaGrammar = formulaParser = null;
            cellInputContainer = cellInputNode = cellUnderlayNode = null;
            paneInputNode = paneUnderlayNode = null;
            textInputNode = textOutputNode = null;
            signatureToolTip = textAreaListMenu = textAreaListItemToolTip = null;
        });

    } }); // class CellTextEditor

    // exports ================================================================

    return CellTextEditor;

});
