/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/customsortdialog', [
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet/main',
    'less!io.ox/office/spreadsheet/view/dialog/customsortdialog'
], function (Forms, KeyCodes, BaseDialog, Iterator, SheetUtils, Labels, Controls, gt) {

    'use strict';

    // constants ==============================================================

    // convenience shortcuts
    var CheckBox = Controls.CheckBox;
    var SortDirection = SheetUtils.SortDirection;

    var UNDEF_LABEL = /*#. sorting: the sorting order is not yet defined */ gt.pgettext('sort', 'undefined');
    var ERR_MISSING_LABEL = gt.pgettext('sort', 'All sort criteria must have a column or row specified.');
    var ERR_MULTI_LABEL = gt.pgettext('sort', 'Some columns or rows are being sorted more than once.');

    // class CustomSortDialog =================================================

    /**
     * A modal dialog with options for sorting.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    var CustomSortDialog = BaseDialog.extend(function (docView, sortRange, tableModel, hasHeader) {

        // base constructor
        BaseDialog.call(this, docView, {
            title: /*#. title for sort dialog box */ gt.pgettext('sort', 'Custom sort'),
            width: 400,
            okLabel: gt('Sort')
        });

        // private properties
        this._docView = docView;
        this._sortRange = sortRange;
        this._tableModel = tableModel;
        this._hasHeader = hasHeader;
        this._sortDir = SortDirection.VERTICAL;
        this._headerDescs = [];
        this._ruleDescs = [];

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // add base menu items
        this.getPopup().addClass('io-ox-office-custom-sort-dialog');

        // dialog elements for regular cell ranges only (not for table ranges)
        if (!tableModel) {

            // header label for the options section
            var optionsLabel = $('<span>').text(/*#. sorting options (has headers, vertical or horizontal) */ gt.pgettext('sort', 'Options'));

            // create and initialize the "has header" check box
            var headerCheckBox = new CheckBox(docView, {
                //#. sorting: selected range has headers which shouldn't be sorted
                label: gt.pgettext('sort', 'Selection has headers'),
                boxed: true
            });
            headerCheckBox.setValue(hasHeader);
            headerCheckBox.on('group:change', function (event, newValue) {
                this._hasHeader = newValue;
                this._initializeSortRules(this._getSortRules());
            }.bind(this));

            // explicitly destroy the check box instance
            this.registerDestructor(function () {
                headerCheckBox.destroy();
            });

            // create and initialize the "sorting direction" drop-down menu
            var directionGroup = $('<div class="dropdown-link">').append(
                $('<label>').append(
                    //#. headline for sorting direction ("top to bottom" or "left to right")
                    $('<span>').text(gt.pgettext('sort', 'Direction')),
                    $('<span>').text(_.noI18n(':'))
                ),
                this._createDropDownMenu([
                    { value: SortDirection.VERTICAL, label: /*#. sorting direction for spreadsheet data */ gt.pgettext('sort', 'Top to bottom') },
                    { value: SortDirection.HORIZONTAL, label: /*#. sorting direction for spreadsheet data */ gt.pgettext('sort', 'Left to right') }
                ], SortDirection.VERTICAL, function (sortDir) {
                    this._sortDir = sortDir;
                    this._initializeSortRules();
                })
            );

            // insert the elements into the dialog body
            this.getBody().append(optionsLabel, headerCheckBox.getNode(), directionGroup, '<hr>');
        }

        // initialize the root element for the sorting rules
        this._rulesRootNode = $('<div class="sort-rule-list">').append(
            $('<div class="sort-rule">').append(
                //#. sorting: sort range by (for example "column B" or "row 4")
                $('<span class="column-sort-by">').text(gt.pgettext('sort', 'Sort by')),
                //#. sorting: the sort order (for example "ascending")
                $('<span class="column-sort-order">').text(gt.pgettext('sort', 'Order'))
            )
        );

        // prepare the "plus"-button to be able to add more sort-rules
        this._addRuleBtn = $('<a href="#" tabindex="0">');
        this._addRuleBtn.text(/*#. add new sorting rule to list */ gt.pgettext('sort', 'Add sort criteria'));
        this._addRuleBtn.on('click keydown', function (event) {
            if ((event.type !== 'keydown') || (event.keyCode === KeyCodes.SPACE)) {
                if (this._canAddSortRule()) {
                    this._appendSortRule();
                    this._updateControls();
                }
                return false;
            }
        }.bind(this));

        // insert the sorting rule nodes into the dialog body
        this.getBody().append(this._rulesRootNode, '<hr>', $('<div class="add-rule-button">').append(this._addRuleBtn));

        // create the initial sorting rules (from table model, or default rule)
        this._initializeSortRules();

        // action handler for the OK button
        this.setOkHandler(function () {
            return { sortDir: this._sortDir, hasHeader: this._hasHeader, sortRules: this._getSortRules() };
        }, { keepOpen: 'fail' });
    });

    // private methods --------------------------------------------------------

    CustomSortDialog.prototype._getSortRules = function () {
        return _.pluck(this._ruleDescs, 'rule');
    };

    /**
     * Returns whether a new sorting rule can be inserted into the list.
     */
    CustomSortDialog.prototype._canAddSortRule = function () {
        return this._ruleDescs.length < this._headerDescs.length;
    };

    /**
     * Returns whether existing sorting rules can be deleted from the list. For
     * a table range, all sorting rules can be deleted. In simple cell ranges,
     * at least one sorting rule must remain in the list.
     */
    CustomSortDialog.prototype._canDeleteSortRule = function () {
        return (this._tableModel !== null) || (this._ruleDescs.length > 1);
    };

    /**
     * generates a dropdown-menu far away from the TK-Forms (bad way)
     *
     * @param {Array<Object>} itemDescs
     *  Descriptor objects for all list items in the drop-down menu. Each array
     *  element is an object with the properties "value", "label", "tooltip"
     *  (optional), and "italic" (optional).
     *
     * @param {Number} activeValue
     *  The value of the active list item (shown in the menu anchor).
     *
     * @param {Function} clickHandler
     *  A callback handler invoked when a drop-down list item has been clicked.
     *
     * @returns {jQuery}
     *  The generated dropdown-html.
     */
    CustomSortDialog.prototype._createDropDownMenu = function (itemDescs, activeValue, clickHandler) {

        // sets label text and tooltip to the passed DOM element
        function initLabel(labelNode, itemDesc) {
            return labelNode.text(itemDesc.label).attr({ title: itemDesc.tooltip || '' }).css('font-style', itemDesc.italic ? 'italic' : '');
        }

        // create the DOM elements for the menu
        var activeDesc = _.findWhere(itemDescs, { value: activeValue }) || { value: activeValue, label: UNDEF_LABEL, italic: true };
        var anchorNode = initLabel($('<a href="#" tabindex="0" aria-haspopup="true" data-toggle="dropdown">'), activeDesc);
        var listNode = $('<ul class="dropdown-menu" role="menu">');
        var menuNode = $('<span class="dropdown">').append(anchorNode, listNode);

        // bind click handler to the dialog instance
        clickHandler = clickHandler.bind(this);

        // create list item elements
        itemDescs.forEach(function (itemDesc) {
            var itemLink = initLabel($('<a class="dropdown-listentry" href="#" role="menuitem">'), itemDesc);
            var listItem = $('<li>').append(itemLink).on('click', function (event) {
                event.preventDefault();
                initLabel(anchorNode, itemDesc);
                clickHandler(itemDesc.value);
            });
            listNode.append(listItem);
        });

        return menuNode;
    };

    /**
     * Appends a new sorting rule to the end of the list.
     */
    CustomSortDialog.prototype._appendSortRule = function (sortRule) {

        // clone the sort rule, and insert it into the internal array
        sortRule = _.extend({ index: -1, descending: false }, sortRule);

        // create the DOM container element for the sorting rule
        var ruleNode = $('<div class="sort-rule">');
        this._rulesRootNode.append(ruleNode);

        // insert sort rule and DOM node into the internal array
        var ruleDesc = { rule: sortRule, node: ruleNode };
        this._ruleDescs.push(ruleDesc);

        // create the drop-down menu with all available columns/rows
        var sortByMenu = this._createDropDownMenu(this._headerDescs, sortRule.index, function (index) {
            sortRule.index = index;
            this._updateControls();
        });
        sortByMenu.addClass('column-sort-by');

        // create the drop-down menu for the sorting order
        var sortOrderMenu = this._createDropDownMenu([
            _.extend({ value: false }, Labels.SORT_ASC_BUTTON_OPTIONS),
            _.extend({ value: true }, Labels.SORT_DESC_BUTTON_OPTIONS)
        ], sortRule.descending, function (value) {
            sortRule.descending = value;
            this._updateControls();
        });
        sortOrderMenu.addClass('column-sort-order');

        // create and initialize the "delete rule" button
        var deleteBtn = $('<a class="delete-rule-button" href="#" tabindex="0">').append(Forms.createIconMarkup('fa-trash-o'));
        deleteBtn.on('click keydown', function (event) {
            if ((event.type !== 'keydown') || (event.keyCode === KeyCodes.SPACE)) {
                this._ruleDescs.splice(this._ruleDescs.indexOf(ruleDesc), 1);
                ruleNode.remove();
                this._updateControls();
                return false;
            }
        }.bind(this));

        // insert the controls into the DOM root node
        ruleNode.append(sortByMenu, sortOrderMenu, deleteBtn);
    };

    /**
     * Updates the state and visibility of various controls ("Add rule" button,
     * OK button, etc.).
     */
    CustomSortDialog.prototype._updateControls = function () {

        // disable the "Add rule" button if the number of rules reaches the number of headers
        this._addRuleBtn.toggleClass('disabled', !this._canAddSortRule());

        // hide the "Delete rule" button of a single rule in a simple cell range
        var deleteButtons = this._rulesRootNode.find('.delete-rule-button');
        Forms.showNodes(deleteButtons, this._canDeleteSortRule());

        // collect multiple rules with the same column/row index
        var ruleGroups = _.countBy(this._ruleDescs, function (ruleDesc) { return ruleDesc.rule.index; });

        // insert a warning icon to the passed drop-down menu
        var hasWarning = false;
        var createWarning = function (menuNode, tooltip) {
            var iconNode = $(Forms.createIconMarkup('fa-warning'));
            iconNode.addClass('rule-warning').attr('title', tooltip);
            menuNode.prepend(iconNode);
            hasWarning = true;
        };

        // show warnings for invalid rule settings
        this._ruleDescs.forEach(function (ruleDesc) {
            var index = ruleDesc.rule.index;
            var menuNode = ruleDesc.node.find('>.column-sort-by');
            menuNode.find('.rule-warning').remove();
            if (index < 0) {
                createWarning(menuNode, ERR_MISSING_LABEL);
            } else if (ruleGroups[index] > 1) {
                createWarning(menuNode, ERR_MULTI_LABEL);
            }
        }, this);

        // update the OK button
        this.enableOkButton(!hasWarning);
    };

    /**
     * Clears all existing sorting rules, and adds the passed sorting rules if
     * specified; otherwise a default sorting rule according to the active cell
     * in the document view, or the existing sorting rules of the table model.
     */
    CustomSortDialog.prototype._initializeSortRules = function (sortRules) {

        // initialize header range data
        var horizontal = this._sortDir === SortDirection.HORIZONTAL;
        var headerRange = this._tableModel ? this._tableModel.getRange().headerRow() : this._sortRange.lineRange(horizontal, 0);
        var cellCollection = this._docView.getCellCollection();

        // collect the settings for all header labels
        this._headerDescs = Iterator.map(headerRange, function (address) {

            // the current column/row index inside the header range
            var index = address.get(!horizontal);
            // use display strings of the header cells if specified
            var display = this._hasHeader ? cellCollection.getDisplayString(address) : null;
            // empty or missing display string (e.g. format error): fall-back to column/row label
            var label = display || Labels.getColRowLabel(index, !horizontal);
            // show fall-back column/row label in header mode in italic style
            var italic = this._hasHeader && !display;

            return { value: index, label: label, italic: italic };
        }, this);

        // delete all sorting rules, and DOM sorting rule elements
        this._ruleDescs.forEach(function (ruleDesc) { ruleDesc.node.remove(); });
        this._ruleDescs = [];

        // use sorting rules of the table model if available
        if (!sortRules && this._tableModel) {
            sortRules = this._tableModel.getSortRules();
        }

        // add existing sorting rules, or a default sorting rule
        if (sortRules && (sortRules.length > 0)) {
            sortRules.forEach(this._appendSortRule, this);
        } else {
            var index = this._docView.getActiveCell().get(!horizontal);
            this._appendSortRule({ index: index, descending: false });
        }

        // update other dialog controls
        this._updateControls();
    };

    // exports ================================================================

    return CustomSortDialog;

});
