/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/mergecellspicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, RadioList, AppObjectMixin, SheetUtils, gt) {

    'use strict';

    // convenience shortcuts
    var MergeMode = SheetUtils.MergeMode;

    // the label or tooltip for this control
    var TOOLTIP_LABEL = gt('Merge or unmerge cells');

    // icon names for merge states
    var MERGE_ICON = 'docs-merged-cells-on';
    var MERGE_HOR_ICON = 'docs-merged-cells-horizontal';
    var MERGE_VERT_ICON = 'docs-merged-cells-vertical';
    var UNMERGE_ICON = 'docs-merged-cells-off';

    // class MergeCellsPicker =================================================

    /**
     * The selector for merged ranges, as drop-down list with split button.
     *
     * @constructor
     *
     * @extends RadioList
     * @extends AppObjectMixin
     *
     * @param {SpreadsheetView} docView
     *  The document view containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioList,
     *  except 'highlight', 'splitValue', and 'updateCaptionMode'.
     */
    var MergeCellsPicker = RadioList.extend({ constructor: function (docView, initOptions) {

        // self reference
        var self = this;

        // whether to show the 'merge' menu item
        var showMergeItem = false;
        // whether to show the 'merge horizontal' menu item
        var showHMergeItem = false;
        // whether to show the 'merge vertical' menu item
        var showVMergeItem = false;
        // whether to show the 'unmerge' menu item
        var showUnmergeItem = false;

        // base constructors --------------------------------------------------

        RadioList.call(this, docView, Utils.extendOptions({
            icon: MERGE_ICON,
            tooltip: TOOLTIP_LABEL,
            dropDownVersion: { label: TOOLTIP_LABEL }
        }, initOptions, {
            highlight: _.identity,
            splitValue: 'toggle',
            updateCaptionMode: 'none'
        }));

        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * Updates the icon according to the merged state of the selection.
         */
        function updateHandler() {
            self.setIcon(showUnmergeItem ? UNMERGE_ICON : MERGE_ICON);
        }

        /**
         * Updates the visibility flags of the list items after the selection
         * in the active sheet has been changed, or after the merge collection
         * of the active sheet has changed.
         */
        function updateVisibilityFlags() {

            // the collection of merged ranges in the active sheet
            var mergeCollection = docView.getMergeCollection();
            // the selected cell ranges
            var ranges = docView.getSelectedRanges();

            // show the 'merge' list item, if at least one range consists of more than a cell or a merged range
            showMergeItem = ranges.some(function (range) {
                var mergedRange = range.single() ? null : mergeCollection.getMergedRange(range.start);
                return !mergedRange || !mergedRange.contains(range);
            });

            // show the 'horizontal' list item, if at least one range does not consist entirely of horizontally merged cells
            showHMergeItem = ranges.some(function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isHorizontallyMerged(range);
            });

            // show the 'vertical' list item, if at least one range does not consist entirely of vertically merged cells
            showVMergeItem = ranges.some(function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isVerticallyMerged(range);
            });

            // show the 'unmerge' list item, if at least one range overlaps with a merged range
            showUnmergeItem = mergeCollection.coversAnyMergedRange(ranges);
        }

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.createOptionButton(MergeMode.MERGE,      { icon: MERGE_ICON,      label: gt('Merge cells'),              visible: function () { return showMergeItem; } })
            .createOptionButton(MergeMode.HORIZONTAL, { icon: MERGE_HOR_ICON,  label: gt('Merge cells horizontally'), visible: function () { return showHMergeItem; } })
            .createOptionButton(MergeMode.VERTICAL,   { icon: MERGE_VERT_ICON, label: gt('Merge cells vertically'),   visible: function () { return showVMergeItem; } })
            .createOptionButton(MergeMode.UNMERGE,    { icon: UNMERGE_ICON,    label: gt('Unmerge cells'),            visible: function () { return showUnmergeItem; } });

        // update the icon in the drop-down button
        this.registerUpdateHandler(updateHandler);

        // update state flags used for visibility of list items (to keep the 'visible' callback functions simple and fast)
        this.waitForImportSuccess(updateVisibilityFlags);
        this.listenTo(docView, 'change:selection insert:merged delete:merged', updateVisibilityFlags);

    } }); // class MergeCellsPicker

    // exports ================================================================

    return MergeCellsPicker;

});
