/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/subtotalresult', function () {

    'use strict';

    // class SubtotalResult ===================================================

    /**
     * The collected subtotal results of the sheet selection.
     *
     * @constructor
     *
     * @property {Number} cells
     *  The total count of non-blank cells (numbers, strings, booleans, and
     *  error codes).
     *
     * @property {Number} numbers
     *  The total count of number cells.
     *
     * @property {Number} sum
     *  The sum of all numbers.
     *
     * @property {Number} min
     *  The minimum of all numbers.
     *
     * @property {Number} max
     *  The maximum of all numbers.
     *
     * @property {Number} average
     *  The arithmetic mean of all numbers.
     */
    function SubtotalResult() {

        this.cells = 0;
        this.numbers = 0;
        this.sum = 0;
        this.min = Number.POSITIVE_INFINITY;
        this.max = Number.NEGATIVE_INFINITY;
        this.average = Number.NaN;

    } // class SubtotalResult

    // public methods ---------------------------------------------------------

    /**
     * Adds a new value to this subtotal result.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The value to be added to this subtotal result. All non-null values will
     *  be counted in the property 'cells'. All numbers will be counted and
     *  aggregated in the remaining properties 'numbers', 'sum', 'min', 'max',
     *  and 'average'.
     *
     * @returns {SubtotalResult}
     *  A reference to this instance.
     */
    SubtotalResult.prototype.addValue = function (value) {

        if (value !== null) {
            this.cells += 1;
        }

        if (typeof value === 'number') {
            this.numbers += 1;
            this.sum += value;
            this.min = Math.min(this.min, value);
            this.max = Math.max(this.max, value);
            this.average = this.sum / this.numbers;
        }

        return this;
    };

    /**
     * Adds the settings of the passed subtotal result object.
     *
     * @param {SubtotalResult} subtotals
     *  The subtotals to be added to this result.
     *
     * @returns {SubtotalResult}
     *  A reference to this instance.
     */
    SubtotalResult.prototype.addSubtotals = function (subtotals) {
        this.cells += subtotals.cells;
        this.numbers += subtotals.numbers;
        this.sum += subtotals.sum;
        this.min = Math.min(this.min, subtotals.min);
        this.max = Math.max(this.max, subtotals.max);
        this.average = this.sum / this.numbers;
        return this;
    };

    /**
     * Adds the settings of the passed subtotal result object.
     *
     * @param {Number|String|Boolean|ErrorCode|Null|SubtotalResult} result
     *  A scalar value, or another instance of this class, to be added to this
     *  result.
     *
     * @returns {SubtotalResult}
     *  A reference to this instance.
     */
    SubtotalResult.prototype.add = function (result) {
        return (result instanceof SubtotalResult) ? this.addSubtotals(result) : this.addValue(result);
    };

    // exports ================================================================

    return SubtotalResult;

});
