/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/rangeset', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/intervalset'
], function (Iterator, ValueSet, Range, RangeArray, IntervalSet) {

    'use strict';

    // constants ==============================================================

    // convenience shortcuts
    var FilterIterator = Iterator.FilterIterator;
    var NestedIterator = Iterator.NestedIterator;
    var SetProto = ValueSet.prototype;

    var MATCHERS = {
        contain:   function (range, element) { return range.contains(element); },
        partial:   function (range, element) { return !range.contains(element); },
        reference: function (range, element) { return range.containsAddress(element.start); }
    };

    // private global functions ===============================================

    /**
     * Returns the key of the passed index interval.
     *
     * @param {Interval} interval
     *  An index interval.
     *
     * @returns {String}
     *  The key of the passed index interval.
     */
    function getKey(interval) {
        return interval.key();
    }

    // class RangeSet =========================================================

    /**
     * A unique set of cell range addresses. This class provides optimized
     * methods for insertion, deletion, and access of cell range addresses,
     * while keeping the set unique (two different range addresses in this set
     * will never have the same start and end address).
     *
     * @constructor
     *
     * @extends ValueSet
     */
    var RangeSet = ValueSet.extend(function () {

        // base constructor
        ValueSet.call(this, getKey);

        // the interval set storing the column intervals of all cell ranges in this set
        this._colSet = new IntervalSet();

    }); // class RangeSet

    // public methods ---------------------------------------------------------

    /**
     * Removes all elements from this set.
     *
     * @returns {RangeSet}
     *  A reference to this instance.
     */
    RangeSet.prototype.clear = function () {
        SetProto.clear.call(this);
        this._colSet.clear();
        return this;
    };
    /**
     * Returns all cell range addresses contained in this set as an array.
     *
     * @returns {RangeArray}
     *  All cell range addresses contained in this set as an array, in no
     *  specific order.
     */
    RangeSet.prototype.values = function () {
        var ranges = new RangeArray();
        this.forEach(function (range) { ranges.push(range); });
        return ranges;
    };

    /**
     * Returns a shallow or deep clone of this range set.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (all
     *  ranges in this set will be cloned); otherwise a shallow clone will be
     *  created (the ranges in this set will be inserted by reference into the
     *  clone). If a function has been passed, it will be invoked for each
     *  cloned cell range, receiving the new cloned cell range inserted into
     *  the cloned set as first parameter, and the old cell range of this set
     *  as second parameter.
     *
     * @param {Object} [context]
     *  The calling context used if the 'deep' parameter is a callback.
     *
     * @returns {RangeSet}
     *  The cloned range set.
     */
    RangeSet.prototype.clone = function (deep, context) {

        // start with a new empty range set
        var clone = new RangeSet();
        // user callback for deep mode
        var callback = (typeof deep === 'function') ? deep : _.noop;

        // create a deep clone of the interval sets (shallow mode only for the final ranges!)
        clone._colSet = this._colSet.clone(function (newColInterval, oldColInterval) {
            newColInterval.rowSet = oldColInterval.rowSet.clone(function (newRowInterval, oldRowInterval) {
                var oldRange = oldRowInterval.range;
                var newRange = deep ? oldRange.clone() : oldRange;
                callback.call(context, newRange, oldRange);
                newRowInterval.range = newRange;
                clone._insert(newRange.key(), newRange);
            });
        });

        return clone;
    };

    /**
     * Returns the cell range address from this set that has the exact position
     * of the passed range address.
     *
     * @param {Range} range
     *  A cell range address to search for.
     *
     * @returns {Range|Null}
     *  An range object from this set with the same start and end address as
     *  the passed range address; otherwise null.
     */
    RangeSet.prototype.get = function (range) {
        return this._get(range.key()) || null;
    };

    /**
     * Returns the cell range address consisting of a single cell from this set
     * that has the exact position of the passed cell address.
     *
     * @param {Address} address
     *  A cell address to search for.
     *
     * @returns {Range|Null}
     *  An range object from this set with the same start and end address as
     *  the passed cell address; otherwise null.
     */
    RangeSet.prototype.getByAddress = function (address) {
        return this._get(Range.key(address, address)) || null;
    };

    /**
     * Returns an iterator that will visit all cell ranges in this set that
     * overlap with the passed cell range address.
     *
     * @param {Range} range
     *  The cell range address used to look for the cell ranges in this set.
     *
     * @param {String} [matchType]
     *  Specifies which overlapping cell ranges will be visited:
     *  - Omitted (default): Visits all cell ranges in this set that overlap
     *      somehow with the passed cell range (partially or completely).
     *  - "contain": Visits only the cell ranges in this set that are
     *      completely  contained in the passed cell range address.
     *  - "partial": Visits only the cell ranges in this set that are NOT
     *      completely contained in the passed cell range address.
     *  - "reference": Visits only the cell ranges in this set whose start
     *      address is contained in the passed cell range address.
     *
     * @returns {Iterator}
     *  An iterator that will visit all cell ranges in this set overlapping
     *  with the passed cell range address, in no specific order.
     */
    RangeSet.prototype.findIterator = function (range, matchType) {

        // visit all column intervals overlapping with the passed range (do not pass "matchType"
        // to interval iterator, e.g. "partial" may not match by column but only by row)
        var iterator = this._colSet.findIterator(range.colInterval());

        // create a nested iterator that visits all row intervals in the columns
        iterator = new NestedIterator(iterator, function (colInterval) {
            return colInterval.rowSet.findIterator(range.rowInterval());
        }, function (colResult, rowResult) {
            return { value: rowResult.value.range };
        });

        // filter by specified match type
        var matcher = matchType ? MATCHERS[matchType] : null;
        return matcher ? new FilterIterator(iterator, matcher.bind(null, range)) : iterator;
    };

    /**
     * Returns whether this set contains a cell range covering the passed cell
     * address.
     *
     * @param {Address} address
     *  The cell address used to look for a cell range in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains a cell range covering the passed address.
     */
    RangeSet.prototype.containsAddress = function (address) {
        return this.findOneByAddress(address) !== null;
    };

    /**
     * Returns an arbitrary cell range in this set that covers the passed cell
     * address.
     *
     * @param {Address} address
     *  The cell address used to look for a cell range in this set.
     *
     * @returns {Range|Null}
     *  An arbitrary cell range address in this set that covers the passed cell
     *  address.
     */
    RangeSet.prototype.findOneByAddress = function (address) {
        return this.findOne(new Range(address));
    };

    /**
     * Returns all cell ranges in this set that cover the passed cell address.
     *
     * @param {Address} address
     *  The cell address used to look for the cell ranges in this set.
     *
     * @returns {RangeArray}
     *  An array with all cell ranges in this set that cover the passed cell
     *  address, in no specific order.
     */
    RangeSet.prototype.findAllByAddress = function (address) {
        return this.findAll(new Range(address));
    };

    /**
     * Returns an arbitrary range that overlaps with the passed cell range.
     *
     * @param {Range} range
     *  The cell range address used to look for overlapping ranges in this set.
     *
     * @param {String} [matchType]
     *  Specifies which overlapping cell ranges from this set will match. See
     *  method RangeSet.findIterator() for details.
     *
     * @returns {Range|Null}
     *  An arbitrary cell range address overlapping with the passed cell range.
     */
    RangeSet.prototype.findOne = function (range, matchType) {
        var result = this.findIterator(range, matchType).next();
        return result.done ? null : result.value;
    };

    /**
     * Returns all ranges in this set that overlap with the passed cell range.
     *
     * @param {Range} range
     *  The cell range used to look for the ranges in this set.
     *
     * @param {String} [matchType]
     *  Specifies which overlapping cell ranges from this set will match. See
     *  method RangeSet.findIterator() for details.
     *
     * @returns {RangeArray}
     *  An array with all ranges in this set overlapping with the passed cell
     *  range, in no specific order.
     */
    RangeSet.prototype.findAll = function (range, matchType) {
        var iterator = this.findIterator(range, matchType);
        return Iterator.toArray(iterator, new RangeArray());
    };

    /**
     * Inserts a new cell range address into this set.
     *
     * @param {Range} range
     *  The cell range address to be inserted into this set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing cell range with equal start and end address
     *  as the passed cell range will be replaced with the new cell range. By
     *  default, the existing cell range will be kept, and the passed cell
     *  range will be ignored. May be important in situations where the ranges
     *  in the set contain additional user settings in arbitrary properties.
     *
     * @returns {Range}
     *  A reference to the cell range contained in this set. This may not be
     *  the same as the passed cell range, e.g. if an equal cell range already
     *  exists in this set, and the flag 'replace' has not been set.
     */
    RangeSet.prototype.insert = function (range, replace) {
        var colInterval = this._colSet.insert(range.colInterval());
        var rowSet = colInterval.rowSet || (colInterval.rowSet = new IntervalSet());
        var rowInterval = rowSet.insert(range.rowInterval());
        if (replace || !rowInterval.range) { rowInterval.range = range; }
        return this._insert(range.key(), rowInterval.range);
    };

    /**
     * Removes the specified cell range address from this set.
     *
     * @param {Range} range
     *  The cell range address to be removed from this set.
     *
     * @returns {Range|Null}
     *  A reference to the removed cell range; or null, if this set does not
     *  contain a cell range that is equal to the passed cell range.
     */
    RangeSet.prototype.remove = function (range) {
        var key = range.key();
        range = this._get(key);
        if (!range) { return null; }
        this._remove(key);
        var colInterval = this._colSet.get(range.colInterval());
        var rowSet = colInterval.rowSet;
        rowSet.remove(range.rowInterval());
        if (rowSet.empty()) { this._colSet.remove(colInterval); }
        return range;
    };

    /**
     * Inserts all cell ranges of the passed set into this set.
     *
     * @param {Object} list
     *  A generic data source with cell range addresses as elements. See static
     *  method Container.forEach() for details.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, cell ranges in the passed list will replace equal cell
     *  ranges in this set. By default, the existing cell ranges of this set
     *  will be preferred.
     *
     * @returns {RangeSet}
     *  A reference to this instance.
     */
    RangeSet.prototype.merge = function (list, replace) {
        ValueSet.forEach(list, function (range) { this.insert(range, replace); }, this);
        return this;
    };

    // exports ================================================================

    return RangeSet;

});
