/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/movedescriptor', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/rangearray'
], function (Utils, Interval, Range, IntervalArray, AddressArray, RangeArray) {

    'use strict';

    // private global functions ===============================================

    /**
     * Transforms a column/row index according to the passed target intervals
     * of a move descriptor in insertion mode.
     *
     * @param {IntervalArray} targetIntervals
     *  The (sorted) target intervals of a move descriptor in insertion mode.
     *
     * @param {Number} index
     *  The column/row index to be transformed.
     *
     * @param {Number} maxIndex
     *  The maximum column/row index in the sheet in move direction.
     *
     * @returns {Number|Null}
     *  The transformed column/row index; or null, if the index has been moved
     *  beyond the specified maximum column/row index in the sheet.
     */
    function transformIndexInsert(targetIntervals, index, maxIndex) {

        // transform the column/row index as long as it remains valid
        for (var i = 0, l = targetIntervals.length; i < l; i += 1) {
            var targetInterval = targetIntervals[i];

            // early exit, if the target intervals are located behind the index (array is sorted)
            if (index < targetInterval.first) { break; }

            // shift index ahead
            index += targetInterval.size();

            // early exit if the index becomes invalid (moved beyond the maximum index)
            if (index > maxIndex) { return null; }
        }
        return index;
    }

    /**
     * Transforms a column/row index according to the passed target intervals
     * of a move descriptor in deletion mode.
     *
     * @param {IntervalArray} targetIntervals
     *  The (sorted) target intervals of a move descriptor in deletion mode.
     *
     * @param {Number} index
     *  The column/row index to be transformed.
     *
     * @returns {Number|Null}
     *  The transformed column/row index; or null, if the index was part of a
     *  target interval and has therefore been deleted.
     */
    function transformIndexDelete(targetIntervals, index) {

        // transform the column/row index as long as it remains valid (in reversed order!)
        for (var i = targetIntervals.length - 1; i >= 0; i -= 1) {
            var targetInterval = targetIntervals[i];

            // nothing to do, if the target intervals are located behind the index
            if (index < targetInterval.first) { continue; }

            // invalidate the index, if it is contained in the target interval
            if (index <= targetInterval.last) { return null; }

            // shift index back
            index -= targetInterval.size();
        }
        return index;
    }

    /**
     * Transforms an index interval according to the passed target intervals of
     * a move descriptor in insertion mode.
     *
     * @param {IntervalArray} targetIntervals
     *  The (sorted) target intervals of a move descriptor in insertion mode.
     *
     * @param {Interval} interval
     *  The index interval to be transformed.
     *
     * @param {Number} maxIndex
     *  The maximum column/row index in the sheet in move direction.
     *
     * @param {Boolean} expandEnd
     *  If set to true, the end position of the interval will be expanded, if
     *  new indexes will be inserted exactly behind the interval.
     *
     * @returns {Interval|Null}
     *  The transformed index interval; or null, if the interval has been moved
     *  beyond the specified maximum column/row index in the sheet.
     */
    function transformIntervalInsert(targetIntervals, interval, maxIndex, expandEnd) {

        // transform the interval as long as it remains valid
        for (var i = 0, l = targetIntervals.length; i < l; i += 1) {
            var targetInterval = targetIntervals[i];
            var targetSize = targetInterval.size();

            // early exit, if the target intervals are located behind the interval (array is sorted)
            if (interval.last + (expandEnd ? 1 : 0) < targetInterval.first) { break; }

            // shift start position ahead, if the interval starts inside or behind the target interval;
            // exit the loop early if the start index becomes invalid (moved beyond the maximum index)
            if (targetInterval.first <= interval.first) {
                interval.first += targetSize;
                if (interval.first > maxIndex) { return null; }
            }

            // shift end position ahead (in expand-end mode: also, if it ends exactly before the target interval),
            // keep the end position inside the limits of the sheet (interval may shrink)
            interval.last = Math.min(interval.last + targetSize, maxIndex);
        }
        return interval;
    }

    /**
     * Transforms an index interval according to the passed target intervals of
     * a move descriptor in deletion mode.
     *
     * @param {IntervalArray} targetIntervals
     *  The (sorted) target intervals of a move descriptor in deletion mode.
     *
     * @param {Interval} interval
     *  The index interval to be transformed.
     *
     * @returns {Interval|Null}
     *  The transformed index interval; or null, if the interval was covered by
     *  a target interval and has therefore been deleted completely.
     */
    function transformIntervalDelete(targetIntervals, interval) {

        // transform the interval index as long as it remains valid (in reversed order!)
        for (var i = targetIntervals.length - 1; i >= 0; i -= 1) {
            var targetInterval = targetIntervals[i];
            var targetSize = targetInterval.size();

            // nothing to do, if the target intervals are located behind the interval
            if (interval.last < targetInterval.first) { continue; }

            // shift start position of the interval back to start of target interval
            if (targetInterval.first < interval.first) {
                interval.first = Math.max(targetInterval.first, interval.first - targetSize);
            }

            // shift end position of the interval back to start of target interval;
            // exit the loop early if the interval becomes invalid
            interval.last = Math.max(interval.last - targetSize, targetInterval.first - 1);
            if (interval.last < interval.first) { return null; }
        }
        return interval;
    }

    // class MoveDescriptor ===================================================

    /**
     * Builds the resulting shifted insertion or deletion intervals, and other
     * useful intervals, needed to generate document operations to insert or
     * delete cells in a worksheet, including inserting/deleting entire columns
     * or rows.
     *
     * Example 1 (Insertion):
     * The columns C:D (two columns) and F:I (4 columns) will be passed to this
     * constructor in insertion mode (insert two columns between B and C, and
     * four columns between E and F). The sheet contains 256 columns (maximum
     * column IV). After applying the insert operations, columns C:D and H:K
     * will be blank. The latter interval F:I has been shifted by two columns
     * to H:K due to the preceding insertion interval C:D. The following
     * intervals will be generated and collected:
     * (1) The shifted insertion instervals (the target intervals) that will be
     *      blank after applying the insert operations (columns C:D and H:K).
     * (2) The original position of the column intervals that will be moved to
     *      the end of the sheet (columns C:E and F:IP). The latter interval
     *      ends six columns before the end of the sheet (column IV) so that it
     *      will not be shifted outside the sheet.
     * (3) The position of the column intervals after they have been moved to
     *      the end of the sheet (columns E:G and L:IV).
     * (4) The deleted intervals shifted outside the sheet (columns IQ:IV).
     *
     * Example 2 (Deletion):
     * The columns C:D (two columns) and F:I (4 columns) will be passed to this
     * constructor in deletion mode (delete columns B and C, and columns F to
     * I). The sheet contains 256 columns (maximum column IV). After applying
     * the delete operations, the specified columns have been deleted, and the
     * remaining entries have been shifted towards the beginning of the sheet.
     * The following intervals will be generated and collected:
     * (1) The deletion intervals (the target intervals) that will be deleted
     *      from the sheet (columns C:D and F:I).
     * (2) The original position of the column intervals that will be moved to
     *      the beginning of the sheet (columns E:E and J:IV).
     * (3) The position of the column intervals after they have been moved to
     *      the beginning of the sheet (columns C:C and D:IP).
     * (4) The intervals deleted from the sheet. in deletion mode, these will
     *      always be copies of the target intervals (columns C:D and F:I).
     *
     * @constructor
     *
     * @property {Interval} bandInterval
     *  The index interval in the crossing direction that restricts the moved
     *  interval, that can be used to convert the target intervals to cell
     *  range addresses. Example: When inserting or deleting columns, this
     *  interval will be interpreted as row interval specifying the row band
     *  with the cells that will be shifted to the left or right.
     *
     * @property {IntervalArray} targetIntervals
     *  The merged, shifted, and shortened intervals, that represent the new
     *  inserted or deleted index intervals.
     *
     * @property {IntervalArray} moveFromIntervals
     *  The index intervals containing the source position of the cells before
     *  they will be moved towards the end (insertion), or to the beginning
     *  (deletion) of the sheet.
     *
     * @priperty {IntervalArray} moveToIntervals
     *  The index intervals containing the final position of the cells after
     *  they have been moved towards the end (insertion), or to the beginning
     *  (deletion) of the sheet. The intervals in this array will have the same
     *  size as the corresponding original intervals from 'moveFromIntervals'.
     *
     * @property {IntervalArray} deleteIntervals
     *  The index intervals containg the position of all entries that will be
     *  deleted, either trailing entries that will be shifted outside the sheet
     *  on insertion, or the target intervals themselved on deletion.
     *
     * @property {Range} dirtyRange
     *  The address of the cell range that will be modified by the move
     *  operation represented by this descriptor. Includes all cells that will
     *  be moved, inserted, or deleted.
     *
     * @property {Boolean} columns
     *  If true, the cells will be moved through columns (i.e. to the left or
     *  right), otherwise through rows (i.e. up or down).
     *
     * @property {Boolean} insert
     *  If set to true, the cells will be moved to the end of the sheet (new
     *  blank cells will be inserted), otherwise to the beginning of the sheet
     *  (existing cells will be deleted).
     */
    function MoveDescriptor(docModel, bandInterval, intervals, columns, insert) {

        // the band interval (crossing direction) restricting the moved target intervals
        this.bandInterval = bandInterval;

        // the intervals affected by the move operation (ensure an array)
        this.targetIntervals = IntervalArray.get(intervals);

        // the original position of all moved intervals
        this.moveFromIntervals = new IntervalArray();

        // the resulting position of all moved intervals
        this.moveToIntervals = new IntervalArray();

        // the intervals that will be deleted (instead of moved)
        this.deleteIntervals = null;

        // the dirty range (moved, inserted, and deleted cells)
        this.dirtyRange = null;

        // the move direction, for convenience
        this.columns = columns;

        // the operation type, for convenience
        this.insert = insert;

        // the maximum column/row index allowed in the specified move direction
        this.maxIndex = docModel.getMaxIndex(columns);

        // initialization -----------------------------------------------------

        if (insert) {

            // sort and unify the intervals, but do not merge them (originating intervals may overlap)
            this.targetIntervals.sort().forEachReverse(function (interval, index, intervals) {
                var nextInterval = intervals[index + 1];
                if (nextInterval && (interval.first === nextInterval.first)) {
                    intervals.splice(index, 1);
                }
            });

            // Shift the following intervals to the end, according to the size of the preceding insertion intervals,
            // shorten or delete the trailing intervals that will be shifted outside the maximum column/row index.
            var boundInterval = new Interval(0, this.maxIndex);
            var insertOffset = 0;
            this.targetIntervals = IntervalArray.map(this.targetIntervals, function (interval) {
                var targetInterval = interval.clone().move(insertOffset).intersect(boundInterval);
                insertOffset += interval.size();
                return targetInterval;
            });

            // The trailing indexes at the end of the sheet will be shifted outside (a single index interval).
            var deleteInterval = new Interval(this.maxIndex - this.targetIntervals.size() + 1, this.maxIndex);
            this.deleteIntervals = new IntervalArray(deleteInterval);

        } else {

            // merge the intervals (the result array will be sorted)
            this.targetIntervals = this.targetIntervals.merge();

            // The target deletion intervals will be deleted (of course).
            this.deleteIntervals = this.targetIntervals.clone(true);
        }

        // Calculate the move intervals between the resulting target intervals.
        var shiftedIntervals = insert ? this.moveToIntervals : this.moveFromIntervals;
        var originalIntervals = insert ? this.moveFromIntervals : this.moveToIntervals;
        var moveOffset = 0;
        this.targetIntervals.some(function (targetInterval, index) {
            if (targetInterval.last >= this.maxIndex) { return true; }

            // the interval between the current and next target interval
            var nextInterval = this.targetIntervals[index + 1];
            var moveInterval = new Interval(targetInterval.last + 1, nextInterval ? (nextInterval.first - 1) : this.maxIndex);
            // the interval is the position after moving (insertion mode); or before moving (deletion mode)
            shiftedIntervals.push(moveInterval);

            // move the interval back by the total size of preceding target intervals
            moveOffset -= targetInterval.size();
            moveInterval = moveInterval.clone().move(moveOffset);
            originalIntervals.push(moveInterval);

        }, this);

        // calculate the dirty range covered by this descriptor
        this.dirtyRange = this.createRange(new Interval(this.targetIntervals.first().first, this.maxIndex));

    } // class MoveDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed cell address is contained in the cell range
     * covered by the band interval of this move descriptor.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed cell address is contained in the cell range covered
     *  by the band interval of this move descriptor.
     */
    MoveDescriptor.prototype.containsAddress = function (address) {
        return this.bandInterval.containsIndex(address.get(!this.columns));
    };

    /**
     * Returns whether the passed range is completely contained in the cell
     * range covered by the band interval of this move descriptor.
     *
     * @param {Range} range
     *  The cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range is completely contained in the cell range
     *  covered by the band interval of this move descriptor.
     */
    MoveDescriptor.prototype.containsRange = function (range) {
        return this.bandInterval.contains(range.interval(!this.columns));
    };

    /**
     * Creates a cell range address located in the band interval of this move
     * descriptor, and covering a single column or row in that band.
     *
     * Example: If the move descriptor represents the row band 1:3 (move cells
     * to the left or right inside these rows), this method will return the
     * cell range address E1:E3 for index 4 (column E) passed to it.
     *
     * @param {Number} index
     *  The column/row index to be used to create the cell range address.
     *
     * @returns {Range}
     *  The cell range address located in the band interval of this move
     *  descriptor, and covering the specified column/row index.
     */
    MoveDescriptor.prototype.createLineRange = function (index) {
        return this.columns ? Range.createFromRowInterval(this.bandInterval, index) : Range.createFromColInterval(this.bandInterval, index);
    };

    /**
     * Creates a cell range address located in the band interval of this move
     * descriptor, and covering the specified index interval in that band.
     *
     * Example: If the move descriptor represents the row band 1:3 (move cells
     * to the left or right inside these rows), this method will return the
     * cell range address E1:F3 for the column interval E:F passed to it.
     *
     * @param {Interval} interval
     *  The interval inside the band covered by this move descriptor.
     *
     * @returns {Range}
     *  The cell range address located in the band interval of this move
     *  descriptor, and covering the specified index interval.
     */
    MoveDescriptor.prototype.createRange = function (interval) {
        return this.columns ? Range.createFromIntervals(interval, this.bandInterval) : Range.createFromIntervals(this.bandInterval, interval);
    };

    /**
     * Creates an array of cell range addresses located in the band interval of
     * this move descriptor, and covering the specified index intervals in that
     * band.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of intervals, or a single interval inside the band covered by
     *  this move descriptor.
     *
     * @returns {RangeArray}
     *  The cell range addresses located in the band interval of this move
     *  descriptor, and covering the specified index intervals.
     */
    MoveDescriptor.prototype.createRanges = function (intervals) {
        return RangeArray.map(intervals, this.createRange, this);
    };

    /**
     * Transforms the passed cell address according to the settings of this
     * move descriptor.
     *
     * @param {Address} address
     *  The cell address to be transformed.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.reverse=false]
     *      Whether to transform the passed cell address with the reversed
     *      operation. If this move descriptor represents an 'insert'
     *      operation, the address will be transformed according to a 'delete'
     *      operation, and vice versa.
     *
     * @returns {Address|Null}
     *  The transformed cell address. If the cell would be deleted completely
     *  while deleting columns or rows, or would be shifted outside the sheet
     *  while inserting columns or rows, null will be returned instead.
     */
    MoveDescriptor.prototype.transformAddress = function (address, options) {

        // nothing to do, if the address is not covered by this descriptor
        if (!this.containsAddress(address)) { return address.clone(); }

        // the effective move operation
        var insert = this.insert !== Utils.getBooleanOption(options, 'reverse', false);
        // the helper function that implements transformation of an address index
        var transformFunc = insert ? transformIndexInsert : transformIndexDelete;
        // transform the correct index of the passed address, according to move direction
        var index = transformFunc(this.targetIntervals, address.get(this.columns), this.maxIndex);

        // create a new address with adjusted column/row index, or return null to indicate deleted cell
        return (index === null) ? null : address.clone().set(index, this.columns);
    };

    /**
     * Transforms the passed cell addresses according to the settings of this
     * move descriptor.
     *
     * @param {AddressArray|Address} addresses
     *  The cell addresses to be transformed, as array of cell addresses, or as
     *  single cell address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.reverse=false]
     *      Whether to transform the passed cell addresses with the reversed
     *      operation. If this move descriptor represents an 'insert'
     *      operation, the addresses will be transformed according to a
     *      'delete' operation, and vice versa.
     *
     * @returns {AddressArray}
     *  The transformed cell addresses. Cells that would be deleted completely
     *  while deleting columns or rows, or would be shifted outside the sheet
     *  while inserting columns or rows, will not be included into the result.
     */
    MoveDescriptor.prototype.transformAddresses = function (addresses, options) {
        // AddressArray.map() filters null values automatically
        return AddressArray.map(addresses, function (address) {
            return this.transformAddress(address, options);
        }, this);
    };

    /**
     * Transforms the passed cell range address according to the settings of
     * this move descriptor.
     *
     * @param {Range} range
     *  The address of the cell range to be transformed.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.transformFull=false]
     *      If set to true, entire column/row ranges will be transformed too.
     *      By default, entire column ranges will not be modified when moving
     *      vertically (through rows), and entire row ranges will not be
     *      modified when moving horizontally (through columns).
     *  - {Boolean} [options.expandEnd=false]
     *      If set to true, the end position of the range will be expanded, if
     *      new columns will be inserted exactly right of the range, or new
     *      rows will be inserted exactly below the range.
     *  - {Boolean} [options.reverse=false]
     *      Whether to transform the passed range with the reversed operation.
     *      If this move descriptor represents an 'insert' operation, the range
     *      will be transformed according to a 'delete' operation, and vice
     *      versa.
     *
     * @returns {Range|Null}
     *  The address of the transformed cell range. If the cell range would be
     *  deleted completely while deleting columns or rows, or would be shifted
     *  outside the sheet while inserting columns or rows, null will be
     *  returned instead.
     */
    MoveDescriptor.prototype.transformRange = function (range, options) {

        // nothing to do, if the range is not covered by this descriptor
        if (!this.containsRange(range)) { return range.clone(); }

        // the effective move operation
        var insert = this.insert !== Utils.getBooleanOption(options, 'reverse', false);
        // whether to expand the trailing border of the range, when inserting behind the range
        var expandEnd = insert && Utils.getBooleanOption(options, 'expandEnd', false);
        // the helper function that implements transformation of an index interval
        var transformFunc = insert ? transformIntervalInsert : transformIntervalDelete;
        // the column/row interval of the range to be transformed
        var interval = range.interval(this.columns);

        // do not transform full column/row intervals unless specified
        var fullInterval = (interval.first === 0) && (interval.last === this.maxIndex);
        if (fullInterval && !Utils.getBooleanOption(options, 'transformFull', false)) {
            return range.clone();
        }

        // transform the correct interval of the passed range, according to move direction
        var transformInterval = transformFunc(this.targetIntervals, interval, this.maxIndex, expandEnd);
        if (!transformInterval) { return null; }

        // create a new range address
        var colInterval = this.columns ? transformInterval : range.colInterval();
        var rowInterval = this.columns ? range.rowInterval() : transformInterval;
        return Range.createFromIntervals(colInterval, rowInterval);
    };

    /**
     * Transforms the passed cell range addresses according to the settings of
     * this move descriptor.
     *
     * @param {RangeArray|Range} ranges
     *  The cell range addresses to be transformed, as array of cell range
     *  addresses, or as single cell range address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.expandEnd=false]
     *      If set to true, the end position of ranges will also be expanded,
     *      if new columns will be inserted exactly right of a range, or new
     *      rows will be inserted exactly below a range.
     *  - {Boolean} [options.transformFull=false]
     *      If set to true, entire column/row ranges will be transformed too.
     *      By default, entire column ranges will not be modified when moving
     *      vertically (through rows), and entire row ranges will not be
     *      modified when moving horizontally (through columns).
     *  - {Boolean} [options.reverse=false]
     *      Whether to transform the passed ranges with the reversed operation.
     *      If this move descriptor represents an 'insert' operation, the
     *      ranges will be transformed according to a 'delete' operation, and
     *      vice versa.
     *
     * @returns {RangeArray}
     *  The transformed cell range addresses. Cell ranges that would be deleted
     *  completely while deleting columns or rows, or would be shifted outside
     *  the sheet while inserting columns or rows, will not be included into
     *  the result.
     */
    MoveDescriptor.prototype.transformRanges = function (ranges, options) {
        // RangeArray.map() filters null values automatically
        return RangeArray.map(ranges, function (range) {
            return this.transformRange(range, options);
        }, this);
    };

    // static methods ---------------------------------------------------------

    /**
     * Transforms the passed cell address according to the settings of all move
     * descriptors.
     *
     * @param {Address} address
     *  The cell address to be transformed.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array with move descriptors that will be used to transform the cell
     *  address.
     *
     * @param {Object} [options]
     *  Optional parameters. See description of the public prototype method
     *  MoveDescriptor.prototype.transformAddress() for details.
     *
     * @returns {Address|Null}
     *  The transformed cell address. If the cell would be deleted completely
     *  while deleting columns or rows, or would be shifted outside the sheet
     *  while inserting columns or rows, null will be returned instead.
     */
    MoveDescriptor.transformAddress = function (address, moveDescs, options) {
        moveDescs.every(function (moveDesc) {
            return (address = moveDesc.transformAddress(address, options));
        });
        return address;
    };

    /**
     * Transforms the passed cell addresses according to the settings of all
     * move descriptors.
     *
     * @param {AddressArray|Address} addresses
     *  The cell addresses to be transformed, as array of cell addresses, or as
     *  single cell address.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array with move descriptors that will be used to transform the cell
     *  addresses.
     *
     * @param {Object} [options]
     *  Optional parameters. See description of the public prototype method
     *  MoveDescriptor.prototype.transformAddresses() for details.
     *
     * @returns {AddressArray}
     *  The transformed cell addresses. Cells that would be deleted completely
     *  while deleting columns or rows, or would be shifted outside the sheet
     *  while inserting columns or rows, will not be included into the result.
     */
    MoveDescriptor.transformAddresses = function (addresses, moveDescs, options) {
        moveDescs.some(function (moveDesc) {
            addresses = moveDesc.transformAddresses(addresses, options);
            return addresses.empty();
        });
        return addresses;
    };

    /**
     * Transforms the passed cell range address according to the settings of
     * all move descriptors.
     *
     * @param {Range} range
     *  The address of the cell range to be transformed.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array with move descriptors that will be used to transform the cell
     *  range address.
     *
     * @param {Object} [options]
     *  Optional parameters. See description of the public prototype method
     *  MoveDescriptor.prototype.transformRange() for details.
     *
     * @returns {Range|Null}
     *  The address of the transformed cell range. If the cell range would be
     *  deleted completely while deleting columns or rows, or would be shifted
     *  outside the sheet while inserting columns or rows, null will be
     *  returned instead.
     */
    MoveDescriptor.transformRange = function (range, moveDescs, options) {
        var reverse = Utils.getBooleanOption(options, 'reverse', false);
        Utils.iterateArray(moveDescs, function (moveDesc) {
            range = moveDesc.transformRange(range, options);
            if (!range) { return Utils.BREAK; }
        }, { reverse: reverse });
        return range;
    };

    /**
     * Transforms the passed cell range addresses according to the settings of
     * all move descriptors.
     *
     * @param {RangeArray|Range} ranges
     *  The cell range addresses to be transformed, as array of cell range
     *  addresses, or as single cell range address.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array with move descriptors that will be used to transform the cell
     *  range addresses.
     *
     * @param {Object} [options]
     *  Optional parameters. See description of the public prototype method
     *  MoveDescriptor.prototype.transformRanges() for details.
     *
     * @returns {RangeArray}
     *  The transformed cell range addresses. Cell ranges that would be deleted
     *  completely while deleting columns or rows, or would be shifted outside
     *  the sheet while inserting columns or rows, will not be included into
     *  the result.
     */
    MoveDescriptor.transformRanges = function (ranges, moveDescs, options) {
        var reverse = Utils.getBooleanOption(options, 'reverse', false);
        Utils.iterateArray(moveDescs, function (moveDesc) {
            ranges = moveDesc.transformRanges(ranges, options);
            if (ranges.empty()) { return Utils.BREAK; }
        }, { reverse: reverse });
        return ranges;
    };

    // exports ================================================================

    return MoveDescriptor;

});
