/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/services/commandmixin', [
    'io.ox/office/spreadsheet/utils/sheetselection'
], function (SheetSelection) {

    'use strict';

    // mix-in class CommandMixin =============================================

    /**
     *
     * @constructor
     */
    function CommandMixin() {

        var self = this;
        var docModel = null;
        var docView = null;

        // private methods ----------------------------------------------------

        /**
         * Sets the optimal column width for the active sheet in the document
         * view.
         *
         * @returns {jQuery.Promise}
         *  A promise that resolves after setting the optimal column widths.
         */
        function setOptimalColumnWidth() {
            // TODO: use model functionality to change optimal column width

            // process the used area of the cell collection
            var usedRange = docView.getCellCollection().getUsedRange();
            if (!usedRange) { return self.createResolvedPromise(); }

            // rescue the old selection of the active sheet
            var oldSelection = docView.getSelection();

            // select the used area of the sheet
            docView.setCellSelection(new SheetSelection(usedRange));

            // set the optimal column width for the used area, restore the old selection afterwards
            return docView.setOptimalColumnWidth().done(function () {
                docView.setCellSelection(oldSelection);
            });
        }

        /**
         * Sets optimal column width for the specified sheet in the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that resolves after setting the optimal column widths.
         */
        function setOptimalColumnWidthForSheet(sheetModel) {

            if (sheetModel.isActive()) {
                return setOptimalColumnWidth();
            }

            var promise = self.waitForEvent(docView, 'update:selection:data').then(setOptimalColumnWidth);
            docModel.setActiveSheet(sheetModel.getIndex());
            return promise;
        }

        /**
         * Sets optimal column width for all sheets in the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that resolves after setting the optimal column widths.
         */
        function setOptimalColumnWidthForDocument() {

            var activeSheet = docModel.getActiveSheet();

            var sheetIt = docModel.createSheetIterator({ cells: true });
            var promise = docModel.iterateSliced(sheetIt, setOptimalColumnWidthForSheet, 'CommandMixin.setOptimalColumnWidthForDocument');

            return promise.done(function () {
                docModel.setActiveSheet(activeSheet);
            });
        }

        function init() {
            var promise = self.getLaunchOption('autoWidth') ? setOptimalColumnWidthForDocument() : null;
            if (promise && (promise.state() === 'pending')) {
                docView.enterBusy({ delay: 0, immediate: true });
                docView.waitForAny(promise, function () { docView.leaveBusy(); });
            }
        }

        // initialization -----------------------------------------------------

        this.onInit(function () {
            docModel = self.getModel();
            docView = self.getView();
        });

        this.waitForImportSuccess(function () {
            docView.one('change:usedrange', function () {
                self.executeDelayed(init, 'CommandMixin.waitForImportSuccess', 1000);
            });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = docView = null;
        });

    } // class CommandMixin

    // exports ================================================================

    return CommandMixin;

});
