/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/sheetmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/editframework/model/viewattributesmixin',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/sheetselection',
    'io.ox/office/spreadsheet/model/namecollection',
    'io.ox/office/spreadsheet/model/colrowcollection',
    'io.ox/office/spreadsheet/model/mergecollection',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/model/hyperlinkcollection',
    'io.ox/office/spreadsheet/model/tablecollection',
    'io.ox/office/spreadsheet/model/validationcollection',
    'io.ox/office/spreadsheet/model/condformatcollection',
    'io.ox/office/spreadsheet/model/drawing/drawingcollection',
    'io.ox/office/spreadsheet/model/drawing/commentcollection',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, Rectangle, Color, AttributedModel, ViewAttributesMixin, Operations, SheetUtils, PaneUtils, SheetSelection, NameCollection, ColRowCollection, MergeCollection, CellCollection, HyperlinkCollection, TableCollection, ValidationCollection, CondFormatCollection, SheetDrawingCollection, CommentCollection, FormulaUtils, TokenArray) {

    'use strict';

    // convenience shortcuts
    var SheetType = SheetUtils.SheetType;
    var SplitMode = SheetUtils.SplitMode;
    var Address = SheetUtils.Address;
    var Interval = SheetUtils.Interval;
    var Range = SheetUtils.Range;
    var RangeArray = SheetUtils.RangeArray;
    var FormulaType = FormulaUtils.FormulaType;

    // additional fixed zoom factor depending on platform type
    var ZOOM_ADJUST_FACTOR = Utils.TOUCHDEVICE ? 1.25 : 1;

    // definitions for sheet view attributes
    var SHEET_VIEW_ATTRIBUTES = {

        /**
         * Sheet selection (cells and drawing objects), as instance of the
         * class SheetSelection. The array of cell range addresses MUST NOT be
         * empty.
         */
        selection: {
            def: SheetSelection.createFromAddress(Address.A1),
            validate: function (selection) {
                return (selection instanceof SheetSelection) ? selection : Utils.BREAK;
            },
            equals: function (selection1, selection2) {
                return selection1.equals(selection2);
            }
        },

        /**
         * Zoom factor in the sheet, as floating-point number. The value 1
         * represents a zoom factor of 100%.
         */
        zoom: {
            def: 1,
            validate: function (zoom) {
                return _.isNumber(zoom) ? PaneUtils.getValidZoom(zoom) : Utils.BREAK;
            }
        },

        /**
         * Whether to display the grid lines between all cells.
         */
        showGrid: {
            def: true,
            validate: function (show) { return !!show; }
        },

        /**
         * The type of the current view split. Has no effect, if both view
         * properties 'splitWidth' and 'splitHeight' are set to the value zero
         * (view is not split at all). Supports the following values:
         * - 'split': Dynamic split mode (movable split lines).
         * - 'frozen': Frozen split mode (a fixed number of columns and/or rows
         *      are frozen, split lines are not movable). When leaving frozen
         *      split mode, the split lines will be hidden.
         * - 'frozenSplit': Frozen split mode (same behavior as 'frozen' mode).
         *      When leaving frozen split mode, the split lines remain visible
         *      and become movable (back to 'split' mode).
         */
        splitMode: {
            def: SplitMode.SPLIT,
            validate: function (splitMode) { return (splitMode instanceof SplitMode) ? splitMode : Utils.BREAK; }
        },

        /**
         * The inner width of the left panes in view split mode. In dynamic
         * split mode, this value is interpreted in 1/100 of millimeters. In
         * frozen split mode, this value specifies the number of columns shown
         * in the frozen panes. Must not be negative. If this value is zero,
         * the view is not split into left and right panes (but may be split
         * into upper and lower panes).
         */
        splitWidth: {
            def: 0,
            validate: function (splitPos) {
                return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
            }
        },

        /**
         * The inner height of the upper panes in view split mode. In dynamic
         * split mode, this value is interpreted in 1/100 of millimeters. In
         * frozen split mode, this value specifies the number of rows shown in
         * the frozen panes. Must not be negative. If this value is zero, the
         * view is not split into top and bottom panes (but may be split into
         * left and right panes).
         */
        splitHeight: {
            def: 0,
            validate: function (splitPos) {
                return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
            }
        },

        /**
         * The identifier of the active pane in split mode. Must be one of
         * 'topLeft', 'bottomLeft', 'topRight', or 'bottomRight'.
         */
        activePane: {
            def: 'bottomRight',
            validate: /^(top|bottom)(Left|Right)$/
        },

        /**
         * The identifier of an entire active pane side. Must be one of the
         * values null, 'left', 'right', 'top', or 'bottom'. Used for example
         * while selecting entire columns or rows.
         */
        activePaneSide: {
            def: null,
            validate: function (paneSide) {
                return (/^(left|right|top|bottom)$/).test(paneSide) ? paneSide : null;
            }
        },

        /**
         * The horizontal scroll position in the left view panes, as
         * floating-point number. The integral part represents the zero-based
         * index of the first column visible in the panes. The fractional part
         * represents the ratio of the width of that column where the exact
         * scroll position (the first visible pixel) is located.
         */
        anchorLeft: {
            def: 0,
            validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
        },

        /**
         * The horizontal scroll position in the right view panes, as
         * floating-point number. The integral part represents the zero-based
         * index of the first column visible in the panes. The fractional part
         * represents the ratio of the width of that column where the exact
         * scroll position (the first visible pixel) is located.
         */
        anchorRight: {
            def: 0,
            validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
        },

        /**
         * The vertical scroll position in the upper view panes, as
         * floating-point number. The integral part represents the zero-based
         * index of the first row visible in the panes. The fractional part
         * represents the ratio of the height of that row where the exact
         * scroll position (the first visible pixel) is located.
         */
        anchorTop: {
            def: 0,
            validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
        },

        /**
         * The vertical scroll position in the lower view panes, as
         * floating-point number. The integral part represents the zero-based
         * index of the first row visible in the panes. The fractional part
         * represents the ratio of the height of that row where the exact
         * scroll position (the first visible pixel) is located.
         */
        anchorBottom: {
            def: 0,
            validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
        },

        /**
         * Specific cell ranges that are in a special activated state, used for
         * example for ranges that will be inserted into a formula while in
         * formula edit mode. These ranges will be rendered in a special way
         * over the regular cell selection (with a marching ants effect).
         * Allowed values are null (no active ranges available), or a sheet
         * selection (an instance of the class SheetSelection). The array of
         * selected drawings objects MUST be empty.
         */
        activeSelection: {
            def: null,
            validate: function (selection) {
                return (selection instanceof SheetSelection) ? selection : null;
            },
            equals: function (selection1, selection2) {
                return (!selection1 && !selection2) || (selection1 && selection2 && selection1.equals(selection2));
            }
        },

        /**
         * Additional data for the current cell range while auto-fill tracking
         * is active. Allowed values are null (auto-fill tracking not active),
         * or an object with the following properties:
         * - {Direction} direction
         *      The direction in which the selected range will be expanded.
         * - {Number} count
         *      The number of columns/rows to extend the selected range into
         *      the specified direction (positive values), or to shrink the
         *      selected range (negative values).
         * This attribute will be ignored, if the current sheet selection
         * consists of more than one cell range.
         */
        autoFillData: {
            def: null,
            validate: function (data) { return _.isObject(data) ? data : null; }
        }
    };

    // private global functions ===============================================

    /**
     * Returns the bounding range of the passed cell range addresses. Either of
     * the passed range addresses may be missing.
     *
     * @param {Range|Null} range1
     *  The first cell range address.
     *
     * @param {Range|Null} range2
     *  The second cell range address.
     *
     * @returns {Range|Null}
     *  The bounding range of the passed cell range addresses; or null, if both
     *  ranges are missing.
     */
    function getBoundRange(range1, range2) {
        return (range1 && range2) ? range1.boundary(range2) : (range1 || range2 || null);
    }

    // class SheetModel =======================================================

    /**
     * Represents a single sheet in the spreadsheet document.
     *
     * Additionally to the events triggered by the base class AttributedModel,
     * an instance of this class triggers the following events:
     * - 'change:viewattributes'
     *      After at least one view attribute has been changed. Event handlers
     *      receive an incomplete attribute map containing all changed view
     *      attributes with their new values.
     * - 'refresh:ranges'
     *      Triggered while invoking the method SheetModel.refreshRanges().
     *      Intended to initiate updating the visual representation of the
     *      notified cell ranges.
     * - 'insert:name', 'change:name', 'delete:name'
     *      The respective event forwarded from the collection of defined names
     *      of this sheet. See class NameCollection for details.
     * - 'insert:columns', 'change:columns', 'delete:columns'
     *      The respective event forwarded from the column collection of this
     *      sheet. See class ColRowCollection for details.
     * - 'insert:rows', 'delete:rows', 'change:rows'
     *      The respective event forwarded from the row collection of this
     *      sheet. See class ColRowCollection for details.
     * - 'insert:merged', 'delete:merged'
     *      The respective event forwarded from the merge collection of this
     *      sheet. See class MergeCollection for details.
     * - 'change:cells', 'move:cells'
     *      The respective event forwarded from the cell collection of this
     *      sheet. See class CellCollection for details.
     *   'insert:rule', 'delete:rule', 'change:rule'
     *      The respective event forwarded from the collection of conditional
     *      formattings of this sheet. See class CondFormatCollection for
     *      details.
     * - 'insert:table', 'change:table', 'delete:table'
     *      The respective event forwarded from the table collection of this
     *      sheet. See class TableCollection for details.
     * - 'insert:drawing', 'delete:drawing', 'move:drawing', 'change:drawing',
     *   'change:drawing:text'
     *      The respective event forwarded from the drawing collection of this
     *      sheet. See class SheetDrawingCollection for details.
     * - 'insert:comment', 'delete:comment', 'move:comment', 'change:comment',
     *   'change:comment:text'
     *      The respective event forwarded from the comment collection of this
     *      sheet. See class CommentCollection for details.
     * - 'change:usedrange'
     *      After the bounding range of this sheet has been changed, e.g. after
     *      inserting or removing cells or merged ranges. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Range|Null} usedRange
     *          The address of the bounding range of this sheet; or null, if
     *          this sheet is empty.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends ViewAttributesMixin
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this sheet.
     *
     * @param {SheetType} sheetType
     *  The type identifier of this sheet.
     *
     * @param {Object} [initialAttrs]
     *  An attribute set with initial formatting attributes for the sheet.
     */
    var SheetModel = AttributedModel.extend({ constructor: function (docModel, sheetType, initialAttrs) {

        // self reference
        var self = this;

        // the collection of defined names in this sheet
        var nameCollection = null;

        // the column and row collections (size and formatting of all columns/rows)
        var colCollection = null;
        var rowCollection = null;

        // the collection of merged cell ranges in this sheet
        var mergeCollection = null;

        // the collection of hyperlink ranges in this sheet
        var hyperlinkCollection = null;

        // the cell collection
        var cellCollection = null;

        // the collection of table ranges (filter, sorting)
        var tableCollection = null;

        // the collection of cell ranges with validation settings
        var validationCollection = null;

        // the collection of cell ranges with conditional formatting
        var condFormatCollection = null;

        // the collection of drawing objects contained in this sheet
        var drawingCollection = null;

        // the collection of comments contained in this sheet
        var commentCollection = null;

        // the effective zoom factor, according to the 'zoom' attribute, and platform type
        var originalZoom = 0;
        var effectiveZoom = 0;

        // base constructors --------------------------------------------------

        AttributedModel.call(this, docModel, initialAttrs, { trigger: 'always', families: 'sheet column row' });
        ViewAttributesMixin.call(this, SHEET_VIEW_ATTRIBUTES, { changeHandler: changeViewAttributesHandler });

        // private methods ----------------------------------------------------

        /**
         * Updates internal settings according to the current effective sheet
         * zoom factor.
         */
        function changeViewAttributesHandler(changedAttributes) {
            if ('zoom' in changedAttributes) {
                originalZoom = changedAttributes.zoom;
                effectiveZoom = originalZoom * ZOOM_ADJUST_FACTOR;
            }
        }

        /**
         * Expands all ranges but entire column/row ranges to the merged ranges
         * in the sheet.
         */
        function getExpandedRanges(ranges) {
            return RangeArray.map(ranges, function (range) {
                return (docModel.isColRange(range) || docModel.isRowRange(range)) ? range : mergeCollection.expandRangeToMergedRanges(range);
            });
        }

        /**
         * Returns the raw value of the specified split size view attribute.
         *
         * @param {Boolean} columns
         *  Whether to return the split width (true, value of the attribute
         *  'splitWidth'), or the split height (false, value of the attribute
         *  'splitHeight').
         *
         * @returns {Number}
         *  The raw value of the specified split size view attribute.
         */
        function getSplitSizeValue(columns) {
            return self.isCellType() ? self.getViewAttribute(columns ? 'splitWidth' : 'splitHeight') : 0;
        }

        /**
         * Returns the column/row interval shown in the leading panes of the
         * split view.
         *
         * @param {Boolean} columns
         *  Whether to return the column interval of the left sheet area
         *  (true), or the row interval of the top sheet area (false).
         *
         * @returns {Interval|Null}
         *  The column/row interval shown in the leading panes. If the view is
         *  not split in the specified direction, this method returns null.
         */
        function getSplitInterval(columns) {

            // the value of the view attribute (number of columns/rows in frozen split);
            // early exit without split
            var splitSize = getSplitSizeValue(columns);
            if (splitSize === 0) { return null; }

            // start scroll anchor in the leading sheet area
            var startAnchor = self.getViewAttribute(columns ? 'anchorLeft' : 'anchorTop');
            // the resulting column/row interval
            var interval = new Interval(Math.floor(startAnchor));

            // build column/row interval in frozen split mode directly
            if (self.hasFrozenSplit()) {
                interval.last = interval.first + splitSize - 1;
                return interval;
            }

            // the column/row collection
            var collection = columns ? colCollection : rowCollection;
            // end scroll anchor in the leading sheet area
            var endAnchor = collection.getScrollAnchorByOffset(collection.convertScrollAnchorToHmm(startAnchor) + splitSize);

            // build the resulting interval
            interval.last = Math.floor(endAnchor);

            // exclude last column/row, if visible less than half of its size
            if (!interval.single() && (endAnchor % 1 < 0.5)) {
                interval.last -= 1;
            }

            // exclude first column/row, if visible less than half of its size
            if (!interval.single() && (startAnchor % 1 > 0.5)) {
                interval.first += 1;
            }

            return interval;
        }

        /**
         * Returns the width or height of the leading area in a sheet in split
         * mode, in 1/100 of millimeters, or in pixels according to the current
         * sheet zoom factor. In frozen split mode, returns the total size of
         * the frozen columns/rows.
         *
         * @param {Boolean} columns
         *  Whether to return the width of the left sheet area (true), or the
         *  height of the top sheet area (false).
         *
         * @param {Boolean} pixel
         *  Whether to return pixels (true) or 1/100 of millimeters (false).
         *
         * @returns {Number}
         *  The width or height of the leading area if the sheet is in split
         *  mode, otherwise zero.
         */
        function getSplitSize(columns, pixel) {

            // return total size of frozen columns/rows
            if (self.hasFrozenSplit()) {
                var interval = getSplitInterval(columns);
                var collection = columns ? colCollection : rowCollection;
                return interval ? collection.getIntervalPosition(interval)[pixel ? 'size' : 'sizeHmm'] : 0;
            }

            // return size of dynamic split
            var splitSize = getSplitSizeValue(columns);
            return pixel ? self.convertHmmToPixel(splitSize) : splitSize;
        }

        /**
         * Updates the frozen split attributes, and notifies the listeners
         * after columns or rows have been inserted into or deleted from this
         * sheet.
         */
        function colRowOperationHandler(eventType, interval, insert, columns) {

            // attribute names for frozen split settings
            var LEADING_ANCHOR_NAME = columns ? 'anchorLeft' : 'anchorTop';
            var TRAILING_ANCHOR_NAME = columns ? 'anchorRight' : 'anchorBottom';
            var SPLIT_SIZE_NAME = columns ? 'splitWidth' : 'splitHeight';

            // the frozen column/row interval (will be null if not frozen)
            var frozenInterval = self.hasFrozenSplit() ? getSplitInterval(columns) : null;
            if (frozenInterval) {

                // the new view attributes
                var attributes = {};

                if (insert) {

                    // the number of inserted columns/rows
                    var insertSize = interval.size();

                    // inserted before frozen interval: update scroll anchor
                    if (interval.first < frozenInterval.first) {
                        attributes[LEADING_ANCHOR_NAME] = frozenInterval.first + insertSize;
                        attributes[TRAILING_ANCHOR_NAME] = frozenInterval.last + 1 + insertSize;

                    // inserted inside frozen interval: update size of frozen interval
                    } else if (interval.first <= frozenInterval.last) {
                        attributes[SPLIT_SIZE_NAME] = frozenInterval.size() + insertSize;
                        attributes[TRAILING_ANCHOR_NAME] = frozenInterval.last + 1 + insertSize;
                    }

                } else {

                    // deleted inside frozen interval: update size of frozen interval
                    var intersectInterval = interval.intersect(frozenInterval);
                    if (intersectInterval) {
                        attributes[SPLIT_SIZE_NAME] = frozenInterval.size() - intersectInterval.size();
                        // bug #55777: Prevent deleting last col/row
                        if (attributes[SPLIT_SIZE_NAME] === 0) { attributes[SPLIT_SIZE_NAME] = 1; }
                    }

                    // deleted before frozen interval: update leading scroll anchor
                    if (interval.first < frozenInterval.first) {
                        attributes[LEADING_ANCHOR_NAME] = interval.first;
                    }

                    // deleted before or inside frozen interval: update trailing scroll anchor
                    if (interval.first <= frozenInterval.last) {
                        attributes[TRAILING_ANCHOR_NAME] = interval.first;
                        // bug #55777: Prevent deleting last col/row
                        if (attributes[TRAILING_ANCHOR_NAME] === 0) { attributes[TRAILING_ANCHOR_NAME] = 1; }
                    }
                }

                // bug 46166: validate view settings when removing a frozen split
                if (attributes[SPLIT_SIZE_NAME] === 0) {

                    // refresh the active pane
                    var oldActivePane = self.getViewAttribute('activePane');
                    var newActivePane = PaneUtils.getNextPanePos(oldActivePane, columns ? 'right' : 'bottom');
                    if (oldActivePane !== newActivePane) {
                        attributes.activePane = newActivePane;
                    }

                    // reset frozen mode if a single frozen split has been removed (no split at all anymore)
                    if (!getSplitInterval(!columns)) {
                        attributes.splitMode = SplitMode.SPLIT;
                    }
                }

                // commit the new view attributes
                self.setViewAttributes(attributes);
            }

            // notify all listeners
            self.trigger(eventType, interval);
        }

        /**
         * Updates the selection after a drawing object has been inserted into
         * this sheet.
         */
        function insertDrawingHandler(event, drawingModel) {

            // the current sheet selection
            var selection = self.getViewAttribute('selection');
            if (selection.drawings.length === 0) { return; }

            // position of the drawing object
            var insertPos = drawingModel.getPosition();
            // the length of the position array
            var length = insertPos.length;
            // the index of the inserted drawing object in its parent collection
            var insIndex = _.last(insertPos);

            // create a deep copy of the current selection to be able to modify it
            selection = selection.clone();

            // update position of all drawings following the inserted drawing
            selection.drawings.forEach(function (selectPos) {

                // nothing to do, if the selected drawing is a parent object
                if (selectPos.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(insertPos, selectPos, length - 1) !== 0) { return; }

                // inserted drawing is a predecessor in the same parent
                if (insIndex <= selectPos[length - 1]) {
                    selectPos[length - 1] += 1;
                }
            });

            self.setViewAttribute('selection', selection);
        }

        /**
         * Updates the selection after a drawing object has been deleted from
         * this sheet.
         */
        function deleteDrawingHandler(event, drawingModel) {

            // the current sheet selection
            var selection = self.getViewAttribute('selection');
            if (selection.drawings.length === 0) { return; }

            // position of the drawing object
            var deletePos = drawingModel.getPosition();
            // the length of the position array
            var length = deletePos.length;
            // the index of the deleted drawing object in its parent collection
            var delIndex = _.last(deletePos);

            // create a deep copy of the current selection to be able to modify it
            selection = selection.clone();

            // update position of all drawings following the deleted drawing
            Utils.iterateArray(selection.drawings, function (selectPos, elemIndex) {

                // nothing to do, if the selected drawing is a parent object
                if (selectPos.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(deletePos, selectPos, length - 1) !== 0) { return; }

                // deleted drawing or any of its children has been deleted
                if (delIndex === selectPos[length - 1]) {
                    selection.drawings.splice(elemIndex, 1);
                // deleted drawing is a predecessor in the same parent
                } else if (delIndex < selectPos[length - 1]) {
                    selectPos[length - 1] -= 1;
                }
            }, { reverse: true });

            self.setViewAttribute('selection', selection);
        }

        /**
         * Updates the selection after a drawing object has been moved to
         * another index.
         */
        function moveDrawingHandler(event, drawingModel, fromIndex) {

            // the current sheet selection
            var selection = self.getViewAttribute('selection');
            if (selection.drawings.length === 0) { return; }

            // current position of the moved drawing object
            var currentPos = drawingModel.getPosition();
            // the length of the position array
            var length = currentPos.length;
            // the new index of the drawing object in its parent collection
            var toIndex = _.last(currentPos);

            // create a deep copy of the current selection to be able to modify it
            selection = selection.clone();

            // update position of all moved drawings (directly or indirectly)
            selection.drawings.forEach(function (selectPos) {

                // nothing to do, if the selected drawing is a parent object
                if (selectPos.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(currentPos, selectPos, length - 1) !== 0) { return; }

                // only the siblings of the moved drawing and their embedded drawings will be moved
                selectPos[length - 1] = Utils.transformArrayIndex(selectPos[length - 1], fromIndex, toIndex);
            });

            self.setViewAttribute('selection', selection);
        }

        // protected methods --------------------------------------------------

        /**
         * Post-processing of the sheet model, after all import operations have
         * been applied successfully.
         *
         * @internal
         *  Called from the application import process. MUST NOT be called from
         *  external code.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when this sheet model has been
         *  post-processed successfully; or rejected when any error has
         *  occurred.
         */
        this.postProcessImport = function () {
            return cellCollection.postProcessCells();
        };

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * settings from the passed source sheet model into this sheet model.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {SheetModel} sourceModel
         *  The source sheet model whose contents will be cloned into this
         *  sheet model.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, sourceModel) {

            // copy all formatting attributes of the sheet
            this.setAttributes(sourceModel.getExplicitAttributeSet(true));

            // clone the contents of all collections
            nameCollection.applyCopySheetOperation(context, sourceModel.getNameCollection());
            colCollection.applyCopySheetOperation(context, sourceModel.getColCollection());
            rowCollection.applyCopySheetOperation(context, sourceModel.getRowCollection());
            mergeCollection.applyCopySheetOperation(context, sourceModel.getMergeCollection());
            hyperlinkCollection.applyCopySheetOperation(context, sourceModel.getHyperlinkCollection());
            cellCollection.applyCopySheetOperation(context, sourceModel.getCellCollection());
            tableCollection.applyCopySheetOperation(context, sourceModel.getTableCollection());
            validationCollection.applyCopySheetOperation(context, sourceModel.getValidationCollection());
            condFormatCollection.applyCopySheetOperation(context, sourceModel.getCondFormatCollection());
            drawingCollection.applyCopySheetOperation(context, sourceModel.getDrawingCollection());
            commentCollection.applyCopySheetOperation(context, sourceModel.getCommentCollection());

            // set the view attributes after cloning all sheet contents (bug 50600: cloning
            // drawing objects would modify the drawing selection if already existing)
            this.setViewAttributes(sourceModel.getViewAttributes());
        };

        /**
         * Callback handler for the document operation 'changeSheet'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeSheet' document operation.
         */
        this.applyChangeSheetOperation = function (context) {
            this.setAttributes(context.getObj('attrs'));
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the type of this sheet.
         *
         * @returns {SheetType}
         *  The type identifier of this sheet.
         */
        this.getType = function () {
            return sheetType;
        };

        /**
         * Returns whether the type of this sheet is supported by the
         * spreadsheet application.
         *
         * @returns {Boolean}
         *  Whether the type of this sheet is supported.
         */
        this.isSupportedType = function () {
            return (sheetType === SheetType.WORKSHEET) || (sheetType === SheetType.CHARTSHEET);
        };

        /**
         * Returns whether this sheet is a worksheet.
         *
         * @returns {Boolean}
         *  Whether this sheet is a worksheet.
         */
        this.isWorksheet = function () {
            return sheetType === SheetType.WORKSHEET;
        };

        /**
         * Returns whether this sheet is a chart sheet.
         *
         * @returns {Boolean}
         *  Whether this sheet is a chart sheet.
         */
        this.isChartsheet = function () {
            return sheetType === SheetType.CHARTSHEET;
        };

        /**
         * Returns whether this sheet contains cells (either a worksheet, or a
         * macro sheet).
         *
         * @returns {Boolean}
         *  Whether this sheet contains cells.
         */
        this.isCellType = function () {
            return (sheetType === SheetType.WORKSHEET) || (sheetType === SheetType.MACROSHEET);
        };

        /**
         * Returns the current sheet index of this sheet model.
         *
         * @returns {Number}
         *  The current zero-based sheet index of this sheet model.
         */
        this.getIndex = function () {
            return docModel.getSheetIndexOfModel(this);
        };

        /**
         * Returns whether this sheet is the active sheet of the document.
         *
         * @returns {Boolean}
         *  Whether this sheet is the active sheet of the document.
         */
        this.isActive = function () {
            return this.getIndex() === docModel.getActiveSheet();
        };

        /**
         * Returns the name of this sheet model.
         *
         * @returns {String}
         *  The name of this sheet model.
         */
        this.getName = function () {
            return docModel.getSheetName(this.getIndex());
        };

        /**
         * Returns the collection of all defined names contained in this sheet.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this sheet.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        /**
         * Returns the collection of all columns in this sheet, containing
         * their size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all columns in this sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the collection of all rows in this sheet, containing their
         * size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all rows in this sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the collection of all merged cell ranges.
         *
         * @returns {MergeCollection}
         *  The collection of all merged cell ranges in this sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        /**
         * Returns the collection of all known cells in this sheet.
         *
         * @returns {CellCollection}
         *  The collection of all known cells in this sheet.
         */
        this.getCellCollection = function () {
            return cellCollection;
        };

        /**
         * Returns the collection of all known hyperlinks in this sheet.
         *
         * @returns {HyperlinkCollection}
         *  The collection of all known hyperlinks in this sheet.
         */
        this.getHyperlinkCollection = function () {
            return hyperlinkCollection;
        };

        /**
         * Returns the collection of all table ranges in this sheet.
         *
         * @returns {TableCollection}
         *  The collection of all table ranges in this sheet.
         */
        this.getTableCollection = function () {
            return tableCollection;
        };

        /**
         * Returns the collection of all cell ranges with validation settings.
         *
         * @returns {ValidationCollection}
         *  The collection of all cell ranges with validation settings in this
         *  sheet.
         */
        this.getValidationCollection = function () {
            return validationCollection;
        };

        /**
         * Returns the collection of all cell ranges with conditional
         * formatting settings.
         *
         * @returns {CondFormatCollection}
         *  The collection of all cell ranges with conditional formatting
         *  settings in this sheet.
         */
        this.getCondFormatCollection = function () {
            return condFormatCollection;
        };

        /**
         * Returns the drawing collection of this sheet, containing the models
         * of all drawing objects contained in this sheet.
         *
         * @returns {SheetDrawingCollection}
         *  The collection of all drawing models in this sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Returns the comment collection of this sheet, containing the models
         * of all cell comments contained in this sheet.
         *
         * @returns {CommentCollection}
         *  The collection of all cell comment models in this sheet.
         */
        this.getCommentCollection = function () {
            return commentCollection;
        };

        /**
         * Creates a new token array for cell formulas located in this sheet.
         *
         * @returns {TokenArray}
         *  A new token array for cell formulas located in this sheet.
         */
        this.createCellTokenArray = function () {
            return new TokenArray(this, FormulaType.CELL);
        };

        // dimensions ---------------------------------------------------------

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return Utils.convertLengthToHmm(length / effectiveZoom, 'px');
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return Utils.convertHmmToLength(length * effectiveZoom, 'px', 1);
        };

        /**
         * Returns the effective zoom factor to be used for this sheet,
         * according to the value of the 'zoom' view attribute, and the type of
         * the current device. On touch devices, the effective zoom factor will
         * be increased for better usability.
         *
         * @returns {Number}
         *  The effective zoom factor for this sheet.
         */
        this.getEffectiveZoom = function () {
            return effectiveZoom;
        };

        /**
         * Returns the maximum width of the digits in the font of the current
         * default cell style.
         *
         * @returns {Number}
         *  The maximum width of the digits in the font of the current default
         *  cell style, in pixels.
         */
        this.getDefaultDigitWidth = function () {
            return docModel.getDefaultDigitWidth(effectiveZoom);
        };

        /**
         * Returns the effective horizontal padding between cell grid lines and
         * the text contents of the cell, according to the current zoom factor
         * of this sheet.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the padding, as used in
         *  document operations.
         *
         * @returns {Number}
         *  The effective horizontal cell content padding, in pixels.
         */
        this.getTextPadding = function (charAttributes) {
            return docModel.getTextPadding(charAttributes, effectiveZoom);
        };

        /**
         * Returns the total size of all horizontal padding occupied in a cell
         * that cannot be used for the cell contents, according to the current
         * zoom factor of this sheet. This value includes the text padding
         * (twice, for left and right border), and additional space needed for
         * the grid lines.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the padding, as used in
         *  document operations.
         *
         * @returns {Number}
         *  The total size of the horizontal cell content padding, in pixels.
         */
        this.getTotalCellPadding = function (charAttributes) {
            return docModel.getTotalCellPadding(charAttributes, effectiveZoom);
        };

        /**
         * Calculates the effective scaled row height in pixels to be used for
         * the passed character attributes and the current zoom factor of this
         * sheet.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height, as
         *  used in document operations.
         *
         * @returns {Number}
         *  The row height for the passed character attributes, in pixels.
         */
        this.getRowHeight = function (charAttributes) {
            return docModel.getRowHeight(charAttributes, effectiveZoom);
        };

        /**
         * Returns the location of this sheet, either in 1/100 of millimeters,
         * or in pixels according to the current sheet zoom factor.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.pixel=false]
         *      If set to true, the location returned by this method will be in
         *      pixels according to the current sheet zoom factor. Otherwise,
         *      the location is in 1/100 of millimeters.
         *
         * @returns {Rectangle}
         *  The location of the entire sheet. The rectangle always starts at
         *  coordinates (0,0). Its size is equal to the total size of all
         *  visible columns and rows.
         */
        this.getSheetRectangle = function (options) {

            // whether to return pixels
            var pixel = Utils.getBooleanOption(options, 'pixel', false);

            return new Rectangle(0, 0,
                pixel ? colCollection.getTotalSize() : colCollection.getTotalSizeHmm(),
                pixel ? rowCollection.getTotalSize() : rowCollection.getTotalSizeHmm()
            );
        };

        /**
         * Returns the location of the passed cell range in the sheet, either
         * in 1/100 of millimeters, or in pixels according to the current sheet
         * zoom factor.
         *
         * @param {Range} range
         *  The cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.pixel=false]
         *      If set to true, the location returned by this method will be in
         *      pixels according to the current sheet zoom factor. Otherwise,
         *      the location is in 1/100 of millimeters.
         *
         * @returns {Rectangle}
         *  The location of the cell range in the sheet.
         */
        this.getRangeRectangle = function (range, options) {

            // whether to return pixels
            var pixel = Utils.getBooleanOption(options, 'pixel', false);
            // the position and size of the column interval
            var colPosition = colCollection.getIntervalPosition(range.colInterval());
            // the position and size of the row interval
            var rowPosition = rowCollection.getIntervalPosition(range.rowInterval());

            return new Rectangle(
                pixel ? colPosition.offset : colPosition.offsetHmm,
                pixel ? rowPosition.offset : rowPosition.offset.Hmm,
                pixel ? colPosition.size : colPosition.sizeHmm,
                pixel ? rowPosition.size : rowPosition.sizeHmm
            );
        };

        /**
         * Returns the location of the specified cell in the sheet, either in
         * 1/100 of millimeters, or in pixels according to the current sheet
         * zoom factor.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.pixel=false]
         *      If set to true, the location returned by this method will be in
         *      pixels according to the current sheet zoom factor. Otherwise,
         *      the location is in 1/100 of millimeters.
         *  - {Boolean} [options.expandMerged=false]
         *      If set to true, and the cell is part of a merged range, the
         *      location of the entire merged range will be returned.
         *
         * @returns {Rectangle}
         *  The location of the cell in the sheet.
         */
        this.getCellRectangle = function (address, options) {

            // try to extend the cell address to a merged range
            var expandMerged = Utils.getBooleanOption(options, 'expandMerged', false);
            var mergedRange = expandMerged ? mergeCollection.getMergedRange(address, 'all') : null;
            if (mergedRange) { return this.getRangeRectangle(mergedRange, options); }

            // whether to return pixels
            var pixel = Utils.getBooleanOption(options, 'pixel', false);
            // the position and size of the column
            var colEntry = colCollection.getEntry(address[0]);
            // the position and size of the row
            var rowEntry = rowCollection.getEntry(address[1]);

            return new Rectangle(
                pixel ? colEntry.offset : colEntry.offsetHmm,
                pixel ? rowEntry.offset : rowEntry.offsetHmm,
                pixel ? colEntry.size : colEntry.sizeHmm,
                pixel ? rowEntry.size : rowEntry.sizeHmm
            );
        };

        /**
         * Returns the address of the cell range covering the passed absolute
         * sheet rectangle.
         *
         * @param {Rectangle} rectangle
         *  The absolute position and size of the sheet rectangle, in pixels,
         *  or in 1/100 of millimeters, according to the 'pixel' option (see
         *  below).
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method ColRowCollection.getEntryByOffset(), especially the boolean
         *  option 'pixel' that causes to interpret the rectangle in pixels.
         *
         * @returns {Range}
         *  The address of the cell range covering the passed rectangle.
         */
        this.getRangeFromRectangle = function (rectangle, options) {

            // the column and row entries covering the start and end corner of the rectangle
            var colEntry1 = colCollection.getEntryByOffset(rectangle.left, options);
            var rowEntry1 = rowCollection.getEntryByOffset(rectangle.top, options);
            var colEntry2 = colCollection.getEntryByOffset(rectangle.right() - 1, options);
            var rowEntry2 = rowCollection.getEntryByOffset(rectangle.bottom() - 1, options);

            return Range.create(colEntry1.index, rowEntry1.index, colEntry2.index, rowEntry2.index);
        };

        /**
         * Returns the range address of the used area in the sheet (including
         * merged ranges).
         *
         * @returns {Range|Null}
         *  The range address of the used area in the sheet; or null, if all
         *  cell-based collections are empty.
         */
        this.getUsedRange = function () {
            var usedRange = cellCollection.getUsedRange();
            usedRange = getBoundRange(usedRange, mergeCollection.getUsedRange());
            return usedRange;
        };

        /**
         * Returns whether the passed cell range address consists of a single
         * cell, or whether it exactly covers a single merged range.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Boolean}
         *  Whether the cell is the only (merged) cell in the range.
         */
        this.isSingleCellInRange = function (range) {
            return range.single() || mergeCollection.isMergedRange(range);
        };

        /**
         * If the passed cell range address represents a single cell, or
         * exactly covers a merged range, the surrounding content range will be
         * returned. Otherwise, the passed range will be returned.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Range}
         *  The address of the content range of a single cell, otherwise the
         *  passed cell range address.
         */
        this.getContentRangeForCell = function (range) {
            return this.isSingleCellInRange(range) ? cellCollection.findContentRange(range) : range;
        };

        /**
         * Returns whether all cells in the passed cell ranges are hidden.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Boolean}
         *  Whether all cells in the passed cell ranges are hidden.
         */
        this.areRangesHidden = function (ranges) {
            return RangeArray.every(ranges, function (range) {
                return colCollection.isIntervalHidden(range.colInterval()) && rowCollection.isIntervalHidden(range.rowInterval());
            });
        };

        /**
         * Returns the visible parts of the passed cell range addresses.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {RangeArray}
         *  The visible parts of the passed cell ranges, in the same order. If
         *  a range is split into multiple parts (hidden columns or rows inside
         *  the range), the range parts are ordered row-by-row. If the entire
         *  range list is located in hidden columns or rows, the returned array
         *  will be empty.
         */
        this.getVisibleRanges = function (ranges) {
            return RangeArray.map(ranges, function (range) {
                var colIntervals = colCollection.getVisibleIntervals(range.colInterval());
                var rowIntervals = rowCollection.getVisibleIntervals(range.rowInterval());
                return RangeArray.createFromIntervals(colIntervals, rowIntervals);
            });
        };

        /**
         * Returns the address of a visible cell located as close as possible
         * to the passed cell address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {String} method
         *  Specifies how to look for another cell, if the specified cell is
         *  hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified cell is
         *      hidden, returns null.
         *  - 'next':
         *      Looks for a visible column and/or row following the cell.
         *  - 'prev':
         *      Looks for a visible column and/or row preceding the cell.
         *  - 'nextPrev':
         *      First, looks for visible column and/or row following the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *  - 'prevNext':
         *      First, looks for visible column and/or row preceding the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *
         * @returns {Address|Null}
         *  The address of a visible cell near the passed cell address; or null
         *  if all columns or all rows are hidden.
         */
        this.getVisibleCell = function (address, method) {

            // collection entries of visible column/row
            var colEntry = colCollection.getVisibleEntry(address[0], method);
            var rowEntry = rowCollection.getVisibleEntry(address[1], method);

            return (colEntry && rowEntry) ? new Address(colEntry.index, rowEntry.index) : null;
        };

        /**
         * Shrinks the passed range address to the visible area, if the leading
         * or trailing columns/rows of the range are hidden.
         *
         * @param {Range} range
         *  The address of the original cell range.
         *
         * @param {String} [method='both']
         *  If set to 'columns', only the columns of the passed range will be
         *  shrunken. If set to 'rows', only the rows of the passed range will
         *  be shrunken. By default, or if set to 'both', columns and rows will
         *  be shrunken.
         *
         * @returns {Range|Null}
         *  The address of the visible area of the passed cell range; or null,
         *  if the range is completely hidden. The first and last column, and
         *  the first and last row of the range are visible (unless a specific
         *  method has been passed to shrink columns or rows only), but some
         *  inner columns/rows in the resulting range may always be hidden.
         */
        this.shrinkRangeToVisible = function (range, method) {

            // shrunken visible column interval
            var colInterval = (method !== 'rows') ? colCollection.shrinkIntervalToVisible(range.colInterval()) : range.colInterval();
            // shrunken visible row interval
            var rowInterval = !colInterval ? null : (method !== 'columns') ? rowCollection.shrinkIntervalToVisible(range.rowInterval()) : range.rowInterval();

            return rowInterval ? Range.createFromIntervals(colInterval, rowInterval) : null;
        };

        /**
         * Returns the address of a range that covers the passed range, and
         * that has been expanded to all hidden columns and rows directly
         * preceding and following the passed range.
         *
         * @param {Range} range
         *  The address of a range to be expanded.
         *
         * @returns {Range}
         *  An expanded cell range address including all leading and trailing
         *  hidden columns and rows.
         */
        this.expandRangeToHidden = function (range) {

            // expanded column interval
            var colInterval = colCollection.expandIntervalToHidden(range.colInterval());
            // expanded row interval
            var rowInterval = rowCollection.expandIntervalToHidden(range.rowInterval());

            return Range.createFromIntervals(colInterval, rowInterval);
        };

        /**
         * Returns the addresses of the merged cell ranges covering the visible
         * parts of the passed ranges. First, the cell ranges will be expanded
         * to the leading and trailing hidden columns and rows. This may reduce
         * the number of ranges in the result, if there are only hidden columns
         * or rows between the passed ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {RangeArray}
         *  The addresses of the merged cell ranges, shrunken to leading and
         *  trailing visible columns and rows (but may contain inner hidden
         *  columns/rows).
         */
        this.mergeAndShrinkRanges = function (ranges) {

            // expand to hidden columns/rows (modify array in-place), merge the ranges
            // (result may be smaller due to expansion to hidden columns/rows)
            ranges = RangeArray.map(ranges, this.expandRangeToHidden, this).merge();

            // reduce merged ranges to visible parts (filter out ranges completely hidden)
            return RangeArray.map(ranges, this.shrinkRangeToVisible, this);
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the columns in
         * this sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the columns in this
         *  sheet.
         */
        this.hasColSplit = function () {
            return getSplitSizeValue(true) > 0;
        };

        /**
         * Returns whether the view split mode is enabled for the rows in this
         * sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the rows in this sheet.
         */
        this.hasRowSplit = function () {
            return getSplitSizeValue(false) > 0;
        };

        /**
         * Returns whether the view split mode is enabled for this sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the sheet.
         */
        this.hasSplit = function () {
            return this.hasColSplit() || this.hasRowSplit();
        };

        /**
         * Returns the current view split mode of this sheet.
         *
         * @returns {SplitMode|Null}
         *  The current view split mode. The value null indicates that the
         *  sheet is not split at all.
         */
        this.getSplitMode = function () {
            return this.hasSplit() ? this.getViewAttribute('splitMode') : null;
        };

        /**
         * Returns whether this sheet is currently split, and dynamic split
         * mode is activated. Returns also true, if the split is currently
         * frozen, but will thaw to dynamic split mode (FROZEN_SPLIT mode).
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (one of the split modes
         *  SPLIT or FROZEN_SPLIT).
         */
        this.hasDynamicSplit = function () {
            var splitMode = this.getSplitMode();
            return (splitMode === SplitMode.SPLIT) || (splitMode === SplitMode.FROZEN_SPLIT);
        };

        /**
         * Returns whether this sheet is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (one of the split modes
         *  FROZEN or FROZEN_SPLIT).
         */
        this.hasFrozenSplit = function () {
            var splitMode = this.getSplitMode();
            return (splitMode === SplitMode.FROZEN) || (splitMode === SplitMode.FROZEN_SPLIT);
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidthHmm = function () {
            return getSplitSize(true, false);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeightHmm = function () {
            return getSplitSize(false, false);
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidth = function () {
            return getSplitSize(true, true);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeight = function () {
            return getSplitSize(false, true);
        };

        /**
         * Returns the column interval shown in the left area of the sheet. If
         * the first or last visible columns are visible less than half of
         * their width, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible column instead, regardless how much of its width
         * is visible.
         *
         * @returns {Interval|Null}
         *  The column interval shown in the left area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitColInterval = function () {
            return getSplitInterval(true);
        };

        /**
         * Returns the row interval shown in the upper area of the sheet. If
         * the first or last visible rows are visible less than half of their
         * height, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible row instead, regardless how much of its height
         * is visible.
         *
         * @returns {Interval|Null}
         *  The row interval shown in the upper area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitRowInterval = function () {
            return getSplitInterval(false);
        };

        /**
         * Sets the split size for dynamic split mode, and updates the scroll
         * anchor view attributes, if the horizontal or vertical split will be
         * enabled.
         *
         * @param {Number} splitWidth
         *  The new value for the 'splitWidth' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between left and
         *  right sheet area.
         *
         * @param {Number} splitHeight
         *  The new value for the 'splitHeight' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between upper
         *  and lower sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (splitWidth, splitHeight, attributes) {

            // the old split size
            var oldSplitWidth = this.getSplitWidthHmm();
            var oldSplitHeight = this.getSplitHeightHmm();

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.clone(attributes) : {};
            _.extend(attributes, { splitMode: SplitMode.SPLIT, splitWidth: splitWidth, splitHeight: splitHeight });

            // Update left/right scroll positions. Copy from left to right, if
            // split has been disabled, or copy from right to left, if split
            // has been enabled.
            if ((oldSplitWidth > 0) && (attributes.splitWidth === 0)) {
                attributes.anchorRight = this.getViewAttribute('anchorLeft');
            } else if ((oldSplitWidth === 0) && (attributes.splitWidth > 0)) {
                attributes.anchorLeft = this.getViewAttribute('anchorRight');
            }

            // Update top/bottom scroll positions. Copy from top to bottom, if
            // split has been disabled, or copy from bottom to top, if split
            // has been enabled.
            if ((oldSplitHeight > 0) && (attributes.splitHeight === 0)) {
                attributes.anchorBottom = this.getViewAttribute('anchorTop');
            } else if ((oldSplitHeight === 0) && (attributes.splitHeight > 0)) {
                attributes.anchorTop = this.getViewAttribute('anchorBottom');
            }

            // set the new attributes
            this.setViewAttributes(attributes);
            return this;
        };

        /**
         * Sets the split size for frozen split mode, and updates the scroll
         * anchor view attributes.
         *
         * @param {Interval|Null} colInterval
         *  If set to an index interval, the new column interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  left and right sheet area.
         *
         * @param {Interval|Null} rowInterval
         *  If set to an index interval, the new row interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  top and bottom sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (colInterval, rowInterval, attributes) {

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.clone(attributes) : {};
            attributes.splitMode = this.hasDynamicSplit() ? SplitMode.FROZEN_SPLIT : SplitMode.FROZEN;

            // set column interval if specified
            if (colInterval) {
                attributes.splitWidth = colInterval.size();
                attributes.anchorLeft = colInterval.first;
                attributes.anchorRight = colInterval.last + 1;
            } else {
                attributes.splitWidth = 0;
            }

            // set row interval if specified
            if (rowInterval) {
                attributes.splitHeight = rowInterval.size();
                attributes.anchorTop = rowInterval.first;
                attributes.anchorBottom = rowInterval.last + 1;
            } else {
                attributes.splitHeight = 0;
            }

            // set the new attributes
            this.setViewAttributes(attributes);
            return this;
        };

        /**
         * Resets all split attributes, effectively disabling any split view
         * that is currently active.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.clearSplit = function () {
            return this.setDynamicSplit(0, 0);
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether this sheet is visible in the user interface.
         *
         * @returns {Boolean}
         *  Whether this sheet is visible.
         */
        this.isVisible = function () {
            return this.getMergedAttributeSet(true).sheet.visible;
        };

        /**
         * Returns whether this sheet is locked. In locked sheets it is not
         * possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether this sheet is locked.
         */
        this.isLocked = function () {
            return !this.isWorksheet() || this.getMergedAttributeSet(true).sheet.locked;
        };

        /**
         * Returns the current raw zoom factor of this sheet (the original
         * value of the view attribute 'zoom'). To get the effective zoom
         * factor that will be used to render the cell contents, see the method
         * SheetModel.getEffectiveZoom().
         *
         * @returns {Number}
         *  The current raw zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         */
        this.getZoom = function () {
            return originalZoom;
        };

        /**
         * Changes the current raw zoom factor of this sheet (the value of the
         * view attribute 'zoom').
         *
         * @param {Number} zoom
         *  The new raw zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setZoom = function (zoom) {
            this.setViewAttribute('zoom', zoom);
            return this;
        };

        /**
         * Returns the current grid color of this sheet.
         *
         * @returns {Color}
         *  The current grid color of this sheet, as instance of the class
         *  Color.
         */
        this.getGridColor = function () {
            var gridColor = Color.parseJSON(this.getMergedAttributeSet(true).sheet.gridColor);
            return gridColor ? gridColor : new Color('auto');
        };

        /**
         * On first invocation, initializes the view attributes for this sheet
         * from the imported sheet view attributes.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.initializeViewAttributes = _.once(_.bind(function () {

            // the merged sheet attributes
            var sheetAttributes = self.getMergedAttributeSet(true).sheet;
            // all selected ranges (convert plain JSON data to instance of RangeArray)
            var selectedRanges = docModel.parseJSONRanges(sheetAttributes.selectedRanges);
            // the array index of the active range (bug 34549: filter may send invalid index)
            var activeIndex = Utils.getIntegerOption(sheetAttributes, 'activeIndex', 0);
            // address of the active cell (convert plain JSON data to instance of Address)
            var activeCell = docModel.parseJSONAddress(sheetAttributes.activeCell);
            // the array of selected drawing objects
            var selectedDrawings =  Utils.getArrayOption(sheetAttributes, 'selectedDrawings', []);
            // selection, built from multiple sheet attributes, expanded to merged ranges
            var selection = (selectedRanges && (selectedRanges.length > 0)) ? new SheetSelection(getExpandedRanges(selectedRanges), activeIndex, activeCell, selectedDrawings) : null;
            // the resulting view attributes
            var viewAttributes = {
                zoom: sheetAttributes.zoom,
                showGrid: sheetAttributes.showGrid,
                splitMode: SplitMode.parse(sheetAttributes.splitMode, SplitMode.SPLIT),
                splitWidth: sheetAttributes.splitWidth,
                splitHeight: sheetAttributes.splitHeight,
                activePane: sheetAttributes.activePane,
                anchorLeft: sheetAttributes.scrollLeft,
                anchorRight: sheetAttributes.scrollRight,
                anchorTop: sheetAttributes.scrollTop,
                anchorBottom: sheetAttributes.scrollBottom
            };

            // show warnings for invalid selection settings (bug 34549)
            if (!selection) {
                Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', invalid selection');
                selection = SheetSelection.createFromAddress(Address.A1);
                selection.ranges = getExpandedRanges(selection.ranges);
            } else {
                if (selection.active !== sheetAttributes.activeIndex) {
                    Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', invalid range index in selection');
                }
                if (!selection.activeRange().containsAddress(selection.address)) {
                    Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', invalid active cell in selection');
                    selection.address = selection.activeRange().start.clone();
                }
            }
            viewAttributes.selection = selection;

            // Sheet view attributes in operations use the top-left grid pane as default pane
            // which is always visible without active view split. Internally, the view uses the
            // bottom-right grid pane as the default pane instead. According to the split settings,
            // the 'activePane' attribute, and the scroll attributes have to be translated.

            // no horizontal split: map all left pane settings to right pane settings
            if (sheetAttributes.splitWidth === 0) {
                viewAttributes.activePane = PaneUtils.getNextPanePos(viewAttributes.activePane, 'right');
                viewAttributes.anchorRight = viewAttributes.anchorLeft;
                delete viewAttributes.anchorLeft;
            }

            // no vertical split: map all top pane settings to bottom pane settings
            if (sheetAttributes.splitHeight === 0) {
                viewAttributes.activePane = PaneUtils.getNextPanePos(viewAttributes.activePane, 'bottom');
                viewAttributes.anchorBottom = viewAttributes.anchorTop;
                delete viewAttributes.anchorTop;
            }

            this.setViewAttributes(viewAttributes);
            return this;
        }, this));

        /**
         * Returns all changed sheet attributes as used in 'changeSheet'
         * document operations according to the current runtime view settings.
         *
         * @returns {Object}
         *  All changed sheet attributes matching the current view settings of
         *  this sheet.
         */
        this.getChangedViewAttributes = function () {

            // the current merged sheet attributes
            var oldSheetAttributes = this.getMergedAttributeSet(true).sheet;
            // the current view settings
            var viewAttributes = this.getViewAttributes();
            // the new sheet attributes according to the view settings
            var newSheetAttributes = {
                selectedRanges: viewAttributes.selection.ranges.toJSON(),
                activeIndex: viewAttributes.selection.active,
                activeCell: viewAttributes.selection.address.toJSON(),
                selectedDrawings: viewAttributes.selection.drawings,
                zoom: viewAttributes.zoom,
                showGrid: viewAttributes.showGrid,
                splitWidth: viewAttributes.splitWidth,
                splitHeight: viewAttributes.splitHeight,
                activePane: viewAttributes.activePane,
                scrollLeft: getScrollIndex(viewAttributes.anchorLeft, true),
                scrollRight: getScrollIndex(viewAttributes.anchorRight, true),
                scrollTop: getScrollIndex(viewAttributes.anchorTop, false),
                scrollBottom: getScrollIndex(viewAttributes.anchorBottom, false)
            };
            // whether a horizontal (left/right) split is active
            var hSplit = viewAttributes.splitWidth > 0;
            // whether a vertical (top/bottom) split is active
            var vSplit = viewAttributes.splitHeight > 0;

            // converts the passed scroll anchor to a simple column/row index
            function getScrollIndex(scrollAnchor, columns) {
                return Math.min(Math.round(scrollAnchor), docModel.getMaxIndex(columns));
            }

            // Sheet view attributes in operations use the top-left grid pane as default pane
            // which is always visible without active view split. Internally, the view uses the
            // bottom-right grid pane as the default pane instead. According to the split settings,
            // the 'activePane' attribute, and the scroll attributes have to be translated.

            // no horizontal split: map all right pane settings to left pane settings
            if (!hSplit) {
                newSheetAttributes.activePane = PaneUtils.getNextPanePos(newSheetAttributes.activePane, 'left');
                newSheetAttributes.scrollLeft = newSheetAttributes.scrollRight;
                delete newSheetAttributes.scrollRight;
            }

            // no vertical split: map all bottom pane settings to top pane settings
            if (!vSplit) {
                newSheetAttributes.activePane = PaneUtils.getNextPanePos(newSheetAttributes.activePane, 'top');
                newSheetAttributes.scrollTop = newSheetAttributes.scrollBottom;
                delete newSheetAttributes.scrollBottom;
            }

            // reset frozen split mode, if all splits are gone
            var splitMode = (hSplit || vSplit) ? viewAttributes.splitMode : SplitMode.SPLIT;
            newSheetAttributes.splitMode = splitMode.toJSON();

            // bug 46166: in frozen split mode, the scrollable pane is always active regardless of view settings
            if (splitMode !== SplitMode.SPLIT) {
                newSheetAttributes.activePane = !hSplit ? 'bottomLeft' : !vSplit ? 'topRight' : 'bottomRight';
            }

            // reduce new sheet attribute to changed ones
            _.each(newSheetAttributes, function (value, name) {
                if (_.isEqual(oldSheetAttributes[name], value)) {
                    delete newSheetAttributes[name];
                }
            });

            return newSheetAttributes;
        };

        /**
         * Returns whether this sheet contains changed view attributes compared
         * to the sheet view attributes imported from the document file.
         *
         * @returns {Boolean}
         *  Whether this sheet contains changed view attributes.
         */
        this.hasChangedViewAttributes = function () {

            // the new sheet attributes according to the view settings
            var sheetAttributes = this.getChangedViewAttributes();

            // delete 'unimportant' view attributes (selection, scroll positions)
            delete sheetAttributes.selectedRanges;
            delete sheetAttributes.activeIndex;
            delete sheetAttributes.activeCell;
            delete sheetAttributes.selectedDrawings;
            delete sheetAttributes.activePane;
            delete sheetAttributes.scrollLeft;
            delete sheetAttributes.scrollRight;
            delete sheetAttributes.scrollTop;
            delete sheetAttributes.scrollBottom;

            // return whether there are changed view attributes left
            return !_.isEmpty(sheetAttributes);
        };

        /**
         * Fires a 'refresh:ranges' event to the listeners of this sheet model,
         * intended to initiate updating the visual representation of the
         * specified ranges. Can be used by model instances that want to
         * refresh the display of specific sheet ranges without generating any
         * document operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.refreshRanges = function (ranges) {
            ranges = RangeArray.get(ranges).merge();
            // notify listeners, if the ranges are not empty
            if (!ranges.empty()) { this.trigger('refresh:ranges', ranges); }
            return this;
        };

        /**
         * Returns whether the contents of the passed cell ranges can be
         * restored completely by undo operations after deleting them.
         *
         * @param {RangeArray} ranges
         *  The addresses of the cell ranges to be deleted.
         *
         * @returns {Boolean}
         *  Whether the cell ranges can be deleted safely.
         */
        this.canRestoreDeletedRanges = function (ranges) {
            // bug 55208: deleting tables cannot be undone
            // bug 56368: check for unsupported drawings
            return tableCollection.canRestoreDeletedRanges(ranges) && drawingCollection.canRestoreDeletedRanges(ranges);
        };

        // operation generators -----------------------------------------------

        /**
         * Creates a new empty operations generator, that has been initialized
         * for this sheet.
         *
         * @param {Object} [options]
         *  Optional parameters that will be passed to the constructor of the
         *  operations generator.
         *
         * @returns {SheetOperationGenerator}
         *  A new empty operations generator for this sheet.
         */
        this.createOperationGenerator = function (options) {
            var generator = docModel.createSheetOperationGenerator(options);
            generator.setSheetIndex(this.getIndex());
            return generator;
        };

        /**
         * Creates a new operations generator associated to this sheet, invokes
         * the callback function, applies all operations contained in the
         * generator, and sends them to the server.
         *
         * @param {Function} callback
         *  The callback function. See description of the public method
         *  SpreadsheetModel.createAndApplyOperations() for details. Will be
         *  called in the context of this sheet model.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  SpreadsheetModel.createAndApplyOperations().
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operations have been
         *  applied and sent successfully (with the result of the callback
         *  function); or rejected, if the callback has returned a rejected
         *  promise (with the result of that promise), or if sending the
         *  operations has failed for internal reasons (with an object
         *  containing a property 'cause' set to 'operation').
         */
        this.createAndApplyOperations = function (callback, options) {
            return docModel.createAndApplySheetOperations(function (generator) {
                generator.setSheetIndex(self.getIndex());
                return callback.call(self, generator);
            }, options);
        };

        /**
         * Generates the operations, and the undo operations, to change the
         * formatting attributes of this sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} attributeSet
         *  The sheet attribute set to be inserted into the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated successfully.
         */
        this.generateChangeSheetOperations = function (generator, attributeSet) {
            generator.generateSheetOperation(Operations.CHANGE_SHEET, { attrs: this.getUndoAttributeSet(attributeSet) }, { undo: true });
            generator.generateSheetOperation(Operations.CHANGE_SHEET, { attrs: attributeSet });
            return this.createResolvedPromise(); // always successful
        };

        /**
         * Generates the undo operation to restore the complete contents of
         * this sheet after it has been deleted.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated successfully.
         */
        this.generateRestoreOperations = function (/*generator*/) {
            // TODO
            return this.createResolvedPromise();
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the contents of this sheet before cells will be moved (especially
         * deleted).
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {MoveDescriptor} moveDescs
         *  An array of move descriptors that specify how to transform the cell
         * collection and its associated contents.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateBeforeMoveCellsOperations = SheetUtils.profileAsyncMethod('SheetModel.generateBeforeMoveCellsOperations()', function (generator, moveDescs) {
            return Utils.invokeChainedAsync(
                function () { return mergeCollection.generateBeforeMoveCellsOperations(generator, moveDescs); },
                function () { return hyperlinkCollection.generateBeforeMoveCellsOperations(generator, moveDescs); },
                function () { return tableCollection.generateBeforeMoveCellsOperations(generator, moveDescs); },
                function () { return drawingCollection.generateBeforeMoveCellsOperations(generator, moveDescs); },
                function () { return commentCollection.generateBeforeMoveCellsOperations(generator, moveDescs); }
            );
        });

        /**
         * Generates the operations and undo operations to update or restore
         * the formulas of all cells, defined names, data validations,
         * formatting rules, and drawing objects in this sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @param {String} [drawingMode]
         *  If set to the value "only", restricts generation of document
         *  operations to the drawing collections (regular drawing objects as
         *  well as cell comments). If set to the value "skip", restricts
         *  generation of document operations to all sheet collections but
         *  drawing collections. If this option is omitted, all collections of
         *  this sheet will be processed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateFormulaOperations = SheetUtils.profileAsyncMethod('SheetModel.generateUpdateFormulaOperations()', function (generator, updateDesc, drawingMode) {

            var drawingsOnly = drawingMode === 'only';
            var skipDrawings = drawingMode === 'skip';

            return Utils.invokeChainedAsync(
                drawingsOnly ? null : function () { return cellCollection.generateUpdateFormulaOperations(generator, updateDesc); },
                drawingsOnly ? null : function () { return nameCollection.generateUpdateFormulaOperations(generator, updateDesc); },
                drawingsOnly ? null : function () { return validationCollection.generateUpdateFormulaOperations(generator, updateDesc); },
                drawingsOnly ? null : function () { return condFormatCollection.generateUpdateFormulaOperations(generator, updateDesc); },
                skipDrawings ? null : function () { return drawingCollection.generateUpdateFormulaOperations(generator, updateDesc); },
                skipDrawings ? null : function () { return commentCollection.generateUpdateFormulaOperations(generator, updateDesc); }
            );
        });

        /**
         * Generates the operations and undo operations to update and restore
         * the inactive anchor attributes of the top-level drawing objects,
         * after the size or visibility of the columns or rows in the sheet has
         * been changed.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateRefreshAnchorOperations = SheetUtils.profileAsyncMethod('SheetModel.generateRefreshAnchorOperations()', function (generator) {
            return Utils.invokeChainedAsync(
                function () { return drawingCollection.generateRefreshAnchorOperations(generator); },
                function () { return commentCollection.generateRefreshAnchorOperations(generator); }
            );
        });

        // initialization -----------------------------------------------------

        // create the collection instances (constructors want to access public methods of this sheet model)
        nameCollection = new NameCollection(this);
        colCollection = new ColRowCollection(this, true);
        rowCollection = new ColRowCollection(this, false);
        mergeCollection = new MergeCollection(this);
        cellCollection = new CellCollection(this);
        hyperlinkCollection = new HyperlinkCollection(this);
        tableCollection = new TableCollection(this);
        validationCollection = new ValidationCollection(this);
        condFormatCollection = new CondFormatCollection(this);
        drawingCollection = new SheetDrawingCollection(this);
        commentCollection = new CommentCollection(this);

        // invoke all registered transformation handlers, after inserting
        // or deleting columns, and forward all column collection events
        colCollection.on({
            'insert:entries': function (event, interval) { colRowOperationHandler('insert:columns', interval, true, true); },
            'delete:entries': function (event, interval) { colRowOperationHandler('delete:columns', interval, false, true); },
            'change:entries': function (event, interval, changeInfo) { self.trigger('change:columns', interval, changeInfo); }
        });

        // invoke all registered transformation handlers, after inserting
        // or deleting rows, and forward all row collection events
        rowCollection.on({
            'insert:entries': function (event, interval) { colRowOperationHandler('insert:rows', interval, true, false); },
            'delete:entries': function (event, interval) { colRowOperationHandler('delete:rows', interval, false, false); },
            'change:entries': function (event, interval, changeInfo) { self.trigger('change:rows', interval, changeInfo); }
        });

        // update cell selection after inserting or deleting merged ranges, and forward the event
        mergeCollection.on('triggered', function () {
            var selection = self.getViewAttribute('selection').clone();
            selection.ranges = getExpandedRanges(selection.ranges);
            self.setViewAttribute('selection', selection);
        });

        // send the position of the entire used area of the sheet
        cellCollection.on('change:usedrange', function () {
            self.trigger('change:usedrange', self.getUsedRange());
        });
        mergeCollection.on('change:usedrange', function () {
            self.trigger('change:usedrange', self.getUsedRange());
        });

        // update drawing selection after inserting or deleting drawing objects, forward events of the collection
        drawingCollection.on({
            'insert:drawing': insertDrawingHandler,
            'delete:drawing': deleteDrawingHandler,
            'move:drawing': moveDrawingHandler
        });

        // forward events of the child collections to own listeners
        this.forwardEvents(nameCollection);
        this.forwardEvents(mergeCollection, { except: 'change:usedrange' });
        this.forwardEvents(cellCollection, { except: 'change:usedrange' });
        this.forwardEvents(hyperlinkCollection);
        this.forwardEvents(condFormatCollection);
        this.forwardEvents(tableCollection);
        this.forwardEvents(drawingCollection);

        // special handling for comment collection
        commentCollection.on({
            'insert:drawing': function (event, commentModel) { self.trigger('insert:comment', commentModel); },
            'delete:drawing': function (event, commentModel) { self.trigger('delete:comment', commentModel); },
            'change:drawing': function (event, commentModel) { self.trigger('change:comment', commentModel); },
            'change:drawing:text': function (event, commentModel) { self.trigger('change:comment:text', commentModel); },
            'move:drawing': function (event, commentModel, oldAnchor) { self.trigger('move:comment', commentModel, oldAnchor); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            commentCollection.destroy();
            drawingCollection.destroy();
            condFormatCollection.destroy();
            validationCollection.destroy();
            tableCollection.destroy();
            cellCollection.destroy();
            hyperlinkCollection.destroy();
            mergeCollection.destroy();
            colCollection.destroy();
            rowCollection.destroy();
            nameCollection.destroy();
            self = docModel = null;
            nameCollection = colCollection = rowCollection = mergeCollection = cellCollection = hyperlinkCollection = null;
            tableCollection = validationCollection = condFormatCollection = drawingCollection = commentCollection = null;
        });

    } }); // class SheetModel

    // exports ================================================================

    return SheetModel;

});
