/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/operationcontext', [
    'io.ox/office/editframework/model/operationcontext',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (OperationContext, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var AddressArray = SheetUtils.AddressArray;

    // class SheetOperationContext ============================================

    /**
     * A small wrapper for a JSON document operation object providing useful
     * helper methods, used as calling context for operation handler callback
     * functions of spreadsheet documents.
     *
     * @constructor
     *
     * @extends OperationContext
     */
    var SheetOperationContext = OperationContext.extend(function (docModel, operation, external, importing) {
        OperationContext.call(this, docModel, operation, external, importing);
        this._maxAddress = docModel.getMaxAddress();
    }); // class SheetOperationContext

    // public methods ---------------------------------------------------------

    /**
     * Parses a cell address from a string in A1 notation.
     *
     * @param {string} parseStr
     *  The string to be parsed.
     *
     * @returns {Address}
     *  The parsed cell address.
     *
     * @throws {OperationError}
     *  If the passed string cannot be parsed to a valid cell address.
     */
    SheetOperationContext.prototype.parseAddress = function (parseStr) {
        var address = Address.parse(parseStr, this._maxAddress);
        this.ensure(address, 'invalid cell address %s', parseStr);
        return address;
    };

    /**
     * Returns the value of a required cell address property (as string in A1
     * notation) of the wrapped JSON operation.
     *
     * @param {string} propName
     *  The name of the operation property.
     *
     * @returns {Address}
     *  The cell address of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid cell address property
     *  with the specified name.
     */
    SheetOperationContext.prototype.getAddress = function (propName) {
        return this.parseAddress(this.getStr(propName));
    };

    /**
     * Returns the value of an optional cell address property (as string in A1
     * notation) of the wrapped JSON operation.
     *
     * @param {string} propName
     *  The name of the operation property.
     *
     * @returns {Address|null}
     *  The cell address of the specified operation property; or null, if the
     *  property is missing.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid cell address.
     */
    SheetOperationContext.prototype.getOptAddress = function (propName) {
        return this.has(propName) ? this.getAddress(propName) : null;
    };

    /**
     * Parses a cell address list from a string of space-separated addresses in
     * A1 notation).
     *
     * @param {string} parseStr
     *  The string to be parsed.
     *
     * @returns {AddressArray}
     *  The list of cell addresses.
     *
     * @throws {OperationError}
     *  If the passed string cannot be parsed to a valid cell address list.
     */
    SheetOperationContext.prototype.parseAddressList = function (parseStr) {
        return AddressArray.map(parseStr.split(/\s+/), this.parseAddress, this);
    };

    /**
     * Returns the value of a required cell address list property (as string of
     * space-separated addresses in A1 notation) of the wrapped JSON operation.
     *
     * @param {string} propName
     *  The name of the operation property.
     *
     * @returns {AddressArray}
     *  The list of cell addresses of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid cell address list
     *  property with the specified name.
     */
    SheetOperationContext.prototype.getAddressList = function (propName) {
        return this.parseAddressList(this.getStr(propName));
    };

    /**
     * Returns the value of an optional cell address list property (as string
     * of space-separated addresses in A1 notation) of the wrapped JSON
     * operation.
     *
     * @param {string} propName
     *  The name of the operation property.
     *
     * @returns {AddressArray|null}
     *  The list of cell addresses of the specified operation property; or
     *  null, if the property is missing.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid cell address list.
     */
    SheetOperationContext.prototype.getOptAddressList = function (propName) {
        return this.has(propName) ? this.getAddressList(propName) : null;
    };

    /**
     * Returns the value of a required column/row interval property of the
     * wrapped JSON operation. The property must be an object with the property
     * 'start' containing a valid column/row index (non-negative, less than or
     * equal to the maximum index of this document). Optionally, the object may
     * contain a property 'end' containing a valid index (non-negative, equal
     * to or greater than 'start', less than or equal to the maximum index of
     * this document).
     *
     * @param {String} [propName]
     *  The name of the operation property. If omitted, the column/row interval
     *  will be extracted directly from the operation properties 'start' and
     *  'end' (the latter assumed to be optional).
     *
     * @param {Boolean} columns
     *  Whether to return a column interval (true), or a row interval (false).
     *
     * @returns {Interval}
     *  The column/row interval of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid column/row interval
     *  property with the specified name.
     */
    SheetOperationContext.prototype.getJSONInterval = function (propName, columns) {
        if (_.isBoolean(propName)) { columns = propName; propName = null; }
        this.ensure(_.isBoolean(columns), 'missing columns flag');
        var interval = this.docModel.parseJSONInterval(propName ? this.getObj(propName) : this.operation, columns, true);
        this.ensure(interval, 'invalid %s interval', columns ? 'column' : 'row');
        return interval;
    };

    /**
     * Returns the value of an optional column/row interval property of the
     * wrapped JSON operation.
     *
     * @param {String} [propName]
     *  The name of the operation property. If omitted, the column/row interval
     *  will be extracted directly from the operation properties 'start' and
     *  'end' (the latter assumed to be optional).
     *
     * @returns {Interval|Null}
     *  The column/row interval of the specified operation property; or null,
     *  if the property is missing.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid column/row interval.
     */
    SheetOperationContext.prototype.getOptJSONInterval = function (propName, columns) {
        return this.has(_.isBoolean(propName) ? 'start' : propName) ? this.getJSONInterval(propName, columns) : null;
    };

    /**
     * Returns the value of a required cell address property of the wrapped
     * JSON operation. See method SpreadsheetModel.parseJSONAddress() for
     * details about the format of valid cell addresses.
     *
     * @param {String} [propName='start']
     *  The name of the operation property. If omitted, the standard property
     *  name 'start' will be used.
     *
     * @returns {Address}
     *  The cell address of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid cell address property
     *  with the specified name.
     */
    SheetOperationContext.prototype.getJSONAddress = function (propName) {
        var address = this.docModel.parseJSONAddress(this.getPos(propName || 'start'));
        this.ensure(address, 'invalid cell address');
        return address;
    };

    /**
     * Returns the value of an optional cell address property of the wrapped
     * JSON operation. See method SpreadsheetModel.parseJSONAddress() for
     * details about the format of valid cell addresses.
     *
     * @param {String} [propName='start']
     *  The name of the operation property. If omitted, the standard property
     *  name 'start' will be used.
     *
     * @returns {Address|Null}
     *  The cell address of the specified operation property; or null, if the
     *  property is missing.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid cell address.
     */
    SheetOperationContext.prototype.getOptJSONAddress = function (propName) {
        return this.has(propName || 'start') ? this.getJSONAddress(propName) : null;
    };

    /**
     * Returns the value of a required cell range address property of the
     * wrapped JSON operation. See method SpreadsheetModel.parseJSONRange() for
     * details about the format of valid cell range addresses.
     *
     * @param {String} [propName]
     *  The name of the operation property. If omitted, the cell range address
     *  will be extracted directly from the operation properties 'start' and
     *  'end' (the latter assumed to be optional).
     *
     * @returns {Range}
     *  The cell range address of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid cell range address
     *  property with the specified name.
     */
    SheetOperationContext.prototype.getJSONRange = function (propName) {
        var range = this.docModel.parseJSONRange(propName ? this.getObj(propName) : this.operation);
        this.ensure(range, 'invalid cell range address');
        return range;
    };

    /**
     * Returns the value of an optional cell range address property of the
     * wrapped JSON operation. See method SpreadsheetModel.parseJSONRange() for
     * details about the format of valid cell range addresses.
     *
     * @param {String} [propName]
     *  The name of the operation property. If omitted, the cell range address
     *  will be extracted directly from the operation properties 'start' and
     *  'end' (the latter assumed to be optional).
     *
     * @returns {Range|Null}
     *  The cell range address of the specified operation property; or null, if
     *  the property is missing.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid cell range address.
     */
    SheetOperationContext.prototype.getOptJSONRange = function (propName) {
        return this.has(propName || 'start') ? this.getJSONRange(propName) : null;
    };

    /**
     * Returns the value of a required cell range address array property of the
     * wrapped JSON operation. See method SpreadsheetModel.parseJSONRanges()
     * for details about the format of valid arrays of cell range addresses.
     *
     * @param {String} [propName='ranges']
     *  The name of the operation property. If omitted, the standard property
     *  name 'ranges' will be used.
     *
     * @returns {RangeArray}
     *  The array of cell range addresses of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid cell range address
     *  array property with the specified name.
     */
    SheetOperationContext.prototype.getJSONRangeList = function (propName) {
        var ranges = this.docModel.parseJSONRanges(this.getArr(propName || 'ranges'));
        this.ensure(ranges, 'invalid cell range address array');
        return ranges;
    };

    /**
     * Returns the value of an optional cell range address array property of
     * the wrapped JSON operation. See method SpreadsheetModel.parseJSONRanges()
     * for details about the format of valid cell range addresses.
     *
     * @param {String} [propName='ranges']
     *  The name of the operation property. If omitted, the standard property
     *  name 'ranges' will be used.
     *
     * @returns {RangeArray|Null}
     *  The array of cell range addresses of the specified operation property;
     *  or null, if the property is missing.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid array of cell range addresses.
     */
    SheetOperationContext.prototype.getOptJSONRangeList = function (propName) {
        return this.has(propName || 'ranges') ? this.getJSONRangeList(propName) : null;
    };

    // exports ================================================================

    return SheetOperationContext;

});
