/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/sheetref', function () {

    'use strict';

    // class SheetRef =========================================================

    /**
     * A structure representing the sheet reference part in a range reference
     * (e.g. in Sheet1!A1:B2), or in a defined name (e.g. Sheet1!name).
     *
     * @constructor
     *
     * @property {Number|String} sheet
     *  The zero-based sheet index, if a sheet with the passed name exists in
     *  the document; or -1 for a broken sheet reference; otherwise the sheet
     *  name as string.
     *
     * @property {Boolean} abs
     *  Whether the sheet reference is marked to be absolute.
     */
    function SheetRef(sheet, abs) {

        this.sheet = sheet;
        this.abs = abs;

    } // class SheetRef

    // static methods ---------------------------------------------------------

    /**
     * Creates a sheet reference structure for the passed simple or complex
     * sheet name. At most one of the passed parameters can be a string, the
     * other parameters must be null or undefined.
     *
     * @param {SpreadsheetDocument} docModel
     *  The spreadsheet document used to look up the sheet indexes for the
     *  passed sheet names.
     *
     * @param {String|Null} simpleSheet
     *  A simple sheet name without any special characters that would cause to
     *  enclose the sheet name in apostrophes.
     *
     * @param {String|Null} complexSheet
     *  A complex sheet name with special characters that requires the sheet
     *  name to be enclosed in apostrophes.
     *
     * @param {String|Null} refError
     *  The error code of a reference error, used a sheet has been deleted,
     *  e.g. in the formula =#REF!!A1.
     *
     * @param {Boolean} abs
     *  Whether the sheet reference will be marked to be absolute.
     *
     * @returns {SheetRef|Null}
     *  A sheet reference structure, if one of the passed parameters is a
     *  string, otherwise null.
     */
    SheetRef.create = function (docModel, simpleSheet, complexSheet, refError, abs) {

        var sheetIndex = null;

        // try to find the passed simple sheet name in the document
        if (_.isString(simpleSheet)) {
            sheetIndex = docModel.getSheetIndex(simpleSheet);
            return new SheetRef((sheetIndex >= 0) ? sheetIndex : simpleSheet, abs);
        }

        // try to find the passed complex sheet name in the document
        if (_.isString(complexSheet)) {
            complexSheet = complexSheet.replace(/''/g, '\'');
            sheetIndex = docModel.getSheetIndex(complexSheet);
            return new SheetRef((sheetIndex >= 0) ? sheetIndex : complexSheet, abs);
        }

        // create a sheet reference object for the #REF! error code
        if (_.isString(refError)) {
            return new SheetRef(-1, abs);
        }

        // no sheet names: sheet reference does not exist
        return null;
    };

    /**
     * Encloses the passed complex sheet name in apostrophes, and duplicates
     * all inner apostrophes contained in the sheet name.
     *
     * @param {String} sheetName
     *  A complex sheet name.
     *
     * @returns {String}
     *  The encoded complex sheet name.
     */
    SheetRef.encodeComplexSheetName = function (sheetName) {
        return '\'' + sheetName.replace(/'/g, '\'\'') + '\'';
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a clone of this sheet reference structure.
     *
     * @returns {SheetRef}
     *  A clone of this sheet reference structure.
     */
    SheetRef.prototype.clone = function () {
        return new SheetRef(this.sheet, this.abs);
    };

    /**
     * Returns whether this sheet reference points to an existing sheet.
     *
     * @returns {Boolean}
     *  Whether this sheet reference points to an existing sheet.
     */
    SheetRef.prototype.valid = function () {
        return (typeof this.sheet === 'number') && (this.sheet >= 0);
    };

    /**
     * Returns whether this sheet reference and the passed sheet reference are
     * equal.
     *
     * @param {SheetRef} sheetRef
     *  The other sheet reference to be compared with this sheet reference.
     *
     * @returns {Boolean}
     *  Whether this sheet reference and the passed sheet reference are equal.
     */
    SheetRef.prototype.equals = function (sheetRef) {
        return (this.abs === sheetRef.abs) && (this.sheet === sheetRef.sheet);
    };

    /**
     * Returns the original name of the sheet referred by this sheet reference.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model needed to resolve sheet names.
     *
     * @returns {String|Null}
     *  The original name of the sheet referred by this sheet reference; or
     *  null, if this sheet reference cannot be resolved to a sheet name.
     */
    SheetRef.prototype.resolve = function (docModel, extDocRef) {
        return (typeof this.sheet !== 'number') ? this.sheet : extDocRef ? null : docModel.getSheetName(this.sheet);
    };

    /**
     * Tries to replace an unresolved sheet name ('sheet' property is a string)
     * with an existing sheet index in the passed spreadsheet document.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model contaiing all existing sheets.
     *
     * @returns {Boolean}
     *  Whether this sheet reference has been changed, i.e. whether it has been
     *  refreshed from an unresolved sheet name ('sheet' property as string) to
     *  an existing sheet index ('sheet' property as integer).
     */
    SheetRef.prototype.refresh = function (docModel) {
        if (typeof this.sheet === 'string') {
            var sheetIndex = docModel.getSheetIndex(this.sheet);
            if (sheetIndex >= 0) {
                this.sheet = sheetIndex;
                return true;
            }
        }
        return false;
    };

    /**
     * Transforms the sheet index in this sheet reference, after a sheet has
     * been inserted, deleted, or moved in the document.
     *
     * @param {Number|Null} fromSheet
     *  The old index of the deleted or moved sheet; or null for inserted
     *  sheets.
     *
     * @param {Number|Null} toSheet
     *  The new index of the inserted or moved sheet; or null for deleted
     *  sheets.
     *
     * @returns {Boolean}
     *  Whether the sheet index in this sheet reference has been modified.
     */
    SheetRef.prototype.transform = function (fromSheet, toSheet) {
        // TODO: also for relative sheet references (ODF)?

        // invalid sheet name in reference, or nothing changes: return false
        if (!this.valid() || (toSheet === fromSheet)) {
            return false;
        }

        // sheet inserted
        if (typeof fromSheet !== 'number') {
            // referenced sheet follows inserted sheet
            if (toSheet <= this.sheet) { this.sheet += 1; return true; }

        // sheet deleted
        } else if (typeof toSheet !== 'number') {
            // referenced sheet follows deleted sheet
            if (fromSheet < this.sheet) { this.sheet -= 1; return true; }
            // referenced sheet has been deleted
            if (fromSheet === this.sheet) { this.sheet = -1; return true; }

        // sheet moved
        } else {
            // referenced sheet moved to another position
            if (fromSheet === this.sheet) { this.sheet = toSheet; return true; }
            // sheet moved backwards, referenced sheet follows
            if ((toSheet < fromSheet) && (toSheet <= this.sheet) && (this.sheet < fromSheet)) { this.sheet += 1; return true; }
            // sheet moved forwards, referenced sheet follows
            if ((fromSheet < toSheet) && (fromSheet < this.sheet) && (this.sheet <= toSheet)) { this.sheet -= 1; return true; }
        }

        return false;
    };

    /**
     * Returns a text description of this sheet reference for debug logging.
     */
    SheetRef.prototype.toString = function () {
        return (this.sheet === -1) ? '#REF' : ((this.abs ? '$' : '') + (_.isString(this.sheet) ? SheetRef.encodeComplexSheetName(this.sheet) : this.sheet));
    };

    // exports ================================================================

    return SheetRef;

});
