/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/formulagrammar', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/parser/formularesource',
    'io.ox/office/spreadsheet/model/formula/parser/referencegrammar'
], function (Utils, LocaleData, Formatter, BaseObject, SheetUtils, FormulaUtils, FormulaResource, ReferenceGrammar) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var MathUtils = FormulaUtils.Math;
    var Scalar = FormulaUtils.Scalar;
    var CellRef = FormulaUtils.CellRef;
    var SheetRef = FormulaUtils.SheetRef;

    // settings for all supported formula grammars, mapped by grammar identifiers
    var GRAMMAR_CONFIG = {
        op: { localized: false, locale: LocaleData.LOCALE },
        ui: { localized: true,  locale: LocaleData.LOCALE },
        en: { localized: true,  locale: 'en_US' }
    };

    // prefix for user-defined macro function in the OOXML file format
    var UDF_PREFIX = '_xludf.';

    // character ranges (without brackets) for a whitespace or other non-printable character
    var WHITESPACE_CHARS = '\\s\\x00-\\x1f\\x80-\\x9f';

    // private global functions ===============================================

    /**
     * Returns a regular expression that matches strings in R1C1 notation with
     * optional absolute row and column parts.
     *
     * @param {String} prefixChars
     *  The actual prefix characters for the row and column part. MUST be a
     *  string with exactly two characters (row and column).
     *
     * @returns {RegExp}
     *  A regular expression that matches absolute references in R1C1 notation.
     *  Either part may be missing completely (references to entire columns or
     *  rows), or the prefix character may be present without index. Matching
     *  strings are for example 'R2C3', 'R2C', 'RC3', 'RC', 'R2', 'C3', 'R',
     *  'C', or the empty string (!).
     */
    var getSimpleRCRegExp = _.memoize(function (prefixChars) {
        return new RegExp('^(?:' + prefixChars[0] + '(\\d*))?(?:' + prefixChars[1] + '(\\d*))?$', 'i');
    });

    /**
     * Returns whether the passed text is a simple R1C1 reference for the
     * passed prefix characters (either absolute column/row index not
     * enclosed in brackets, or missing column/row index).
     *
     * @param {SpreadsheetModel} docModel
     *  The document model needed to check the maximum column/row index of
     *  sheet names that look like cell addresses.
     *
     * @param {String} prefixChars
     *  The actual prefix characters for the row and column part. MUST be a
     *  string with exactly two characters (row and column).
     *
     * @param {String} text
     *  The string to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed text is a simple R1C1 reference for the passed
     *  prefix characters.
     */
    function isSimpleRCReference(docModel, prefixChars, text) {

        // do not accept empty strings (would be matched by the regular expression)
        if (text.length === 0) { return false; }

        // check that text matches the regular expression
        var matches = getSimpleRCRegExp(prefixChars).exec(text);
        if (!matches) { return false; }

        // extract column and row index (row before column in R1C1 notation!)
        var row = matches[1] ? (parseInt(matches[1], 10) - 1) : 0;
        var col = matches[2] ? (parseInt(matches[2], 10) - 1) : 0;

        // column and row index must be valid in the document
        return docModel.isValidAddress(new Address(col, row));
    }

    // class FormulaGrammar ===================================================

    /**
     * An instance of this class represents all settings of a specific formula
     * grammar and file format.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @property {String} GRAMMAR
     *  The identifier of the formula grammar, as passed to the constructor.
     *
     * @property {Boolean} UI
     *  Whether the formula grammar is used to handle formula expressions in
     *  the user interface.
     *
     * @property {String} REF_SYNTAX
     *  The identifier of the syntax of cell references:
     *  - 'ooxml' for the syntax used in native and localized OOXML grammars
     *      (references in A1 or R1C1 style, exclamation mark as sheet name
     *      separator, sheets are always absolute, apostrophes enclose entire
     *      range of sheet names, e.g.: ='Sheet1:Sheet 2'!A1:B2).
     *  - 'odfui' for the syntax used in the localized ODF grammar (references
     *      in A1 or R1C1 style, exclamation mark as sheet name separator,
     *      sheets can be absolute or relative, apostrophes enclose single
     *      sheet names in a range of sheets, e.g.: =Sheet1:$'Sheet 2'!A1:B2).
     *  - 'of' for the native OpenFormula syntax (entire reference enclosed in
     *      brackets, references in A1 style, row or column index can be #REF!
     *      errors, period as sheet name separator, sheets can be absolute or
     *      relative, apostrophes enclose single sheet names in a range of
     *      sheets, e.g.: =[Sheet1.A1:$'Sheet 2'.B#REF!]).
     *
     * @property {Boolean} RC_STYLE
     *  Whether this instance supports cell references in A1 notation (false),
     *  or in R1C1 notation (true).
     *
     * @property {String} DEC
     *  The decimal separator character.
     *
     * @property {String} SEP
     *  The separator character in function parameter lists.
     *
     * @property {String} MAT_OPEN
     *  The leading character of a matrix literal.
     *
     * @property {String} MAT_ROW
     *  The separator character between rows in a matrix literal.
     *
     * @property {String} MAT_COL
     *  The separator character between row elements in a matrix literal.
     *
     * @property {String} MAT_CLOSE
     *  The trailing character of a matrix literal.
     *
     * @property {String} TABLE_SEP
     *  The list seperator used inside table references.
     *
     * @property {String} PREFIX_CHARS
     *  The prefix characters used in cell references in R1C1 notation (this
     *  property exists always regardless of the reference mode passed to the
     *  constructor of this instance).
     *
     * @property {Object} RE
     *  A map with regular expressions needed to parse formula expressions.
     */
    var FormulaGrammar = BaseObject.extend({ constructor: function (grammarId, fileFormat, rcStyle) {

        // self reference
        var self = this;

        // special behavior depending on file format
        var odf = fileFormat === 'odf';

        // static configuration settings for the passed formula grammar
        var grammarConfig = GRAMMAR_CONFIG[grammarId];

        // the formula resource data according to file format and grammar
        var formulaResource = FormulaResource.create(fileFormat, grammarConfig.locale);

        // whether to used localized strings
        var localized = grammarConfig.localized;

        // the collection of boolean descriptors
        var booleanCollection = formulaResource.getBooleanCollection();

        // the collection of error code descriptors
        var errorCollection = formulaResource.getErrorCollection();

        // the collection of table region descriptors
        var regionCollection = formulaResource.getRegionCollection();

        // the collection of operator descriptors
        var operatorCollection = formulaResource.getOperatorCollection();

        // the collection of function descriptors
        var functionCollection = formulaResource.getFunctionCollection();

        // specific separator/delimiter characters used in regular expressions
        var DEC = formulaResource.getDec(localized);
        var GROUP = formulaResource.getGroup(localized);
        var SEP = formulaResource.getSeparator(localized);
        var TABLE_SEP = SEP;

        // specific error codes used in regular expressions, or when generating text representations
        var REF_ERROR = errorCollection.getName(ErrorCode.REF.key, localized);
        var NA_ERROR = errorCollection.getName(ErrorCode.NA.key, localized);

        // a simple number formatter to generate the string representation of numbers
        var formatter = new Formatter(BaseObject.SINGLETON, { dec: DEC, group: GROUP });

        // reference syntax identifier
        var refSyntax = !odf ? 'ooxml' : localized ? 'odfui' : 'of';

        // prefix characters for R1C1 notation
        var prefixChars = formulaResource.getRCPrefixChars(localized);

        // reference syntax configuration
        var refGrammar = ReferenceGrammar.create(formulaResource, localized, rcStyle);
        var sheetConfig = refGrammar.sheetConfig;
        var sheetPatterns = sheetConfig.PATTERNS;

        // Character ranges (without brackets) for characters that need to be escaped in table column names.
        var TABLE_COL_ESCAPE_CHARS = '\\[\\]\'#' + (localized ? '@' : '');

        // Character ranges (without brackets) for characters of complex table column names. The comma
        // will be a complex table character, if it is also used as separator in the current grammar.
        var TABLE_COL_COMPLEX_CHARS = '\\x00-\\x2B\\x2D-\\x2F\\x3A-\\x40\\x5B-\\x60\\x7B\\x7D\\x7E' + ((SEP === ',') ? ',' : '');

        // RE pattern for a simple column name in a structured table reference.
        // - Group 1: The column name.
        var TABLE_SIMPLE_COL_NAME_PATTERN = '([^' + TABLE_COL_COMPLEX_CHARS + ']+)';

        // RE pattern for a complex column name in a structured table reference.
        // - Group 1: The column name.
        var TABLE_COL_NAME_PATTERN = '((?:[^' + TABLE_COL_ESCAPE_CHARS + ']|\'.)+)';

        // RE pattern for a simple column name without brackets, or any column name with brackets in a
        // structured table reference, e.g. in the formula =Table1[@Col1:[Col 2]]).
        // - Group 1: The simple column name found without brackets.
        // - Group 2: The complex column name found inside brackets, without the brackets.
        var TABLE_MIXED_COL_NAME_PATTERN = '(?:' + TABLE_SIMPLE_COL_NAME_PATTERN + '|\\[' + TABLE_COL_NAME_PATTERN + '\\])';

        // RE pattern for optional whitespace characters in a structured table reference
        // - Group 1: The whitespace characters.
        var TABLE_OPTIONAL_WS_PATTERN = '([' + WHITESPACE_CHARS + ']*)';

        // RE pattern for whitespace characters following a token in a structured table reference. Will only be recognized
        // if specific control characters that are valid inside a tbale reference will follow.
        // - Group 1: The whitespace characters.
        var TABLE_TRAILING_WS_PATTERN = '(?:([' + WHITESPACE_CHARS + ']+)(?=\\[|\\]|' + _.escapeRegExp(TABLE_SEP) + (localized ? '|@' : '') + '|$))?';

        // Matches a valid label for a defined name.
        var VALID_NAME_RE = new RegExp('^' + sheetPatterns.DEFINED_NAME + '$', 'i');

        // Matches all characters in table column names that need to be escaped with an apostrophe.
        var TABLE_COL_ESCAPE_RE = new RegExp('[' + TABLE_COL_ESCAPE_CHARS + ']', 'g');

        // Matches a simple table column name that may appear without brackets in a complex structured table reference.
        var SIMPLE_TABLE_COLUMN_NAME_RE = new RegExp('^[^' + TABLE_COL_COMPLEX_CHARS + ']+$', 'i');

        // base constructor ---------------------------------------------------

        BaseObject.call(this, BaseObject.SINGLETON);

        // public properties --------------------------------------------------

        // the grammar identifier
        this.GRAMMAR = grammarId;

        // the user interface mode
        this.UI = localized;

        // the reference syntax and style identifiers
        this.REF_SYNTAX = refSyntax;
        this.RC_STYLE = rcStyle;

        // reference syntax configuration
        this.refGrammar = refGrammar;

        // separator characters
        this.DEC = DEC;
        this.SEP = SEP;
        this.MAT_OPEN = '{';
        this.MAT_ROW = (odf || localized) ? '|' : ';';
        this.MAT_COL = (odf || localized) ? ';' : ',';
        this.MAT_CLOSE = '}';
        this.TABLE_SEP = TABLE_SEP;
        this.PREFIX_CHARS = prefixChars;

        // map of all generic regular expressions needed to parse formula expressions
        this.RE = {

            // Any whitespace or other non-printable characters.
            WHITESPACE: new RegExp('^[' + WHITESPACE_CHARS + ']+'),

            // String literals: Enclosed in double quotes, embedded double quotes
            // are represented by a sequence of two double quote characters.
            STRING_LIT: /^(".*?")+/,

            // Boolean literals: Must not be followed by any character valid inside an identifier.
            // This prevents to match 'trueA' or 'false_1' or similar name tokens, as well as TRUE()
            // and FALSE() function calls, or references with sheet names such as 'TRUE!A1'.
            BOOLEAN_LIT: new RegExp('^' + booleanCollection.getAllPattern(localized) + sheetPatterns.TERMINATE_REF, 'i'),

            // Error code literals (only supported error codes, not any string
            // starting with a hash character).
            ERROR_LIT: new RegExp('^' + errorCollection.getAllPattern(localized) + '(?!' + sheetPatterns.NAME_INNER_CHAR_CLASS + ')', 'i'),

            // Number literals as fraction with integral part, e.g. '1 2/3'.
            // - Group 1: The leading integer (always non-negative).
            // - Group 2: The numerator (always non-negative).
            // - Group 3: The denominator (always positive).
            FRACTION_LIT: /^(\d+) (\d+)\/(0*[1-9]\d*)/,

            // Opening parenthesis of a matrix literal.
            MATRIX_OPEN: new RegExp('^' + _.escapeRegExp(this.MAT_OPEN)),

            // The column separator in a matrix literal.
            MATRIX_COL_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_COL)),

            // The row separator in a matrix literal.
            MATRIX_ROW_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_ROW)),

            // Closing parenthesis of a matrix literal.
            MATRIX_CLOSE: new RegExp('^' + _.escapeRegExp(this.MAT_CLOSE)),

            // The list/parameter separator character.
            SEPARATOR: new RegExp('^' + _.escapeRegExp(SEP)),

            // All unary and binary operators.
            // - Group 1: The name of the matched operator.
            OPERATOR: new RegExp('^' + operatorCollection.getAllPattern(localized)),

            // The opening parenthesis.
            OPEN: /^\(/,

            // The closing parenthesis.
            CLOSE: /^\)/,

            // The opening bracket of a structured table reference.
            // Group 1: The whitespace characters following the opening bracket.
            TABLE_OPEN: new RegExp('^\\[' + TABLE_TRAILING_WS_PATTERN),

            // The closing bracket of a structured table reference.
            // Group 1: The whitespace characters preceding the closing bracket.
            // Group 2: The closing bracket, or an empty string if the bracket is missing at the end of the formula.
            TABLE_CLOSE: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + '(\\]|$)'),

            // Any whitespace or other non-printable characters inside a table reference.
            // Group 1: The whitespace characters preceding the separator.
            // Group 2: The whitespace characters following the separator.
            TABLE_SEPARATOR: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + _.escapeRegExp(TABLE_SEP) + TABLE_TRAILING_WS_PATTERN),

            // A single predefined table region in structured table references (without brackets), followed
            // by a closing bracket, e.g. =Table[#All].
            // - Group 1: The name of the table region, with heading hash character.
            TABLE_SINGLE_REGION_NAME: new RegExp('^' + regionCollection.getAllPattern(localized) + '(?=\\])', 'i'),

            // A single column name (also complex names) in structured table references (without brackets),
            // followed by a closing bracket, e.g. =Table[Col'#2].
            // - Group 1: The column name.
            TABLE_SINGLE_COLUMN_NAME: new RegExp('^' + TABLE_COL_NAME_PATTERN + '(?=\\])', 'i'),

            // A simple column name (no special characters) in structured table references that can be used
            // without brackets in a complex table reference with separators and column ranges.
            // - Group 1: The column name.
            TABLE_SIMPLE_COLUMN_NAME: new RegExp('^' + TABLE_SIMPLE_COL_NAME_PATTERN, 'i'),

            // A predefined table region in structured table references, enclosed in brackets.
            // - Group 1: The name of the table region, without brackets, with heading hash character.
            TABLE_REGION_REF: new RegExp('^\\[' + regionCollection.getAllPattern(localized) + '\\]', 'i'),

            // A column reference in structured table references, enclosed in brackets.
            // - Group 1: The column name, without brackets.
            TABLE_COLUMN_REF: new RegExp('^\\[' + TABLE_COL_NAME_PATTERN + '\\]', 'i'),

            // A column reference in a structured table reference. The column name may occur without brackets
            // (simple column names only), or inside brackets (any column name). Example: the table reference
            // Table1[@Col1] contains the simple column name 'Col1', but the table reference Table1[@[Col 2]]
            // contains the complex column name 'Col 2' in brackets.
            // - Group 1: Leading whitespace characters before the column name.
            // - Group 2: The simple column name found without brackets (undefined for complex).
            // - Group 3: The complex column name without the brackets (undefined for simple).
            TABLE_MIXED_COLUMN_REF: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + TABLE_MIXED_COL_NAME_PATTERN, 'i'),

            // A column range in structured table references, each column name enclosed in brackets.
            // - Group 1: The first column name, without brackets.
            // - Group 2: The second column name, without brackets.
            TABLE_COLUMN_RANGE_REF: new RegExp('^\\[' + TABLE_COL_NAME_PATTERN + '\\]:\\[' + TABLE_COL_NAME_PATTERN + '\\]', 'i'),

            // A column range in a structured table reference. Each column name may occur without brackets
            // (simple column names only), or inside brackets (any column name). Example: the table reference
            // Table1[@Col1:[Col 2]] contains a column range with the simple column name 'Col1', and the
            // complex column name 'Col 2' in brackets.
            // - Group 1: Leading whitespace characters before the column range.
            // - Group 2: The simple name of the first column found without brackets (undefined for complex).
            // - Group 3: The complex name of the first column without the brackets (undefined for simple).
            // - Group 4: The simple name of the second column found without brackets (undefined for complex).
            // - Group 5: The complex name of the second column without the brackets (undefined for simple).
            TABLE_MIXED_COLUMN_RANGE_REF: new RegExp('^' + TABLE_OPTIONAL_WS_PATTERN + TABLE_MIXED_COL_NAME_PATTERN + ':' + TABLE_MIXED_COL_NAME_PATTERN, 'i'),

            // An at-sign, used in UI grammars as shortcut for [#This Row] in structured table references.
            TABLE_THISROW_REF: /^@/
        };

        FormulaUtils.info('FormulaGrammar(): grammar=' + grammarId + ' config=', this);

        // private methods ----------------------------------------------------

        /**
         * Returns whether the passed text is a boolean literal according to
         * this formula grammar.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is a boolean literal.
         */
        function isBooleanLiteral(text) {
            return booleanCollection.getKey(text, localized) !== null;
        }

        /**
         * Returns whether the passed text is the representation of a relative
         * cell reference in A1 notation, or a cell reference in R1C1 notation
         * (either English, e.g. 'R1C1', or according to this formula grammar,
         * e.g. 'Z1S1' in the German UI grammar).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is the representation of a  cell reference
         *  in A1 or R1C1 notation.
         */
        function isReferenceSymbol(docModel, text) {

            // cell address in A1 notation
            var address = Address.parse(text);
            if (address && docModel.isValidAddress(address)) { return true; }

            // cell address in native R1C1 notation (regardless of the grammar)
            var nativePrefixChars = formulaResource.getRCPrefixChars(false);
            if (isSimpleRCReference(docModel, nativePrefixChars, text)) { return true; }

            // localized R1C1 references (e.g. the German S1Z1)
            if ((nativePrefixChars !== prefixChars) && isSimpleRCReference(docModel, prefixChars, text)) { return true; }

            // the passed text is not a cell reference
            return false;
        }

        /**
         * Encodes the passed sheet name, if it is a complex name, and adds an
         * absolute marker if needed.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to recognize simple sheet names.
         *
         * @param {String} sheetName
         *  The sheet name.
         *
         * @param {Boolean} abs
         *  Whether the sheet reference is absolute.
         *
         * @returns {String}
         *  The encoded sheet name. Complex sheet names will be enclosed in
         *  apostrophes. If the absolute flag has been set, and the sheet name
         *  is not empty, a dollar sign will be added before the (encoded)
         *  sheet name.
         */
        function encodeAbsSheetName(docModel, sheetName, abs) {

            // use the #REF! error code for invalid sheets, enclose complex sheet names in apostrophes
            if (sheetName && !self.isSimpleSheetName(docModel, sheetName)) {
                sheetName = SheetRef.encodeComplexSheetName(sheetName);
            }

            // add the leading absolute marker in ODF mode
            return ((abs && sheetName) ? '$' : '') + sheetName;
        }

        /**
         * Returns the string representation of the passed sheet references for
         * regular token syntax, i.e. as range of sheet names, and with an
         * exclamation mark as sheet name separator.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {Number|Null} extDocRef
         *  The index of the external document, e.g. in [1]!extname or
         *  [1]Sheet1!A1. The value zero represents a global reference into the
         *  own document, e.g. in [0]!globalname. If set to null, the token
         *  represents a reference in the own document (e.g. globalname).
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference. If set to null, an empty string will be
         *  returned.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference. Ignored, if the value null has been
         *  passed to parameter 'sheet1Ref'. If set to null, a single sheet
         *  name as described by the parameter 'sheet1Ref' will be used.
         *
         * @returns {String}
         *  The string representation for valid sheet references, with a
         *  trailing exclamation mark; the value null, if the sheet references
         *  are invalid (deleted sheet); or an empty string, if no sheet
         *  references have been passed (sheet-local reference).
         */
        function generateSheetRangePrefix(docModel, extDocRef, sheet1Ref, sheet2Ref) {

            // start with the external document reference
            var result = (extDocRef !== null) ? ('[' + extDocRef + ']') : '';

            // empty string for missing sheet reference
            if (sheet1Ref) {

                // convert first sheet name
                var sheet1Name = sheet1Ref.resolve(docModel, extDocRef);
                if (sheet1Name === null) { return null; }

                // convert second sheet name
                var sheet2Name = (sheet2Ref && !sheet1Ref.equals(sheet2Ref)) ? sheet2Ref.resolve(docModel, extDocRef) : '';
                if (sheet2Name === null) { return null; }

                // the resulting sheet names
                result += odf ? encodeAbsSheetName(docModel, sheet1Name, sheet1Ref.abs) : sheet1Name;
                if (sheet2Name) {
                    result += ':' + (odf ? encodeAbsSheetName(docModel, sheet2Name, sheet2Ref.abs) : sheet2Name);
                }

                // OOXML: enclose sheet range in apostrophes, if one of the sheet names is complex
                if (!odf && (!self.isSimpleSheetName(docModel, sheet1Name) || (sheet2Name && !self.isSimpleSheetName(docModel, sheet2Name)))) {
                    result = SheetRef.encodeComplexSheetName(result);
                }
            }

            return result ? (result + '!') : '';
        }

        /**
         * Returns the string representation of the passed sheet reference for
         * the OpenFormula token syntax, i.e. with absolute sheet marker.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef} sheetRef
         *  The sheet reference.
         *
         * @returns {String}
         *  The string representation for the passed sheet reference; the value
         *  null, if the sheet reference is invalid (deleted sheet); or an
         *  empty string, if no sheet reference has been passed (sheet-local
         *  reference).
         */
        function generateOFSheetPrefix(docModel, sheetRef) {

            // empty string for missing sheet reference
            if (!sheetRef) { return ''; }

            // convert reference to sheet name, use the #REF! error code for invalid sheets,
            // enclose complex sheet names in apostrophes
            var sheetName = sheetRef.resolve(docModel, null);
            return sheetName ? (encodeAbsSheetName(docModel, sheetName, sheetRef.abs) + '.') : null;
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the passed text is a reserved symbol according to
         * this formula grammar. Boolean literals, and strings that look like
         * cell addresses are considered to be reserved symbols. Cell addresses
         * include the representation of a relative cell reference in A1
         * notation, or a cell reference in R1C1 notation (either English, e.g.
         * 'R1C1', or according to this formula grammar, e.g. 'Z1S1' in the
         * German UI grammar).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index of
         *  strings that look like cell addresses.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is a reserved symbol according to this
         *  formula grammar.
         */
        this.isReservedSymbol = function (docModel, text) {
            return isBooleanLiteral(text) || isReferenceSymbol(docModel, text);
        };

        /**
         * Returns whether the passed sheet name is a simple sheet name, i.e.
         * it does not need to be enclosed in a pair of apostrophes in formula
         * expressions. Simple sheet names do not contain any reserved
         * characters (e.g. whitespace, or operators used in formulas), and do
         * not look like other reserved names (e.g. boolean literals, or cell
         * references).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index of
         *  sheet names that look like cell addresses.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.external=false]
         *      If set to true, the passed text may contain a leading simple
         *      file name enclosed in brackets, e.g. '[doc.xlsx]Sheet1', or
         *      even '[path\to\doc.xlsx]Sheet1'.
         *  - {Boolean} [options.range=false]
         *      If set to true, the passed text may contain a sheet range (two
         *      simple sheet names separated by a colon), e.g. 'Sheet1:Sheet2'.
         *
         * @returns {Boolean}
         *  Whether the passed sheet name is a simple sheet name.
         */
        this.isSimpleSheetName = function (docModel, text, options) {

            // the result of the regular expression test
            var matches = sheetConfig.RE.SIMPLE_EXT_SHEET_NAME.exec(text);
            if (!matches) { return false; }

            // without 'external' flag, external document name is not allowed
            if (matches[1] && !Utils.getBooleanOption(options, 'external', false)) {
                return false;
            }

            // without 'range' flag, second sheet name is not allowed
            if (matches[3] && !Utils.getBooleanOption(options, 'range', false)) {
                return false;
            }

            // reserved symbols are not simple sheet names
            if (this.isReservedSymbol(docModel, matches[2])) { return false; }
            if (matches[3] && this.isReservedSymbol(docModel, matches[3])) { return false; }

            return true;
        };

        /**
         * Checks if the passed string can be used as label for a defined name.
         *
         * @param {String} text
         *  The string to be checked.
         *
         * @returns {String}
         *  The empty string, if the passed text is a valid label for a defined
         *  name; otherwise one of the following error codes:
         *  - 'name:empty': The passed label is empty.
         *  - 'name:invalid': The passed label contains invalid characters.
         *  - 'name:address': The passed label would be a valid, but conflicts
         *      with the representation of a relative cell reference in A1
         *      notation, or a cell reference in R1C1 notation (either English,
         *      e.g. 'R1C1', or according to current UI language, e.g. 'Z1S1'
         *      in German).
         */
        this.validateNameLabel = function (docModel, text) {

            // the passed string must not be empty
            if (text.length === 0) { return 'name:empty'; }

            // check that the passed string does not contain invalid characters
            if (!VALID_NAME_RE.test(text)) { return 'name:invalid'; }

            // check that the text does not look like a boolean literal (TODO: own error code?)
            if (!odf && isBooleanLiteral(text)) { return 'name:invalid'; }

            // bug 38786: check that the text does not look like a cell address
            if (isReferenceSymbol(docModel, text)) { return 'name:address'; }

            // the passed text is a valid label for a defined name
            return '';
        };

        /**
         * Converts the passed scalar value to its text representation in
         * formulas according to this formula grammar.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The value to be converted.
         *
         * @returns {String}
         *  The string representation of the passed value.
         */
        this.formatScalar = function (value) {

            switch (Scalar.getType(value)) {

                case Scalar.Type.NUMBER:
                    return !isFinite(value) ? this.getErrorName(ErrorCode.NUM) :
                        MathUtils.isZero(value) ? '0' :
                        formatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT);

                case Scalar.Type.STRING:
                    // duplicate inner quotes, enclose in quotes
                    return '"' + value.replace(/"/g, '""') + '"';

                case Scalar.Type.BOOLEAN:
                    return this.getBooleanName(value);

                case Scalar.Type.ERROR:
                    return this.getErrorName(value);

                case Scalar.Type.NULL:
                    return '';
            }

            Utils.warn('FormulaGrammar.formatScalar(): unsupported scalar value type');
            return '';
        };

        /**
         * A generic formatter for a cell reference that takes preformatted
         * cell address components, and adds the sheet names of the specified
         * sheet references according to the formula grammar of this instance.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference (e.g. the particle 'Sheet1' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). If set to null, no sheet
         *  names will be inserted into the returned string.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference (e.g. the particle 'Sheet3' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). Ignored, if the value null
         *  has been passed to parameter 'sheet1Ref'. If set to null, a single
         *  sheet name as described by the parameter 'sheet1Ref' will be used
         *  (e.g. 'Sheet1!A1:C3').
         *
         * @param {String} startStr
         *  The text representation of the start cell address (e.g. the text
         *  'A1' for the reference 'Sheet1!A1:C3').
         *
         * @param {String|Null} [endStr]
         *  The text representation of the end cell address (e.g. the text 'C3'
         *  for the reference 'Sheet1!A1:C3'). If omitted, or set to null, or
         *  to an empty string, a single cell address will be generated (e.g.
         *  'Sheet1!A1').
         *
         * @returns {String}
         *  The string representation of the cell range reference.
         */
        this.formatGenericRange = function (docModel, sheet1Ref, sheet2Ref, startStr, endStr) {

            // OpenFormula syntax: sheets of a sheet interval will be added individually to the
            // start and end addresses of the range, e.g.: Sheet1.A1:Sheet2.B2
            if (this.REF_SYNTAX === 'of') {

                // the first sheet name with separator character (simple #REF! error for sheet reference errors)
                var sheet1Prefix = generateOFSheetPrefix(docModel, sheet1Ref);
                if (sheet1Prefix === null) { return REF_ERROR; }

                // the second sheet name with separator character (simple #REF! error for sheet reference errors)
                var sheet2Prefix = sheet2Ref ? generateOFSheetPrefix(docModel, sheet2Ref) : '';
                if (sheet2Prefix === null) { return REF_ERROR; }

                // the first sheet name with separator character, and the first cell address
                var result = sheet1Prefix + startStr;

                // add second cell address for sheet ranges, or for cell ranges (repeat
                // single cell address in a sheet range, e.g. Sheet1.A1:Sheet2.A1)
                if (sheet2Prefix || endStr) {
                    result += ':' + sheet2Prefix + (endStr || startStr);
                }

                return result;
            }

            // start with all sheet names before the cell range address, e.g. Sheet1:Sheet2!A1:C3
            // (sheet reference errors will result in a simple #REF! error without anything else)
            var sheetPrefix = generateSheetRangePrefix(docModel, null, sheet1Ref, sheet2Ref);
            return (sheetPrefix === null) ? REF_ERROR : (sheetPrefix + startStr + (endStr ? (':' + endStr) : ''));
        };

        /**
         * Converts the passed reference structures to the text representation
         * of a complete cell range reference in formulas according to this
         * formula grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {Address} refAddress
         *  The reference address needed to create column/row offsets as used
         *  in R1C1 notation, e.g. R[1]C[-1].
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference (e.g. the particle 'Sheet1' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). If set to null, no sheet
         *  names will be inserted into the returned string.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference (e.g. the particle 'Sheet3' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). Ignored, if the value null
         *  has been passed to parameter 'sheet1Ref'. If set to null, a single
         *  sheet name as described by the parameter 'sheet1Ref' will be used
         *  (e.g. 'Sheet1!A1:C3').
         *
         * @param {CellRef|Null} cell1Ref
         *  The first cell reference (e.g. the cell address 'A1' in 'A1:C3').
         *  If set to null, the #REF! error code will be returned instead of a
         *  cell range address.
         *
         * @param {CellRef|Null} cell2Ref
         *  The second cell reference (e.g. the cell address 'C3' in 'A1:C3').
         *  If set to null, a single cell address will be returned.
         *
         * @returns {String}
         *  The string representation of the cell range reference.
         */
        this.formatReference = function (docModel, refAddress, sheet1Ref, sheet2Ref, cell1Ref, cell2Ref) {

            // R1C1 notation not supported in native formula grammars (UI only)
            if (!localized && rcStyle) {
                Utils.error('FormulaGrammar.formatReference(): R1C1 notation not supported in native grammar');
                return NA_ERROR;
            }

            // the cell range address (with adjusted column/row indexes)
            var range = cell1Ref ? Range.createFromAddresses(cell1Ref.toAddress(), cell2Ref && cell2Ref.toAddress()) : null;
            // the text representation of the start and end cell address
            var startStr = null, endStr = null;

            // OpenFormula syntax: references are enclosed in brackets, absolute markers for
            // sheets, sheet names are separated by periods, e.g.: [$Sheet1.A1:$'Sheet 2'.B2]
            if (this.REF_SYNTAX === 'of') {

                // in OpenFormula, reference errors will be shown as simple #REF! error without sheet reference
                if (!range) { return '[' + REF_ERROR + ']'; }

                // the text representation of the start address
                startStr = cell1Ref.refText();
                // start address will always contain a leading period, also without sheet name
                if (!sheet1Ref) { startStr = '.' + startStr; }

                // the text representation of the end address
                if (cell2Ref) {
                    endStr = cell2Ref.refText();
                    // end address will always contain a leading period, also without sheet name
                    if (!sheet2Ref) { endStr = '.' + endStr; }
                }

                // enclose the entire reference into brackets
                return '[' + this.formatGenericRange(docModel, sheet1Ref, sheet2Ref, startStr, endStr) + ']';
            }

            // no valid range: generate a #REF! error with (optional) leading sheet name (ODF: simple #REF! error only)
            if (!range) {
                return odf ? REF_ERROR : this.formatGenericRange(docModel, sheet1Ref, sheet2Ref, REF_ERROR);
            }

            if (cell1Ref.absCol && cell2Ref && cell2Ref.absCol && docModel.isRowRange(range)) {
                // generate row interval (preferred over column interval for entire sheet range)
                startStr = rcStyle ? cell1Ref.rowTextRC(prefixChars, refAddress) : cell1Ref.rowText();
                if (!rcStyle || !range.singleRow() || (cell1Ref.absRow !== cell2Ref.absRow)) {
                    endStr = rcStyle ? cell2Ref.rowTextRC(prefixChars, refAddress) : cell2Ref.rowText();
                }
            } else if (cell1Ref.absRow && cell2Ref && cell2Ref.absRow && docModel.isColRange(range)) {
                // generate column interval
                startStr = rcStyle ? cell1Ref.colTextRC(prefixChars, refAddress) : cell1Ref.colText();
                if (!rcStyle || !range.singleCol() || (cell1Ref.absCol !== cell2Ref.absCol)) {
                    endStr = rcStyle ? cell2Ref.colTextRC(prefixChars, refAddress) : cell2Ref.colText();
                }
            } else {
                // generate range or cell address
                startStr = rcStyle ? cell1Ref.refTextRC(prefixChars, refAddress) : cell1Ref.refText();
                if (cell2Ref) {
                    endStr = rcStyle ? cell2Ref.refTextRC(prefixChars, refAddress) : cell2Ref.refText();
                }
            }

            return this.formatGenericRange(docModel, sheet1Ref, sheet2Ref, startStr, endStr);
        };

        /**
         * Converts the passed sheet reference and label of a defined name to
         * the text representation of a complete defined name reference in
         * formulas according to this formula grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {Number|Null} extDocRef
         *  The index of the external document, e.g. the number in the brackets
         *  in the formula [1]!extname. The value zero represents a globally
         *  defined name with document reference in the passed document model,
         *  e.g. the zero in [0]!globalname. If null or undefined, the token
         *  represents a name in the passed document without document reference
         *  (e.g. globalname).
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference (e.g. the particle 'Sheet1' in 'Sheet1!name').
         *  If set to null, no sheet name will be inserted into the returned
         *  string.
         *
         * @param {String} label
         *  The label of the defined name.
         *
         * @returns {String}
         *  The string representation of the defined name.
         */
        this.formatName = function (docModel, extDocRef, sheetRef, label) {

            // no external or sheet-local names in ODF files
            if (odf && (extDocRef || sheetRef)) { return NA_ERROR; }

            // sheet reference error will be replaced by a global workbook reference
            var sheetPrefix = generateSheetRangePrefix(docModel, extDocRef, sheetRef);
            if (sheetPrefix === null) { sheetPrefix = '[' + (extDocRef || 0) + ']!'; }
            return sheetPrefix + label;
        };

        /**
         * Converts the passed sheet reference and macro function name to the
         * text representation of a complete macro call in formulas according
         * to this formula grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {Number|Null} extDocRef
         *  The index of the external document, e.g. the number in the brackets
         *  in the formula [1]!macro(). The value zero represents a macro with
         *  document reference in the passed document model, e.g. the zero in
         *  [0]!macro(). If null or undefined, the token represents a macro in
         *  the passed document without document reference (e.g. macro()).
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference (e.g. the particle 'Sheet1' in the formula
         *  'Sheet1!macro()'). If set to null, no sheet name will be inserted
         *  into the returned string.
         *
         * @param {String} label
         *  The name of a macro function.
         *
         * @returns {String}
         *  The string representation of the macro call.
         */
        this.formatMacro = function (docModel, extDocRef, sheetRef, label) {

            // no external or sheet-local macros in ODF files
            if (odf) { return (extDocRef || sheetRef) ? NA_ERROR : label; }

            // OOXML: macro names equal to the internal name of a built-in function will be prefixed
            // Example: the formula '=SUMME(1)+SUM(1)' in a German UI results in the expression 'SUM(1)+_xludf.SUM(1)'
            var hasUdfPrefix = (UDF_PREFIX.length < label.length) && (label.substr(0, UDF_PREFIX.length).toLowerCase() === UDF_PREFIX);

            // add a missing prefix for operation grammer, or remove existing prefix for UI grammar
            if (localized && hasUdfPrefix) {
                label = label.substr(UDF_PREFIX.length);
            } else if (!localized && !hasUdfPrefix && functionCollection.getKey(label, false)) {
                label = UDF_PREFIX + label;
            }

            // format the macro call like a defined name
            return this.formatName(docModel, extDocRef, sheetRef, label);
        };

        /**
         * Returns the name of the passed boolean value.
         *
         * @param {Boolean} value
         *  The boolean value to be converted to its string representation.
         *
         * @returns {String}
         *  The name of the passed boolean value.
         */
        this.getBooleanName = function (value) {
            return booleanCollection.getName(value ? 't' : 'f', localized);
        };

        /**
         * Converts the passed name of a boolean value to the boolean value.
         *
         * @param {String} name
         *  The name of a boolean value (case-insensitive).
         *
         * @returns {Boolean|Null}
         *  The boolean value for the passed name of a boolean value; or null,
         *  if the passed string is not the name of a boolean value.
         */
        this.getBooleanValue = function (name) {
            var key = booleanCollection.getKey(name, localized);
            return key ? (key === 't') : null;
        };

        /**
         * Returns the name of the passed error code.
         *
         * @param {ErrorCode} error
         *  The error code to be converted to its string representation.
         *
         * @returns {String|Null}
         *  The name of the passed error code; or null, if the passed error
         *  code cannot be resolved to a name.
         */
        this.getErrorName = function (error) {
            return errorCollection.getName(error.key, localized);
        };

        /**
         * Converts the passed error name to the error code instance.
         *
         * @param {String} name
         *  The name of an error code (case-insensitive).
         *
         * @returns {ErrorCode|Null}
         *  The error code instance for the passed error name; or null, if the
         *  passed string is not the name of a supported error code.
         */
        this.getErrorCode = function (name) {
            var key = errorCollection.getKey(name, localized);
            return key ? ErrorCode.create(key) : null;
        };

        /**
         * Returns the name of an operator for the passed unique resource key.
         *
         * @param {String} key
         *  The unique resource key of an operator.
         *
         * @returns {String|Null}
         *  The operator name for the passed resource key; or null, if the
         *  passed resource key cannot be resolved to the name of an operator.
         */
        this.getOperatorName = function (key) {
            return operatorCollection.getName(key, localized);
        };

        /**
         * Converts the passed operator name to the unique resource key of the
         * operator.
         *
         * @param {String} name
         *  The name of an operator.
         *
         * @returns {String|Null}
         *  The resource key for the passed operator name; or null, if the
         *  passed string is not the name of a supported operator.
         */
        this.getOperatorKey = function (name) {
            return operatorCollection.getKey(name, localized);
        };

        /**
         * Returns the name of a function for the passed unique resource key.
         *
         * @param {String} key
         *  The unique resource key of a function.
         *
         * @returns {String|Null}
         *  The function name for the passed resource key; or null, if the
         *  passed resource key cannot be resolved to the name of a function.
         */
        this.getFunctionName = function (key) {
            return functionCollection.getName(key, localized);
        };

        /**
         * Converts the passed function name to the unique resource key of the
         * function.
         *
         * @param {String} name
         *  The name of a function (case-insensitive).
         *
         * @returns {String|Null}
         *  The resource key for the passed function name; or null, if the
         *  passed string is not the name of a supported function.
         */
        this.getFunctionKey = function (name) {
            return functionCollection.getKey(name, localized);
        };

        /**
         * Returns the name of a table region for the passed unique resource
         * key.
         *
         * @param {String} key
         *  The unique resource key of a table region.
         *
         * @returns {String|Null}
         *  The name of a table region for the passed resource key; or null, if
         *  the passed resource key cannot be resolved to the name of a table
         *  region.
         */
        this.getTableRegionName = function (key) {
            return regionCollection.getName(key, localized);
        };

        /**
         * Converts the passed name of a table region to the unique resource
         * key of the table region.
         *
         * @param {String} name
         *  The name of a table region (with leading hash character).
         *
         * @returns {String|Null}
         *  The resource key for the passed table region; or null, if the
         *  passed string is not the name of a supported table region.
         */
        this.getTableRegionKey = function (name) {
            return regionCollection.getKey(name, localized);
        };

        /**
         * Returns whether the passed table column name is considered to be
         * simple, i.e. it can be used without brackets in complex structured
         * table references in UI formula grammars.
         *
         * @param {String} colName
         *  The table column name to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed table column name is considered to be simple.
         */
        this.isSimpleTableColumn = function (colName) {
            return SIMPLE_TABLE_COLUMN_NAME_RE.test(colName);
        };

        /**
         * Converts the passed text representation of a table column name (used
         * in formula expressions) to the original column name.
         *
         * @param {String} colName
         *  The encoded name of a table column.
         *
         * @returns {String}
         *  The original table column name.
         */
        this.decodeTableColumn = function (colName) {
            return colName.replace(/'(.)/g, '$1');
        };

        /**
         * Converts the passed table column name to the text representation of
         * a table column reference as used in structured table references in
         * formulas according to this formula grammar.
         *
         * @param {String} colName
         *  The name of a table column.
         *
         * @returns {String}
         *  The string representation of the table column reference.
         */
        this.encodeTableColumn = function (colName) {
            return colName.replace(TABLE_COL_ESCAPE_RE, '\'$&');
        };

        /**
         * Returns whether the intersection operator is a space character, and
         * thus needs special handling to disambiguate it from white-space.
         *
         * @returns {Boolean}
         *  Whether the intersection operator is a space character.
         */
        this.isSpaceIntersection = function () {
            return this.getOperatorName('isect') === ' ';
        };

        /**
         * Generates the formula expression for a generic subtotal formula.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to generate references.
         *
         * @param {String} funcKey
         *  The resource key of a function.
         *
         * @param {Range} range
         *  The address of the cell range to be inserted into the formula.
         *
         * @returns {String}
         *  The formula expression of the subtotal formula.
         */
        this.generateAutoFormula = function (docModel, funcKey, range, refAddress) {

            var formula = this.getFunctionName(funcKey) + '(';

            if (range) {
                var cell1Ref = new CellRef(range.start[0], range.start[1], false, false);
                var cell2Ref = range.single() ? null : new CellRef(range.end[0], range.end[1], false, false);
                formula += this.formatReference(docModel, refAddress, null, null, cell1Ref, cell2Ref);
            }

            return formula + ')';
        };

    } }); // class FormulaGrammar

    // static methods ---------------------------------------------------------

    /**
     * Returns an existing formula grammar singleton from the internal cache,
     * if available; otherwise creates a new instance.
     *
     * @param {String} grammarId
     *  The identifier of a formula grammar. Supported values are:
     *  - 'op': Fixed token representations as used in operations, and the file
     *      format.
     *  - 'ui': The localized token representations according to the current
     *      UI language of the application.
     *  - 'en': The token representations according to the American English UI
     *      language (locale 'en_US').
     *
     * @param {String} fileFormat
     *  The identifier of the file format related to the formula resource data
     *  used by the formula grammar.
     *
     * @param {Boolean} rcStyle
     *  Whether the formula grammar represents references in A1 notation
     *  (false), or in R1C1 notation (true).
     *
     * @returns {FormulaGrammar}
     *  A formula grammar singleton for the passed parameters.
     */
    FormulaGrammar.create = (function () {

        // create a hash key for the memoize() cache
        function getGrammarKey(grammarId, fileFormat, rcStyle) {
            return grammarId + ':' + fileFormat + '!' + (rcStyle ? 'rc' : 'a1');
        }

        // create a new instance of FormulaGrammar
        function createGrammar(grammarId, fileFormat, rcStyle) {
            return new FormulaGrammar(grammarId, fileFormat, rcStyle);
        }

        // create a function that caches all created formula grammars
        return _.memoize(createGrammar, getGrammarKey);
    }());

    // exports ================================================================

    return FormulaGrammar;

});
