/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/sheetdescriptor', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils'
], function (Utils, ValueMap, SheetUtils, DependencyUtils) {

    'use strict';

    // convenience shortcuts
    var AddressSet = SheetUtils.AddressSet;
    var FormulaSet = DependencyUtils.FormulaSet;

    // class SheetDescriptor ==================================================

    /**
     * This descriptor object collects all dependency settings for a sheet in
     * a spreadsheet document.
     *
     * @constructor
     *
     * @property {SheetModel} sheetModel
     *  The model of the sheet represented by this descriptor object.
     *
     * @property {FormulaSet} formulaSet
     *  The formula descriptors (instances of class FormulaDescriptor) for all
     *  known token arrays of the sheet.
     *
     * @property {AddressSet} cellFormulaSet
     *  The formula descriptors (instances of class FormulaDescriptor) for all
     *  known cell formulas of the sheet, as property 'formula' in the cell
     *  address objects of an address set.
     *
     * @property {ValueMap<FormulaSet>} modelFormulasMap
     *  The descriptors of all formulas attached to a token array embedded in a
     *  model object (e.g. conditions of formatting rules, source link formulas
     *  of chart data series). The elements of this map are formula descriptor
     *  sets. The sets are keyed by the UIDs of the parent token model.
     *
     * @property {ValueMap<FormulaSet>} missingNamesMap
     *  The formula descriptors for all known token arrays that contain
     *  references to non-existing defined names. These formulas are considered
     *  to be dirty, if a defined name has been inserted, or an existing defind
     *  name has changed its label, which may result in validating a #NAME?
     *  error. The property is a map with the name keys (the upper-case labels)
     *  of missing names as key, and stores a formula set per missing name.
     *
     * @property {FormulaSet} recalcAlwaysSet
     *  The formula descriptors for all known token arrays with recalculation
     *  mode 'always'. These formulas will always be recalculated whenever
     *  something in the document has changed, regardless of their
     *  dependencies.
     *
     * @property {FormulaSet} recalcOnceSet
     *  The formula descriptors for all known token arrays with recalculation
     *  mode 'once'. These formulas will be recalculated once after the
     *  document has been imported, regardless of their dependencies.
     *
     * @property {ValueMap<Number>} missingFuncMap
     *  The resource keys of all unimplemented functions that occur in any
     *  formula of the sheet, and the number of occurences.
     */
    function SheetDescriptor(sheetModel) {

        this.sheetModel = sheetModel;

        this.formulaSet = new FormulaSet();
        this.cellFormulaSet = new AddressSet();
        this.modelFormulasMap = new ValueMap();
        this.missingNamesMap = new ValueMap();
        this.recalcAlwaysSet = new FormulaSet();
        this.recalcOnceSet = new FormulaSet();
        this.missingFuncMap = new ValueMap();

    } // class SheetDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Inserts the passed formula descriptor into the formula caches of this
     * sheet descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The formula descriptor to be inserted into the formula caches.
     */
    SheetDescriptor.prototype.insertFormula = function (formulaDesc) {

        // insert the formula descriptor into the own maps
        this.formulaSet.insert(formulaDesc);
        switch (formulaDesc.recalc) {
            case 'always': this.recalcAlwaysSet.insert(formulaDesc); break;
            case 'once':   this.recalcOnceSet.insert(formulaDesc);   break;
        }

        // insert the formula into the type-specific containers
        if (formulaDesc.tokenModel) {
            var modelUid = formulaDesc.tokenModel.getUid();
            this.modelFormulasMap.getOrConstruct(modelUid, FormulaSet).insert(formulaDesc);
        } else if (formulaDesc.cellAddress) {
            var address = formulaDesc.cellAddress.clone();
            this.cellFormulaSet.insert(address).formula = formulaDesc;
        } else {
            Utils.error('SheetDescriptor.insertFormula(): unknown formula type');
        }

        // map the formula descriptor by missing name labels
        ValueMap.forEachKey(formulaDesc.missingNames, function (key) {
            var formulaSet = this.missingNamesMap.getOrConstruct(key, FormulaSet);
            formulaSet.insert(formulaDesc);
        }, this);

        // register all unimplemented functions
        ValueMap.forEachKey(formulaDesc.missingFuncs, function (key) {
            this.missingFuncMap.update(key, 0, function (count) { return count + 1; });
        }, this);

        DependencyUtils.withLogging(function () {
            DependencyUtils.log('new dependencies for ' + formulaDesc + ': ' + formulaDesc.references);
        });
    };

    /**
     * Deletes the specified formula descriptor from the formula caches of this
     * sheet descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor of the formula to be deleted.
     */
    SheetDescriptor.prototype.removeFormula = function (formulaDesc) {

        // remove the formula descriptor from the own maps
        this.formulaSet.remove(formulaDesc);
        this.recalcAlwaysSet.remove(formulaDesc);
        this.recalcOnceSet.remove(formulaDesc);
        ValueMap.forEachKey(formulaDesc.missingNames, function (key) {
            this.missingNamesMap.with(key, function (formulaSet) {
                formulaSet.remove(formulaDesc);
                if (formulaSet.empty()) { this.missingNamesMap.remove(key); }
            }, this);
        }, this);

        // remove the formula from the type-specific containers
        if (formulaDesc.tokenModel) {
            var modelUid = formulaDesc.tokenModel.getUid();
            this.modelFormulasMap.with(modelUid, function (formulaSet) {
                formulaSet.remove(formulaDesc);
                if (formulaSet.empty()) {
                    this.modelFormulasMap.remove(modelUid);
                }
            }, this);
        } else if (formulaDesc.cellAddress) {
            this.cellFormulaSet.remove(formulaDesc.cellAddress);
        } else {
            Utils.error('SheetDescriptor.removeFormula(): unknown formula type');
        }

        // unregister all unimplemented functions
        ValueMap.forEachKey(formulaDesc.missingFuncs, function (key) {
            var count = this.missingFuncMap.update(key, 0, function (count) { return count - 1; });
            if (count <= 0) { this.missingFuncMap.remove(key); }
        }, this);

        DependencyUtils.log('removed dependencies for ' + formulaDesc);
    };

    // exports ================================================================

    return SheetDescriptor;

});
