/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingmodelmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils'
], function (Utils, DrawingUtils, Operations, SheetUtils, TextFrameUtils) {

    'use strict';

    // convenience shortcuts
    var UpdateMode = SheetUtils.UpdateMode;
    var Range = SheetUtils.Range;

    // class DrawingModelMixin ================================================

    /**
     * A mix-in class that extends the public API of any drawing model object
     * that has been inserted into a sheet of a spreadsheet document.
     *
     * @constructor
     *
     * @internal
     *  This is a mix-in class supposed to extend an existing instance of the
     *  class DrawingModel or any of its sub classes. Expects the symbol 'this'
     *  to be bound to an instance of DrawingModel.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing the drawing collection with this drawing
     *  model object.
     *
     * @param {Function} cloneConstructor
     *  A callback function invoked from the public method clone() which has to
     *  create and return a new instance cloned from this drawing model.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  - {Function} [initOptions.generateRestoreOperations]
     *      A callback function that will be invoked from the own public method
     *      DrawingModelMixin.generaterestoreOperations() while generating
     *      document operations that will delete this drawing object. An
     *      implementation must generate more undo operations to restore the
     *      complete drawing object.
     *  - {Function} [initOptions.generateUpdateFormulaOperations]
     *      A callback function that will be invoked from the own public method
     *      DrawingModelMixin.generateUpdateFormulaOperations() while
     *      generating document operations that involve to update the formula
     *      expressions in this drawing object. Receives all parameters passed
     *      to the public method mentioned before. May return an attribute set
     *      with additional formatting attributes to be applied at this drawing
     *      object, or null.
     */
    function DrawingModelMixin(sheetModel, cloneConstructor, initOptions) {

        // the document model instance
        var docModel = sheetModel.getDocModel();

        // the attribute pool for drawing formatting
        var attributePool = docModel.getDrawingAttributePool();

        // the drawing collection (or comment collection) containing this drawing model
        var rootCollection = this.getRootCollection();

        // callback handler to generate additional restore operations
        var generateRestoreOperationsHandler = Utils.getFunctionOption(initOptions, 'generateRestoreOperations', null);

        // callback handler to generate additional operations to update formula expressions
        var generateUpdateFormulaOperationsHandler = Utils.getFunctionOption(initOptions, 'generateUpdateFormulaOperations', null);

        // whether this drawing object is embedded in another drawing object (e.g. group object)
        var embedded = this.isEmbedded();

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this drawing object model
         * for the specified sheet.
         *
         * @internal
         *  Used by the class SheetDrawingCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the target
         *  drawing collection.
         *
         * @param {DrawingCollection} targetCollection
         *  The drawing collection that will own the clone returned by this
         *  method. May be the root drawing collection of the target sheet, or
         *  an embedded drawing collection of a group object.
         *
         * @returns {DrawingModelMixin}
         *  A clone of this drawing model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel, targetCollection) {

            // the new clone of this drawing model
            var cloneModel = cloneConstructor.call(this, targetModel, targetCollection);
            // the embedded child collection of the new drawing model
            var cloneCollection = cloneModel.getChildCollection();

            // clone the child drawing collection (e.g. group objects)
            this.forEachChildModel(function (childModel, childIndex) {
                var childCloneModel = childModel.clone(targetModel, cloneCollection);
                cloneCollection.implInsertModel([childIndex], childCloneModel);
            });

            return cloneModel;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the sheet model instance that contains this drawing model.
         *
         * @returns {SheetModel}
         *  The sheet model instance that contains this drawing model.
         */
        this.getSheetModel = function () {
            return sheetModel;
        };

        /**
         * Returns the global document position of this drawing model, as used
         * in document operations, including the leading sheet index.
         *
         * @returns {Array<Number>}
         *  The global document position of this drawing model.
         */
        this.getDocPosition = function () {
            return [sheetModel.getIndex()].concat(this.getPosition());
        };

        /**
         * Returns the address of the cell range covered by this drawing model
         * object.
         *
         * @returns {Range}
         *  The address of the cell range covered by this drawing model.
         */
        this.getRange = function () {

            // the drawing attributes contain the cell anchor values
            var anchorAttrs = this.getMergedAttributeSet(true).drawing;
            // build the cell range address from the anchor attributes
            var range = Range.create(anchorAttrs.startCol, anchorAttrs.startRow, anchorAttrs.endCol, anchorAttrs.endRow);

            // shorten by one column/row, if the end offset is zero
            if ((anchorAttrs.endColOffset === 0) && !range.singleCol()) { range.end[0] -= 1; }
            if ((anchorAttrs.endRowOffset === 0) && !range.singleRow()) { range.end[1] -= 1; }

            return range;
        };

        /**
         * Returns the rectangle in pixels covered by this drawing object,
         * either relative to its parent area (the sheet area for top-level
         * drawing objects, or the rectangle of the parent drawing object); or
         * always absolute in the sheet area.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.absolute=false]
         *      If set to true, this method will return the absolute sheet
         *      position also for drawing objects embedded into other drawing
         *      objects. By default, this method returns the location of the
         *      specified drawing object relative to its parent object.
         *
         * @returns {Rectangle|Null}
         *  The location of this drawing object in pixels according to the
         *  current sheet zoom factor, either relative to its parent object
         *  (sheet area for top-level drawing objects), or always absolute in
         *  the sheet area (according to the passed options); or null, if the
         *  drawing object is located entirely in hidden columns or rows and
         *  thus has no meaningful location.
         */
        this.getRectangle = function (options) {
            return rootCollection.getRectangleForModel(this, options);
        };

        /**
         * Returns the minimum pixel size this drawing object needs for proper
         * rendering. For area-like objects, the minimum size is 5 pixels. For
         * connector shapes, the minimum size is 1 pixel in order to be able to
         * show horizontal or vertical straight lines.
         *
         * @returns {Number}
         *  The minimum pixel size this drawing object needs for proper
         *  rendering.
         */
        this.getMinSize = function () {
            return this.isTwoPointShape() ? 1 : 5;
        };

        /**
         * Returns whether the transformation attributes of this drawing model
         * will lead to swapped width and height of the rendered drawing frame.
         *
         * @returns {Boolean}
         *  Whether the transformation attributes will lead to swapped width
         *  and height of the rendered drawing frame.
         */
        this.hasSwappedDimensions = function () {
            return DrawingUtils.hasSwappedDimensions(this.getMergedAttributeSet(true).drawing);
        };

        /**
         * Returns the DOM drawing frame that represents this drawing model.
         * The DOM drawing frame is used to store the text contents of this
         * drawing object.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for this drawing model; or null, if no DOM
         *  drawing frame could be found.
         */
        this.getModelFrame = function () {
            return rootCollection.getDrawingFrameForModel(this);
        };

        /**
         * Refreshes the settings of the DOM drawing frame associated with this
         * drawing model, and returns the cached rendering settings.
         *
         * @returns {Object|Null}
         *  A descriptor with rendering data for this drawing model; or null,
         *  if no data is available for the drawing model. See description of
         *  the method SheetDrawingCollection.refreshDrawingFrameForModel() for
         *  details.
         */
        this.refreshModelFrame = function () {
            return rootCollection.refreshDrawingFrameForModel(this);
        };

        /**
         * Extracts the anchor attributes from this drawing object.
         *
         * @returns {Object}
         *  The anchor attributes of this drawing object.
         */
        this.getAnchorAttributes = function () {
            return rootCollection.getAnchorAttributesForModel(this);
        };

        /**
         * Returns an attribute set that will restore the current state of the
         * explicit attributes, after they have been changed according to the
         * passed attribute set. Intended to be used to create undo operations.
         * This method overwrites the implementation of the base class
         * AttributedModel, and always adds a complete set of anchor attributes
         * as expected by the export filters.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set to be merged over the current
         *  explicit attributes of this instance.
         *
         * @returns {Object}
         *  An attribute set that will restore the current explicit attributes
         *  after they have been changed according to the passed attributes.
         */
        this.getUndoAttributeSet = _.wrap(this.getUndoAttributeSet, function (baseMethod, attributeSet) {

            // create the undo attributes
            var undoAttrSet = baseMethod.call(this, attributeSet);

            // bug 57983: always add complete set of anchor attributes
            if (!_.isEmpty(undoAttrSet) && attributeSet.drawing && attributeSet.drawing.anchorType) {
                _.extend(undoAttrSet.drawing, this.getAnchorAttributes());
            }

            return undoAttrSet;
        });

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to change the
         * attributes of this drawing object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} attributeSet
         *  The incomplete attribute set with all formatting attributes to be
         *  changed.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateChangeOperations = function (generator, attributeSet) {

            // create the undo attributes; nothing to do if no attribute actually changes
            var undoAttrSet = this.getUndoAttributeSet(attributeSet);
            if (_.isEmpty(undoAttrSet)) { return this; }

            // the position of the drawing object to be inserted into the operation
            var position = this.getPosition();

            generator.generateDrawingOperation(Operations.CHANGE_DRAWING, position, { attrs: undoAttrSet }, { undo: true });
            generator.generateDrawingOperation(Operations.CHANGE_DRAWING, position, { attrs: attributeSet });
            return this;
        };

        /**
         * Generates the undo operations needed to restore this drawing object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator) {

            // bug 47948: do not restore unsupported drawing types
            if (!this.isDeepRestorable()) { return this; }

            // the operation position of the drawing object in the root collection
            var position = this.getPosition();

            // create an "insertDrawing" operation to restore the drawing model with its attributes
            var undoProps = { type: this.getType() };
            var attrSet = this.getExplicitAttributeSet(true);
            if (!_.isEmpty(attrSet)) { undoProps.attrs = attrSet; }
            generator.generateDrawingOperation(Operations.INSERT_DRAWING, position, undoProps, { undo: true });

            // additional undo operations for complex drawing contents
            if (generateRestoreOperationsHandler) {
                generateRestoreOperationsHandler.call(this, generator, position);
            }

            // generate the operations to restore the text contents
            var modelFrame = this.getModelFrame();
            if (TextFrameUtils.getTextFrame(modelFrame)) {
                var textGenerator = docModel.createOperationGenerator();
                textGenerator.generateContentOperations(modelFrame, [sheetModel.getIndex()].concat(position));
                generator.appendOperations(textGenerator.getOperations(), { undo: true });
            }

            // generate the operations to restore all embedded drawing objects
            this.forEachChildModel(function (childModel) {
                childModel.generateRestoreOperations(generator);
            });

            return this;
        };

        /**
         * Generates the operations to delete, and the undo operations to
         * restore this drawing object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateDeleteOperations = function (generator) {
            this.generateRestoreOperations(generator);
            generator.generateDrawingOperation(Operations.DELETE_DRAWING, this.getPosition());
            return this;
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the inactive anchor attributes of this drawing object, after the
         * size or visibility of the columns or rows in the sheet has been
         * changed.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateRefreshAnchorOperations = function (generator) {

            // nothing to do for embedded drawing objects
            if (embedded) { return this; }

            // recalculate the inactive anchor attributes according to the anchor type
            var anchorAttrs = rootCollection.generateAnchorAttributes(this);

            // create a change operation, if some attributes will be changed in this drawing
            if (anchorAttrs) {
                this.generateChangeOperations(generator, { drawing: anchorAttrs });
            }

            return this;
        };

        /**
         * Generates the operations and undo operations to change the anchor
         * mode of this drawing object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {AnchorMode} anchorMode
         *  The new display anchor mode for the drawing object.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateChangeAnchorModeOperations = function (generator, anchorMode) {

            // nothing to do for embedded drawing objects
            if (embedded) { return this; }

            // recalculate the inactive anchor attributes according to the anchor type
            var anchorAttrs = rootCollection.generateAnchorAttributes(this, anchorMode);

            // create a change operation, if some attributes will be changed in this drawing
            if (anchorAttrs) {
                this.generateChangeOperations(generator, { drawing: anchorAttrs });
            }

            return this;
        };

        /**
         * Generates the undo operations to restore this drawing object if it
         * will be deleted implicitly by moving cells (e.g. deleting columns or
         * rows) in the sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {MoveDescriptor} moveDescs
         *  An array of move descriptors that specify how the cells in the
         *  sheet will be moved.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateBeforeMoveCellsOperations = function (generator, moveDescs) {

            // nothing to do for embedded drawing objects
            if (embedded) { return this; }

            // calculate the moved anchor attributes
            var anchorAttrs = rootCollection.generateAnchorAttributesForMove(this, moveDescs);

            // delete the top-level drawing object if all its parent cells have been deleted
            if (anchorAttrs === 'delete') {
                this.generateDeleteOperations(generator);
            }

            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source links in this drawing object
         * that refer to some cells in this document.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateUpdateFormulaOperations = function (generator, updateDesc) {

            // whether a top-level drawing object is affected by a move operation in its own sheet
            var topLevelMove = !embedded && (updateDesc.type === UpdateMode.MOVE_CELLS) && (updateDesc.sheet === sheetModel.getIndex());
            // update the anchor of top-level drawing objects, if cells in the own sheet will be moved (may delete a complete object!)
            var anchorAttrs = topLevelMove ? rootCollection.generateAnchorAttributesForMove(this, updateDesc.moveDescs) : null;

            // bug 58155: should not happen anymore, deleting drawings moved to generateBeforeMoveCellsOperations()
            if (anchorAttrs === 'delete') {
                Utils.error('DrawingModelMixin.generateUpdateFormulaOperations(): unexpected deleted drawing');
                return this;
            }

            // the formatting attributes to be applied for this drawing object
            var attributeSet = null;

            // derived classes may generate new operations for child objects, and additional formatting attributes
            if (generateUpdateFormulaOperationsHandler) {
                attributeSet = generateUpdateFormulaOperationsHandler.call(this, generator, this.getPosition(), updateDesc);
            }

            // reduce the attribute to the attributes that will actually change
            if (attributeSet) {
                attributeSet = this.getReducedAttributeSet(attributeSet);
            }

            // add the new anchor attributes to the attribute set (bug 57983: do not reduce these attributes!)
            if (anchorAttrs) {
                attributeSet = attributePool.extendAttributeSet(attributeSet || {}, { drawing: anchorAttrs });
            }

            // create a change operation, if some attributes will be changed in this drawing
            if (!_.isEmpty(attributeSet)) {
                this.generateChangeOperations(generator, attributeSet, topLevelMove ? updateDesc.moveDescs : null);
            }

            // generate the operations to update all embedded drawing objects
            this.forEachChildModel(function (childModel) {
                childModel.generateUpdateFormulaOperations(generator, updateDesc);
            });

            return this;
        };

        // initialization -----------------------------------------------------

        // add missing inactive anchor attributes that can be calculated from the existing anchor attributes
        if (!embedded) {
            var anchorAttrs = rootCollection.getInactiveAnchorAttributes(this.getMergedAttributeSet(true).drawing);
            this.setAttributes({ drawing: anchorAttrs }, { notify: 'never' });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = sheetModel = attributePool = rootCollection = cloneConstructor = null;
        });

    } // class DrawingModelMixin

    // exports ================================================================

    return DrawingModelMixin;

});
