/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/tablemixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, SheetUtils) {

    'use strict';

    // mix-in class TableMixin ================================================

    /**
     * Implementations of all controller items for manipulating the selected
     * table in the active sheet (either a 'real' table range with styling, or
     * the auto-filter of the sheet), intended to be mixed into a document
     * controller instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the active sheet.
     */
    function TableMixin(docView) {

        // self reference (the controller instance)
        var self = this;

        // the model and collections of the active sheet
        var sheetModel = null;
        var cellCollection = null;
        var mergeCollection = null;
        var tableCollection = null;

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // returns the model instance of the table range containing the active cell
            'sheet/realtable': {
                parent: 'sheet/operation/unlocked/cell',
                get: function () {
                    var tableModel = tableCollection.findTable(docView.getActiveCell());
                    return (tableModel && !tableModel.isAutoFilter()) ? tableModel : null;
                }
            },

            // returns the model instance of the active table (selected table range, or existing auto-filter)
            'sheet/anytable': {
                parent: 'sheet/realtable',
                get: function (tableModel) {
                    return tableModel || tableCollection.getAutoFilter();
                }
            },

            // the style flags of the table range
            'table/flags': {
                parent: 'sheet/realtable',
                get: function (tableModel) {
                    return tableModel ? tableModel.getStyleFlags() : null;
                }
            },

            // toggles the table filter, if a table is selected in the active
            // sheet, otherwise toggles the auto-filter of the active sheet
            'table/filter': {
                parent: 'sheet/anytable',
                // TODO: filtering real tables has been disabled, until OOXML filter supports import/export of filter settings
                enable: function (tableModel) { return !tableModel || tableModel.isAutoFilter(); },
                get: function (tableModel) { return !!tableModel && tableModel.isAutoFilter(); },
                set: function (state) {

                    // the active table model
                    var tableModel = this.getParentValue();
                    // no filter manipulation in locked sheets
                    var promise = docView.ensureUnlockedSheet();

                    // toggle filter state of table range; or toggle entire auto-filter
                    promise = promise.then(function () {
                        return sheetModel.createAndApplyOperations(function (generator) {

                            // toggle filter of a real table range (not auto-filter)
                            if (tableModel && !tableModel.isAutoFilter()) {
                                // TODO
                                //return tableModel.generateChangeTableOperations(generator, { table: { filtered: state } });
                                Utils.error('TableMixin.toggleTableFilter(): missing implementation');
                                return self.createRejectedPromise();
                            }

                            // concatenate optional asynchronous methods
                            var promise2 = self.createResolvedPromise();

                            // remove existing auto-filter (regardless of passed state)
                            if (tableModel) {
                                promise2 = promise2.then(function () {
                                    return tableModel.generateDeleteOperations(generator);
                                });
                            }

                            // create a new table for the auto-filter
                            if (state) {
                                promise2 = promise2.then(function () {

                                    // the current selection ranges (auto-filter cannot be created on a multi selection)
                                    var ranges = docView.getSelectedRanges();
                                    if (ranges.length !== 1) {
                                        return SheetUtils.makeRejected('autofilter:multiselect');
                                    }

                                    // expand single cell to content range
                                    var range = sheetModel.getContentRangeForCell(ranges.first());
                                    // bug 36606: restrict range to used area of the sheet (range may become null)
                                    var usedRange = cellCollection.getUsedRange();
                                    range = usedRange ? range.intersect(usedRange) : null;

                                    // bug 36227: selection must not be entirely blank
                                    if (!range || cellCollection.areRangesBlank(range)) {
                                        return SheetUtils.makeRejected('autofilter:blank');
                                    }

                                    // generate the operations to insert the new table range
                                    return tableCollection.generateInsertTableOperations(generator, '', range, { table: { headerRow: true } });
                                });
                            }

                            // translate error codes for auto-filter
                            return promise2.catch(function (result) {
                                result.cause = result.cause.replace(/^table:/, 'autofilter:');
                                throw result;
                            });
                        }, { storeSelection: true });
                    });

                    // show warning alert if necessary
                    return docView.yellOnFailure(promise);
                }
            },

            // refreshes the filter and sorting of the selected table range
            'table/refresh': {
                parent: 'sheet/anytable',
                enable: function (tableModel) { return !!tableModel && tableModel.isRefreshable(); },
                set: function () {

                    // the active table model
                    var tableModel = this.getValue();
                    // ensure unlocked sheet (no filter manipulation in locked sheets)
                    var promise = docView.ensureUnlockedSheet();

                    // create and apply the operations to refresh the active table
                    promise = promise.then(function () {
                        return sheetModel.createAndApplyOperations(function (generator) {
                            return tableModel.generateRefreshOperations(generator, { updateFilter: true, updateSort: true });
                        }, { storeSelection: true });
                    });

                    // show warning alert if necessary
                    return docView.yellOnFailure(promise);
                }
            },

            // getter returns the style sheet identifier of the active table (table containing the active cell; empty string: no style);
            // setter will change the style sheet of ALL tables overlapping with the cell selection
            'table/stylesheet': {
                parent: 'sheet/realtable',
                enable: function (tableModel) { return !!tableModel; },
                get: function (tableModel) { return (tableModel && tableModel.getStyleId()) || ''; },
                set: function (styleId) {

                    // ensure unlocked sheet (no filter manipulation in locked sheets)
                    var promise = docView.ensureUnlockedSheet();

                    // create and apply the operations for all selected tables
                    promise = promise.then(function () {
                        return sheetModel.createAndApplyOperations(function (generator) {
                            var ranges = docView.getSelectedRanges();
                            var iterator = tableCollection.createModelIterator();
                            styleId = styleId || null; // use value null to clear table style
                            return self.iterateSliced(iterator, function (tableModel) {
                                if (!tableModel.isAutoFilter() && (tableModel.getStyleId() !== styleId) && ranges.overlaps(tableModel.getRange())) {
                                    tableModel.generateChangeOperations(generator, { styleId: styleId });
                                }
                            }, 'TableMixin.changeTableStyle');
                        }, { storeSelection: true });
                    });

                    // show warning alert if necessary
                    return docView.yellOnFailure(promise);
                }
            },

            // Applies formatting attributes for the specified table column. Setter expects an object
            // with the properties 'tableName', 'tableCol', and 'attributes' (incomplete attribute set
            // with filter and sorting attributes).
            'table/column/attributes': {
                parent: 'sheet/operation/unlocked/cell', // item does not depend on selected tables
                set: function (data) {

                    // ensure unlocked sheet (no filter manipulation in locked sheets)
                    var promise = docView.ensureUnlockedSheet();

                    // create and apply the operations for the specified table
                    promise = promise.then(function () {

                        // get the specified table range
                        var tableModel = tableCollection.getTable(data.tableName);
                        if (!tableModel) { return $.Deferred().reject(); }

                        // the data range of the table (may be null)
                        var dataRange = tableModel.getDataRange();
                        // the sorting attributes
                        var sortAttrs = Utils.getObjectOption(data.attributes, 'sort', null);
                        // whether some sort attributes have been changed
                        var sortAttrsChanged = false;

                        // check whether sort attributes will change
                        if (sortAttrs) {
                            var columnModel = tableModel.getColumnModel(data.tableCol);
                            var oldSortAttrs = columnModel.getMergedAttributeSet(true).sort;
                            sortAttrsChanged = _.some(sortAttrs, function (value, name) {
                                return oldSortAttrs[name] !== value;
                            });
                        }

                        // bug 56313: check for merged ranges before generating the operations
                        if (sortAttrsChanged && dataRange && mergeCollection.coversAnyMergedRange(dataRange)) {
                            return SheetUtils.makeRejected('sort:merge:overlap');
                        }

                        // create and apply the operations for the specified table
                        return sheetModel.createAndApplyOperations(function (generator) {
                            // if changing sort settings of a column, reset the sort settings of all other columns
                            var resetSort = sortAttrsChanged && (sortAttrs.type !== 'none');
                            // add new settings to current column
                            return tableModel.generateChangeColumnOperations(generator, data.tableCol, data.attributes, { resetSort: resetSort });
                        }, { storeSelection: true });
                    });

                    // show warning alert if necessary
                    return docView.yellOnFailure(promise);
                }
            }
        });

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
            cellCollection = sheetModel.getCellCollection();
            mergeCollection = sheetModel.getMergeCollection();
            tableCollection = sheetModel.getTableCollection();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = sheetModel = cellCollection = mergeCollection = tableCollection = null;
        });

    } // class TableMixin

    // exports ================================================================

    return TableMixin;

});
