/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/cellmixin', [
    'io.ox/office/spreadsheet/view/labels'
], function (Labels) {

    'use strict';

    // mix-in class CellMixin =================================================

    /**
     * Implementations of all controller items for manipulating cell contents
     * and formatting in the active sheet, intended to be mixed into a document
     * controller instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the cell selection.
     */
    function CellMixin(docView) {

        // self reference
        var self = this;

        // the document model and style collections
        var docModel = docView.getDocModel();
        var tableStyles = docModel.getTableStyles();

        // the models of the active sheet
        var sheetModel = null;
        var cellCollection = null;
        var tableCollection = null;

        // private methods ----------------------------------------------------

        /**
         * Returns whether the format painter is currently active.
         *
         * @returns {Boolean}
         *  Whether the format painter is currently active.
         */
        function isFormatPainterActive() {
            return docView.isCustomSelectionMode('painter');
        }

        /**
         * Activates or deactivates the format painter.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the format painter.
         */
        function activateFormatPainter(state) {

            // deactivate format painter: cancel custom selection mode
            if (!state) {
                docView.cancelCustomSelectionMode();
                return;
            }

            // the address of the active cell (used as formatting source)
            var address = docView.getActiveCell();
            // fill the auto-style of the active cell into the range that will be selected
            var contents = { s: cellCollection.getStyleId(address) };

            // do not lock editable cells by copying the 'unlocked' attribute (in locked sheets only, as in Excel)
            if (sheetModel.isLocked()) {
                contents.a = { cell: { unlocked: true } };
            }

            // resolve table style attributes from table model covered by the active cell
            var tableModel = tableCollection.findTable(address);
            var tableStyleId = tableModel ? tableModel.getStyleId() : null;
            if (tableStyleId) {
                contents.table = tableStyles.resolveCellAttributeSetInTable(tableStyleId, tableModel, address);
            }

            // start custom selection mode (wait for range selection)
            var promise = docView.enterCustomSelectionMode('painter', {
                selection: docView.getActiveCellAsSelection(),
                statusLabel: Labels.FORMAT_PAINTER_LABEL
            });

            // select the target range (bug 35295: check that the cells are not locked)
            promise = promise.then(function (selection) {
                docView.setCellSelection(selection);
                return docView.ensureUnlockedSelection();
            });

            // copy the formatting attributes to the target range
            promise.done(function () {
                self.executeItem('cell/fill', contents);
            });

            // do not return the promise (would block the GUI)
        }

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            'cell/painter': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                get: isFormatPainterActive,
                set: activateFormatPainter
            }

        });

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
            cellCollection = sheetModel.getCellCollection();
            tableCollection = sheetModel.getTableCollection();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = docModel = tableStyles = null;
            sheetModel = cellCollection = tableCollection = null;
        });

    } // class CellMixin

    // exports ================================================================

    return CellMixin;

});
