/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define('io.ox/office/settings/spellchecklanguagenotification', [
    'io.ox/core/event',
    'io.ox/office/tk/config',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/tk/forms',
    'gettext!io.ox/office/settings/main',
    'less!io.ox/office/settings/spellchecklanguagenotification'
], function (Event, Config, LocaleData, BaseDialog, Forms, gt) {

    'use strict';

    var
        // the keys for save/get the user's spell-check language blacklist/whitelist to the user settings
        BLACKLIST_SETTINGS_KEY = 'module/unsupportedspellchecklanguageblacklist',
        WHITELIST_SETTINGS_KEY = 'module/unsupportedspellchecklanguagewhitelist',

        // the user's language blacklist/whitelist
        languageBlacklist,
        languageWhitelist,

        editSettingsActionRegistry = {};

    // constructor ------------------------------------------------------------

    function SpellcheckNotification() {

        var newSection = false;

        // public methods -----------------------------------------------------
        this.newSection = function () {
            return newSection;
        };

        this.refresh = function (settings) {
            if (SpellcheckNotification.doesLanguageListExist()) {
                settings.appendButton(/*#. Button label for open the user spell-check languages dialog */gt('Set language notification'), (new Dialog()).show).attr('data-action', 'editspellchecknotification');
            }
        };
    }

    // static functions -------------------------------------------------------

    /**
     * does dynamically at runtime register an action (related to an action link)
     * once per application and keeps track on each application specific registration.
     * @param app
     */
    SpellcheckNotification.registerEditSettingsAction = function (app) {
        var
            applicationId = app.cid,
            registryItem  = editSettingsActionRegistry[applicationId];

        if (!registryItem || !registryItem.isRegistered) {

            app.getController().registerDefinition('document/settings/editspellchecknotification', {
                set: triggerEditSpellcheckNotification
            });

            editSettingsActionRegistry[applicationId] = {
                isRegistered: true
            };
        }
    };

    SpellcheckNotification.saveToLanguageWhitelist = function (languageList) {
        // sanitize received data - see BUG 50606 [https://bugs.open-xchange.com/show_bug.cgi?id=50606]
        languageList = languageList.filter(isValidLanguageKey);

        // sort/sanitize language list - see BUG 54502 [https://bugs.open-xchange.com/show_bug.cgi?id=54502] / 'Set language notification' not stored
        languageWhitelist = _.unique(languageList.concat(SpellcheckNotification.getLanguageWhitelist())).sort(ascendingAlphabeticalOrder);

        // save list as converted string - see BUG 54502
        Config.set(WHITELIST_SETTINGS_KEY, languageWhitelist.join(','));
    };

    /**
     * Return the Whitelist of Spell-check Languages.
     * @returns {String[]} the whitelist of spell-check languages
     */
    SpellcheckNotification.getLanguageWhitelist = function () {
        if (!languageWhitelist) {

            // - create sanitized and sorted language list from persistent string-value setting
            // - see BUG 54502 [https://bugs.open-xchange.com/show_bug.cgi?id=54502] / 'Set language notification' not stored

            languageWhitelist = Config.getStr(WHITELIST_SETTINGS_KEY) || Config.getArray(WHITELIST_SETTINGS_KEY);

            // stay defensive due to older settings of former releases that might have been stored as string type arrays.
            if (typeof languageWhitelist === 'string') {
                languageWhitelist = languageWhitelist.split(',');
            }
            languageWhitelist = languageWhitelist.filter(isValidLanguageKey).sort(ascendingAlphabeticalOrder);
        }
        return languageWhitelist;
    };

    /**
     * Return the Blacklist of Spell-check Languages.
     * @returns {String[]} the blacklist of spell-check languages
     */
    SpellcheckNotification.getLanguageBlacklist = function () {
        if (!languageBlacklist) {

            // - create sanitized and sorted language list from persistent string-value setting
            // - see BUG 54502 [https://bugs.open-xchange.com/show_bug.cgi?id=54502] / 'Set language notification' not stored

            languageBlacklist = Config.getStr(BLACKLIST_SETTINGS_KEY) || Config.getArray(BLACKLIST_SETTINGS_KEY);

            // stay defensive due to older settings of former releases that might have been stored as string type arrays.
            if (typeof languageBlacklist === 'string') {
                languageBlacklist = languageBlacklist.split(',');
            }
            languageBlacklist = languageBlacklist.filter(isValidLanguageKey).sort(ascendingAlphabeticalOrder);
        }
        return languageBlacklist;
    };

    /**
     * Does return whether or not there are any spell-check language items listed specifically under a user's settings.
     * @returns {boolean}
     * whether or not there are any user specific spell-check language items already listed/available.
     */
    SpellcheckNotification.doesLanguageListExist = function () {
        var
            currentBlacklist        = _.clone(SpellcheckNotification.getLanguageBlacklist()),
            currentWhitelist        = _.clone(SpellcheckNotification.getLanguageWhitelist()),

            unsupportedLanguageList = _.unique(currentBlacklist.concat(currentWhitelist)),
            isLanguageListDoesExist = (unsupportedLanguageList.length >= 1);

        return isLanguageListDoesExist;
    };

    // private functions ------------------------------------------------------

    function isValidLanguageKey(languageKey) {
        return (_.isString(languageKey)/* && (languageKey !== '')*/ && !!$.trim(languageKey) && (languageKey !== 'none'));
    }

    function triggerEditSpellcheckNotification() {
        if (SpellcheckNotification.doesLanguageListExist()) {

            (new Dialog()).show();
        }
    }

    function ascendingAlphabeticalOrder(a, b) {
        var
            a_name = a.toLowerCase(),
            b_name = b.toLowerCase();

        return (((a_name < b_name) && -1) || ((a_name > b_name) && 1) || 0);
    }

    /**
     * The Language Notification Edit Dialog which is visible if the user presses the setting's edit button.
     */
    function Dialog() {

        var
            $fieldset,
            $languageList,

            currentBlacklist,
            currentWhitelist,

            checkboxRegistry = {};

        /**
         * Fill the UI list with the words from the dictionary.
         */
        function renderCheckboxList(languageList) {

            languageList.map(function (languageKey) {
                return {
                    key: languageKey,
                    name: LocaleData.getLanguageName(languageKey.replace('-', '_'))
                };
            }).filter(function (languageTuple) { // sanitize list - see BUG 50606 [https://bugs.open-xchange.com/show_bug.cgi?id=50606]

                return (!!languageTuple.name && isValidLanguageKey(languageTuple.key));

            }).sort(function (a, b) {
                var
                    a_name = a.name.toLowerCase(),
                    b_name = b.name.toLowerCase();

                return (((a_name < b_name) && -1) || ((a_name > b_name) && 1) || 0);

            }).forEach(function (languageTuple) {
                var
                    languageKey = languageTuple.key,
                    languageName = languageTuple.name,
                    checkboxMarkup = Forms.createButtonMarkup({ attributes: { class: languageKey, role: 'checkbox' }, label: languageName }),
                    $checkbox = checkboxRegistry[languageKey] = $(checkboxMarkup);

                $languageList.append($checkbox);

                Forms.checkButtonNodes($checkbox, false, { design: 'boxed', ambiguous: false });
                Forms.setButtonKeyHandler($checkbox);

                $checkbox.on('click', function () {
                    var state = !$checkbox.find('i').hasClass('checked');
                    Forms.checkButtonNodes($checkbox, state, { design: 'boxed', ambiguous: false });
                });
            });
            currentBlacklist.forEach(function (languageKey) {
                var
                    $checkbox = checkboxRegistry[languageKey];

                if ($checkbox) {
                    Forms.checkButtonNodes($checkbox, true, { design: 'boxed', ambiguous: false });
                }
            });
        }

        /**
         * Show the edit dialog of how to handle notification(s) about unsupported spell-check language(s).
         */
        this.show = function () {
            var
                dialog = new BaseDialog('io-ox-office-settings', {
                    //#. Button label for the save button for the user dictionary dialog.
                    okLabel: gt('Save'),
                    //#. Title of the user's spell-check language blacklist dialog.
                    title: gt('Unsupported spell checker notification')
                }),
                copyEditList  = /*#. Text that describes a black-list of languages, a user wishes to be informed about, in case each of it is unsupported by spellchecking */ gt('Suppress missing spell checker warnings for the following languages:'),

                unsupportedLanguageList;

            currentBlacklist = _.clone(SpellcheckNotification.getLanguageBlacklist());
            currentWhitelist = _.clone(SpellcheckNotification.getLanguageWhitelist());

            unsupportedLanguageList = _.unique(currentBlacklist.concat(currentWhitelist));

            $fieldset = $([

                '<fieldset>',
                '<legend class="sectiontitle expertmode">',

                copyEditList,

                '</legend>',
                '</fieldset>'

            ].join(''));

            $languageList = $('<div class="form-group expertmode">')/*.css({
                height: _.device('smartphone') ? $(window).height() - 300 : 307,
                overflowY: 'auto',
                minHeight: 60
            })*/;

            renderCheckboxList(unsupportedLanguageList);

            dialog.append($fieldset.append($languageList));

            dialog.getBody().addClass('settings-details');

            // action handler for the OK button
            dialog.setOkHandler(function () {
                var
                    updatedBlacklist = [],
                    updatedWhitelist = [];

                Object.keys(checkboxRegistry).forEach(function (languageKey) {
                    var
                        $checkbox = checkboxRegistry[languageKey];

                    if ($checkbox.find('i').hasClass('checked')) {
                        updatedBlacklist.push(languageKey);
                    //updatedWhitelist.push(languageKey);
                    } else {
                    //updatedBlacklist.push(languageKey);
                        updatedWhitelist.push(languageKey);
                    }
                });
                // sort/sanitize language lists - see BUG 54502 [https://bugs.open-xchange.com/show_bug.cgi?id=54502] / 'Set language notification' not stored
                languageBlacklist = updatedBlacklist.sort(ascendingAlphabeticalOrder);
                languageWhitelist = updatedWhitelist.sort(ascendingAlphabeticalOrder);

                // save lists as converted strings - see BUG 54502
                Config.set(BLACKLIST_SETTINGS_KEY, languageBlacklist.join(','));
                Config.set(WHITELIST_SETTINGS_KEY, languageWhitelist.join(','));
            });

            dialog.show();
        };
    }

    // initialization ---------------------------------------------------------

    Event.extend(SpellcheckNotification);

    return SpellcheckNotification;

});
