/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/slideformatmanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/presentation/utils/presentationutils'
], function (Utils, TriggerObject, DrawingFrame, AttributeUtils, Config, DOM, PresentationUtils) {

    'use strict';

    // class SlideFormatManager ===============================================

    /**
     * An instance of this class represents the slide formatting manager. This manager
     * keeps track of unformatted slides. They are formatted on demand for performance
     * reasons.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {PresentationApplication} app
     *  The application instance.
     */
    function SlideFormatManager(app) {

        var // self reference
            self = this,
            // a reference to the model
            model = null,
            // a collector for all unformatted slides
            // -> this is an object, whose keys are the IDs of the unformatted slides
            allUnformattedSlides = null,
            // a collector for all unformatted slides loaded from local storage
            localStorageSlides = null,
            // a collector for all slides that are currently formatted
            // -> avoiding that slides, whose formatting takes a long time, are formatted more than once
            runningTasks = {},
            // the order of the document slides
            docOrder = null,
            // the order of the master and layout slides
            layoutOrder = null,
            // a function that checks for idle time to update slides in background
            idleTimeChecker = $.noop,
            // the idle time (in ms), after that a slide formatting is triggered
            idleTime = Config.AUTOTEST ? 10 : 2500,
            // a collector for all those slides, that need to be shifted during formatting of the active slide
            shiftedSlides = null,
            // a collector for slide IDs, that are waiting to inform the listeners, because dependent slides are currently formatted
            waitingIDs = null,
            // whether ther first tab is already activated after loading the document (performance)
            firstTabActivated = false,
            // a queue for slide IDs that are externally demanded during the loading phase (performance)
            loadingQueue = [];

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Removing a task from the collector for all unformatted slides.
         *
         * @param {String} id
         *  The ID of the slide that is no longer unformatted.
         */
        function removeTask(id) {

            // removing slide from the collector of unformatted slides
            if (allUnformattedSlides && allUnformattedSlides[id]) {
                delete allUnformattedSlides[id];
                if (localStorageSlides) { delete localStorageSlides[id]; }
            } else {
                if (allUnformattedSlides) {
                    Utils.warn('SlideFormatManager: Trying to remove non existing task: ' + id);
                } else {
                    Utils.warn('SlideFormatManager: Trying to remove task: ' + id + ' from non existent container of unformatted slides.');
                }
            }

            if (model.isStandardSlideId(id)) {
                docOrder = _.without(docOrder, id);
            } else {
                layoutOrder = _.without(layoutOrder, id);
            }

            if (_.isEmpty(allUnformattedSlides)) {
                allUnformattedSlides = null;
                docOrder = null;
                layoutOrder = null;
                localStorageSlides = null;
            }
        }

        /**
         * Updating the document slides belonging to a specified layout slide in that way, that
         * the correct customized template texts are shown.
         *
         * @param {String} id
         *  The ID of a (layout) slide.
         */
        function updateAllCustomizedTemplateTextOnSlide(id) {
            _.each(model.getAllSlideIDsWithSpecificTargetParent(id), function (slideId) {
                model.getDrawingStyles().handleAllEmptyPlaceHolderDrawingsOnSlide(model.getSlideById(slideId), { ignoreTemplateText: true });
            });
        }

        /**
         * Element formatting for one slide and its content.
         *
         * @param {jQuery} slide
         *  The jQuerified slide node.
         *
         * @param {String} id
         *  The ID of the slide that will be formatted.
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after the specified slide is formatted
         *  completely.
         */
        function formatOneSlide(slide, id) {

            if (app.isInQuit()) {
                return $.Deferred().reject();
            }

            var // whether the visibility of the slide has changed for formatting reasons
                visibilityChanged = false,
                // the paragraphs in the slide
                paragraphContainer = $(),
                // the slide styles object
                drawingStyles = model.getDrawingStyles(),
                // collector for all drawings on the slide
                allDrawings = slide.find('.drawing'),
                // whether the document slides need to be updated because of a customized template text
                updateCustomizedTemplateTexts = false,
                // a helper index for body place holder drawings without index (after loading)
                odfBodyIndex = 0,
                // whether the slide was loaded from local storage and can be formatted faster (and it has been formatted before)
                useLessFormatting = localStorageSlides && localStorageSlides[id] && slide.is('.slideformatted') && !model.externalLocalStorageOperationsExist(),
                // a jQuery promise object
                promise = null;

            // updating one drawing on the slide
            function updateOneDrawing(oneDrawing) {

                var // the paragraphs in one drawing
                    // -> reading the paragraphs after drawingStyles.updateElementFormatting because of implicit paragraphs in ODP
                    paragraphs = null,
                    // the attribute object of family 'presentation' of the drawing
                    presentationAttrs = null,
                    // the jQueryfied drawing node
                    drawing = $(oneDrawing),
                    // whether this is an image drawing
                    isImage = drawing.attr('data-type') === 'image',
                    // whether this is a table drawing
                    isTable = drawing.attr('data-type') === 'table';

                if (!model || model.destroyed) { return; } // the model might have been destroyed in the meantime

                // Performance: When loaded from local storage or fast load the 'src' property has to be renamed
                if (isImage) { model.prepareImageNodeFromStorage(drawing); }

                // Preformance: When loaded from local storage, only drawings with Canvas need to be restored
                if (useLessFormatting && drawing.find(DrawingFrame.CANVAS_NODE_SELECTOR).length === 0) { return; }

                if (app.isODF() && PresentationUtils.isBodyPlaceHolderWithoutIndexDrawing(drawing)) { // repairing the missing index
                    presentationAttrs = AttributeUtils.getExplicitAttributes(drawing, { family: 'presentation', direct: true });
                    presentationAttrs.phIndex = odfBodyIndex++; // modifying the original presentation attribute
                }

                // Performance: Grouped images and grouped tables need to be formatted once completely
                if (isImage || isTable) { drawing.addClass('formatrequired'); }

                drawingStyles.updateElementFormatting(drawing);

                drawing.removeClass('slideformat'); // removing marker after updateElementFormatting, it is no longer needed

                paragraphs = drawing.find(DOM.PARAGRAPH_NODE_SELECTOR); // but not in tables

                // also updating all paragraphs inside text frames (not searching twice for the drawings)
                // -> using option 'suppressFontSize', so that the dynamic font size calculation is NOT
                //    started during formatting of a complete slide.
                // The paragraphs inside tables will be updated by table formatting. Additional formatting
                // leads to problems with tabulators (53672)
                if (!DrawingFrame.isTableDrawingFrame(drawing)) {
                    model.implParagraphChangedSync(paragraphs, { suppressFontSize: true });
                }

                paragraphContainer = paragraphContainer.add(paragraphs);

                // marking collecting all drawings with customized template text
                if (model.isLayoutSlideId(id) && PresentationUtils.isCustomPromptPlaceHolderDrawing(drawing)) { updateCustomizedTemplateTexts = true; }
            }

            // Performance: Changing order of drawings, so that the group drawings are formatted after
            //              all child drawings are added to the group.
            function handleGroupNodes() {

                // collecting all drawings of type 'group'
                var splittedValues = _.partition(allDrawings, function (drawing) { return $(drawing).attr('data-type') === 'group'; });
                // a container for all drawings of type 'group' on the slide
                var groupDrawings = splittedValues[0];

                // partition generates two separated arrays
                allDrawings = splittedValues[1];  // not group drawings

                if (groupDrawings.length > 0) {
                    // adding the group drawings to the end of the collection
                    groupDrawings.reverse(); // formatting inner groups before outer groups
                    allDrawings = allDrawings.concat(groupDrawings);
                }
            }

            // attaching the slide node to the DOM, if required
            model.appendSlideToDom(id);

            // Performance: Removing marker class from slides that they received after fast load handling
            slide.removeClass(DOM.HIDDENAFTERATTACH_CLASSNAME);

            // making slide visible for formatting
            visibilityChanged = model.handleContainerVisibility(slide, { makeVisible: true });

            slide.addClass('leftshift');

            // setting a marker to group nodes, so that this process is recognized as slide format manager formatting
            allDrawings.addClass('slideformat');

            // appending drawings of type 'group' to the end of the drawing list, so that it is not necessary
            // to update the complete group after each client
            handleGroupNodes();

            // Important: Using a local promise, not an external created by 'updateElementFormatting'
            promise = self.createResolvedPromise();

            // updating the slide ...
            promise = promise.then(function () { return model.getSlideStyles().updateElementFormatting(slide); });

            // updating all drawings via iteration (in case of success, but also in error case (58241))
            promise = promise.then(function () {
                return self.iterateArraySliced(allDrawings, updateOneDrawing, 'SlideFormatManager.formatOneSlide');
            }, function () {
                return self.iterateArraySliced(allDrawings, updateOneDrawing, 'SlideFormatManager.formatOneSlide');
            });

            promise.done(function () {
                // updating the lists also synchronously
                model.updateLists(null, paragraphContainer);
                // updating the customized template texts on document slides
                if (updateCustomizedTemplateTexts) { updateAllCustomizedTemplateTextOnSlide(id); }
            });

            promise.always(function () {

                // shifting back the slide after formatting
                slide.css('left', '');
                slide.removeClass('leftshift');

                if (visibilityChanged) {
                    // making the slide invisible again
                    if (!model || model.destroyed) { return; } // the model might have been destroyed in the meantime
                    model.handleContainerVisibility(slide, { makeVisible: false });
                }

                // let the new formatted slide stay in the DOM for some seconds
                model.getSlideVisibilityManager().registerOperationSlideId(id);

                // this task is no longer running
                if (runningTasks[id]) { delete runningTasks[id]; }

                // remove slide from local storage collector in any case
                if (localStorageSlides) { delete localStorageSlides[id]; }
            });

            promise.done(function () {
                removeTask(id); // updating the collector of unformatted slides
                // Setting marker class 'slideformatted' for following loading with local storage.
                // This marker is saved in local storage and specifies, that the drawings on this
                // slide do not need to be formatted (except those drawings that contain a Canvas
                // node).
                slide.addClass('slideformatted');
            });

            promise.fail(function () {
                Utils.error('Error during slide formatting! ID: ' + id);
                removeTask(id); // updating the collector of unformatted slides -> avoiding endless formatting (58241)
            });

            return promise;
        }

        /**
         * Formatting a full set of slides. The caller of this function must make sure,
         * that after all specified slides (in container slideIDs) are formatted, the listener
         * can be informed about a fully formatted slide (all required layout and master
         * slides are also formatted).
         *
         * Info: All slide IDs that are specified in the container 'slideIDs' must exist
         *       in the global collector 'allUnformattedSlides'.
         *
         * @param {String[]) slideIDs
         *  The IDs of the slide(s) that need to be updated. It is necessary, that this
         *  is always
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after all specified slides
         *  are formatted completely.
         */
        function formatSlides(slideIDs) {

            // -> check for running tasks, so that a slide is not formatted again
            var newSlideIDs = _.reject(slideIDs, function (id) {
                return runningTasks[id];
            });

            // blocking these IDs for further formatting
            _.each(newSlideIDs, function (id) {
                runningTasks[id] = 1;
            });

            // -> iterating asynchronously over the collected slide IDs (formatting slide(s) asynchronously)
            return self.iterateArraySliced(newSlideIDs, function (id) {
                return formatOneSlide(model.getSlideById(id), id);
            }, 'SlideFormatManager.formatSlides');
        }

        /**
         * Registering the handler to find idle time for formatting slides.
         *
         * @returns {undefined}
         */
        function registerIdleTimeChecker() {
            idleTimeChecker = self.createDebouncedMethod('SlideFormatManager.idleTimeChecker', null, idleSlideFormatHandler, { delay: idleTime });
            model.getNode().on('keydown mousedown touchstart', idleTimeChecker);
        }

        /**
         * Deregistering the handler to find idle time for formatting slides. This
         * can be done, if there are no more open tasks.
         */
        function deregisterIdleTimeChecker() {
            model.getNode().off('keydown mousedown touchstart', idleTimeChecker);
            idleTimeChecker = _.noop;
        }

        /**
         * Determining the ID of the slide that needs to be formatted next. This is done
         * with an idle time, so that it is possible to format any of the remaining
         * unformatted slides.
         *
         * @returns {String}
         *  The ID of the slide that shall be formatted.
         */
        function getNextFormatSlideId() {

            var // whether the layout/master view is visible
                isMasterView = model.isMasterView(),
                // the id of the slide, that shall be formatted
                slideID = isMasterView ? _.first(layoutOrder) : _.first(docOrder);

            // Fallback: simply take one arbitrary slide ID
            if (!slideID || !allUnformattedSlides[slideID]) { slideID = _.first(_.keys(allUnformattedSlides)); }

            return slideID;
        }

        /**
         * Handler to make slide formatting, if there was no user input for a specified time.
         */
        function idleSlideFormatHandler() {

            var // the ID of the slide that will be formatted
                slideID = null,
                // the set of affected slides
                set = null,
                // a collector for the unformatted slides
                formatCollector = null;

            if (!model || model.destroyed) { return; } // handling closing of application

            if (!firstTabActivated) { // handling loading of document
                idleTimeChecker();
                return;
            }

            if (self.unformattedSlideExists()) {

                // Trigger formatting one slide
                slideID = getNextFormatSlideId();

                // -> this must be a full set of slides
                set = model.getSlideIdSet(slideID);

                if (set.id && allUnformattedSlides[set.id] && !runningTasks[set.id]) {
                    formatCollector = [set.id];
                }

                if (set.layoutId && allUnformattedSlides[set.layoutId] && !runningTasks[set.layoutId]) {
                    if (formatCollector) {
                        formatCollector.push(set.layoutId);
                    } else {
                        formatCollector = [set.layoutId];
                    }
                }

                if (set.masterId && allUnformattedSlides[set.masterId] && !runningTasks[set.masterId]) {
                    if (formatCollector) {
                        formatCollector.push(set.masterId);
                    } else {
                        formatCollector = [set.masterId];
                    }
                }

                if (formatCollector) {
                    // formatting the collected slides
                    formatSlides(formatCollector).done(function () {
                        // informing the listeners
                        informListeners(formatCollector);
                        // check for waiting listeners
                        checkWaitingListeners(formatCollector); // checking, if waiting listeners can be informed
                        // triggering next slide update
                        if (self.unformattedSlideExists()) {
                            idleTimeChecker();
                        } else {
                            deregisterIdleTimeChecker();
                        }
                    });
                } else {
                    // triggering next slide update
                    if (self.unformattedSlideExists()) {
                        idleTimeChecker();
                    } else {
                        deregisterIdleTimeChecker();
                    }
                }

            } else {
                deregisterIdleTimeChecker();
            }
        }

        /**
         * Listener for the event 'change:slide', that is triggered from the model,
         * if the active slide was modified.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.newSlideId='']
         *      The id of the standard slide that will be activated.
         *  @param {String} [options.newLayoutId='']
         *      The id of the layout slide that will be activated.
         *  @param {String} [options.newMasterId='']
         *      The id of the master slide that will be activated.
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after all specified
         *  slides are formatted completely.
         *  Info: This promise should not be used, except in the case of the function
         *        'forceSlideFormatting'. This is required for the first visible slide
         *        that needs to block the document load process. Typically all affected
         *        processes must be informed via the event 'slidestate:formatted'.
         */
        function changeSlideHandler(event, options) {

            var // a collector for the unformatted slides
                formatCollector = null;

            if (!self.unformattedSlideExists()) { return $.when(); }

            if (options.newSlideId && allUnformattedSlides[options.newSlideId] && !runningTasks[options.newSlideId]) {
                formatCollector = [options.newSlideId];
            }

            if (options.newLayoutId && allUnformattedSlides[options.newLayoutId] && !runningTasks[options.newLayoutId]) {
                if (formatCollector) {
                    formatCollector.push(options.newLayoutId);
                } else {
                    formatCollector = [options.newLayoutId];
                }
            }

            if (options.newMasterId && allUnformattedSlides[options.newMasterId] && !runningTasks[options.newMasterId]) {
                if (formatCollector) {
                    formatCollector.push(options.newMasterId);
                } else {
                    formatCollector = [options.newMasterId];
                }
            }

            if (formatCollector) {

                // formatting the collected slides
                return formatSlides(formatCollector).done(function () {
                    // informing the listeners
                    informListeners(formatCollector);
                    // check for waiting listeners
                    checkWaitingListeners(formatCollector); // checking, if waiting listeners can be informed
                });
            } else {
                // ignoring that slides might be currently formatted (this has to be handled by the caller)
                return $.when();
            }
        }

        /**
         * Listener for the event 'request:formatting' that can be triggered by an
         * external source to request a formatting update of the specified slides.
         * external trigger of slide formatting
         *
         * The formatting of the slides is done asynchronously.
         *
         * This function also handles that only 'complete' formatting is done. If
         * a document slide is in the specified list 'allSlides', the corresponding
         * layout and master slide are also formatted, if this is necessary.
         *
         * Listeners are informed with the event 'slidestate:formatted' from the
         * model, that the specified slides are formatted.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {String|String[]|Object} allSlides
         *  The IDs of the slide(s) that need to be updated. This can be a single string,
         *  an array of strings or an object, whose values are interpreted as string IDs.
         *  Such an object is created by the function 'getSlideIdSet' from the model.
         */
        function externalFormattingHandler(event, allSlides) {

            var // the container for the slide IDs
                container = null;

            if (_.isString(allSlides)) {
                container = [allSlides];
            } else if (_.isArray(allSlides)) {
                container = allSlides;
            } else if (_.isObject(allSlides)) {
                container = _.values(allSlides);
            }

            // shortcut, if there are no unformatted slides
            if (!self.unformattedSlideExists()) {
                // informing the listeners that all slides are formatted
                informListeners(container);
                return;
            }

            if (!firstTabActivated) {
                _.each(container, function (id) { loadingQueue.push(id); });
                return; // do nothing, until first tab is activated
            }

            _.each(container, function (id) {

                var // the set of affected slides
                    set = model.getSlideIdSet(id),
                    // a collector for the unformatted slides
                    formatCollector = null,
                    // whether the triggering needs to be debounced
                    waitingRequired = false,
                    // a collector for all IDs of those slides, that are currently formatted
                    waitForIDs = null;

                if (set.id && allUnformattedSlides[set.id]) {
                    if (runningTasks[set.id]) {
                        waitForIDs = {};
                        waitForIDs[set.id] = 1;
                    } else {
                        formatCollector = [set.id];
                    }
                }

                if (set.layoutId && allUnformattedSlides[set.layoutId]) {
                    if (runningTasks[set.layoutId]) {
                        waitForIDs = waitForIDs || {};
                        waitForIDs[set.layoutId] = 1;
                    } else {
                        if (formatCollector) { // the document slide is already in the format collector
                            waitForIDs = waitForIDs || {};
                            waitForIDs[set.layoutId] = 1; // the document slide needs to wait for finished formatting of layout slide
                        }

                        if (formatCollector) {
                            formatCollector.push(set.layoutId);
                        } else {
                            formatCollector = [set.layoutId];
                        }
                    }
                }

                if (set.masterId && allUnformattedSlides[set.masterId]) {
                    if (runningTasks[set.masterId]) {
                        waitForIDs = waitForIDs || {};
                        waitForIDs[set.masterId] = 1;
                    } else {
                        if (formatCollector) { // the document and/or layout slides are already in the format collector
                            waitForIDs = waitForIDs || {};
                            waitForIDs[set.masterId] = 1; // the document and/or layout slide needs to wait for finished formatting of master slide
                        }

                        if (formatCollector) {
                            formatCollector.push(set.masterId);
                        } else {
                            formatCollector = [set.masterId];
                        }
                    }
                }

                // Is it necessary to inform the listener, even if all slides are already formatted -> yes

                if (formatCollector) {
                    if (waitForIDs) { // some slides are currently formatted
                        // adding the IDs of the format collector to the 'waitForIDs' collector
                        _.each(formatCollector, function (oneID) { waitForIDs[oneID] = 1; });
                        informListenersLater(waitForIDs, id); // the listeners need to be informed later
                        waitingRequired = true;
                    }

                    // formatting the collected slides
                    formatSlides(formatCollector).done(function () {
                        checkWaitingListeners(formatCollector); // checking, if waiting listeners can be informed
                        if (!waitingRequired) {
                            informListeners([id]); // informing the listeners, that the requested slide can be used
                        }
                    });
                } else {
                    if (waitForIDs) {
                        informListenersLater(waitForIDs, id); // the listeners need to be informed later
                    } else {
                        informListeners([id]); // informing the listeners, that the requested slide can be used
                    }
                }
            });

        }

        /**
         * Registering a slide ID for deferred information of listeners. The listeners can only be
         * informed, if all dependent slides (saved in waitObject) are also formatted.
         *
         * @param {Object} waitObject
         *  An object containing as keys all those slide IDs of those slides, that need to be formatted,
         *  before the listeners can be informed, that the slide with the specified ID is formatted
         *  completely.
         *
         * @param {String} id
         *  The ID of that slide, that can only be used by listeners, if all dependent slides (whose
         *  ID is saved in waitObject) are also formatted.
         */
        function informListenersLater(waitObject, id) {
            waitingIDs = waitingIDs || {};
            waitingIDs[id] = waitObject;
        }

        /**
         * Checking, if listeners can be informed, that a specified slide can be used.
         *
         * @param {String[]} formattedIDs
         *  An array containing the IDs of all new formatted slides.
         */
        function checkWaitingListeners(formattedIDs) {
            if (waitingIDs) {
                _.each(waitingIDs, function (waitIDs, id) {

                    _.each(formattedIDs, function (formattedID) {
                        delete waitIDs[formattedID];
                    });

                    if (_.isEmpty(waitIDs)) {
                        delete waitingIDs[id];
                        informListeners([id]); // finally the waiting listeners can be informed
                    }
                });
                if (_.isEmpty(waitingIDs)) { waitingIDs = null; }
            }
        }

        /**
         * Informing the listeners, that the slide with the specified IDs are formatted.
         * This includes a check, if the model was destroyed in the meantime, because the trigger
         * is executed with a delay (required for performance reasons).
         *
         * @param {String[]} allSlideIDs
         *  An array with all formatted slide IDs.
         */
        function informListeners(allSlideIDs) {
            self.executeDelayed(function () {
                if (model && !model.destroyed) { model.trigger('slidestate:formatted', allSlideIDs); }
            }, 'SlideFormatManager.informListeners', 50);
        }

        /**
         * Registering a listener function for the 'tab:activated' event, that registers
         * the first finished activation of a toolbar tab.
         */
        function registerFirstTabActivatedHandler() {

            function registerTabActivation() {
                // adding delay for slide formatting, so that toolbar is really rendered by the browser
                self.executeDelayed(function () {
                    firstTabActivated = true;
                    if (!_.isEmpty(loadingQueue)) {
                        externalFormattingHandler(null, loadingQueue); // calling external handler directly
                    }
                }, 'SlideFormatManager.registerFirstTabActivatedHandler', 100);
            }

            if (app.getUserSettingsValue('showToolbars', true)) {
                app.getView().getToolBarTabs().one('tab:activated', registerTabActivation);
            } else {
                model.waitForImportSuccess(registerTabActivation); // alternative event, triggered earlier
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Whether there is at least one unformatted slide.
         *
         * @returns {Boolean}
         *  Whether there is at least one unformatted slide.
         */
        this.unformattedSlideExists = function () {
            return allUnformattedSlides && !_.isEmpty(allUnformattedSlides);
        };

        /**
         * Whether all slides are formatted.
         *
         * @returns {Boolean}
         *  Whether all slides are formatted.
         */
        this.allSlidesFormatted = function () {
            return !self.unformattedSlideExists();
        };

        /**
         * Whehter at least one slide is currently formatted.
         *
         * @returns {Boolean}
         *  Whehter at least one slide is currently formatted.
         */
        this.runningTasksExist = function () {
            return !_.isEmpty(runningTasks);
        };

        /**
         * Whether the slide specified by its ID is an unformatted slide. It is
         * possible, that the slide is currently formatted.
         *
         * @param {String} id
         *  The id of a slide.
         *
         * @returns {Boolean}
         *  Whether the slide specified by its ID is an unformatted slide.
         */
        this.isUnformattedSlide = function (id) {
            return allUnformattedSlides && !_.isUndefined(allUnformattedSlides[id]);
        };

        /**
         * Whether the slide specified ty its ID is already formatted.
         *
         * @param {String} id
         *  The id of a slide.
         *
         * @returns {Boolean}
         *  Whether the slide specified by its ID is formatted.
         */
        this.isFormattedSlide = function (id) {
            return !self.isUnformattedSlide(id);
        };

        /**
         * Checks whether a specified slide is currently formatted.
         *
         * @param {String} id
         *  The ID of the slide, that needs to be formatted.
         *
         * @returns {Boolean}
         *  Whether the slide with the specified id is currently formatted.
         */
        this.isRunningTask = function (id) {
            return runningTasks[id] === 1;
        };

        /**
         * Whether the specified slide set is already completely formatted.
         *
         * @param {String[]} set
         *  The set of slide IDs.
         *
         * @returns {Boolean}
         *  Whether all the slides specified by its IDs are formatted.
         */
        this.isSlideSetFormatted = function (set) {
            return _.every(set, function (id) {
                return self.isFormattedSlide(id);
            });
        };

        /**
         * Whether the specified slide set is already completely formatted
         * or the slides are currently formatted. In this case no new formatting
         * process must be triggered.
         *
         * @param {String[]} set
         *  The set of slide IDs.
         *
         * @returns {Boolean}
         *  Whether all the slides specified by its IDs are already formatted
         *  or are currently formatted.
         */
        this.isSlideSetCurrentlyFormatted = function (set) {
            return _.every(set, function (id) {
                return self.isFormattedSlide(id) || self.isRunningTask(id);
            });
        };

        /**
         * Adding additional IDs to the container of unformatted slide IDs.
         *
         * @param {String|String[]} IDs
         *  A single string or an array of strings with the IDs of slides, that need
         *  to be formatted later.
         */
        this.addTasks = function (IDs) {

            if (_.isString(IDs)) {
                if (model.isValidSlideId(IDs)) {
                    if (!allUnformattedSlides) { allUnformattedSlides = {}; }
                    allUnformattedSlides[IDs] = 1;
                }
            } else if (_.isArray(IDs)) {
                if (!allUnformattedSlides) { allUnformattedSlides = {}; }
                _.each(IDs, function (id) {
                    allUnformattedSlides[id] = 1;
                });
            }
        };

        /**
         * Shifting a specified slide set far to the left, so that it cannot be seen by the user. This is necessary
         * during formatting of the slide. In this case the complete set of slide IDs need to be shifted, even though
         * some slides of it might already be completely formatted.
         *
         * @param {String[]} idSet
         *  The set of slide IDs that will be shifted to the left, so that the user cannot see it.
         */
        this.shiftSlideSetOutOfView = function (idSet) {

            if (shiftedSlides) { return; } // only one set of shifted slides is supported

            _.each(idSet, function (id) {
                var oneSlide = model.getSlideById(id);
                if (oneSlide) {
                    oneSlide.addClass('leftshift');  // using negative left offset, so that the scroll bar is not influenced
                    shiftedSlides = shiftedSlides || [];
                    shiftedSlides.push(id); // avoing 'null' values in shiftSlides container
                }
            });
        };

        /**
         * Shifting back the slides that were shifted using 'shiftSlideSetOutOfView'. This must be done, if the active
         * slide does not need this shifting, because all slides are formatted.
         */
        this.shiftSlideSetIntoView = function () {

            _.each(shiftedSlides, function (id) {
                var oneSlide = null;
                if (!self.isRunningTask(id)) { // not shifting into view, if the slide is currently formatted
                    oneSlide = model.getSlideById(id);
                    if (oneSlide) { oneSlide.removeClass('leftshift'); } // using negative left offset, so that the scroll bar is not influenced
                }
            });

            shiftedSlides = null;
        };

        /**
         * Whether there are some slides shifted due to formatting reasons. During a long running formatting process
         * all three slide nodes are shifted far to the left, so that the user cannot see half formatted slides. Even
         * if master or layout slide are completely formatted, it makes no sense to show them until the document
         * slide is completely formatted. Therefore the array 'shiftedSlides' contains the complete collection
         * of a slide ID set.
         *
         * @returns {Boolean}
         *  Whether there are some slides shifted due to formatting reasons.
         */
        this.shiftedSlideExist = function () {
            return shiftedSlides !== null;
        };

        /**
         * Whether a slide with a specified ID is shifted due to formatting reasons. This slide can already be
         * completely formatted, but it is shifted, because another slide of the slide ID set of the active slide
         * is not completely formatted.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {Boolean}
         *  Whether the specified slide is shifted due to formatting reasons.
         */
        this.isShiftedSlide = function (id) {
            return _.contains(shiftedSlides, id);
        };

        /**
         * Forcing an update of one specified slide and its layout and master slide (if required).
         * This process is especially required for formatting the first visible slide.
         *
         * Important: The caller of this function has to check, if the slide with the specified
         *            id is not currently formatted. Otherwise it might happen, that the function
         *            'changeSlideHandler' returns a resolved promise, even though the slide is
         *            currently formatted and this formatting has not finished yet.
         *            Check for the caller: if (!slideFormatManager.isRunningTask(id)) { ...
         *
         * @param {String|String[]} id
         *  The ID of the slide, that needs to be formatted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the specified slide and all its parent slides
         *  are formatted.
         */
        this.forceSlideFormatting = function (id) {

            var // the set of affected slides
                set = _.isString(id) ? model.getSlideIdSet(id) : id;

            return changeSlideHandler(null, { newSlideId: set.id, newLayoutId: set.layoutId, newMasterId: set.masterId });
        };

        /**
         * Setting the initial tasks for the slide formatting manager. This is typically done with
         * the object 'idTypeConnection' from the model.
         *
         * @param {Object} idObject
         *  An object, that contains the IDs of the slides, that need to be formatted as keys.
         *
         * @param {Array} docSlideOrder
         *  Copy of the container with the document slide IDs in the correct order.
         *
         * @param {Array} layoutSlideOrder
         *  Copy of the container with the slide IDs of master and layout slides in the correct order.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.LocalStorage=false]
         *      Whether the document was loaded from local storage.
         */
        this.initializeManager = function (idObject, docSlideOrder, layoutSlideOrder, options) {

            allUnformattedSlides = idObject;
            docOrder = docSlideOrder;
            layoutOrder = layoutSlideOrder;

            // slides loaded from local storage can be formatted faster
            if (Utils.getBooleanOption(options, 'localStorage', false)) { localStorageSlides = _.copy(allUnformattedSlides, true); }

            // registering a listener for the event 'request:formatting', that can be triggered
            // to force the slide format manager to format some slides specified by their IDs.
            self.listenTo(model, 'request:formatting', externalFormattingHandler);

            // registering a listener for the activation of the first tab
            registerFirstTabActivatedHandler();

            // starting the process to format slides, if there is no user input for a specified time
            registerIdleTimeChecker();
            idleTimeChecker(); // starting process

            //model.trigger('formatmanager:init');
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            deregisterIdleTimeChecker();
            model = allUnformattedSlides = localStorageSlides = docOrder = layoutOrder = null;
        });

    } // class SlideFormatManager

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SlideFormatManager });
});
