/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/toolbartabgroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiogroup'
], function (Utils, KeyCodes, Forms, RadioGroup) {

    'use strict';

    // class ToolBarTabGroup ==================================================

    /**
     * A button group in the top-pane used to select the active tool bar.
     *
     * @constructor
     *
     * @extends RadioGroup
     *
     * @param {ToolBarTabCollection} toolBarTabs
     *  The collection with all tool bars registered for the application.
     */
    function ToolBarTabGroup(docView, toolBarTabs) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, docView, { classes: 'tab-group', role: 'tablist' });

        // private methods ----------------------------------------------------

        function tabCreateHandler(event, tabId, options) {
            self.createOptionButton(tabId, Utils.extendOptions(options, { attributes: { role: 'tab' } }));
        }

        function tabLabelHandler(event, tabId, label) {
            Forms.setCaptionText(self.findOptionButtons(tabId), label);
        }

        function keyDownHandler(event) {

            var startNode = $(event.target),
                focusableControls = Forms.findFocusableNodes(self.getNode()),
                newSelectedNode = null;

            if (KeyCodes.matchKeyCode(event, 'LEFT_ARROW')) {
                newSelectedNode = Utils.moveNodeFocus(startNode, focusableControls, 'left');
            } else if (KeyCodes.matchKeyCode(event, 'RIGHT_ARROW')) {
                newSelectedNode = Utils.moveNodeFocus(startNode, focusableControls, 'right');
            }

            if (newSelectedNode) {
                self.triggerChange(Forms.getButtonValue(newSelectedNode), { preserveFocus: true });
                event.preventDefault();
            }
        }

        // initialization -----------------------------------------------------

        // update control according to the tab collection
        this.listenTo(toolBarTabs, {
            // create button element for a new tab
            'tab:create': tabCreateHandler,
            'tab:label': tabLabelHandler,
            // show/hide tab buttons
            'tab:show': function (event, tabId) { Forms.showNodes(self.findOptionButtons(tabId), true); },
            'tab:hide': function (event, tabId) { Forms.showNodes(self.findOptionButtons(tabId), false); },
            // update selected button
            'tab:activate': function (event, tabId) { self.setValue(tabId); }
        });

        // add own keyboard handling
        this.getNode().on('keydown', keyDownHandler);

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docView = toolBarTabs = null;
        });

    } // class ToolBarTabGroup

    // exports ================================================================

    // derive this class from class RadioGroup
    return RadioGroup.extend({ constructor: ToolBarTabGroup });

});
