/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/mixedborder', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border'
], function (Utils, Color, Border) {

    'use strict';

    // the names of all border properties in cells, mapped by border mode properties
    var CELL_BORDER_ATTRIBUTES = {
        left: 'borderLeft',
        right: 'borderRight',
        top: 'borderTop',
        bottom: 'borderBottom',
        insideh: 'borderInsideHor',
        insidev: 'borderInsideVert'
    };

    // the names of all border mode properties in cells, mapped by border properties
    var CELL_BORDER_MODE_PROPERTIES = {
        borderLeft: 'left',
        borderRight: 'right',
        borderTop: 'top',
        borderBottom: 'bottom',
        borderInsideHor: 'insideh',
        borderInsideVert: 'insidev'
    };

    // the names of all border properties in paragraphs, mapped by border mode properties
    var PARAGRAPH_BORDER_ATTRIBUTES = {
        left: 'borderLeft',
        right: 'borderRight',
        top: 'borderTop',
        bottom: 'borderBottom',
        insideh: 'borderInside'
    };

    // private static functions ===============================================

    /**
     * Returns a map with names of border attributes used in operations, mapped
     * by border mode keys as used in the UI.
     */
    function getBorderAttributeNames(options) {
        return Utils.getBooleanOption(options, 'paragraph', false) ? PARAGRAPH_BORDER_ATTRIBUTES : CELL_BORDER_ATTRIBUTES;
    }

    // static class MixedBorder ===============================================

    /**
     * Provides helper function to mix several border attributes, e.g. from a
     * selection containing multiple elements with borders, and to convert
     * between mixed borders and an abstract border mode used in GUI border
     * controls.
     */
    var MixedBorder = {};

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed mixed border object is valid and visible
     * (line style ambiguous or different to 'none', and line width ambiguous
     * or greater than 0).
     *
     * @param {Object} [mixedBorder]
     *  The mixed border object, as returned for example by the method
     *  'MixedBorder.mixBorders()'. If the parameter is missing or not an
     *  object, it is considered invisible.
     *
     * @returns {Boolean}
     *  Whether the mixed border object represents a visible border style.
     */
    MixedBorder.isVisibleBorder = function (mixedBorder) {
        return _.isObject(mixedBorder) &&
            // in mixed state, there are always some borders visible
            (!!mixedBorder.mixed || (mixedBorder.style !== 'none')) &&
            // border width may be ambiguous, but must not be 0)
            ((mixedBorder.width === null) || (_.isNumber(mixedBorder.width) && (mixedBorder.width > 0)));
    };

    /**
     * Returns whether the passed mixed border object contains any ambiguous
     * properties, i.e. whether either the 'style', the 'width', or the 'color'
     * property is set to null. The property 'mixed' will be ignored.
     *
     * @param {Object} [mixedBorder]
     *  The mixed border object, as returned for example by the method
     *  'MixedBorder.mixBorders()'. If the parameter is missing or not an
     *  object, false will be returned (a missing border object is considered
     *  to be invisible, and therefore not ambiguous).
     *
     * @returns {Boolean}
     *  Whether the passed mixed border object contains ambiguous properties.
     */
    MixedBorder.hasAmbiguousProperties = function (mixedBorder) {
        return _.isObject(mixedBorder) && ((mixedBorder.style === null) || (mixedBorder.width === null) || (mixedBorder.color === null));
    };

    /**
     * Returns whether the passed mixed border object is completely ambiguous
     * (whether the 'mixed' flag is set, and all border properties are set to
     * the value null).
     *
     * @param {Object} [mixedBorder]
     *  The mixed border object, as returned for example by the method
     *  'MixedBorder.mixBorders()'. If the parameter is missing or not an
     *  object, false will be returned (a missing border object is considered
     *  to be invisible, and therefore not ambiguous).
     *
     * @returns {Boolean}
     *  Whether the passed mixed border object is completely ambiguous.
     */
    MixedBorder.isFullyAmbiguousBorder = function (mixedBorder) {
        return _.isObject(mixedBorder) && !!mixedBorder.mixed && (mixedBorder.style === null) && (mixedBorder.width === null) && (mixedBorder.color === null);
    };

    /**
     * Mixes the passed border objects, and returns a mixed border object
     * representing the states of the single border properties.
     *
     * @param {Object|Array<Object>} [...]
     *  An unlimited number of parameters, each parameter can be a single mixed
     *  border object and/or a regular border attribute value, or an array of
     *  these objects.
     *
     * @returns {Object}
     *  A mixed border object representing the states of the properties of all
     *  passed borders. Contains the following properties:
     *  - {Boolean} mixed
     *      Whether the visibility state of the borders is ambiguous. If true,
     *      a few but not all borders are visible, and the other borders are
     *      invisible.
     *  - {String|Null} style
     *      The line style of all visible border lines. Will be null, if the
     *      line style is ambiguous. Will be 'none', if ALL borders are
     *      invisible.
     *  - {Number|Null} [width]
     *      The line width of all visible border lines, in 1/100 mm. Will be
     *      null, if the line width is ambiguous. Will be omitted, if ALL
     *      borders are invisible.
     *  - {Object|Null} [color]
     *      The line color of all visible border lines. Will be null, if the
     *      line color is ambiguous. Will be omitted, if ALL borders are
     *      invisible.
     */
    MixedBorder.mixBorders = function () {

        // the resulting mixed border object
        var mixedBorder = null;

        // process all parameters
        _.each(arguments, function (arg) {
            _.getArray(arg).forEach(function (border) {

                // start with a copy of the first border object
                if (!mixedBorder) {
                    mixedBorder = _.clone(border);
                    return;
                }

                // whether the passed border is visible
                var visible = ('mixed' in border) ? MixedBorder.isVisibleBorder(border) : Border.isVisibleBorder(border);

                // update the mixed visibility state
                if (!mixedBorder.mixed) {
                    mixedBorder.mixed = (border.mixed === true) || (MixedBorder.isVisibleBorder(mixedBorder) !== visible);
                }

                // skip the other border line properties, if the border is invisible
                if (!visible) { return; }

                // mix line style (null represents mixed state of visible styles)
                if (mixedBorder.style === 'none') {
                    mixedBorder.style = border.style;
                } else if (mixedBorder.style !== border.style) {
                    mixedBorder.style = null;
                }

                // update line width (null represents mixed state)
                if (!('width' in mixedBorder)) {
                    mixedBorder.width = border.width;
                } else if (_.isNumber(mixedBorder.width) && (mixedBorder.width !== border.width)) {
                    mixedBorder.width = null;
                }

                // update line color (null represents mixed state)
                if (!('color' in mixedBorder)) {
                    mixedBorder.color = border.color;
                } else if (_.isObject(mixedBorder.color) && !Color.isEqual(mixedBorder.color, border.color)) {
                    mixedBorder.color = null;
                }
            });
        });

        // create a complete mixed border object
        mixedBorder = _.extend({ mixed: false, style: 'none' }, mixedBorder);

        // do not return unused attributes for invisible borders retrieved from input data
        return (mixedBorder.style === 'none') ? { mixed: mixedBorder.mixed, style: 'none' } : mixedBorder;
    };

    /**
     * Converts the passed mixed border attributes to an object describing the
     * visibility states of all borders.
     *
     * @param {Object} borderAttributes
     *  The attribute map containing mixed border objects, as returned for
     *  example by the method 'MixedBorder.mixBorders()', mapped by border
     *  attribute names (e.g. 'borderLeft', 'borderTop', etc.).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.paragraph=false]
     *      If set to true, a single inner border attribute 'borderInside' will
     *      be used instead of the two attributes 'borderInsideHor' and
     *      'borderInsideVert' (for paragraphs which are stacked vertically and
     *      do not have vertical inner borders).
     *
     * @returns {Object}
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  or Null values specifying whether the respective borders are visible.
     *  If the visibility state of a mixed border object in the passed
     *  attribute map is missing or ambiguous (property 'mixed' is true), the
     *  map value will be null. Otherwise, the Boolean value of the property
     *  states whether the border is visible.
     */
    MixedBorder.getBorderMode = function (borderAttributes, options) {

        // the border attribute names
        var attributeNames = getBorderAttributeNames(options);
        // result mapping border attribute names to boolean values
        var borderMode = {};

        _.each(attributeNames, function (attrName, propName) {
            var border = borderAttributes[attrName];
            if (_.isObject(border)) {
                borderMode[propName] = (border.mixed === true) ? null : MixedBorder.isVisibleBorder(border);
            }
        });

        return borderMode;
    };

    /**
     * Converts the passed object describing the visibility states of cell
     * border attributes to a table attribute map containing all border
     * attributes.
     *
     * @param {Object|String} borderMode
     *  An object with position keys ('left', 'top', etc.), mapping boolean
     *  values specifying whether the respective border is visible. Properties
     *  not contained in this object will not occur in the returned attribute
     *  map. Alternatively, the passed value can be a single position key as
     *  string, which causes to toggle the current visibility state of the
     *  respective border attribute.
     *
     * @param {Object} origAttributes
     *  The original attribute map of the target object (table, cell, ...). May
     *  also be a map of mixed borders with ambiguous border properties. If any
     *  of the border properties is ambiguous (value null), the default border
     *  will be used instead.
     *
     * @param {Object} defaultBorder
     *  Default border style style used when making a border attribute visible
     *  without having a corresponding visible border value in the passed
     *  original border attributes.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.paragraph=false]
     *      If set to true, a single inner border attribute 'borderInside' will
     *      be used (for paragraphs which are stacked vertically and do not
     *      have vertical inner borders), instead of the two attributes
     *      'borderInsideHor' and 'borderInsideVert'.
     *
     *  @param {Boolean} [options.clearBorderNoneFromAttributes=false]
     *      If set to true, and the border is not visible (e.g. the paragraph
     *      border is removed in text), it will clear the related border
     *      attribute from the paragraph attributes list by returning 'null'
     *      for the related operation, instead of returning the styles from
     *      'Border.NONE'. This is conform with the behavior in Word.
     *
     *  @param {Boolean} [options.fullBorderSet=false]
     *      If set to true, a full set of borders shall be returned. This is
     *      the case in the presentation app.
     *
     * @returns {Object}
     *  The table/cell attributes containing all border attributes that are
     *  mentioned in the passed border mode. In case the passed border mode is
     *  a string, the returned attributes will contain the corresponding border
     *  attribute only.
     */
    MixedBorder.getBorderAttributes = function (borderMode, origAttributes, defaultBorder, options) {

        // the border attribute names
        var attributeNames = getBorderAttributeNames(options);
        // the resulting border attributes
        var borderAttributes = {};
        // indicates if 'Border.NONE' or 'null' is returned for an invisible border
        var clearBorderNoneFromAttributes = Utils.getBooleanOption(options, 'clearBorderNoneFromAttributes', false);
        // whether a full set of borders shall be returned
        var fullBorderSet = Utils.getBooleanOption(options, 'fullBorderSet', false);

        function addBorderAttribute(visible, propName, obj, keepState) {

            // ignore invalid properties
            if (!(propName in attributeNames)) { return; }

            // the real name of the border attribute
            var borderAttrName = attributeNames[propName];
            // the original border attribute value or mixed border object
            var origBorder = origAttributes[borderAttrName];
            // whether the original border is visible
            var origVisible = MixedBorder.isVisibleBorder(origBorder);

            // toggle visibility if specified (but switch from 'mixed' state to visible)
            visible = (visible === null) ? ((origVisible && origBorder.mixed) || !origVisible) : visible;

            // no toggling, but keeping the state (only useful in connection with 'fullBorderSet')
            if (keepState) { visible = origVisible; }

            // return 'null' when the border is not visible, this removes
            // the border attribute completely from the paragraph attributes
            if (clearBorderNoneFromAttributes && !visible) {
                borderAttributes[borderAttrName] = null;

            // the old, normal behaviour
            } else {
                // use default border, if any of the original border properties is ambiguous
                var newBorder = !visible ? Border.NONE : (origVisible && !MixedBorder.hasAmbiguousProperties(origBorder)) ? origBorder : defaultBorder;

                // build the border attribute value according to visibility and original borders
                borderAttributes[borderAttrName] = _.copy(newBorder, true);
                delete borderAttributes[borderAttrName].mixed;
            }
        }

        if (_.isObject(borderMode)) {
            // fill the resulting attribute map
            _.each(borderMode, addBorderAttribute);

            // if full border mode is set, all border attributes must be specified
            if (fullBorderSet) {
                _.each(CELL_BORDER_ATTRIBUTES, function (type) {
                    if (!borderAttributes[type]) { borderAttributes[type] = { style: 'none' }; }
                });
            }

        } else if (_.isString(borderMode)) {
            // toggle the visibility of a single border
            addBorderAttribute(null, borderMode);

             // if full border mode is set, all border attributes must be specified (keeping the visibility of all other borders)
            if (fullBorderSet) {
                _.each(CELL_BORDER_ATTRIBUTES, function (type) {
                    if (!borderAttributes[type]) { addBorderAttribute(null, CELL_BORDER_MODE_PROPERTIES[type], null, true); }
                });
            }
        }

        return borderAttributes;
    };

    // exports ================================================================

    return MixedBorder;

});
