/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/fontcollection', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/font',
    'io.ox/office/baseframework/model/modelobject'
], function (Config, Utils, Scheduler, ValueMap, Font, ModelObject) {

    'use strict';

    // all known fonts, mapped by font name
    var PREDEFINED_FONTS = {

        Helvetica:              { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] }, display: true },
        Arial:                  { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] }, display: true },
        'Arial Black':          { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['sans serif', 'Arial'] } },
        'Arial Narrow':         { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Calibri Light'] } },
        Verdana:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] }, display: true },
        Tahoma:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] }, display: true },
        Calibri:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Carlito', 'Arial'] }, display: true },
        'Calibri Light':        { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Carlito', 'Arial'] }, display: true },
        Carlito:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Calibri', 'Arial'] } },
        Albany:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Liberation Sans', 'Arial'] } }, // old OOo default for sans-serif
        'Liberation Sans':      { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Albany', 'Arial'] }, display: 'odf' }, // LO default for sans-serif
        // bug 49056: in Linux, "sans serif" is not the same as "sans-serif"
        Impact:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['sans serif', 'Arial'] }, display: true },
        'Helvetica Neue':       { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] } },
        'Helvetica Light':      { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] } },
        'Open Sans':            { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica Neue'] } },
        'Open Sans Light':      { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans'] } },
        'Open Sans Semibold':   { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans'] } },
        'Open Sans Extrabold':  { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans Semibold'] } },
        Euphemia:               { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Calibri', 'Arial'] } },

        Times:                  { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] } },
        'Times New Roman':      { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times'] }, display: true },
        Georgia:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] }, display: true },
        Palatino:               { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] }, display: true },
        'Book Antiqua':         { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Palatino'] }, display: true },
        Cambria:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Caladea', 'Times New Roman'] }, display: true },
        Caladea:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Cambria', 'Times New Roman'] } },
        Thorndale:              { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Liberation Serif', 'Times New Roman'] } }, // old OOo default for serif
        'Liberation Serif':     { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Thorndale', 'Times New Roman'] }, display: 'odf' }, // LO default for serif
        'Calisto MT':           { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Book Antiqua'] } },

        Courier:                { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] } },
        'Courier New':          { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier'] },     display: true },
        'Andale Mono':          { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] }, display: true },
        Consolas:               { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] }, display: true },
        Cumberland:             { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Liberation Mono', 'Courier New'] } }, // old OOo default for monospace
        'Liberation Mono':      { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Cumberland', 'Courier New'] }, display: 'odf' } // LO default for monospace
    };

    // a regular expression that matches valid font names
    var RE_VALID_FONTNAME = new RegExp('^[^<>[\\]()"\'&/\\\\]+$');

    // key in localStorage were the font data is saved
    var LOCAL_STORAGE_FONT_KEY = 'appsuite.office-fonts';

    // key in localStorage were blockDownloadToken token is saved
    var LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY = LOCAL_STORAGE_FONT_KEY + '-do-not-download-again';

    // web font name that is used in debug mode
    var DEBUG_FONT_NAME = 'oxdebugwebfontox';

    // private global functions ===============================================

    // When used with 'office:testautomation=true', the importReplacementFonts is tested with a own, unique font (see 'DEBUG_FONT_NAME') that doesn't exists on Windows and Linux.
    // Since Calibri and Cambria are system fonts in Windows, it would not work otherwise. In Linux 'Fontconfig' may generate Calibri replacement fonts on OS level that are not possible to detect.
    function isTestFontInstalled() {
        return !Config.AUTOTEST || Font.isInstalled(DEBUG_FONT_NAME, 'AB');
    }

    // class FontCollection ===================================================

    /**
     * Contains the definitions of all known fonts in a document.
     *
     * Instances of this class trigger the following events:
     * - 'insert:font'
     *      After a new font definition has been inserted into this collection,
     *      or an existing font definition has been changed. Event handlers
     *      receive the name of the changed font.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     */
    var FontCollection = ModelObject.extend({ constructor: function (docModel) {

        // all registered fonts, mapped by capitalized font name
        var fontMap = new ValueMap();

        // special handling for different file formats
        var fileFormat = docModel.getApp().getFileFormat();

        var debugStateImportReplacementFonts = {
            fontLoadedWithAjax: false,
            fontReadyInDOM: false,
            fontAttachedFromLocalStorage: false,
            installedOrReplacementAvailable: false
        };

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Removes the calculated CSS font families from all registered fonts.
         */
        function deleteAllCssFontFamilies() {
            fontMap.forEach(function (font) { delete font.cssFontFamily; });
        }

        /**
         * Calculates and returns the CSS font family attribute containing the
         * specified font name and all alternative font names.
         */
        function calculateCssFontFamily(fontName) {

            // the resulting collection of CSS font names
            var cssFontNames = [fontName];
            // the last found CSS default font family
            var lastCssDefault = null;

            // collects all alternative font names of the fonts in the passed array
            function collectAltNames(fontNames) {

                // alternative font names of the passed fonts
                var altFontNames = [];

                // collect all alternative font names
                fontNames.forEach(function (fontName2) {
                    fontMap.with(fontName2, function (fontDesc) {
                        if (_.isArray(fontDesc.fontProps.altNames)) {
                            altFontNames = altFontNames.concat(fontDesc.fontProps.altNames);
                            if (typeof fontDesc.cssDefault === 'string') {
                                lastCssDefault = fontDesc.cssDefault;
                            }
                        }
                    });
                });

                // unify the new alternative font names and remove font names
                // already collected in cssFontNames (prevent cyclic references)
                altFontNames = _.chain(altFontNames).unique().difference(cssFontNames).value();

                // insert the new alternative font names into result list, and
                // call recursively until no new font names have been found
                if (altFontNames.length > 0) {
                    cssFontNames = cssFontNames.concat(altFontNames);
                    collectAltNames(altFontNames);
                }
            }

            // collect alternative font names by starting with passed font name
            collectAltNames([fontName]);

            // finally, add the CSS default family
            if (_.isString(lastCssDefault)) {
                cssFontNames.push(lastCssDefault);
            }

            // remove invalid/dangerous characters
            cssFontNames = cssFontNames.filter(RE_VALID_FONTNAME.test, RE_VALID_FONTNAME);

            // enclose all fonts with white-space into quote characters
            cssFontNames.forEach(function (cssFontName, index) {
                if (/\s/.test(cssFontName)) {
                    cssFontNames[index] = '\'' + cssFontName.replace(/\s+/g, ' ') + '\'';
                }
            });

            // return the CSS font-family attribute value
            return cssFontNames.join(',');
        }

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'insertFontDescription'.
         * Inserts a new font entry into this collection.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertFontDescription' document
         *  operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyInsertFontOperation = function (context) {

            // fonts are keyed by capitalized names
            var fontName = Utils.capitalizeWords(context.getStr('fontName'));

            // store passed font attributes
            var fontProps = _.copy(context.getObj('attrs'), true);
            var fontDesc = fontMap.insert(fontName, { fontProps: fontProps, display: true });

            // validate alternative names
            if (_.isArray(fontProps.altNames)) {
                fontProps.altNames = fontProps.altNames.filter(_.identity).map(Utils.capitalizeWords);
            } else {
                fontProps.altNames = [];
            }

            // add predefined alternative names
            if ((fontName in PREDEFINED_FONTS) && _.isArray(PREDEFINED_FONTS[fontName].fontProps.altNames)) {
                fontProps.altNames = fontProps.altNames.concat(PREDEFINED_FONTS[fontName].fontProps.altNames);
            }

            // unify alternative names, remove own font name
            fontProps.altNames = _.chain(fontProps.altNames).unique().without(fontName).value();

            // add default CSS font family if known
            if (fontName in PREDEFINED_FONTS) {
                fontDesc.cssDefault = PREDEFINED_FONTS[fontName].cssDefault;
            }
            // remove calculated CSS font families of all fonts, notify listeners
            deleteAllCssFontFamilies();
            this.trigger('insert:font', fontName);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the names of all predefined and registered fonts. Predefined
         * fonts with the hidden flag that have not been registered will not be
         * included.
         *
         * @returns {Array<String>}
         *  The names of all predefined and registered fonts, as array.
         */
        this.getFontNames = function () {
            return fontMap.reduce([], function (fontNames, fontDesc, fontName) {
                if ((fontDesc.display === true) || (fontDesc.display === fileFormat)) {
                    fontNames.push(fontName);
                }
                return fontNames;
            });
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName) {

            if (!fontName) { return ''; } // increasing resilience

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // default for unregistered fonts: just the passed font name
            var fontDesc = fontMap.get(fontName, null);
            if (!fontDesc) { return fontName; }

            // calculate and cache the CSS font family
            if (!_.isString(fontDesc.cssFontFamily)) {
                fontDesc.cssFontFamily = calculateCssFontFamily(fontName);
            }

            return fontDesc.cssFontFamily;
        };

        /**
         * Make replacement fonts for 'Calibri' and 'Cambria' available in the browser,
         * when neither these two nor the replacement fonts exists on the client.
         *
         * Loading:
         * When they are not existing, the replacement fonts are loaded from the server,
         * stored in localStorage and added to the DOM to be useable.
         *
         * font data:
         * The replacement fonts (io.ox/office/tk/render/fonts.css) are base64 encoded woff
         * files that are attached directly to the <head> in the DOM.
         *
         * Already stored:
         * When the replacement fonts are already in localStorage, they don't need to be loaded
         * again. And when they are already in the DOM they even must not attached again
         * (font data would be two times in DOM then).
         *
         * LocalStorage and logout:
         * It's important that in the 'LOCAL_STORAGE_FONT_KEY' ('appsuite.office-fonts')
         * is added to the 'fileList' in '/core/cache/localstorage.js', see web repo
         * commit: 5a5a47c6d9a220ca501563cdc226c09ac9f34a99), so that this key is
         * not deleted after a logout from the appSuite.
         *
         * Prevent multiple downloads:
         * Also, to prevent constant attempts to download fonts when they can't be stored
         * in the localStorage, e.g. because of the size (which should not happen normally), only a
         * small token is set in the localStorage to prevent downloading until it is cleared
         * (e.g. when the localStorage is cleared, this token is cleared too and therefore a
         * new attempt is made to download and store the font in localStorage).
         *
         * Debug-Mode:
         * There is a special debug-mode to test font-loading with an own created test font.
         * This is useful to test the font loading in windows (selenium) where the Calibri and Cambria
         * fonts are always available. Or in Linux when 'Fontconfig' has Calibri replacement
         * fonts on OS level that are not possible to detect (for more information see our wiki).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the font is available in the browser. It's also resolved
         *  when font loading didn't work, because everything works like before the
         *  replacement fonts existed. There is no error handling possible in a fail handler.
         */
        this.importReplacementFonts = function () {

            // whether 'Calibri' or the metric compatible replacement 'Carlito' is available at the device
            var calibriOrReplacementAvailible = Font.isInstalled('Calibri') || Font.isInstalled('Carlito');
            // whether 'Cambria' or the metric compatible replacement 'Caladea' is available at the device
            var cambriaOrReplacementAvailible = Font.isInstalled('Cambria') || Font.isInstalled('Caladea');

            // only for debug, but very helpful to detect failures
            Utils.withLogging(function () {
                Utils.log('ImportReplacementFonts - Calibri:', Font.isInstalled('Calibri'), ' - Carlito:', Font.isInstalled('Carlito'), ' | Cambria:', Font.isInstalled('Cambria'), ' - Caladea:', Font.isInstalled('Caladea'), '| FontDebugMode:', Config.AUTOTEST, '- ' + DEBUG_FONT_NAME + ':', isTestFontInstalled());
            });

            // check if font or replacement is installed locally, or already available in the browser
            if (calibriOrReplacementAvailible && cambriaOrReplacementAvailible && isTestFontInstalled()) {
                Utils.log('ImportReplacementFonts - Needed fonts are installed or a replacement fonts is already available for rendering in the browser');
                debugStateImportReplacementFonts.installedOrReplacementAvailable = true;
                return this.createResolvedPromise();
            }

            // the deferred object representing the font import
            var loadDeferred = Scheduler.createDeferred(this, 'FontCollection.importReplacementFonts');

            // Attach the CSS content which contains the web font to DOM
            function addFontToDOM(cssFontData) {

                // when this node exists, the font data is also attached to the DOM, so don't do it again
                if (Utils.findHiddenNodes('.font-preloading-helper').length < 1) {
                    $('head').append('<style type="text/css">' + cssFontData + '</style>');

                    // insert a hidden node to reference the replacement fonts in the DOM. This starts using the font-face rule (note: 'A' is a used glyph in 'debugFontName' font )
                    Utils.insertHiddenNodes($('<div class="font-preloading-helper"><div style="font-family:Carlito;">g</div><div style="font-family:Caladea;">g</div><div style="font-family:' +  DEBUG_FONT_NAME + ';">A</div></div>'));

                    // trigger a reflow in the browser, at least Chrome needs this to make
                    // the font really available for rendering in the browser
                    Utils.findHiddenNodes('.font-preloading-helper').outerWidth();
                }

                // Check and resolve the Promise if the font is available in the browser. This must
                // be checked async, because when the code runs synchronous e.g. Chrome is not able to make
                // the font available for rendering in the browser. This check should normally resolve with
                // true in the first round, but browser can behave differently here. So better be save and
                // check until the font is finally available or the interval is timed out.
                var pollingStartTime = Date.now();
                var isFontLoadedInBrowserPolling = window.setInterval(function () {

                    // exit condition: break when the font is still not ready in the browser after 2000ms
                    if (2000 < (Date.now() - pollingStartTime)) {
                        //console.log(Date.now() - pollingStartTime);
                        window.clearInterval(isFontLoadedInBrowserPolling);
                        loadDeferred.resolve();

                        Utils.warn('ImportReplacementFonts - Timeout while checking font availability for rendering in the Browser after appending font data to DOM');

                    // check if the font is available now
                    } else if (Font.isInstalled('Carlito') && Font.isInstalled('Caladea') && isTestFontInstalled()) {

                        window.clearInterval(isFontLoadedInBrowserPolling);
                        loadDeferred.resolve();
                        Utils.log('ImportReplacementFonts - Font is finally available for rendering in the browser after appending font data to DOM');
                        // set debug state
                        debugStateImportReplacementFonts.fontReadyInDOM = true;

                    }
                }, 100);
            }

            // add font when it's cached in localStorage
            var storageFontData = localStorage.getItem(LOCAL_STORAGE_FONT_KEY);
            if (storageFontData) {

                // add font to DOM to be useable
                addFontToDOM(storageFontData);

                Utils.log('ImportReplacementFonts - Attaching font data from localStorage to DOM');
                // set debug state
                debugStateImportReplacementFonts.fontAttachedFromLocalStorage = true;

            // when not cached, load them from server
            } else {

                // whether a possible font download should be allowed
                var blockDownloadToken = localStorage.getItem(LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY) === 'true';

                // Check if the localStorage is available and really writable (possible problem with browser incognito mode).
                // Also do not try to download again when localStorage has no free size left
                // important: the token and also the whole localStorage, besides the fonts,
                // should be cleared after a logout. Since the token is also deleted, it will
                // try to download it again after that.
                if (window.Storage && Utils.isLocalStorageSupported() && !blockDownloadToken) {

                    // get the css file which contains the base64 encoded fonts
                    $.ajax({
                        url: ox.root + '/apps/io.ox/office/tk/render/fonts.css',
                        dataType: 'text',
                        timeout: 40000 // with 3G 512 kbit/s the 1.7mb font.css needs about 27s to download, add some safety buffer
                    }).done(function (data) {

                        Utils.log('ImportReplacementFonts - Loaded fonts via AJAX-request, attach data to localStorage');
                        // set debug state
                        debugStateImportReplacementFonts.fontLoadedWithAjax = true;

                        try {
                            // put to local storage
                            localStorage.setItem(LOCAL_STORAGE_FONT_KEY, data);
                            // add font to DOM to be useable
                            addFontToDOM(data);

                        } catch (ex) {
                            Utils.warn('ImportReplacementFonts - Failed to store data in localStorage', ex);

                            try {
                                // set token
                                localStorage.setItem(LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY, true);
                            } catch (ex2) {
                                Utils.warn('ImportReplacementFonts - Failed to store do-not-download-again token in localStorage', ex2);
                            }

                            loadDeferred.resolve();
                        }

                    }).fail(function () {

                        // fail mean that just the replacement is not available, everything works like before the replacement fonts existed
                        loadDeferred.resolve();

                        Utils.warn('ImportReplacementFonts - Loading font via AJAX-request failed, probably timed out');
                    });

                // do nothing when no localStorage is available, everything works like before the replacement fonts existed
                } else {
                    loadDeferred.resolve();
                    Utils.warn('ImportReplacementFonts - No localStorage supported or "do-not-download-again" token set', blockDownloadToken);
                }
            }

            return this.createAbortablePromise(loadDeferred, function () {
                Utils.warn('ImportReplacementFonts -  Promise aborted');
            });
        };

        /**
         * Deletes all web replacement fonts related items from local storage.
         */
        this.deleteWebFontsFromLocalStorage = function () {
            localStorage.removeItem(LOCAL_STORAGE_BLOCK_FONT_DOWNLOAD_KEY);
            localStorage.removeItem(LOCAL_STORAGE_FONT_KEY);
        };

        this.getImportReplacementFontsState = function () {
            return debugStateImportReplacementFonts;
        };

        // initialization -----------------------------------------------------

        // start with all predefined fonts
        _.each(PREDEFINED_FONTS, function (fontDesc, fontName) {
            fontMap.insert(fontName, _.copy(fontDesc, true));
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = fontMap = null;
        });

    } }); // class FontCollection

    // exports ================================================================

    return FontCollection;

});
