/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import java.io.File;
import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang.StringUtils;
import org.im4java.core.ConvertCmd;
import org.im4java.core.IM4JavaException;
import org.im4java.core.IMOperation;
import org.im4java.core.IMOps;
import org.im4java.process.ProcessExecutor;
import org.im4java.process.ProcessStarter;
import org.im4java.process.ProcessTask;
import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageFormat;
import com.openexchange.imagetransformation.ScaleType;

/**
 * {@link ImageProcessor}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
class ImageProcessor {

    /**
     * Initializes a new {@link ImageProcessor}.
     */
    ImageProcessor(@NonNull final String searchPath, boolean useGraphicsMagick) {
        super();

        ProcessStarter.setGlobalSearchPath(searchPath);
        m_useGraphicsMagick = useGraphicsMagick;

        implCheckAvailability(searchPath);
    }

    /**
     *
     */
    void shutdown() {
        m_processExecutor.shutdownNow();

        try {
            m_processExecutor.awaitTermination(ImageConverterUtils.AWAIT_TERMINATION_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS);
        } catch (@SuppressWarnings("unused") InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    /**
     * @param inputFile
     * @param outputFile
     * @param imageFormat
     * @return
     * @throws ImageConverterException
     */
    boolean scale(
        final File inputFile,
        @NonNull final File outputFile,
        @NonNull final ImageFormat imageFormat) throws ImageConverterException {

        boolean ret = false;

        if (isAlive()) {
            final IMOperation imOp = new IMOperation();
            final Integer imWidth = ImageConverterUtils.getIM4JExtent(imageFormat.getWidth());
            final Integer imHeight = ImageConverterUtils.getIM4JExtent(imageFormat.getHeight());
            final ScaleType icScaleType = imageFormat.getScaleType();

            imOp.addImage();

            if (imageFormat.isAutoRotate()) {
                imOp.autoOrient();
            }

            switch (icScaleType) {
                case CONTAIN: {
                    imOp.resize(imWidth, imHeight, imageFormat.isShrinkOnly() ? Character.valueOf('>') : null);
                    break;
                }

                case CONTAIN_FORCE_DIMENSION: {
                    // http://www.imagemagick.org/Usage/resize/
                    // http://www.imagemagick.org/Usage/thumbnails/#fit_summery
                    imOp.resize(imWidth, imHeight, imageFormat.isShrinkOnly() ? Character.valueOf('>') : null);
                    imOp.gravity("center");
                    imOp.extent(imWidth, imHeight);
                    break;
                }

                case COVER: {
                    imOp.resize(imWidth, imHeight, Character.valueOf('^'));
                    break;
                }

                case COVER_AND_CROP: {
                    // Fill Area Flag ('^' flag)
                    imOp.resize(imWidth, imHeight, Character.valueOf('^'));
                    imOp.gravity("center");
                    imOp.extent(imWidth, imHeight);
                    break;
                }

                case AUTO:
                default:
                    imOp.scale(imWidth, imHeight);
                    break;
            }

            ret = submitTask(inputFile, outputFile, imageFormat, imOp);
        }

        return ret;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     */
    protected boolean isAlive() {
        return !m_processExecutor.isTerminated() && !m_processExecutor.isTerminating();
    }

    /**
     * @param inputFile
     * @param outputFile
     * @param imageFormat
     * @param imageOperation
     * @return
     * @throws ImageConverterException
     */
    protected boolean submitTask(
        @NonNull final File inputFile,
        @NonNull final File outputFile,
        @NonNull final ImageFormat imageFormat,
        @NonNull final IMOps imageOps) throws ImageConverterException {

        final ConvertCmd convertCmd = new ConvertCmd(m_useGraphicsMagick);
        ImageConverterException rethrowable = null;

        try {
            imageOps.quality(Double.valueOf(imageFormat.getQuality()));
            imageOps.addImage();

            final ProcessTask processTask = convertCmd.getProcessTask(imageOps, ImageConverterUtils.getIM4JInput(inputFile), ImageConverterUtils.getIM4JOutput(outputFile, imageFormat));

            if (null != processTask) {
                Future<?> future = m_processExecutor.submit(processTask);

                if (null != future.get()) {
                    rethrowable = new ImageConverterException("IC error while executing IM4Java ProcessTask");
                }
            }
        } catch (IM4JavaException e) {
            rethrowable = new ImageConverterException("IC error while creating IM4Java ProcessTask", e);
        } catch (IOException e) {
            rethrowable = new ImageConverterException("IC error while creating IM4Java ProcessTask", e);
        } catch (ExecutionException e) {
            rethrowable = new ImageConverterException("IC error while running IM4Java ProcessTask", e);
        } catch (@SuppressWarnings("unused") InterruptedException e) {
            LOG.trace("IC interrupted while running IM4Java ProcessTask");
        } catch (@SuppressWarnings("unused") RejectedExecutionException e) {
            LOG.trace("IC rejected execution while running IM4Java ProcessTask");
        }

        if (null != rethrowable) {
            throw rethrowable;
        }

        return true;
    }

    // - Implementation --------------------------------------------------------

    void implCheckAvailability(@NonNull final String searchPath) {
        if (StringUtils.isNotBlank(searchPath)) {
            m_isAvaible = new File(searchPath, "convert").canExecute() &&
                (!m_useGraphicsMagick || new File(searchPath, "gm").canExecute());
        }

        if (!m_isAvaible) {
            LOG.error("IC was not able to detecd a valid " +
                (m_useGraphicsMagick ? "graphicsmagick" : "imagemagick") + " executable at " + searchPath +
                ". Image Processing will not be possible as long as these programs are executable!");
        }
    }

    // - Members ---------------------------------------------------------------

    private ProcessExecutor m_processExecutor = new ProcessExecutor();

    private boolean m_useGraphicsMagick = false;

    private boolean m_isAvaible = false;
}
