/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.ElementLock;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItem;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageConverterPriority;
import com.openexchange.imageconverter.api.ImageFormat;
import com.openexchange.imageconverter.api.MetadataImage;

/**
 * {@link ImageConverterQueue}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
class ImageConverterQueue extends Thread {

    @SuppressWarnings("serial")
    class ImageConverterQueueImpl extends LinkedList<ImageConverterRunnable> {
        /**
         * Initializes a new {@link ImageConverterQueue}.
         */
        ImageConverterQueueImpl(ImageConverterPriority imageConverterPriority, int capacity) {
            super();

            m_imageConverterPriority = imageConverterPriority;
            m_capacity = capacity;
        }

        /**
         * @return
         */
        ImageConverterPriority getImageConverterPriority() {
            return m_imageConverterPriority;
        }

        /**
         * @return
         */
        int getCapacity() {
            return m_capacity;
        }

        // - Members -----------------------------------------------------------

        private ImageConverterPriority m_imageConverterPriority;

        private int m_capacity;
    }

    /**
     * Initializes a new {@link ImageConverterQueue}.
     */
    ImageConverterQueue(@NonNull final ImageConverter imageConverter,
        @NonNull final IFileItemService fileItemService,
        @NonNull final IMetadataReader metadataReader) {

        super("ImageConverter Queue");

        m_imageConverter = imageConverter;
        m_fileItemService = fileItemService;
        m_metadataReader = metadataReader;

        // create an ordered BidiMap to map ImageFormats to image format strings and vice versa
        m_imageFormats = Arrays.asList(m_imageConverter.getImageFormats());

        m_imageProcessor = new ImageProcessor(ImageConverterConfig.IMAGECONVERTER_SEARCHPATH, ImageConverterConfig.IMAGECONVERTER_USE_GRAPHICSMAGICK);

        for (final ImageConverterPriority curPriority : ImageConverterPriority.values()) {
            m_imageConverterQueuesImpl.put(curPriority, new ImageConverterQueueImpl(curPriority, ImageConverterConfig.IMAGECONVERTER_QUEUE_LENGTH));
        }

        m_executorThreads = Executors.newFixedThreadPool(ImageConverterConfig.IMAGECONVERTER_THREAD_COUNT);

        if (LOG.isInfoEnabled()) {
            LOG.info(new StringBuilder("IC conversion queue started (thread count: ").
                append(ImageConverterConfig.IMAGECONVERTER_THREAD_COUNT).
                append(", queue entries: ").append(ImageConverterConfig.IMAGECONVERTER_QUEUE_LENGTH).
                append(')').toString());
        }

    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Thread#run()
     */
    @Override
    public void run() {
        ImageConverterRunnable runnable = null;
        ImageConverterPriority runnablePriority = ImageConverterPriority.BACKGROUND;
        boolean trace = LOG.isTraceEnabled();

        while (!isTerminated()) {
            trace = LOG.isTraceEnabled();
            runnable = null;

            try {
                // check prioritized queues from highest to lowest priority
                synchronized (m_imageConverterQueuesImpl) {
                    try {
                        m_imageConverterQueuesImpl.wait(10);
                    } catch (@SuppressWarnings("unused") InterruptedException e) {
                        // OK
                    }

                    // iterate over all prioritzed queues from highest to lowest
                    // priority, in order to retrieve the next runnable candidate
                    for (final ImageConverterPriority curPriority : PRIORITY_ORDER) {
                        final ImageConverterQueueImpl curQueue = m_imageConverterQueuesImpl.get(curPriority);

                        if (!curQueue.isEmpty()) {
                            // continue, if valid runnable has been found
                            // resource warning has been checked, runnable is closed
                            // after will have been executed as FutureTask
                            if (null != (runnable = curQueue.remove())) {
                                if (trace) {
                                    LOG.trace(new StringBuilder(128).
                                            append("IC entry count after take from ").
                                            append(curPriority.name().toUpperCase()).
                                            append(" queue: ").append(curQueue.size()).toString());
                                }

                                runnablePriority = curPriority;
                                break;
                            }
                        }
                    }
                }

                if (null != runnable) {
                    ImageConverterUtils.IC_MONITOR.addQueueTimeMillis(runnable.getCurrentMeasurementTimeMillis(), runnablePriority);
                    m_executorThreads.submit(runnable);
                }
            } catch (@SuppressWarnings("unused") RejectedExecutionException e) {
                if (trace) {
                    LOG.trace("IC queue runnable execution rejected after termination => jobs are not processed anymore");
                }
            }
        }

        if (trace) {
            LOG.trace("IC queue worker finished");
        }
    }

    // - Public API ------------------------------------------------------------

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     */
    void cache(@NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @Nullable final String context,
        @Nullable ImageConverterPriority priority) throws ImageConverterException {

        if (!m_imageConverter.isAvailable(imageKey)) {
            implCache(imageKey, srcImageStm, null, context, (null != priority) ? priority : ImageConverterPriority.BACKGROUND);
        }
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     * @return
     */
    MetadataImage cacheAndGetMetadataImage(@NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @NonNull final String requestFormat,
        @Nullable final String context,
        @Nullable final ImageConverterPriority priority) throws ImageConverterException {

        MetadataImage ret = null;

        try {
            ret = getMetadataImage(imageKey, requestFormat, context);
        } catch (ImageConverterException e) {
            LOG.trace("IC not able to read MetadataImage", e);
        }

        if (null == ret) {
            return implCache(imageKey, srcImageStm, requestFormat, context, (null != priority) ? priority : ImageConverterPriority.MEDIUM);
        }

        return ret;
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     * @return
     */
    MetadataImage getMetadataImage(@NonNull final String imageKey, @NonNull final String requestFormat, @Nullable final String context) throws ImageConverterException {
        if (m_imageConverter.isAvailable(imageKey)) {
            final IMetadata imageMetadata = getMetadata(imageKey, context);

            if (null == imageMetadata) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace(new StringBuilder("IC could not read metadata for ").
                        append(imageKey).append(": ").
                        append(requestFormat).toString());
                }

                return null;
            }

            final List<ImageFormat> availableImageFormats = implGetAvailableImageKeyFormats(imageKey);

            // if there're no available formats, the entry must be invalid
            if (0 == availableImageFormats.size()) {
                return null;
            }

            final ImageFormat requestTargetFormat = ImageConverterUtils.getBestMatchingFormat(
                availableImageFormats,
                imageMetadata,
                ImageFormat.parseImageFormat(requestFormat));

            if (null == requestTargetFormat) {
                throw new ImageConverterException(new StringBuilder("IC error while parsing requested image format for ").
                    append(imageKey).append(": ").
                    append(requestFormat).toString());
            }

            ElementLocker.lock(imageKey, ElementLock.LockMode.WAIT_IF_PROCESSED);

            try {
                return new MetadataImage(ImageConverterUtils.readImageByTargetFormat(m_fileItemService, imageKey, requestTargetFormat), imageMetadata);
            } catch (Exception e) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace(new StringBuilder("IC could not read image for ").
                        append(imageKey).append(": ").
                        append(requestFormat).append(" / ").
                        append(requestTargetFormat.getFormatString()).toString(),
                        Throwables.getRootCause(e));
                }
            } finally {
                ElementLocker.unlock(imageKey);
            }
        }

        return null;
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     * @return
     */
    IMetadata getMetadata(@NonNull final String imageKey, @Nullable final String context) throws ImageConverterException {
        IMetadata ret = null;

        if (m_imageConverter.isAvailable(imageKey)) {
            // try to use a current runnable first, if possible
            // resource warning has been checked!
            @SuppressWarnings("resource") ImageConverterRunnable imageConverterRunnableImpl =
                m_processingMap.get(imageKey);

            ret = (null != imageConverterRunnableImpl) ?
                imageConverterRunnableImpl.getImageMetadata() :
                    null;

            if (null == ret) {
                ElementLocker.lock(imageKey, ElementLock.LockMode.WAIT_IF_PROCESSED);

                try {
                    ret = ImageConverterUtils.readMetadata(m_fileItemService, m_metadataReader, imageKey);
                } finally {
                    ElementLocker.unlock(imageKey);
                }
            }
        }

        return ret;
    }

    /**
     *
     */
    void shutdown() {
        if (m_terminated.compareAndSet(false, true)) {
            final long shutdownStartTimeMillis = System.currentTimeMillis();
            final boolean trace = LOG.isTraceEnabled();

            if (trace) {
                LOG.trace("IC queue starting shutdown");
            }

            try {
                final List<Runnable> openRunnableList = m_executorThreads.shutdownNow();

                if (null != openRunnableList) {
                    if (trace) {
                        LOG.trace("IC closing already queued entries in shutdown: " + openRunnableList.size());
                    }

                    // cleanup/close of all pending Runnables
                    openRunnableList.forEach(new Consumer<Runnable>() {

                        @Override
                        public void accept(Runnable imageConversionRunnable) {
                            if (null != imageConversionRunnable) {
                                ((ImageConverterRunnable) imageConversionRunnable).close();
                            }
                        }
                    });
                }

                interrupt();

                m_imageProcessor.shutdown();
                m_executorThreads.awaitTermination(ImageConverterUtils.AWAIT_TERMINATION_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS);
            } catch (@SuppressWarnings("unused") InterruptedException e) {
                LOG.trace("IC queue received interrupt while awaiting termination");

                Thread.currentThread().interrupt();
            }

            LOG.trace("IC queue shutdown open Runnable count after shutdown: {}" + m_processingMap.size());
            LOG.trace("IC queue shutdown finished in {}ms", (System.currentTimeMillis() - shutdownStartTimeMillis));
        }
    }

    /**
     * @return
     */
    boolean isTerminated() {
        return m_terminated.get() || interrupted();
    }

    /**
     * @return <code>true</code> if the given imageKey is queued OR processed
     */
    boolean isProcessing(@NonNull final String imageKey) {
        return m_processingMap.containsKey(imageKey);
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param imageKey
     */
    protected MetadataImage implCache(@NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @Nullable final String requestFormat,
        @Nullable final String context,
        @NonNull final ImageConverterPriority imageConverterPriority) throws ImageConverterException {

        if (m_processingMap.containsKey(imageKey)) {
            LOG.trace("IC queue processing map already contains imageKey while start queueing: {}", imageKey);
        }

        ImageConverterRunnable imageConverterRunnable = null;
        MetadataImage ret = null;
        boolean closeRunnable = true;

        // wait for a possible other thread, currently processing
        // the given imageKey and finally start a new processing cycle
        ElementLocker.lock(imageKey, ElementLock.LockMode.WAIT_IF_PROCESSED_AND_BEGIN_PROCESSING);

        try {
            imageConverterRunnable = new ImageConverterRunnable(this, m_fileItemService, m_metadataReader, m_imageProcessor, m_imageFormats, imageKey, srcImageStm, context);

            if (null != requestFormat) {
                ret = imageConverterRunnable.processRequestFormat(requestFormat);
            }

            // add runnable to appropriate queue
            implAddRunnableToQueue(imageConverterPriority, imageConverterRunnable);

            // Runnable was put into queue, set flag to not close runnable and start processing
            closeRunnable = false;
        } catch (ImageConverterException e) {
            // Resource warning has been checked, runnable is
            // closed in finally block, if indicated
            throw e;
        } finally {
            if (closeRunnable) {
                closeRunnable(imageKey, imageConverterRunnable, true);
            }

            ElementLocker.unlock(imageKey, closeRunnable);
        }

        // Resource warning has been checked;
        // runnable is either closed in
        // finally block, if an exception has been encountered,
        // or in runnable worker itself
        return ret;
    }

    /**
     * @param imageKey
     * @return
     * @throws ImageConverterException
     */
    List<ImageFormat> implGetAvailableImageKeyFormats(@NonNull final String imageKey) throws ImageConverterException {
        IFileItem[] fileItems = null;

        try {
            fileItems = m_fileItemService.get(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey);
        } catch (FileItemException e) {
            throw new ImageConverterException(e);
        }

        if (null != fileItems) {
            final List<ImageFormat> imageFormats = new ArrayList<>(fileItems.length);

            for (final IFileItem curFileItem : fileItems) {
                final String fileId = curFileItem.getFileId();

                if ((null != fileId) && !fileId.startsWith(ImageConverterUtils.IMAGECONVERTER_METADATA_FILEID)) {
                    imageFormats.add(ImageFormat.parseImageFormat(fileId));
                }
            }

            return imageFormats;
        }

        return new ArrayList<>();
    }

    /**
     * @param priority
     * @param runnable
     */
    void implAddRunnableToQueue(@NonNull final ImageConverterPriority priority, @NonNull ImageConverterRunnable runnable) {
        int updatedQueueSize = 0;

        // add runnable to appropriate, prioritized
        // reset measurement time of runnable in order to retrieve the
        // queue time, after which the runnable is taken from queue
        runnable.initMeasurementTime();

        synchronized (m_imageConverterQueuesImpl) {
            final ImageConverterQueueImpl queue = m_imageConverterQueuesImpl.get(priority);

            queue.add(runnable);
            updatedQueueSize = queue.size();

            m_imageConverterQueuesImpl.notify();
        }

        ImageConverterUtils.IC_MONITOR.setQueueCount(priority, updatedQueueSize);

        if (LOG.isTraceEnabled()) {
            LOG.trace(new StringBuilder(128).
                append("IC entry count after put to ").
                append(priority.name().toUpperCase()).
                append(" queue: ").append(updatedQueueSize).toString());
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     * @throws ImageConverterException
     */
    void closeRunnable(@NonNull String imageKey, @Nullable final ImageConverterRunnable imageConverterRunnableImpl, final boolean remove) {
        try {
            if (remove) {
                m_fileItemService.removeSubGroup(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey);
            }
        } catch (FileItemException e) {
            LOG.trace("IC error while removing imageKey when closing Runnable: {}", imageKey, e);
        } finally {
            if (null != imageConverterRunnableImpl) {
                imageConverterRunnableImpl.close();
            }
        }

        LOG.trace("IC queue ended Runnable: {}", imageKey);
    }

    /**
     * @param imageKey
     * @param runnable
     */
    void startProcessing(@NonNull final String imageKey, @NonNull ImageConverterRunnable runnable) {
        m_processingMap.put(imageKey, runnable);
    }

    /**
     * @param imageKey
     */
    void endProcessing(@NonNull final String imageKey) {
        m_processingMap.remove(imageKey);
    }


    // - Static members --------------------------------------------------------

    final private static ImageConverterPriority[] PRIORITY_ORDER = {
        ImageConverterPriority.INSTANT,
        ImageConverterPriority.MEDIUM,
        ImageConverterPriority.BACKGROUND
    };

    // - Members ---------------------------------------------------------------

    final private ImageConverter m_imageConverter;

    final private IFileItemService m_fileItemService;

    final private IMetadataReader m_metadataReader;

    final private List<ImageFormat> m_imageFormats;

    final private Map<ImageConverterPriority, ImageConverterQueueImpl> m_imageConverterQueuesImpl = new HashMap<>();

    final private ExecutorService m_executorThreads;

    final private AtomicBoolean m_terminated = new AtomicBoolean(false);

    final private ConcurrentMap<String, ImageConverterRunnable> m_processingMap = new ConcurrentHashMap<>();

    final private ImageProcessor m_imageProcessor;
}
