/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter.client.json;

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import com.openexchange.ajax.container.FileHolder;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.documentconverter.DocumentConverterManager;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.Properties;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link GetDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */

@DispatcherNotes(defaultFormat = "file", allowPublicSession = true)
public class GetDocumentAction extends DocumentConverterAction {

    final private static String KEY_ERRORCODE = "errorCode";

    final private static int ERRORCODE_NONE = 0;
    final private static int ERRORCODE_BAD_REQUEST = 400;

    /**
     * Initializes a new {@link GetDocumentAction}.
     *
     * @param services
     */
    public GetDocumentAction(ServiceLookup services) {
        super(services);
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.json.actions.DocumentConverterAction#perform(com.openexchange.ajax.requesthandler.AJAXRequestData, com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(@NonNull AJAXRequestData request, @NonNull ServerSession session) {
        String fileName = request.getParameter("filename");
        String mimeType = request.getParameter("mimetype");
        final String async = request.getParameter("async");
        final boolean convertToPDF = "pdf".equalsIgnoreCase(request.getParameter("documentformat")) && !"application/pdf".equalsIgnoreCase(mimeType);
        final boolean isAsync = (null != async) && async.equalsIgnoreCase("true");
        final JobError[] jobError = { JobError.NONE };
        ManagedFile managedResultFile = null;

        if ((!convertToPDF || (null == (managedResultFile = getPDFFile(request, session, isAsync, jobError)))) && !isAsync) {
            final IDocumentConverter docConverter = m_services.getService(IDocumentConverter.class);
            final String encTest = request.getParameter("enctest");

            // In case of PDF input and output, the result file is equal to the source file
            managedResultFile = getRequestSourceFile(request, session, jobError);

            // If source file is valid and encTest is set to 'pdf',
            // do an encryption test to get a valid error code in
            // case of an encrypted PDF
            if ((null != managedResultFile) && (null != encTest) && encTest.equalsIgnoreCase("pdf") && (null != docConverter)) {
                final HashMap<String, Object> jobProperties = new HashMap<>(12);
                final HashMap<String, Object> resultProperties = new HashMap<>(8);

                jobProperties.put(Properties.PROP_INPUT_FILE, managedResultFile.getFile());
                jobProperties.put(Properties.PROP_PRIORITY, JobPriority.fromString(request.getParameter("priority")));

                if (null != fileName) {
                    jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
                }

                // do the PDF encryption test and close possible result stream immediately
                IOUtils.closeQuietly(docConverter.convert("pdfenctest", jobProperties, resultProperties));

                jobError[0] = isAsync ?
                    JobError.NONE :
                        JobError.fromObject(resultProperties.get(Properties.PROP_RESULT_ERROR_CODE));
            }
        }

        // create appropriate response result
        IFileHolder resultFileHolder = null;
        JSONObject jsonResult = null;

        try {
            if (isAsync) {
                (jsonResult = new JSONObject()).put("async", true);
            } else if (null != managedResultFile) {
                final String sessionId = request.getParameter("session");
                final String saveAsFileName = request.getParameter("saveas_filename");
                final String saveAsFolderId = request.getParameter("saveas_folder_id");
                final File tempFile = managedResultFile.getFile();

                if (StringUtils.isBlank(fileName) || !"pdf".equalsIgnoreCase(FilenameUtils.getExtension(fileName))) {
                    final StringBuilder fileNameBuilder = new StringBuilder(StringUtils.isBlank(fileName) ? "file" : fileName);
                    final int curExtPos = fileNameBuilder.lastIndexOf(".");

                    // set or replace extension with ".pdf"
                    mimeType = "application/pdf";
                    fileName = ((curExtPos > -1) ?
                                    fileNameBuilder.replace(curExtPos, fileNameBuilder.length(),".pdf") :
                                        fileNameBuilder.append(".pdf")).toString();
                }

                jsonResult = getJSONResultForSaveAs(session, sessionId, tempFile, saveAsFileName, saveAsFolderId, fileName, mimeType);

                if (null == jsonResult) {
                    resultFileHolder = new FileHolder(FileHolder.newClosureFor(tempFile), tempFile.length(), mimeType, fileName);
                } else if (jsonResult.getInt(KEY_ERRORCODE) > ERRORCODE_NONE) {
                    return (new AJAXRequestResult().setHttpStatusCode(ERRORCODE_BAD_REQUEST));
                }
            }
        } catch (Exception e) {
            logException(e);
        }

        return getRequestResult(request, resultFileHolder, jsonResult, jobError[0]);
    }

    /**
     * @param session
     * @param resultFile
     * @param saveAsFileName
     * @param saveAsFolderId
     * @param givenFileName
     * @param givenMimeType
     * @return
     */
    protected JSONObject getJSONResultForSaveAs(@NonNull ServerSession session, String sessionId,
        File resultFile, String saveAsFileName, String saveAsFolderId, String givenFileName, String givenMimeType) throws Exception {

        JSONObject ret = null;

        if ((null != resultFile) && resultFile.canRead() && (resultFile.length() > 0) && !StringUtils.isEmpty(saveAsFileName) && !StringUtils.isEmpty(saveAsFolderId)) {
            // #60025: we need to check for an illegal CSRF attempt by checking the
            // correct sessionId; set errorcode to 400 in case of an invalid request
            if ((null == sessionId) || !sessionId.equals(session.getSessionID())) {
                DocumentConverterManager.logError("DC Ajax handler was called with invalid sessionId for save request: " + sessionId);
                ret = (new JSONObject()).put(KEY_ERRORCODE, ERRORCODE_BAD_REQUEST);
            } else {
                final IDBasedFolderAccessFactory folderFactory = m_services.getService(IDBasedFolderAccessFactory.class);
                final IDBasedFolderAccess folderAccess = (null != folderFactory) ? folderFactory.createAccess(session) : null;

                if (null != folderAccess) {
                    final FileStorageFolder folder = folderAccess.getFolder(saveAsFolderId);
                    final FileStoragePermission permission = (null != folder) ? folder.getOwnPermission() : null;

                    if ((null != permission) && (permission.getFolderPermission() >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER)) {
                        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                        final IDBasedFileAccess fileAccess = (null != fileFactory) ? fileFactory.createAccess(session) : null;

                        if (null != fileAccess) {
                            final com.openexchange.file.storage.File saveAsFileAccessFile = new DefaultFile();

                            saveAsFileAccessFile.setId(FileStorageFileAccess.NEW);
                            saveAsFileAccessFile.setFolderId(saveAsFolderId);
                            saveAsFileAccessFile.setFileName(saveAsFileName);
                            saveAsFileAccessFile.setFileMIMEType(givenMimeType);

                            try (final InputStream resultFileInputStm = FileUtils.openInputStream(resultFile)) {
                                String resultFileId = fileAccess.saveDocument(saveAsFileAccessFile, resultFileInputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());

                                if (!StringUtils.isEmpty(resultFileId)) {
                                    (ret = new JSONObject()).
                                        put("id", resultFileId).
                                        put("filename", saveAsFileAccessFile.getFileName()).
                                        put("errorCode", 0);
                                }
                            }

                            fileAccess.commit();
                        }
                    }
                }
            }
        }

        return ret;
    }
}
