/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.json;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.EnumSet;
import java.util.HashMap;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.crypto.CryptographicServiceAuthenticationFactory;
import com.openexchange.config.ConfigurationService;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.Nullable;
import com.openexchange.documentconverter.PDFExtractor;
import com.openexchange.documentconverter.Properties;
import com.openexchange.documentconverter.client.impl.ClientConfig;
import com.openexchange.documentconverter.client.impl.ClientManager;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageCapability;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.FolderID;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographicAwareIDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographyMode;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.groupware.ldap.User;
import com.openexchange.mail.MailServletInterface;
import com.openexchange.mail.dataobjects.MailPart;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link DocumentConverterAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public abstract class DocumentConverterAction implements AJAXActionService {

    // - Statics ---------------------------------------------------------------

    // TODO (KA): check list of parameters for completeness prior to every release!!!
    // The parameter list is used for the fast detection of ManagedFiles, based on
    // the set of all given attributes to create a unique identifier. The ManagedFile itself
    // is created for the speedup of chunked requests, using the same source again and again
    final static private String[] idParamArray = { "uid", "id", "folder_id", "version", "revtag", "source", "attached", "module", "cryptoAuth", "attachmentId" };

    /**
     * Initializes a new {@link DocumentConverterAction}.
     *
     * @param services
     */
    public DocumentConverterAction(ServiceLookup services) {
        super();

        m_services = services;

        if (null != m_services) {
            final ConfigurationService configService = m_services.getService(ConfigurationService.class);

            // read some config entries
            if (null != configService) {
                final int curIntEntry = configService.getIntProperty("com.openexchange.documentconverter.maxSourceFileSizeMB", Integer.MIN_VALUE);

                // ENGINE_MAX_SOURCEFILESIZE_MB
                if (curIntEntry != Integer.MIN_VALUE) {
                    m_maxSourceFileSizeMB = curIntEntry;
                }
            }

            m_managedFileManagement = m_services.getService(ManagedFileManagement.class);
        }
    }

    /**
     * @param request
     * @param session
     * @param jobError
     * @return
     */
    protected ManagedFile getRequestSourceFile(@NonNull final AJAXRequestData request, final @NonNull ServerSession session, @NonNull JobError[] jobError) {
        final FileIdLocker fileIdLocker = FileIdLocker.get(m_managedFileManagement);
        final String documentId = getDocumentId(request);

        ManagedFile ret = null;

        try {
            fileIdLocker.lock(documentId);

            if (null == (ret = getManagedFile(documentId))) {
                final String fileId = request.getParameter("id");
                final String source = request.getParameter("source");
                final String attached = request.getParameter("attached");
                final String module = request.getParameter("module");
                final String folderId = request.getParameter("folder_id");
                final String version = request.getParameter("version");
                final String cryptoAuth = request.getParameter("cryptoAuth");
                final boolean mailCheck = ((attached != null) && (folderId != null) && (attached.length() > 0));
                final boolean taskCheck = mailCheck && (module != null);
                MailServletInterface mailServletInterface = null; // closing done in finally block
                InputStream documentStm = null; // closing done in finally block

                try {
                    if (mailCheck && "mail".equals(source)) {
                        // mail attachment

                        if (StringUtils.isNotEmpty(cryptoAuth)) {
                            // If GuardMail, pull authentication from request
                            final CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = m_services.getOptionalService(CryptographicServiceAuthenticationFactory.class);
                            String authentication = null;

                            if (null != encryptionAuthenticationFactory) {
                                authentication = encryptionAuthenticationFactory.createAuthenticationFrom(request.optHttpServletRequest());
                            }

                            mailServletInterface = MailServletInterface.getInstanceWithDecryptionSupport(session, authentication);
                        } else {
                            mailServletInterface = MailServletInterface.getInstance(session);
                        }

                        if (null != mailServletInterface) {
                            final MailPart mailPart = mailServletInterface.getMessageAttachment(folderId, fileId, attached, false);

                            if (null != mailPart) {
                                documentStm = mailPart.getInputStream();
                            }
                        }
                    } else if (taskCheck && ("tasks".equals(source) || "calendar".equals(source) || "contacts".equals(source))) {
                        // other attachment from tasks, calendar or contact

                        documentStm = Attachments.getInstance().getAttachedFile(
                            session,
                            Integer.parseInt(folderId),
                            Integer.parseInt(attached),
                            Integer.parseInt(module),
                            Integer.parseInt(fileId),
                            session.getContext(),
                            session.getUser(),
                            session.getUserConfiguration());
                    } else {
                        // use FileStore as default data source
                        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                        IDBasedFileAccess fileAccess = (null != fileFactory) ? fileFactory.createAccess(session) : null;

                        if ((null != fileAccess) && "guard".equals(source)) {
                            if (StringUtils.isNotEmpty(cryptoAuth)) {
                                // If GuardMail, get authentication
                                final CryptographicAwareIDBasedFileAccessFactory encryptionAwareFileAccessFactory = m_services.getOptionalService(CryptographicAwareIDBasedFileAccessFactory.class);
                                final CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = m_services.getOptionalService(CryptographicServiceAuthenticationFactory.class);

                                if ((null != encryptionAwareFileAccessFactory) && (null != encryptionAuthenticationFactory)) {
                                    final EnumSet<CryptographyMode> cryptMode = EnumSet.of(CryptographyMode.DECRYPT);

                                    fileAccess = encryptionAwareFileAccessFactory.createAccess(fileAccess, cryptMode, session,
                                        encryptionAuthenticationFactory.createAuthenticationFrom(request.optHttpServletRequest()));
                                } else {
                                    // file is encrypted, but we can't decrypt...
                                    fileAccess = null;
                                }
                            } else {
                                fileAccess = null;
                            }

                            // set output paramter jobError to JobError.PASSWORD in
                            // case we got no valid fileAccess for given Guard parameters
                            if ((null == fileAccess) && (jobError.length > 0)) {
                                jobError[0] = JobError.PASSWORD;
                            }
                        }

                        if (null != fileAccess) {
                            // #46555 map version String("null") to null
                            String versionToUse = "null".equalsIgnoreCase(version) ? FileStorageFileAccess.CURRENT_VERSION : version;

                            // #46555 check for versioning support
                            if (FileStorageFileAccess.CURRENT_VERSION != versionToUse) {
                                final FolderID folderIdentifier = new FolderID(folderId);

                                if (!fileAccess.supports(folderIdentifier.getService(), folderIdentifier.getAccountId(), FileStorageCapability.FILE_VERSIONS)) {
                                    versionToUse = FileStorageFileAccess.CURRENT_VERSION;
                                }
                            }

                            documentStm = fileAccess.getDocument(fileId, versionToUse);
                        }
                    }

                    if (null != documentStm) {
                        ret = m_managedFileManagement.createManagedFile(documentId, documentStm);
                    }
                } catch (final Exception e) {
                    logException(e);
                } finally {
                    IOUtils.closeQuietly(documentStm);
                    if (mailServletInterface != null) {
                        mailServletInterface.close();
                    }
                }
            }
        } finally {
            fileIdLocker.unlock(documentId);
        }

        return ret;
    }

    /**
     * @param request
     * @param session
     * @param isAsync
     * @param resultJobError
     * @return
     */
    protected ManagedFile getPDFFile(@NonNull AJAXRequestData request, @NonNull ServerSession session, boolean isAsync, final JobError[] resultJobError) {
        final FileIdLocker fileIdLocker = FileIdLocker.get(m_managedFileManagement);
        final IDocumentConverter docConverter = m_services.getService(IDocumentConverter.class);
        ManagedFile managedResultFile = null;

        if (null != docConverter) {
            final String locale = getPreferredLanguage(session);
            final String resultDocumentId = getDocumentId(request, "pdf", locale);

            try {
                fileIdLocker.lock(resultDocumentId);

                if (null == (managedResultFile = getManagedFile(resultDocumentId))) {
                    final ManagedFile managedSourceFile = getRequestSourceFile(request, session, resultJobError);

                    if(null != managedSourceFile) {
                        final String fileName = request.getParameter("filename");

                        if ((null != fileName) && "pdf".equalsIgnoreCase(FilenameUtils.getExtension(fileName))) {
                            // return source file in case of a PDF source
                            managedResultFile = managedSourceFile;
                        } else {
                            // convert to PDF
                            final HashMap<String, Object> jobProperties = new HashMap<>(12);
                            final HashMap<String, Object> resultProperties = new HashMap<>(8);

                            jobProperties.put(Properties.PROP_INPUT_FILE, managedSourceFile.getFile());
                            jobProperties.put(Properties.PROP_PRIORITY, JobPriority.fromString(request.getParameter("priority")));
                            jobProperties.put(Properties.PROP_LOCALE, locale);

                            if (null != fileName) {
                                jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
                            }

                            if (isAsync) {
                                jobProperties.put(Properties.PROP_ASYNC, Boolean.TRUE);
                                jobProperties.put(Properties.PROP_PRIORITY, JobPriority.BACKGROUND);
                            } else {
                                // this is a user request in synchronous case
                                jobProperties.put(Properties.PROP_USER_REQUEST, Boolean.TRUE);
                            }

                            try (InputStream pdfResultStm = docConverter.convert("pdf", jobProperties, resultProperties)) {
                                if (null != pdfResultStm) {
                                    managedResultFile = m_managedFileManagement.createManagedFile(resultDocumentId, pdfResultStm);
                                    resultJobError[0] = JobError.NONE;
                                } else {
                                    resultJobError[0] = isAsync ? JobError.NONE : JobError.fromObject(resultProperties.get(Properties.PROP_RESULT_ERROR_CODE));
                                }
                            }
                        }
                    } else {
                        // async case for PDF documents
                        resultJobError[0] = isAsync ? JobError.NONE : JobError.NO_CONTENT;
                    }
                }
            } catch (Exception e) {
                logException(e);
            } finally {
                fileIdLocker.unlock(resultDocumentId);
            }
        } else {
            resultJobError[0] = JobError.GENERAL;
        }

        return managedResultFile;
    }

    /**
     * @param request
     * @param fileHolder
     * @return
     */
    AJAXRequestResult getRequestResult(@NonNull AJAXRequestData request, IFileHolder fileHolder, JSONObject jsonObject, JobError jobError) {
        AJAXRequestResult ret = null;

        if ((jobError == JobError.NONE) && (null != fileHolder)) {
            request.setFormat("file");
            ret = new AJAXRequestResult(fileHolder, "file");
        } else {
            final JSONObject jsonResult = (null != jsonObject) ? jsonObject : new JSONObject();

            try {
                String errorCause = null;

                switch (jobError) {
                    case NONE: {
                        break;
                    }

                    case TIMEOUT: {
                        errorCause = "timeout";
                        break;
                    }

                    case PASSWORD: {
                        errorCause = "passwordProtected";
                        break;
                    }

                    case NO_CONTENT: {
                        errorCause = "noContent";
                        break;
                    }

                    case MAX_SOURCESIZE: {
                        errorCause = "maxSourceSizeExceeded";
                        jsonResult.put("MaxSourceSizeMB", "" + m_maxSourceFileSizeMB);
                        break;
                    }

                    case DISPOSED: {
                        errorCause = "engineDisposed";
                        break;
                    }

                    case UNSUPPORTED: {
                        errorCause = "unsupportedType";
                        break;
                    }

                    case OUT_OF_MEMORY: {
                        errorCause = "outOfMemory";
                        break;
                    }

                    case ABORT: {
                        errorCause = "abort";
                        break;
                    }

                    case PDFTOOL: {
                        errorCause = "pdfTool";
                        break;
                    }

                    case GENERAL:
                    default: {
                        errorCause = "filterError";
                        break;
                    }
                }

                if (null != errorCause) {
                    jsonResult.put("cause", errorCause);
                }
            } catch (final JSONException e) {
                logException(e);
            }

            request.setFormat("json");
            ret = new AJAXRequestResult(jsonResult, "json");
        }

        return ret;
    }

    /**
     * @return
     * @throws IOException
     */
    protected @Nullable PDFExtractor getPDFExtractor() throws IOException {
        final ClientConfig clientConfig = ClientManager.get().getClientConfig();
        final File pdfExtractorExecutable = clientConfig.PDFEXTRACTOR_EXECUTABLE_PATH;

        return (null != pdfExtractorExecutable) ?
            PDFExtractor.get(pdfExtractorExecutable, clientConfig.PDFEXTRACTOR_WORKDIR) :
                null;
    }

    /**
     * @param pdfInputFile
     * @return
     */
    protected int getPDFPageCount(@NonNull final File pdfInputFile) throws IOException {
        final PDFExtractor pdfExtractor = getPDFExtractor();

        return (null != pdfExtractor) ?
            pdfExtractor.getPageCount(pdfInputFile) :
                -1;
    }

    /**
     * @param session
     * @return
     */
    protected @NonNull String getPreferredLanguage(ServerSession session) {
        final User sessionUser = session.getUser();
        String prefLanguage = null;

        // user language
        if (null != sessionUser) {
            prefLanguage = sessionUser.getPreferredLanguage();
        }

        return StringUtils.isBlank(prefLanguage) ? "en_US" : prefLanguage;
    }

    /**
     * @param request
     * @return
     */
    protected String getDocumentId(@NonNull AJAXRequestData request, String... additionalData) {
        final StringBuilder idBuilder = new StringBuilder(128);
        final String[][] allParamArrays = { idParamArray, additionalData };

        for (final String[] curParamArray : allParamArrays) {
            if (null != curParamArray) {
                final boolean isParam = curParamArray.equals(idParamArray);

                for (final String curParam : curParamArray) {
                    String curValue = curParam;

                    if (isParam) {
                        curValue = request.getParameter(curParam);
                    }

                    idBuilder.append(curValue).append('#');
                }
            }
        }

        return idBuilder.toString();
    }

    /**
     * @param id
     * @return
     */
    @Nullable protected ManagedFile getManagedFile(@NonNull final String id) {
        ManagedFile ret = null;

        try {
            ret = m_managedFileManagement.optByID(id);

            // TODO: clarify with Thorben, why this size check seems to be necessary nowadays
            if ((null != ret) && (ret.getSize() <= 0)) {
                ret.delete();
                ret = null;
            }
        } catch (final OXException e) {
            logException(e);
        }

        return ret;
    }


    /**
     * @param e
     */
    protected void logException(Exception e) {
        if (null != LOG) {
            LOG.error(((null != e) && (null != e.getMessage())) ? e.getMessage() : "Unknown exception occured");
        }
    }

    // - Members ---------------------------------------------------------------

    private Logger LOG = LoggerFactory.getLogger(this.getClass());

    protected ServiceLookup m_services = null;

    protected ManagedFileManagement m_managedFileManagement = null;

    protected long m_maxSourceFileSizeMB = 32;
}
