/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.impl;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import com.openexchange.documentconverter.DocumentConverterManager;
import com.openexchange.documentconverter.HttpHelper;
import com.openexchange.documentconverter.HttpResponseProcessor;
import com.openexchange.documentconverter.LogData;
import com.openexchange.documentconverter.NonNull;

/**
 * {@link RemoteDocumentConverterCheck}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
class RemoteValidator {

    /**
     * ENABLE_LOG_TIMEOUT
     */
    final private static long LOG_TIMEOUT = 60000;

    protected class RemoteValidatorRunnable implements Runnable {

        /**
         *
         */
        public RemoteValidatorRunnable() {
            super();
        }

        /**
         * @return
         * @throws Exception
         */
        @Override
        public void run() {
            final AtomicInteger remoteApiVersion = new AtomicInteger(-1);

            // check remote connection API; may be checked repeatedly, if a valid remote URL is set
            if (null != m_remoteUrl) {
                try {
                    HttpHelper.get().executeGet(m_remoteUrl, new HttpResponseProcessor() {

                        @Override
                        public void processResponse(int responseCode, InputStream responseInputStream, String resultCookie) {
                            try (final BufferedReader inputReader = new BufferedReader(new InputStreamReader(responseInputStream))) {
                                for (String readLine = null; (readLine = inputReader.readLine()) != null;) {
                                    readLine = readLine.toLowerCase();

                                    if (readLine.contains("open") && readLine.contains("xchange") && (readLine.contains("documentconverter"))) {
                                        if (remoteApiVersion.get() < 0) {
                                            remoteApiVersion.set(1);
                                        }
                                    } else if (readLine.contains("api:")) {
                                        final Matcher matcher = apiVersionPattern.matcher(readLine);

                                        if (matcher.find()) {
                                            remoteApiVersion.set(Integer.parseInt(matcher.group(1)));
                                        }
                                    }
                                }
                            } catch (@SuppressWarnings("unused") Exception e) {
                                // #52867: no action intended here
                            }
                        }
                    });
                } catch (@SuppressWarnings("unused") Exception e) {
                    // #52867: no action intended here
                }

                if (remoteApiVersion.get() < 0) {
                    // no connection established until now
                    final long curLogTime = System.currentTimeMillis();

                    if ((0 == m_lastLogTime) || ((curLogTime - m_lastLogTime) >= LOG_TIMEOUT)) {
                        DocumentConverterManager.logError("DC remote connection could not be established => please check remote converter setup",
                            new LogData("remoteurl", m_remoteUrl.toString()),
                            new LogData("async_request", "true"));
                    }

                    m_lastLogTime = curLogTime;
                } else {
                    // connection established
                    DocumentConverterManager.logInfo("DC established remote connection",
                        new LogData("remoteurl", m_remoteUrl.toString()),
                        new LogData("async_request", "true"));
                }
            }

            m_remoteAPIVersion.set(remoteApiVersion.get());
            m_isRunning.compareAndSet(true, false);
        }
    }

    /**
     * Initializes a new {@link RemoteValidator}.
     * @param serverType
     * @param remoteURL
     */
    RemoteValidator(@NonNull final URL remoteURL) {
        super();

        implSetRemoteURL(remoteURL);
    }

    /**
     * @param remoteURL
     */
    void updateRemoteURL(@NonNull final URL remoteURL) {
        implSetRemoteURL(remoteURL);
    }

    /**
     * @return
     */
    boolean isRemoteServerSupported() {
        return (null != m_remoteUrl);
    }

    /**
     * @return
     */
    int getRemoteAPIVersion() {
        return m_remoteAPIVersion.get();
    }

    /**
     * @return
     */
    synchronized void trigger() {
        if ((null != m_runnable) && m_isRunning.compareAndSet(false, true)) {
            Executors.newSingleThreadExecutor().execute(m_runnable);
        }
    }

    /**
     *
     */
    synchronized void resetConnection() {
        if ((m_remoteAPIVersion.get() >= 1) && !m_isRunning.get()) {
            setConnectionLost();
        }
    }

    /**
     *
     */
    void setConnectionLost() {
        m_remoteAPIVersion.set(-1);

        if (DocumentConverterManager.isLogError()) {
            DocumentConverterManager.logError("DC lost remote connection:", new LogData("remoteurl", m_remoteUrl.toString()));
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param remoteUrl
     */
    void implSetRemoteURL(@NonNull final URL remoteUrl) {
        if ((null == m_remoteUrl) || !remoteUrl.equals(m_remoteUrl)) {
            m_remoteAPIVersion.set(-1);
            m_isRunning.set(false);
            m_remoteUrl = remoteUrl;

            // trigger new connection validation
            trigger();
        }
    }

    // - Static members --------------------------------------------------------

    protected static Pattern apiVersionPattern = Pattern.compile("api\\:\\s*v([0-9]+)");

    // - Members ---------------------------------------------------------------

    protected RemoteValidatorRunnable m_runnable = new RemoteValidatorRunnable();

    protected AtomicInteger m_remoteAPIVersion = new AtomicInteger(-1);

    protected AtomicBoolean m_isRunning = new AtomicBoolean(false);

    protected URL m_remoteUrl = null;

    protected long m_lastLogTime = 0;
}
