/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter;

import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.util.HashMap;
import java.util.UUID;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * {@link BasicManager}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
/**
 * {@link DocumentConverterManager}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
public class DocumentConverterManager {

    final private static int HTTP_DOCUMENTCONVERTER_CONNECT_TIMEOUT_MS=60000;

    final private static int HTTP_DOCUMENTCONVERTER_READ_TIMEOUT_MS=(int) ((HTTP_DOCUMENTCONVERTER_CONNECT_TIMEOUT_MS * 210L) / 100L);

    final private static int HTTP_CACHESERVER_CONNECT_TIMEOUT_MS=5000;

    final private static int HTTP_CACHESERVER_READ_TIMEOUT_MS=60000;

    /**
     * Initializes a new {@link DocumentConverterManager}.
     */
    public DocumentConverterManager() {
        super();
    }

    /**
     *
     */
    public void resetRemoteConnection() {
        // nothing to do by default
    }

    /**
     * @param file
     * @param locale
     * @return
     */
    public static StringBuilder getFileHashBuilder(File file, String locale) {
        StringBuilder hashBuilder = null;

        if ((null != file) && file.canRead()) {
            hashBuilder = new StringBuilder();

            try {
                hashBuilder.append(Long.toString(file.length())).append('_').
                    append(Long.toString(FileUtils.checksumCRC32(file))).append('-');
            } catch (final Exception e) {
                logExcp(e);
            }

            // 'en*' is considered to be the default locale; add all other
            // locales to the
            // input file hash in order to be able to perform locale based
            // conversions
            if (!StringUtils.isEmpty(locale)) {
                hashBuilder.append(locale).append('-');
            }
        }

        return hashBuilder;
    }

    /**
     * @return
     */
    public static int getHTTPConnectTimeoutMilliseconds(ServerType serverType) {
        switch (serverType) {
            case CACHESERVER: {
                return HTTP_CACHESERVER_CONNECT_TIMEOUT_MS;
            }

            case DOCUMENTCONVERTER:
            default: {
                return HTTP_DOCUMENTCONVERTER_CONNECT_TIMEOUT_MS;
            }
        }
    }

    /**
     * @return
     */
    public static int getHTTPReadTimeoutMilliseconds(ServerType serverType) {
        switch (serverType) {
            case CACHESERVER: {
                return HTTP_CACHESERVER_READ_TIMEOUT_MS;
            }

            case DOCUMENTCONVERTER:
            default: {
                return HTTP_DOCUMENTCONVERTER_READ_TIMEOUT_MS;
            }
        }
    }

    /**
     * @param prefix
     * @param fileName
     * @return
     */
    public static File createTempFile(String prefix, String fileName) {
        return createTempFile(FileUtils.getTempDirectoryPath(), prefix, fileName);
    }

    /**
     * @param prefix
     * @param fileName
     * @return
     */
    public static File createTempFile(@NonNull String tmpRoot, String prefix, String fileName) {
        File tmpFile = null;

        try {
            if ((null != fileName) && (fileName.length() > 0)) {
                final File scratchDir = new File(tmpRoot);
                final String dirPrefix = ((((null != prefix) && (prefix.length() > 0)) ? prefix : Long.valueOf(System.currentTimeMillis()).toString())) + "_";
                final long number = System.nanoTime();
                final long maxIterations = 10000;

                for (long i = number, end = number + maxIterations; i < end; ++i) {
                    final File tempDir = new File(scratchDir, dirPrefix + i);

                    if (tempDir.mkdirs()) {
                        tmpFile = new File(tempDir.getAbsolutePath(), fileName);
                        break;
                    }
                }
            } else {
                final File scratchDir = new File(tmpRoot);

                if ((scratchDir.isDirectory() && scratchDir.canWrite()) || mkdirs(scratchDir)) {
                    // create a temp. file with the name 'prefix*.tmp' within the configured documentconverter cache directory
                    tmpFile = File.createTempFile(prefix, ".tmp", scratchDir);
                }
            }
        } catch (final IOException e) {
            logExcp(e);
        }

        return tmpFile;
    }

    /**
     * @param tmpRoot
     * @return
     */
    public static File createTempDir(@NonNull String tmpRoot) {
        final File root = new File(tmpRoot);
        File tempDirFile = null;

        if (root.canWrite()) {
            while (null == tempDirFile) {
                tempDirFile = new File(tmpRoot, UUID.randomUUID().toString());

                try {
                    FileUtils.forceMkdir(tempDirFile);
                } catch (IOException e) {
                    DocumentConverterManager.logExcp(e);
                }

                if (!tempDirFile.exists()) {
                    tempDirFile = null;
                }
            }
        }

        return tempDirFile;
    }

    /**
     * @param dirFile
     * @return
     */
    public static boolean mkdirs(File directory) {
        boolean ret = false;

        if (null != directory) {
            if (!directory.exists()) {
                try {
                    FileUtils.forceMkdir(directory);
                } catch (final IOException e) {
                    logExcp(e);
                }

                FileUtils.waitFor(directory, 3);
            }

            ret = directory.canWrite();
        }

        return ret;
    }

    /**
     * @param filename
     * @return
     */
    public static byte[] getResourceBuffer(String filename) {
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        byte[] resourceBuffer = null;

        if (null == loader) {
            loader = DocumentConverterManager.class.getClassLoader();
        }

        final URL url = loader.getResource(filename);

        if (null != url) {
            URLConnection connection = null;

            try {
                connection = url.openConnection();

                if (null != connection) {
                    connection.connect();

                    try (InputStream resourceInputStm = connection.getInputStream()) {
                        if (null != resourceInputStm) {
                            resourceBuffer = IOUtils.toByteArray(resourceInputStm);
                        }
                    }
                }
            } catch (final IOException e) {
                logExcp(e);
            }
        }

        return resourceBuffer;
    }

    /**
     * @param jobProperties
     * @param deleteInputFile
     * @return
     */
    public static File getJobInputFile(@NonNull HashMap<String, Object> jobProperties, @NonNull boolean[] deleteInputFile) {
        File inputFile = (File) jobProperties.get(Properties.PROP_INPUT_FILE);

        deleteInputFile[0] = false;

        // if no input file is given, write content of possible InputStream
        // property to a temp file; the conversion is always done using a real
        // file as input source due to very bad performance with InputStream
        // reads via the UNO bridge (at least with latest LO ReaderEngine)
        if (null == inputFile) {
            try (final InputStream inputStm = (InputStream) jobProperties.get(Properties.PROP_INPUT_STREAM)) {
                if (null != inputStm) {
                    if (null != (inputFile = DocumentConverterManager.createTempFile(FileUtils.getTempDirectoryPath(), "oxcs", null))) {
                        try {
                            FileUtils.copyInputStreamToFile(inputStm, inputFile);

                            // set the just filled inputFile as new source at our
                            // job
                            // and remove the previously set input stream;
                            jobProperties.put(Properties.PROP_INPUT_FILE, inputFile);
                            jobProperties.remove(Properties.PROP_INPUT_STREAM);

                            // the temporarily created file needs to be deleted by
                            // the caller of this method later in time
                            deleteInputFile[0] = true;
                        } catch (final IOException e) {
                            logExcp(e);

                            FileUtils.deleteQuietly(inputFile);
                            inputFile = null;
                        }
                    }
                }
            } catch (final IOException e) {
                logExcp(e);
            }
        }

        // check for input type %PDF-x.x, since this input
        // type is explicitly checked at various locations
        if (null != inputFile) {
            try (final InputStream inputStm = FileUtils.openInputStream(inputFile)) {
                final byte[] buffer = new byte[256];

                if ((inputStm.read(buffer) > 0) && new String(buffer, "UTF-8").trim().toLowerCase().startsWith("%pdf-")) {
                    jobProperties.put(Properties.PROP_INPUT_TYPE, "pdf");
                }
            } catch (final IOException e) {
                logExcp(e);
            }
        }

        return inputFile;
    }

    // - Logging ---------------------------------------------------------------

    /**
     * @param logProvider
     * @return
     */
    public static boolean isLogTrace() {
        return ((null != LOG) && LOG.isTraceEnabled());
    }

    /**
     * @return true, if the log level 'info' is enabled
     */
    public static boolean isLogDebug() {
        return ((null != LOG) && LOG.isDebugEnabled());
    }

    /**
     * @return true, if the log level 'info' is enabled
     */
    public static boolean isLogInfo() {
        return ((null != LOG) && LOG.isInfoEnabled());
    }

    /**
     * @return true, if the log level 'warn' is enabled
     */
    public static boolean isLogWarn() {
        return ((null != LOG) && LOG.isWarnEnabled());
    }

    /**
     * @return true, if the log level 'error' is enabled
     */
    public static boolean isLogError() {
        return ((null != LOG) && LOG.isErrorEnabled());
    }

    /**
     * @param info
     * @param extraData
     */
    static public void logInfo(final String info, LogData... extraData) {
        logInfo(info, null, extraData);
    }

    /**
     * @param info
     * @param message
     * @param jobProperties
     * @param extraData
     */
    static public void logInfo(final String info, final HashMap<String, Object> jobProperties, LogData... extraData) {
        if (null != LOG) {
            LOG.info(implGetLogMessage(info, jobProperties, extraData));
        }
    }

    /**
     * @param warning
     * @param extraData
     */
    static public void logWarn(final String warning, LogData... extraData) {
        logWarn(warning, null, extraData);
    }

    /**
     * @param warning
     * @param jobProperties
     * @param extraData
     */
    static public void logWarn(final String warning, final HashMap<String, Object> jobProperties, LogData... extraData) {
        if (null != LOG) {
            LOG.warn(implGetLogMessage(warning, jobProperties, extraData));
        }
    }

    /**
     * @param error
     * @param extraData
     */
    static public void logError(final String error, LogData... extraData) {
        logError(error, null, extraData);
    }

    /**
     * @param error
     * @param jobProperties
     * @param extraData
     */
    static public void logError(final String error, final HashMap<String, Object> jobProperties, LogData... extraData) {
        if (null != LOG) {
            LOG.error(implGetLogMessage(error, jobProperties, extraData));
        }
    }

    /**
     * @param error
     * @param extraData
     */
    static public void logDebug(final String debugMsg, LogData... extraData) {
        logDebug(debugMsg, null, extraData);
    }

    /**
     * @param error
     * @param jobProperties
     * @param extraData
     */
    static public void logDebug(final String debugMsg, final HashMap<String, Object> jobProperties, LogData... extraData) {
        if (null != LOG) {
            LOG.debug(implGetLogMessage(debugMsg, jobProperties, extraData));
        }
    }

    /**
     * @param error
     * @param extraData
     */
    static public void logTrace(final String traceMsg, LogData... extraData) {
        logTrace(traceMsg, null, extraData);
    }

    /**
     * @param error
     * @param jobProperties
     * @param extraData
     */
    static public void logTrace(final String traceMsg, final HashMap<String, Object> jobProperties, LogData... extraData) {
        if (null != LOG) {
            LOG.trace(implGetLogMessage(traceMsg, jobProperties, extraData));
        }
    }

    /**
     * @param e
     */
    static public void logExcp(Exception e) {
        if (isLogError()) {
            String message = (null != e) ? e.getMessage() : null;
            final String lowerMessage = (null != message) ? message.toLowerCase() : null;

            if ((e instanceof SocketException) ||
                (e instanceof SocketTimeoutException) ||
                (e instanceof EOFException) ||
                ((e instanceof IOException) && (null != lowerMessage) && ((lowerMessage.contains("broken pipe") || lowerMessage.contains(("connection reset")) || lowerMessage.contains(("invalid stream")))))) {
                // ignore but trace socket exceptions; these may occur, when a connection timeout has happened
                LOG.trace(implGetLogMessage("DC caught an ignorable connection exception: \n" + message, null));
            } else {
                if (null != message) {
                    if (message.length() > Properties.MAX_ERROR_MESSAGE_TOTAL_LENGTH) {
                        final String fillStr = "...";
                        final int beginLength = Properties.MAX_ERROR_MESSAGE_TOTAL_LENGTH - Properties.MAX_ERROR_MESSAGE_END_LENGTH - fillStr.length();
                        final int endLength = Math.min(message.length() - beginLength, Properties.MAX_ERROR_MESSAGE_END_LENGTH);
                        message = message.substring(0, beginLength) + ((endLength > 0) ? ("..." + message.substring(message.length() - endLength)) : "");
                    }
                } else if (null != e){
                    message = e.getClass().getName();
                }

                LOG.error(implGetLogMessage("DC caught an exception: \n" + message, null));
            }
        }
    }

    /**
     * @param logType
     * @param message
     * @param jobProperties
     * @param extraData
     */
    public static String implGetLogMessage(final String message, HashMap<String, Object> jobProperties, LogData... extraData) {
        final StringBuilder logMessageBuilder = (new StringBuilder(message.length() << 1)).append(message);
        StringBuilder infoStrBuilder = null;
        String fileName = null;

        if (null != jobProperties) {
            fileName = (String) jobProperties.get(Properties.PROP_INFO_FILENAME);
        }

        if (extraData.length > 0) {
            logMessageBuilder.append(" (");

            for (int i = 0; i < extraData.length; ++i) {
                final LogData logData = extraData[i];

                if (null != logData) {
                    if (i > 0) {
                        logMessageBuilder.append(", ");
                    }

                    logMessageBuilder.append(logData.getKey());
                    logMessageBuilder.append('=');

                    final String value = logData.getValue();
                    logMessageBuilder.append((null != value) ? value : "null");
                }
            }
        }

        if (null != fileName) {
            try {
                fileName = URLDecoder.decode(fileName, "UTF-8");
            } catch (final UnsupportedEncodingException e) {
                logExcp(e);
            }

            if (extraData.length > 0) {
                logMessageBuilder.append(", filename=");
            } else {
                logMessageBuilder.append(" (filename=");
            }

            logMessageBuilder.append(fileName).append(')');
        } else if (extraData.length > 0) {
            logMessageBuilder.append(')');

        }

        if ((null != jobProperties) && !jobProperties.isEmpty()) {
            for (final String key : jobProperties.keySet()) {
                // log all info_ properties but info_Filename only in case of an error
                if (key.startsWith(Properties.INFO_PREFIX) && !key.equals(Properties.PROP_INFO_FILENAME)) {
                    final Object obj = jobProperties.get(key);

                    if (null != infoStrBuilder) {
                        infoStrBuilder.append(", ");
                    } else {
                        infoStrBuilder = new StringBuilder(" (jobproperties: ");
                    }

                    infoStrBuilder.append(key.substring(Properties.INFO_PREFIX.length())).append('=').append(
                        (null != obj) ? obj.toString() : "null");
                }
            }

            if (null != infoStrBuilder) {
                infoStrBuilder.append(')');
            }
        }

        if (null != infoStrBuilder) {
            logMessageBuilder.append(infoStrBuilder);
        }

        return logMessageBuilder.toString();
    }

    // - Static members --------------------------------------------------------

    private static Logger LOG = LoggerFactory.getLogger(DocumentConverterManager.class);
}
