/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.websockets.grizzly.osgi;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.openexchange.websockets.WebSocketListener;
import com.openexchange.websockets.grizzly.impl.DefaultGrizzlyWebSocketApplication;
import com.openexchange.websockets.grizzly.impl.WebSocketListenerAdapter;
import com.openexchange.websockets.grizzly.impl.WebSocketListenerRegistry;


/**
 * {@link WebSocketListenerTracker}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.8.3
 */
public class WebSocketListenerTracker implements ServiceTrackerCustomizer<Object, Object>, WebSocketListenerRegistry {

    /*
     * Uses a ReadWriteLock to prevent from calling getListeners() while a new listener appears in addingService() call-back.
     * Non-synchronized access might end-up in WebSockets having duplicate listeners added.
     */

    private final BundleContext context;
    private final CopyOnWriteArrayList<org.glassfish.grizzly.websockets.WebSocketListener> adapters;
    private volatile DefaultGrizzlyWebSocketApplication application;

    /**
     * Initializes a new {@link WebSocketListenerTracker}.
     */
    public WebSocketListenerTracker(BundleContext context, WebSocketListener... initialListeners) {
        super();
        this.context = context;
        if (null != initialListeners && initialListeners.length > 0) {
            List<org.glassfish.grizzly.websockets.WebSocketListener> initialAdapters = new ArrayList<>(initialListeners.length);
            for (WebSocketListener listener : initialListeners) {
                initialAdapters.add(WebSocketListenerAdapter.newAdapterFor(listener));
            }
            adapters = new CopyOnWriteArrayList<>(initialAdapters);
        } else {
            adapters = new CopyOnWriteArrayList<>();
        }
    }

    /**
     * Sets the application
     *
     * @param application The application to set
     */
    public void setApplication(DefaultGrizzlyWebSocketApplication application) {
        this.application = application;
    }

    @Override
    public Object addingService(ServiceReference<Object> reference) {
        Object service = context.getService(reference);
        if (false == (service instanceof WebSocketListener)) {
            context.ungetService(reference);
            return null;
        }

        WebSocketListener listener = (WebSocketListener) service;
        WebSocketListenerAdapter adapter = WebSocketListenerAdapter.newAdapterFor(listener);
        if (adapters.addIfAbsent(adapter)) {
            DefaultGrizzlyWebSocketApplication application = this.application;
            if (null != application) {
                application.addWebSocketListener(adapter);
            }
            return listener;
        }
        context.ungetService(reference);
        return null;
    }

    @Override
    public void modifiedService(ServiceReference<Object> reference, Object service) {
        // Nothing
    }

    @Override
    public void removedService(ServiceReference<Object> reference, Object service) {
        if (false == (service instanceof WebSocketListener)) {
            context.ungetService(reference);
            return;
        }

        WebSocketListener listener = (WebSocketListener) service;
        WebSocketListenerAdapter adapter = WebSocketListenerAdapter.newAdapterFor(listener);
        if (adapters.remove(adapter)) {
            DefaultGrizzlyWebSocketApplication application = this.application;
            if (null != application) {
                application.removeWebSocketListener(adapter);
            }
        }
        context.ungetService(reference);
    }

    @Override
    public List<org.glassfish.grizzly.websockets.WebSocketListener> getListeners() {
        return adapters;
    }

}
