package com.openexchange.office.tools.osgi;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleListener;
import org.osgi.framework.ServiceRegistration;
import org.slf4j.Logger;

import com.openexchange.ajax.requesthandler.AJAXActionServiceFactory;
import com.openexchange.ajax.requesthandler.ExtendableAJAXActionServiceFactory;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
public abstract class ActivatorBase implements BundleActivator
										     , BundleListener
{
	//-------------------------------------------------------------------------
	public static final Logger LOG = LogFactory.getSlfLog (ActivatorBase.class);
	
    //-------------------------------------------------------------------------
    private static final boolean VERIFY_SERVICE_INTEGRITY = true;
    
	//-------------------------------------------------------------------------
	public ActivatorBase (final String sBundleName)
	    throws Exception
	{
		m_sBundleName = sBundleName;
        LOG.trace("["+m_sBundleName+"] : activator created new ...");
	}
	
	//-------------------------------------------------------------------------
	protected synchronized String getBundleContextId ()
		throws Exception
	{
		Validate.notNull(m_aBundleContext, "Cant calculate ID for this activator if no bundle context is well known. Dont call this method before start() or after stop() ;-)");
		final String sId = ObjectUtils.identityToString(m_aBundleContext);
		return sId;
	}

	//-------------------------------------------------------------------------
	protected synchronized List< Class< ? > > getMandatoryServiceImports ()
	    throws Exception
	{
		return mem_MandatoryServiceImports();
	}
	
	//-------------------------------------------------------------------------
	protected synchronized List< Class< ? > > getOptionalServiceImports ()
	    throws Exception
	{
		return mem_OptionalServiceImports();
	}

	//-------------------------------------------------------------------------
	protected synchronized List< Class< ? > > getServiceExports ()
	    throws Exception
	{
		return mem_ServiceExports ();
	}

	//-------------------------------------------------------------------------
	/** INTERNAL - has not to be called from derived class nor from any other place.
	 *  It's called from the OSGI runtime to start this activator / bundle.
	 *
	 *  @param  aBundleContext [IN]
	 *  		the bundle context this activator is bound to.
	 */
	@Override
	public /* no synchronized */ void start(final BundleContext aBundleContext)
		throws Exception
	{
		try
		{
			synchronized (this)
			{
				if (m_bIsActive)
					return;

				m_bIsActive      = true;
				m_aBundleContext = aBundleContext;
			}

			if (VERIFY_SERVICE_INTEGRITY)
				impl_verifyNoDoubleRegistration ();

	        LOG.trace("["+m_sBundleName+"] : register mandatory service imports ...");
			impl_transferMandatoryServiceImports ();

			LOG.trace("["+m_sBundleName+"] : register optional service imports ...");
			impl_transferOptionalServiceImports  ();

	        LOG.trace("["+m_sBundleName+"] : activate service ...");
			impl_activate ();
			
	        LOG.trace("["+m_sBundleName+"] : start service tracking");
	        impl_startTracking ();

	        LOG.trace("["+m_sBundleName+"] : register in lookup registry");
			impl_registerLookup ();
			
			LOG.trace("["+m_sBundleName+"] : is active now");
		}
		catch (Throwable ex)
		{
			LOG.error(ex.getMessage(), ex);
			impl_deregisterLookup ();

			synchronized (this)
			{
				m_bIsActive = false;
			}
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);
			throw new RuntimeException (ex);
		}
	}

	//-------------------------------------------------------------------------
	/** INTERNAL - has not to be called from derived class nor from any other place.
	 *  It's called from the OSGI runtime to stop this activator / bundle.
	 *
	 *  @param  aBundleContext [IN]
	 *  		the bundle context this activator is bound to.
	 */
	@Override
	public /* no synchronized */ void stop(final BundleContext aBundleContext)
		throws Exception
	{
		try
		{
			synchronized (this)
			{
				if ( ! m_bIsActive)
					return;
				m_bIsActive = false;
			}

	        LOG.trace("["+m_sBundleName+"] : deregister from lookup registry");
			impl_deregisterLookup ();

			LOG.trace("["+m_sBundleName+"] : deactivate service ...");
			impl_deactivate ();

	        LOG.trace("["+m_sBundleName+"] : stop service tracking");
			impl_stopTracking ();

			LOG.trace("["+m_sBundleName+"] : is inactive now");
		}
		catch (Throwable ex)
		{
			// reset of m_bIsActive makes no sense ...
			// we don't know in which state this bundle is (if an exception occurred) ...
			
			LOG.error(ex.getMessage(), ex);
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);

			throw new RuntimeException (ex);
		}
	}

	//-------------------------------------------------------------------------
	/** INTERNAL - has not to be called from derived class nor from any other place.
	 *  It's called from the OSGI runtime to notify this activator / bundle about changes at runtime.
	 *
	 *  @param  aEvent [IN]
	 *  		describe the change in detail.
	 */
	@Override
	public void bundleChanged(final BundleEvent aEvent)
	{
	}

	//-------------------------------------------------------------------------
	protected /* no synchronized */ < S > void registerService(final Class< S > aService ,
									                           final S          aInstance)
    {
		registerService (aService, aInstance, null);
	}

    //-------------------------------------------------------------------------
    protected /* no synchronized */ < S > void registerService(final Class< S >              aService   ,
    									                       final S                       aInstance  ,
    									                       final Dictionary< String, ? > aDictionary)
    {
		try
		{
			Validate.notNull(aService , "Invalid argument 'service'." );
			Validate.notNull(aInstance, "Invalid argument 'instance'.");
			// Dictionary is optional
			
			LOG.trace ("["+m_sBundleName+"] : register service : [service="+aService+", instance="+aInstance+"] ...");

			ActivatorCore     aImpl    = null;
			List< Class< ? >> lExports = null;
			
			synchronized (this)
			{
				aImpl    = mem_Impl           ();
				lExports = mem_ServiceExports ();
			}

			aImpl   .registerService(aService, aInstance, aDictionary);
			lExports.add            (aService                        );
		}
		catch (Throwable ex)
		{
			LOG.error(ex.getMessage(), ex);
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);

			throw new RuntimeException(ex);
		}
    }

	//-------------------------------------------------------------------------
	protected /* no synchronized */void registerAJAXModule(final AJAXActionServiceFactory aModule    ,
									  					   final String                   sModuleName,
									  					   final boolean                  bMultiple  )
	{
		try
		{
			Validate.notNull (aModule    , "Invalid argument 'module'."    );
			Validate.notEmpty(sModuleName, "Invalid argument 'moduleName'.");

			final Dictionary< String, Object > aDictionary = new Hashtable< String, Object >();
	        aDictionary.put("module"  , sModuleName                 );
	        aDictionary.put("multiple", bMultiple ? "true" : "false");

			LOG.trace("["+m_sBundleName+"] : register ajax module '"+sModuleName+"' ...");
	        registerService(AJAXActionServiceFactory.class, aModule, aDictionary);

	        if (aModule instanceof ExtendableAJAXActionServiceFactory)
	            registerService(ExtendableAJAXActionServiceFactory.class, (ExtendableAJAXActionServiceFactory)aModule, null);
		}
		catch (Throwable ex)
		{
			LOG.error(ex.getMessage(), ex);
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);

			throw new RuntimeException (ex);
		}
    }

    //-------------------------------------------------------------------------
	protected /* no synchronized */ boolean hasService (final Class< ? > aService)
	{
		try
		{
			final List< Class< ? > > lMandatoryServices = mem_MandatoryServiceImports ();
			final List< Class< ? > > lOptionalServices  = mem_OptionalServiceImports  ();
			final List< Class< ? > > lExportedServices  = mem_ServiceExports          ();
			
			final boolean bIsInMandatoryList = impl_isServiceInList (aService, lMandatoryServices);
			final boolean bIsInOptionalList  = impl_isServiceInList (aService, lOptionalServices );
			final boolean bIsInExportList    = impl_isServiceInList (aService, lExportedServices );
			final boolean bIsInAnyList       = (bIsInMandatoryList || bIsInOptionalList || bIsInExportList);
			
			return bIsInAnyList;
		}
		catch (Throwable ex)
		{
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);
		}

		return false;
	}

    //-------------------------------------------------------------------------
    protected /* no synchronized */ <S extends Object> S getService(final Class< ? extends S > aService)
	{
		return (S) impl_getService (aService, /* mandatory */ true);
	}

    //-------------------------------------------------------------------------
	protected /* no synchronized */ < S > S getOptionalService(final Class< ? extends S > aService)
    {
		return (S) impl_getService (aService, /* mandatory */ false);
	}

	//-------------------------------------------------------------------------
	protected abstract void listMandatoryServiceImports (final List< Class< ? >> lServices)
		throws Exception;

	//-------------------------------------------------------------------------
	protected abstract void listOptionalServiceImports (final List< Class< ? >> lServices)
		throws Exception;

	//-------------------------------------------------------------------------
	protected abstract void activate ()
        throws Exception;

	//-------------------------------------------------------------------------
	protected abstract void deactivate ()
        throws Exception;

    //-------------------------------------------------------------------------
	private /* no synchronized */ void impl_transferMandatoryServiceImports ()
		throws Exception
	{
		final Class< ? >[]  lServcies = impl_convertServiceListToArray(mem_MandatoryServiceImports());
		ActivatorCore aImpl = null;
		synchronized(this)
		{
			aImpl = mem_Impl ();
		}
		aImpl.setMandatoryServiceImports(lServcies);
	}
	
    //-------------------------------------------------------------------------
	private /* no synchronized */ void impl_transferOptionalServiceImports ()
		throws Exception
	{
		final Class< ? >[]  lServcies = impl_convertServiceListToArray(mem_OptionalServiceImports());
		ActivatorCore aImpl = null;
		synchronized(this)
		{
			aImpl = mem_Impl ();
		}
		aImpl.setOptionalServiceImports(lServcies);
	}

	//-------------------------------------------------------------------------
	private /* no synchronized */ void impl_activate ()
		throws Exception
	{
		LOG.trace("["+m_sBundleName+"] impl_activate");
		ActivatorCore aImpl = null;
		synchronized(this)
		{
			aImpl = mem_Impl ();
		}
		aImpl.start(m_aBundleContext);
	}

	//-------------------------------------------------------------------------
	private /* no synchronized */ void impl_deactivate ()
		throws Exception
	{
		LOG.trace("["+m_sBundleName+"] impl_deactivate");
		ActivatorCore aImpl = null;
		synchronized(this)
		{
			aImpl = mem_Impl ();
		}
		aImpl.stop(m_aBundleContext);
	}

	//-------------------------------------------------------------------------
	private /* no synchronized */ void impl_startTracking ()
		throws Exception
	{
		LOG.trace ("["+m_sBundleName+"] start servcie tracking ...");
		ActivatorCore aImpl = null;
		synchronized(this)
		{
			aImpl = mem_Impl ();
		}
		aImpl.openTrackers();
	}

	//-------------------------------------------------------------------------
	private /* no synchronized */ void impl_stopTracking ()
		throws Exception
	{
		LOG.trace ("["+m_sBundleName+"] stop servcie tracking ...");
		ActivatorCore aImpl = null;
		synchronized(this)
		{
			aImpl = mem_Impl ();
		}
		aImpl.closeTrackers();
	}

    //-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	private /* no synchronized */ < S > S impl_getService(final Class< ? extends S > aService  ,
														  final boolean              bMandatory)
		/* no throws */
    {
		try
		{
			LOG.trace("["+m_sBundleName+"] get service : '"+aService+"' ...");;
			
			if (VERIFY_SERVICE_INTEGRITY)
				impl_verifyServiceIsInAnyList (aService);

			ActivatorCore aImpl          = null;
			Object        aInstance      = null;
			BundleContext aBundleContext = null;
			
			synchronized(this)
			{
				aImpl          = mem_Impl ();
				aBundleContext = m_aBundleContext;
			}
			
			LOG.trace("["+m_sBundleName+"] ... look in imports");;
			
			if (bMandatory)
				aInstance = aImpl.getService         (aService);
			else
				aInstance = aImpl.getOptionalService (aService);
			
			if (aInstance == null)
			{
				LOG.trace("["+m_sBundleName+"] ... look in exports");;
				final ServiceRegistration< ? > aServiceReg = aImpl.getServiceRegistrationFor(aService);
				if (aServiceReg != null)
					aInstance = aBundleContext.getService(aServiceReg.getReference());
			}

			if (aInstance == null && VERIFY_SERVICE_INTEGRITY)
			{
				final ServiceRegistration< ? > aServiceReg        = aImpl.getServiceRegistrationFor(aService);
				final boolean                  bIsInMandatoryList = impl_isServiceInList (aService, mem_MandatoryServiceImports ());
				final boolean                  bIsInOptionalList  = impl_isServiceInList (aService, mem_OptionalServiceImports  ());
				final boolean                  bIsInExportList    = impl_isServiceInList (aService, mem_ServiceExports          ());
				
				final StringBuffer sMsg = new StringBuffer (256);
				sMsg.append ("===============================================\n");
				sMsg.append ("getService () did not found requested service '"+aService+"'\n");
				sMsg.append ("\n");
				sMsg.append ("activator inst  = "+this                +"\n");
				sMsg.append ("activator scope = "+this.m_sBundleName +"\n");
				sMsg.append ("service         = "+aService            +"\n");
				sMsg.append ("instance        = "+aInstance           +"\n");
				sMsg.append ("registration    = "+aServiceReg         +"\n");
				sMsg.append ("\n");
				sMsg.append ("in mandatory    = "+bIsInMandatoryList  +"\n");
				sMsg.append ("in optional     = "+bIsInOptionalList   +"\n");
				sMsg.append ("in exports      = "+bIsInExportList     +"\n");
				sMsg.append ("\n");
				sMsg.append ("stack :\n"+ExceptionUtils.getStackTrace(new Exception ()));

				LOG.error (sMsg.toString ());
				LOG.error (impl_describeService (aService, aInstance));
			}
			
			return (S) aInstance;
		}
		catch (Throwable ex)
		{
			LOG.error(ex.getMessage(), ex);
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);

			throw new RuntimeException (ex);
		}
	}

	//-------------------------------------------------------------------------
	public class ActivatorCoreBinding implements IActivatorCoreBinding
	{
		//-------------------------------------------------------------------------
		@Override
		public void inActivate()
			throws Exception
		{
			LOG.trace("["+m_sBundleName+"] binding : in activate");
			activate ();
		}
	
		//-------------------------------------------------------------------------
		@Override
		public void inDeactivate()
			throws Exception
		{
			LOG.trace("["+m_sBundleName+"] binding : in deactivate");
			deactivate ();
		}
	
		//-------------------------------------------------------------------------
		@Override
		public void serviceIsUnavailable (final Class< ? > aService)
			throws Exception
		{
			final boolean bIsMandatory = impl_isServiceInList(aService, mem_MandatoryServiceImports ());
			if (bIsMandatory)
				stop (m_aBundleContext);
		}
	
		//-------------------------------------------------------------------------
		@Override
		public void serviceIsAvailable (final Class< ? > aService)
			throws Exception
		{
			final boolean bAllAvailable = mem_Impl ().allServciesAvailable();
			if (bAllAvailable)
				start (m_aBundleContext);
		}
	}

	//-------------------------------------------------------------------------
	private String impl_describeService (final Class< ? > aService ,
										 final Object     aInstance)
	{
//		try
//		{
//			final Bundle                   aBundle     = FrameworkUtil.getBundle(aService);
//			final BundleContext            aContextA   = aBundle.getBundleContext();
//			final BundleContext            aContextB   = super.context;
//			final ServiceRegistration< ? > aServiceReg = super.getServiceRegistrationFor(aService);
//			
//			final StringBuffer sDesc = new StringBuffer (256);
//			sDesc.append ("===============================================\n");
//			sDesc.append ("service descriptor :\n");
//			sDesc.append ("activator inst  : "+this                     +"\n");
//			sDesc.append ("activator scope : "+this.m_sServiceName      +"\n");
//			sDesc.append ("service inst    : "+aInstance                +"\n");
//			sDesc.append ("service class   : "+aService                 +"\n");
//			sDesc.append ("service reg     : "+aServiceReg              +"\n");
//			sDesc.append ("context (util)  : "+aContextA                +"\n");
//			sDesc.append ("context (base)  : "+aContextB                +"\n");
//			sDesc.append ("bundle          : "+aBundle                  +"\n");
//			sDesc.append ("bundle-id       : "+aBundle.getBundleId()    +"\n");
//			sDesc.append ("bundle-name     : "+aBundle.getSymbolicName()+"\n");
//			return sDesc.toString ();
//		}
//		catch (Throwable ex)
//		{}
		
		return "???";
	}
	
    //-------------------------------------------------------------------------
    private synchronized void impl_verifyServiceIsInAnyList (final Class< ? > aService)
        throws Exception
    {
		final String             sService           = aService.getName ();
		final List< Class< ? > > lMandatoryServices = mem_MandatoryServiceImports ();
		final List< Class< ? > > lOptionalServices  = mem_OptionalServiceImports  ();
		final List< Class< ? > > lExportedServices  = mem_ServiceExports          ();
		
		final boolean bIsInMandatoryList = impl_isServiceInList (aService, lMandatoryServices);
		final boolean bIsInOptionalList  = impl_isServiceInList (aService, lOptionalServices );
		final boolean bIsInExportList    = impl_isServiceInList (aService, lExportedServices );
		final boolean bIsValid           = (bIsInMandatoryList || bIsInOptionalList || bIsInExportList);

		if ( ! bIsValid)
		{
			final StringBuffer sMsg = new StringBuffer (256);
			sMsg.append ("["+m_sBundleName+"] : service '"+sService+"' is not in list of mandatory or optional services nor in it's own export list. Dependency tree not valid.");
			sMsg.append ("\ndetails:");
			sMsg.append ("\nmandatory services : "+impl_convertServiceListToString(lMandatoryServices ));
			sMsg.append ("\noptional  services : "+impl_convertServiceListToString(lMandatoryServices ));
			sMsg.append ("\nexported  services : "+impl_convertServiceListToString(lExportedServices  ));
			sMsg.append ("\nin mandatory list  : "+bIsInMandatoryList                                  );
			sMsg.append ("\nin optional  list  : "+bIsInOptionalList                                   );
			sMsg.append ("\nin export    list  : "+bIsInExportList                                     );
			sMsg.append ("\nis valid           : "+bIsValid                                            );
			
			throw new Exception(sMsg.toString ());
		}
    }
    
	//-------------------------------------------------------------------------
	private synchronized void impl_verifyNoDoubleRegistration ()
		throws Exception
	{
		final List< Class< ? >> lMandatoryServices = mem_MandatoryServiceImports ();
		final List< Class< ? >> lOptionalServices  = mem_OptionalServiceImports  ();
		
		for (final Class< ? > aMandatory : lMandatoryServices)
		{
			if (lOptionalServices.contains(aMandatory))
				throw new Exception ("["+m_sBundleName+"] : mandatory service '"+aMandatory+"' was also registered as optional one.");
		}
	}
	
	//-------------------------------------------------------------------------
	private /* no synchronized */ void impl_registerLookup ()
	    /* no throws */
	{
		try
		{
			ServiceLookupRegistry.get().registerServiceLookup(this);
		}
		catch (Throwable ex)
		{
			LOG.error(ex.getMessage(), ex);
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);

			throw new RuntimeException (ex);
		}
	}
	
	//-------------------------------------------------------------------------
	private /* no synchronized */ void impl_deregisterLookup ()
	    /* no throws */
	{
		try
		{
			ServiceLookupRegistry.get().deregisterServiceLookup(this);
		}
		catch (Throwable ex)
		{
			LOG.error(ex.getMessage(), ex);
        	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);

			throw new RuntimeException (ex);
		}
	}
	
	//-------------------------------------------------------------------------
	private static boolean impl_isServiceInList (final Class< ? >         aService,
												 final List< Class< ? > > lList   )
	    throws Exception
	{
		final boolean bIsInList = lList.contains(aService);
		return bIsInList;
	}

	//-------------------------------------------------------------------------
	private static Class< ? >[] impl_convertServiceListToArray (final List< Class< ? >> lServices)
	    throws Exception
	{
		final int          c      = lServices.size();
		final Class< ? >[] lArray = new Class< ? >[c];
		lServices.toArray(lArray);
		return lArray;
	}

	//-------------------------------------------------------------------------
	private static String impl_convertServiceListToString (final List< Class< ? >> lServices)
	    throws Exception
	{
		final StringBuffer sString       = new StringBuffer (256);
		      boolean      bAddSeparator = false;
		for (final Class< ? > aService : lServices)
		{
			if (bAddSeparator)
				sString.append (",");
			else
				bAddSeparator = true;
			sString.append (aService.getName());
		}
		return sString.toString ();
	}
	
	//-------------------------------------------------------------------------   
	private synchronized ActivatorCore mem_Impl ()
		throws Exception
	{
		if (m_aImpl == null)
		{
			m_aImpl = new ActivatorCore ();
			m_aImpl.setBinding(new ActivatorCoreBinding ());
		}
		return m_aImpl;
	}
	
	//-------------------------------------------------------------------------   
	private synchronized List< Class< ? >> mem_MandatoryServiceImports ()
	    throws Exception
	{
		if (m_aMandatoryServiceImports == null)
		{
	        LOG.trace("["+m_sBundleName+"] : list mandatory service imports ...");
			final List< Class< ? >> lServices = new Vector< Class< ? >> ();
			listMandatoryServiceImports (lServices);

			LOG.trace("["+m_sBundleName+"] : mandatory = {"+impl_convertServiceListToString (lServices)+"}");
			m_aMandatoryServiceImports = lServices;
		}
		return m_aMandatoryServiceImports;
	}

	//-------------------------------------------------------------------------   
	private synchronized List< Class< ? >> mem_OptionalServiceImports ()
	    throws Exception
	{
		if (m_aOptionalServiceImports == null)
		{
	        LOG.trace("["+m_sBundleName+"] : list optional service imports ...");
			final List< Class< ? >> lServices = new Vector< Class< ? >> ();
			listOptionalServiceImports (lServices);

			LOG.trace("["+m_sBundleName+"] : optional = {"+impl_convertServiceListToString (lServices)+"}");
			m_aOptionalServiceImports = lServices;
		}
		return m_aOptionalServiceImports;
	}

	//-------------------------------------------------------------------------   
	private synchronized List< Class< ? >> mem_ServiceExports ()
	    throws Exception
	{
		if (m_aServiceExports == null)
			m_aServiceExports = new Vector< Class< ? >> ();
		return m_aServiceExports;
	}

	//-------------------------------------------------------------------------   
	private ActivatorCore m_aImpl = null;

	//-------------------------------------------------------------------------   
	private BundleContext m_aBundleContext = null;

	//-------------------------------------------------------------------------   
	private String m_sBundleName = null;

	//-------------------------------------------------------------------------   
	private boolean m_bIsActive = false;
	
	//-------------------------------------------------------------------------   
	private List< Class< ? >> m_aMandatoryServiceImports = null;
	
	//-------------------------------------------------------------------------   
	private List< Class< ? >> m_aOptionalServiceImports = null;

	//-------------------------------------------------------------------------   
	private List< Class< ? >> m_aServiceExports = null;
}
