/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl;

import org.apache.commons.lang3.StringUtils;
import org.json.JSONObject;

/**
 * Helper class to extract properties from the connect data sent by the frontend
 * to correctly initialize the connection instance.
 *
 * @author Carsten Driesner
 *
 */
public class ConnectDataHelper {
    private final String PROP_CONNECTDATA_FOLDER_ID            = "folder_id";
    private final String PROP_CONNECTDATA_FILE_ID              = "id";
    private final String PROP_CONNECTDATA_OLD_RTID             = "oldrtId";
    private final String PROP_CONNECTDATA_SESSION_ID           = "session";
    private final String PROP_CONNECTDATA_LOCALSTORAGE_VERSION = "storageVersion";
    private final String PROP_CONNECTDATA_LOCALSTORAGE_OSN     = "storageOSN";

    private final JSONObject connectData;

    /**
     * Initializes a new ConnectDataHelper instance.
     *
     * @param connectData
     */
    public ConnectDataHelper(final JSONObject connectData) {
        this.connectData = connectData;
    }

    /**
     * Determines, if the connect data have valid resource data (folder & file-id) or
     * not.
     *
     * @return
     */
    public boolean hasValidResource() {
        boolean result = false;

        if ((null != connectData) && connectData.has(PROP_CONNECTDATA_FOLDER_ID) && connectData.has(PROP_CONNECTDATA_FILE_ID)) {
            result = (StringUtils.isNotEmpty(connectData.optString(PROP_CONNECTDATA_FOLDER_ID)) &&
                      StringUtils.isNotEmpty(connectData.optString(PROP_CONNECTDATA_FILE_ID)));
        }

        return result;
    }

    /**
     * Provides the folder-id of the connect data or null.
     *
     * @return
     */
    public final String getFolderId() {
        return (null != connectData) ? connectData.optString(PROP_CONNECTDATA_FOLDER_ID) : null;
    }

    /**
     * Provides the folder-id of the connect data or null.
     *
     * @return
     */
    public final String getFileId() {
        return (null != connectData) ? connectData.optString(PROP_CONNECTDATA_FILE_ID) : null;
    }

    /**
     * Provides the old-rt id of the client.
     *
     * @return
     */
    public final String getOldRTId() {
        return (null != connectData) ? connectData.optString(PROP_CONNECTDATA_OLD_RTID) : null;
    }

    /**
     * Provides the session id of the client.
     *
     * @return
     */
    public final String getSessionId() {
        return (null != connectData) ? connectData.optString(PROP_CONNECTDATA_SESSION_ID) : null;
    }

    /**
     * Provides the session id of the client.
     *
     * @return
     */
    public final boolean hasValidLocalStorageData() {
        return (null != connectData) ? StringUtils.isNotEmpty(connectData.optString(PROP_CONNECTDATA_LOCALSTORAGE_VERSION)) : false;
    }

    /**
     * Provides the session id of the client.
     *
     * @return
     */
    public final String getLocalStoragVersion() {
        return (null != connectData) ? connectData.optString(PROP_CONNECTDATA_LOCALSTORAGE_VERSION) : null;
    }

    /**
     * Provides the session id of the client.
     *
     * @return
     */
    public final int getLocalStorageOSN() {
        return (null != connectData) ? connectData.optInt(PROP_CONNECTDATA_LOCALSTORAGE_OSN) : -1;
    }

}
