/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.docx.components;

import javax.xml.bind.JAXBException;
import javax.xml.datatype.XMLGregorianCalendar;
import org.docx4j.Child;
import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTBookmark;
import org.docx4j.wml.CTMarkupRange;
import org.docx4j.wml.CTPPrChange;
import org.docx4j.wml.CTParaRPrOriginal;
import org.docx4j.wml.CTTrackChange;
import org.docx4j.wml.CommentRangeEnd;
import org.docx4j.wml.CommentRangeStart;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Drawing;
import org.docx4j.wml.FldChar;
import org.docx4j.wml.IText;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.PPrBase.PStyle;
import org.docx4j.wml.ParaRPr;
import org.docx4j.wml.ParaRPrChange;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.RunDel;
import org.docx4j.wml.RunIns;
import org.docx4j.wml.RunMoveFrom;
import org.docx4j.wml.RunMoveTo;
import org.docx4j.wml.STBrType;
import org.docx4j.wml.STFldCharType;
import org.docx4j.wml.SdtBlock;
import org.docx4j.wml.SdtRun;
import org.json.JSONException;
import org.json.JSONObject;
import com.google.common.collect.ImmutableSet;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.ooxml.components.Component;
import com.openexchange.office.filter.ooxml.components.ComponentContext;
import com.openexchange.office.filter.ooxml.components.IParagraph;
import com.openexchange.office.filter.ooxml.docx.tools.Character;
import com.openexchange.office.filter.ooxml.docx.tools.DrawingML;
import com.openexchange.office.filter.ooxml.docx.tools.DrawingML.DrawingML_root;
import com.openexchange.office.filter.ooxml.docx.tools.Paragraph;
import com.openexchange.office.filter.ooxml.docx.tools.TextUtils;
import com.openexchange.office.filter.ooxml.docx.tools.Utils;

public class ParagraphComponent extends DocxComponent implements IParagraph {

    public ParagraphComponent(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);
    }

    @Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
        final IndexedNode<Object> paragraphNode = getNode();
        final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)paragraphNode.getData()).getContent();
        final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;
        IndexedNode<Object> childNode = previousChildContext!=null ? nodeList.getNextNode(previousChildContext.getNode()) : nodeList.getFirstNode();

        Component nextComponent = null;
        for(; nextComponent==null&&childNode!=null; childNode = nodeList.getNextNode(childNode)) {
            final Object o = getContentModel(childNode, paragraphNode.getData());
            if(o instanceof R) {
                final TextRunContext textRunContext = new TextRunContext(this, childNode);
                nextComponent = textRunContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof P.Hyperlink) {
                final HyperlinkContext hyperlinkContext = new HyperlinkContext(this, childNode);
                nextComponent = hyperlinkContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof RunIns) {
                final RunInsContext runInsContext = new RunInsContext(this, childNode);
                nextComponent = runInsContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof RunDel) {
                final RunDelContext runDelContext = new RunDelContext(this, childNode);
                nextComponent = runDelContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof RunMoveTo) {
                final RunMoveToContext runMoveToContext = new RunMoveToContext(this, childNode);
                nextComponent = runMoveToContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof RunMoveFrom) {
                final RunMoveFromContext runMoveFromContext = new RunMoveFromContext(this, childNode);
                nextComponent = runMoveFromContext.getNextChildComponent(null, previousChildComponent);
            }
            else if(o instanceof CommentRangeStart) {
            	nextComponent = new CommentRangeStartComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof CommentRangeEnd) {
            	nextComponent = new CommentRangeEndComponent(this, childNode, nextComponentNumber);
            }
            else if(o.getClass()==CTBookmark.class) {
                nextComponent = new BookmarkStartComponent(this, childNode, nextComponentNumber);
            }
            else if(o.getClass()==CTMarkupRange.class) {
                nextComponent = new BookmarkEndComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof SdtRun) {
                final SdtParagraphContext sdtParagraphContext = new SdtParagraphContext(this, childNode);
                nextComponent = sdtParagraphContext.getNextChildComponent(null, previousChildComponent);
            }
        }
        return nextComponent;
    }
    @Override
    public Component insertChildComponent(int textPosition, JSONObject attrs, Type childType)
        throws JSONException, JAXBException, InvalidFormatException, PartUnrecognisedException {

        switch(childType) {
            case TAB : return insertChildWithTextRun(Context.getWmlObjectFactory().createRTab(), textPosition, attrs);
            case HARDBREAK_DEFAULT :
            case HARDBREAK_PAGE :
            case HARDBREAK_COLUMN : {
                final Child newChild = Context.getWmlObjectFactory().createBr();
                if(childType==Type.HARDBREAK_PAGE) {
                    ((Br)newChild).setType(STBrType.PAGE);
                }
                else if(childType==Type.HARDBREAK_COLUMN) {
                    ((Br)newChild).setType(STBrType.COLUMN);
                }
                return insertChildWithTextRun(newChild, textPosition, attrs);
            }
            case COMPLEX_FIELD_START :
            case COMPLEX_FIELD_SEPARATE :
            case COMPLEX_FIELD_END : {
            	final Child newChild = Context.getWmlObjectFactory().createFldChar();
            	((FldChar)newChild).setFldCharType(STFldCharType.BEGIN);
            	if(childType==Type.COMPLEX_FIELD_END) {
                	((FldChar)newChild).setFldCharType(STFldCharType.END);
            	}
            	else if(childType==Type.COMPLEX_FIELD_SEPARATE) {
                	((FldChar)newChild).setFldCharType(STFldCharType.SEPARATE);
            	}
            	else {
            		((FldChar)newChild).setFldLock(false);
            	}
            	final FldChar_Base component = (FldChar_Base)insertChildWithTextRun(newChild, textPosition, attrs);
            	component.removeInvalidParents();
            	return component;
            }
            case COMMENT_REFERENCE :        return insertChildWithTextRun(Context.getWmlObjectFactory().createRCommentReference(), textPosition, attrs);
            case COMMENT_RANGE_START :      return insertChildWithoutTextRun(Context.getWmlObjectFactory().createCommentRangeStart(), textPosition, attrs);
            case COMMENT_RANGE_END :        return insertChildWithoutTextRun(Context.getWmlObjectFactory().createCommentRangeEnd(), textPosition, attrs);
            case BOOKMARK_START:            return insertChildWithoutTextRun(new CTBookmark(), textPosition, attrs);
            case BOOKMARK_END:              return insertChildWithoutTextRun(new CTMarkupRange(), textPosition, attrs);
            case AC_IMAGE : {
                final Drawing drawing = Context.getWmlObjectFactory().createDrawing();
                drawing.getAnchorOrInline().add(DrawingML_root.createInline(false, "", 500000, 500000));
                return insertChildWithTextRun(drawing, textPosition, attrs);
            }
            case AC_SHAPE :                 return insertChildWithTextRun(DrawingML.createAlternateContentShape(operationDocument.getPackage()), textPosition, attrs);
            case AC_CONNECTOR :             return insertChildWithTextRun(DrawingML.createAlternateContentConnector(operationDocument.getPackage()), textPosition, attrs);
            case AC_GROUP :                 return insertChildWithTextRun(DrawingML.createAlternateContentGroup(operationDocument.getPackage()), textPosition, attrs);
            default : {
                throw new UnsupportedOperationException();
            }
        }
    }

    private Component insertChildWithTextRun(Child newChild, int textPosition, JSONObject attrs)
        throws JSONException, InvalidFormatException, PartUnrecognisedException, JAXBException {

        final P paragraph = (P)getObject();
        final IndexedNodeList<Object> paragraphContent = paragraph.getContent();

        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1, SplitMode.ATTRIBUTE);
            }
            if(childComponent instanceof TextRun_Base) {

                // the new child can be added into an existing textRun
                final R run = ((TextRun_Base)childComponent).getTextRun();
                newChild.setParent(run);
                run.getContent().addNode(((TextRun_Base)childComponent).getNode(), new IndexedNode<Object>(newChild), textPosition==0);
            }
            else {

                // we need to create a new textRun, check if we can get referenceAttributes that we will clone for the new run. this is
                // only possible if the childComponent is a CommentRangeStartComponent or an CommentRangeEndComponent
                final R newRun = Context.getWmlObjectFactory().createR();
                newChild.setParent(newRun);
                newRun.getContent().add(newChild);

                final ContentAccessor parentContextObject = (ContentAccessor)childComponent.getParentContext().getNode().getData();
                ((IndexedNodeList<Object>)parentContextObject.getContent()).addNode(childComponent.getNode(), new IndexedNode<Object>(newRun), textPosition==0);
            }
        }
        else {
            final R newRun = Context.getWmlObjectFactory().createR();
            newRun.setParent(paragraph);
            paragraphContent.add(newRun);
            newChild.setParent(newRun);
            newRun.getContent().add(newChild);
            if(paragraph.getPPr(false)!=null) {
                final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr(false).getRPr(false), operationDocument.getPackage());
                if(rPr!=null) {
                    newRun.setRPr(rPr);
                }
            }
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(childComponent instanceof TextRun_Base) {
            if(((TextRun_Base)childComponent).hasChangeTracking()) {
                attrs = TextUtils.forceRemoveChangeTrack(attrs);
            }
        }
        if(attrs!=null) {
            childComponent.splitStart(textPosition, SplitMode.ATTRIBUTE);
            childComponent.splitEnd(textPosition, SplitMode.ATTRIBUTE);
            childComponent.applyAttrsFromJSON(attrs);
        }
        return childComponent;
    }

    private Component insertChildWithoutTextRun(Child newChild, int textPosition, JSONObject attrs)
        throws JSONException, InvalidFormatException, PartUnrecognisedException, JAXBException {

        final P paragraph = (P)getObject();
        final IndexedNodeList<Object> paragraphContent = paragraph.getContent();

        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1, SplitMode.ATTRIBUTE);
            }
            final ComponentContext contextChild = childComponent.getContextChild(parentContextList);
            final ComponentContext parentContext = contextChild.getParentContext();
            final ContentAccessor parentObject = (ContentAccessor)parentContext.getNode().getData();
            newChild.setParent(parentObject);
            ((IndexedNodeList<Object>)parentObject.getContent()).addNode(contextChild.getNode(), new IndexedNode<Object>(newChild), textPosition==0);
        }
        else {
            newChild.setParent(paragraph);
            paragraphContent.add(newChild);
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(childComponent instanceof TextRun_Base) {
            if(((TextRun_Base)childComponent).hasChangeTracking()) {
                attrs = TextUtils.forceRemoveChangeTrack(attrs);
            }
        }
        if(attrs!=null) {
            childComponent.splitStart(textPosition, SplitMode.ATTRIBUTE);
            childComponent.splitEnd(textPosition, SplitMode.ATTRIBUTE);
            childComponent.applyAttrsFromJSON(attrs);
        }
        return childComponent;
    }

    static final private ImmutableSet<Class<?>> parentContextList = ImmutableSet.<Class<?>> builder()
        .add(SdtParagraphContext.class)
        .add(HyperlinkContext.class)
        .add(RunMoveToContext.class)
        .add(RunMoveFromContext.class)
        .add(RunInsContext.class)
        .add(RunDelContext.class)
        .build();

    public static void splitAccessor(Component component, boolean splitStart, SplitMode splitMode) {
        IndexedNode<Object> node = component.getNode();
        ComponentContext parentContext = component.getParentContext();
        while(parentContext!=null) {
        	if(parentContext instanceof ParagraphComponent) {
        		break;
        	}
        	if(splitMode==SplitMode.ATTRIBUTE) {
        		if(parentContext instanceof SdtParagraphContext) {
        			break;
        		}
        	}

            final IndexedNode<Object> parentNode = parentContext.getNode();
            final IndexedNodeList<Object> parentNodeContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
            if((splitStart&&!node.isFirst())||(!splitStart&&!node.isLast(parentNodeContent))) {
                IndexedNode<Object> newParentNode = null;
                if(parentContext instanceof TextRunContext) {
                    final R parent = (R)parentNode.getData();
                    final R newParent = Context.getWmlObjectFactory().createR();
                    newParentNode = new IndexedNode<Object>(newParent);
                    final RPr rPr = parent.getRPr();
                    if(rPr!=null) {
                    	final RPr rPrNew = XmlUtils.deepCopy(rPr, component.getOperationDocument().getPackage());
                    	TextUtils.resetMarkupId(rPrNew);
                        newParent.setRPr(rPrNew);
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                        ((TextRun_Base)component).setTextRunNode(newParentNode);
                    }
                }
                else if(parentContext instanceof HyperlinkContext) {
                    final P.Hyperlink parent = (P.Hyperlink)parentNode.getData();
                    final P.Hyperlink newParent = Context.getWmlObjectFactory().createPHyperlink();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAnchor(parent.getAnchor());
                    newParent.setDocLocation(parent.getDocLocation());
                    newParent.setHistory(parent.isHistory());
                    newParent.setId(parent.getId());
                    newParent.setTgtFrame(parent.getTgtFrame());
                    newParent.setTooltip(parent.getTooltip());
                    if(splitStart&&component instanceof TextRun_Base) {
                    	((TextRun_Base)component).setHyperlinkNode(newParentNode);
                    }
                }
                else if(parentContext instanceof RunMoveToContext) {
                    final RunMoveTo parent = (RunMoveTo)parentNode.getData();
                    final RunMoveTo newParent = Context.getWmlObjectFactory().createRunMoveTo();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAuthor(parent.getAuthor());
                    final XMLGregorianCalendar date = parent.getDate();
                    if(date!=null) {
                        newParent.setDate((XMLGregorianCalendar) date.clone());
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                        ((TextRun_Base)component).setRunMoveToNode(newParentNode);
                    }
                }
                else if(parentContext instanceof RunMoveFromContext) {
                    final RunMoveFrom parent = (RunMoveFrom)parentNode.getData();
                    final RunMoveFrom newParent = Context.getWmlObjectFactory().createRunMoveFrom();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAuthor(parent.getAuthor());
                    final XMLGregorianCalendar date = parent.getDate();
                    if(date!=null) {
                        newParent.setDate((XMLGregorianCalendar) date.clone());
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                        ((TextRun_Base)component).setRunMoveFromNode(newParentNode);
                    }
                }
                else if(parentContext instanceof RunInsContext) {
                    final RunIns parent = (RunIns)parentNode.getData();
                    final RunIns newParent = Context.getWmlObjectFactory().createRunIns();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAuthor(parent.getAuthor());
                    final XMLGregorianCalendar date = parent.getDate();
                    if(date!=null) {
                        newParent.setDate((XMLGregorianCalendar) date.clone());
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                    	((TextRun_Base)component).setRunInsNode(newParentNode);
                    }
                }
                else if(parentContext instanceof RunDelContext) {
                    final RunDel parent = (RunDel)parentNode.getData();
                    final RunDel newParent = Context.getWmlObjectFactory().createRunDel();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setAuthor(parent.getAuthor());
                    final XMLGregorianCalendar date = parent.getDate();
                    if(date!=null) {
                        newParent.setDate((XMLGregorianCalendar)date.clone());
                    }
                    if(splitStart&&component instanceof TextRun_Base) {
                    	((TextRun_Base)component).setRunDelNode(newParentNode);
                    }
                }
                else if(parentContext instanceof SdtParagraphContext&&splitMode==SplitMode.DELETE) {
                    final SdtRun parent = (SdtRun)parentNode.getData();
                    final SdtRun newParent = Context.getWmlObjectFactory().createSdtRun();
                    newParentNode = new IndexedNode<Object>(newParent);
                    newParent.setSdtEndPr(parent.getSdtEndPr());
                    newParent.setSdtPr(parent.getSdtPr());
                }
                if(newParentNode!=null) {
                    ((Child)newParentNode.getData()).setParent(parentNode.getData());
                    final IndexedNodeList<Object> rContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
                    final IndexedNodeList<Object> rContentNew = (IndexedNodeList<Object>)((ContentAccessor)newParentNode.getData()).getContent();
                    final Object pParent = ((Child)parentNode.getData()).getParent();
                    final Object pNewParent = newParentNode.getData();
                    ((Child)pNewParent).setParent(pParent);
                    final IndexedNodeList<Object> pParentNode = (IndexedNodeList<Object>)((ContentAccessor)pParent).getContent();
                    pParentNode.addNode(parentNode, newParentNode, false);
                    final IndexedNode<Object> sourceRefNode = !splitStart ? rContent.getNextNode(node) : node;
                    if(sourceRefNode!=null) {
                        rContent.moveNodes(sourceRefNode, -1, rContentNew, rContentNew.getLastNode(), false, pNewParent);
                    }
                    if(splitStart) {
                        parentContext.setNode(newParentNode);
                    }
                }
            }
            node = parentContext.getNode();
            parentContext = parentContext.getParentContext();
        }
    }

    @Override
	public void applyAttrsFromJSON(JSONObject attrs) {
		if(attrs==null) {
			return;
		}
		try {
	        final P paragraph = (P)getObject();
	        final PPr pPr = paragraph.getPPr(true);
	        if(attrs.has(OCKey.STYLE_ID.value())) {
	            final Object styleId = attrs.get(OCKey.STYLE_ID.value());
	            if(styleId instanceof String) {
	                final PPrBase.PStyle pStyle = pPr.getPStyle(true);
	                pStyle.setVal((String)styleId);
	                pPr.setPStyle(pStyle);
	            }
	            else {
	                pPr.setPStyle(null);
	            }
	        }
	        Paragraph.applyParagraphProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, attrs.optJSONObject(OCKey.PARAGRAPH.value()), pPr);
	        final Object characterAttrs = attrs.opt(OCKey.CHARACTER.value());
	        if(characterAttrs instanceof JSONObject) {
	            com.openexchange.office.filter.ooxml.docx.tools.Character.applyCharacterProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)characterAttrs, pPr.getRPr(true));
	        }
            final Object changes = attrs.opt(OCKey.CHANGES.value());
            if(changes!=null) {
	            if(changes instanceof JSONObject) {
	                final ParaRPr paraRPr = pPr.getRPr(true);
	            	final Object attrsInserted = ((JSONObject)changes).opt(OCKey.INSERTED.value());
	            	if(attrsInserted!=null) {
	            		if(attrsInserted instanceof JSONObject) {
	            		    if(paraRPr.getIns(false)==null) {
	            		        paraRPr.setIns(paraRPr.getMoveTo(false));
	            		        paraRPr.setMoveTo(null);
	            		    }
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)attrsInserted, paraRPr.getIns(true));
	            		}
	            		else {
	            			paraRPr.setIns(null);
	            		}
	            		paraRPr.setMoveTo(null);
	            	}
	            	final Object attrsRemoved = ((JSONObject)changes).opt(OCKey.REMOVED.value());
	            	if(attrsRemoved!=null) {
	            		if(attrsRemoved instanceof JSONObject) {
	            		    if(paraRPr.getDel(false)==null) {
	            		        paraRPr.setDel(paraRPr.getMoveFrom(false));
	            		        paraRPr.setMoveFrom(null);
	            		    }
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)attrsRemoved, paraRPr.getDel(true));
	            		}
	            		else {
	            			paraRPr.setDel(null);
	            		}
	            		paraRPr.setMoveFrom(null);
	            	}
	            	final Object modified = ((JSONObject)changes).opt(OCKey.MODIFIED.value());
	            	if(modified!=null) {
	            		if(modified instanceof JSONObject) {
	            		    final ParaRPrChange rPrChange = paraRPr.getRPrChange(true);
	            		    final CTPPrChange pPrChange = pPr.getPPrChange(true);
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)modified, rPrChange);
                			Utils.applyTrackInfoFromJSON((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)modified, pPrChange);
                			final Object attrsModified = ((JSONObject)modified).opt(OCKey.ATTRS.value());
                			if(attrsModified!=null) {
                				if(attrsModified instanceof JSONObject) {
		                			// Style
                					if(((JSONObject) attrsModified).has(OCKey.STYLE_ID.value())) {
        					            final Object styleId = attrs.get(OCKey.STYLE_ID.value());
        					            final PPrBase pPrBase = pPrChange.getPPr(true);
         					            if(styleId instanceof String) {
         					                final PPrBase.PStyle pStyle = pPrBase.getPStyle(true);
        					                pStyle.setVal((String)styleId);
        					                pPrBase.setPStyle(pStyle);
        					            }
        					            else {
        					                pPrBase.setPStyle(null);
        					            }
                					}
                					final Object characterChanges = ((JSONObject)attrsModified).opt(OCKey.CHARACTER.value());
			            			if(characterChanges!=null) {
			            				if(characterChanges instanceof JSONObject) {
			            					Character.applyCharacterProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)characterChanges, rPrChange.getRPr(true));
			            				}
			            				else {
			            					rPrChange.setRPr(null);
			            				}
			            			}
			            			final Object paragraphChanges = ((JSONObject)attrsModified).opt(OCKey.PARAGRAPH.value());
			            			if(paragraphChanges!=null) {
			            				if(paragraphChanges instanceof JSONObject) {
	            							Paragraph.applyParagraphProperties((com.openexchange.office.filter.ooxml.docx.DocxOperationDocument)operationDocument, (JSONObject)paragraphChanges, pPrChange.getPPr(true));
			            				}
			            				else {
			            					pPr.setPPrChange(null);
			            				}
			            			}
                				}
                				else {
                					paraRPr.setRPrChange(null);
                					pPr.setPPrChange(null);
                				}
                			}
	            		}
	            		else {
	            			paraRPr.setRPrChange(null);
	            			pPr.setPPrChange(null);
	            		}
	            	}
	            }
	            else {
	            	final ParaRPr paraRPr = pPr.getRPr(false);
	            	if(paraRPr!=null) {
	            		paraRPr.setIns(null);
	            		paraRPr.setDel(null);
	            		paraRPr.setRPrChange(null);
	            	}
	            }
            }
		}
        catch (Exception e) {
            //
		}
	}

	@Override
	public JSONObject createJSONAttrs(JSONObject attrs)
		throws JSONException {

		final PPr pPr = ((P)getObject()).getPPr(false);
        if(pPr!=null) {
            final PPrBase.PStyle pStyle = pPr.getPStyle(false);
            if(pStyle!=null) {
                attrs.put(OCKey.STYLE_ID.value(), pStyle.getVal());
            }
            final JSONObject paragraphProperties = Paragraph.createParagraphProperties(pPr);
            if(paragraphProperties!=null) {
                attrs.put(OCKey.PARAGRAPH.value(), paragraphProperties);
            }
            final ParaRPr paraRPr = pPr.getRPr(false);
            if(paraRPr!=null) {
            	// only creating paragraph character attributes if there are no childs with character attributes elements #35344#, #35571#
                Component childComponent = getNextChildComponent(null, null);
                while(childComponent!=null) {
                    if(childComponent instanceof TextRun_Base) {
                        break;
                    }
                    childComponent = childComponent.getNextComponent();
                }
                if(childComponent==null) {
                    final JSONObject characterProperties = Character.createCharacterProperties(operationDocument.getThemeFonts(), paraRPr);
                    if(characterProperties!=null) {
                        attrs.put(OCKey.CHARACTER.value(), characterProperties);
                    }
                }
            }

            CTTrackChange rPrIns = null;
        	CTTrackChange rPrDel = null;
        	ParaRPrChange rPrChange = null;
        	if(paraRPr!=null) {
        		rPrIns = paraRPr.getIns(false);
                if(rPrIns==null) {
                    rPrIns = paraRPr.getMoveTo(false);
                }
        		rPrDel = paraRPr.getDel(false);
        		if(rPrDel==null) {
        		    rPrDel = paraRPr.getMoveFrom(false);
        		}
        		rPrChange = paraRPr.getRPrChange(false);
        	}
        	final CTPPrChange pPrChange = pPr.getPPrChange(false);
            if(rPrIns!=null||rPrDel!=null||rPrChange!=null||pPrChange!=null) {
            	final JSONObject changes = new JSONObject(2);
            	if(rPrIns!=null) {
            		changes.put(OCKey.INSERTED.value(), Utils.createJSONFromTrackInfo(getOperationDocument(), rPrIns));
                }
                if(rPrDel!=null) {
                	changes.put(OCKey.REMOVED.value(), Utils.createJSONFromTrackInfo(getOperationDocument(), rPrDel));
                }
                if(rPrChange!=null||pPrChange!=null) {
                	JSONObject modified = null;
                	JSONObject modifiedAttrs = new JSONObject(3);
                	if(rPrChange!=null) {
                		modified = Utils.createJSONFromTrackInfo(getOperationDocument(), rPrChange);
	                	final CTParaRPrOriginal rPrOriginal  = rPrChange.getRPr(false);
                    	if(rPrOriginal!=null) {
	                    	final JSONObject changedJsonCharacterAttributes = Character.createCharacterProperties(operationDocument.getThemeFonts(), rPrOriginal);
	                    	if(changedJsonCharacterAttributes!=null) {
	                    		modifiedAttrs.put(OCKey.CHARACTER.value(), changedJsonCharacterAttributes);
	                    	}
                    	}
                	}
                	if(pPrChange!=null) {
 	                	if(modified==null) {
 	                		modified = Utils.createJSONFromTrackInfo(getOperationDocument(), pPrChange);
 	                	}
 	                	final PPrBase pPrOriginal = pPrChange.getPPr(false);
 	                	if(pPrOriginal!=null) {
	 	                	final PStyle originalStyle = pPrOriginal.getPStyle(false);
		 	   	            if(originalStyle!=null) {
		 		                modifiedAttrs.put(OCKey.STYLE_ID.value(), originalStyle.getVal());
		 		            }
 	                	}
 	                	final JSONObject changedJsonParagraphAttributes = Paragraph.createParagraphProperties(pPrOriginal);
 	                	if(changedJsonParagraphAttributes!=null) {
 	                		modifiedAttrs.put(OCKey.PARAGRAPH.value(), changedJsonParagraphAttributes);
 	                	}
                	}

                	if(!modifiedAttrs.isEmpty()) {
                		modified.put(OCKey.ATTRS.value(), modifiedAttrs);
                	}
                	if(!modified.isEmpty()) {
                		changes.put(OCKey.MODIFIED.value(), modified);
                	}
                }
                if(!changes.isEmpty()) {
                	attrs.put(OCKey.CHANGES.value(), changes);
                }
            }
        }
        return attrs;
	}

	@Override
	public void insertText(int textPosition, String text, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(text.length()>0) {
        	final P paragraph = (P)getObject();

            IText t = null;
            Component childComponent = getNextChildComponent(null, null);
            Component cRet = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    t = (IText)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getValue());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setValue(s.toString());
                    cRet = childComponent;
                }
                else {

                	t = Context.getWmlObjectFactory().createText();
                    t.setValue(text);
                	if(childComponent instanceof TextRun_Base) {

                		// jackpot, we can insert the text into a existing run
                    	final R oldRun = ((TextRun_Base)childComponent).getTextRun();
                        t.setParent(oldRun);
                        oldRun.getContent().addNode(childComponent.getNode(), new IndexedNode<Object>(t), textPosition==0);
                        cRet = childComponent;
                	}
                	else {

                		// childComponent will be CommentRangeStartComponent, CommentRangeEndComponent or FldSimpleComponent
                        final R newRun = Context.getWmlObjectFactory().createR();
                        t.setParent(newRun);
                        newRun.getContent().add(t);

                        final IndexedNode<Object> parentNode = childComponent.getParentContext().getNode();
                		final ContentAccessor parentContent = (ContentAccessor)parentNode.getData();
                		newRun.setParent(parentNode.getData());
                		((IndexedNodeList<Object>)parentContent.getContent()).addNode(childComponent.getNode(), new IndexedNode<Object>(newRun), textPosition==0);
                	}
                    if(textPosition>0) {
                        cRet = childComponent.getNextComponent();
                    }
                    else {
                        cRet = getNextChildComponent(null, null);
                    }
                }
            }
            else {

            	// the paragraph is empty, we have to create R and its text
                final R newRun = Context.getWmlObjectFactory().createR();
                newRun.setParent(getObject());
                paragraph.getContent().add(newRun);
                if(paragraph.getPPr(false)!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr(false).getRPr(false), operationDocument.getPackage());
                    if(rPr!=null) {
                        newRun.setRPr(rPr);
                    }
                }
                t = Context.getWmlObjectFactory().createText();
                t.setParent(newRun);
                t.setValue(text);
                newRun.getContent().add(t);
                cRet = getNextChildComponent(null, null);
            }
			if(t!=null) {
				TextComponent.preserveSpace(t);
			}
			if(cRet instanceof TextRun_Base&&((TextRun_Base)cRet).hasChangeTracking()) {
				attrs = TextUtils.forceRemoveChangeTrack(attrs);
			}
            if(attrs!=null) {
            	cRet.splitStart(textPosition, SplitMode.ATTRIBUTE);
            	cRet.splitEnd(textPosition+text.length()-1, SplitMode.ATTRIBUTE);
            	cRet.applyAttrsFromJSON(attrs);
            }
        }
	}

    @Override
    public void splitParagraph(int textPosition) {

    	final P paragraph = (P)getObject();

        RPr lastRPr = null;
    	Component component = getNextChildComponent(null, null);
    	while(component!=null&&component.getNextComponentNumber()<=textPosition) {
    		if(component instanceof TextRun_Base) {
    			lastRPr = ((TextRun_Base)component).getTextRun().getRPr();
    		}
    		component = component.getNextComponent();
    	}
        if(component!=null) {
        	component.splitStart(textPosition, SplitMode.ATTRIBUTE);
        }
        // creating our new paragraph
        final P destParagraph = Context.getWmlObjectFactory().createP();
        destParagraph.setParent(paragraph.getParent());
        final IndexedNode<Object> destParagraphNode = new IndexedNode<Object>(destParagraph);
        Utils.getContent(paragraph.getParent()).addNode(getNode(), destParagraphNode, false);
        final PPr sourceParagraphProperties = paragraph.getPPr(false);
        if(sourceParagraphProperties!=null){
            final PPr destParagraphProperties = XmlUtils.deepCopy(sourceParagraphProperties, getOperationDocument().getPackage());
            // no inheritance for pageBreakBefore attribute... this attribute is always false

            TextUtils.resetMarkupId(destParagraphProperties);

            destParagraphProperties.setPageBreakBefore(null);
            destParagraph.setPPr(destParagraphProperties);
        }

        // TODO: to be optimized.. the component is already the correct source component,
        // the iteration is only needed to get the previous RPr
        while(component!=null) {
            if(textPosition>=component.getComponentNumber()&&textPosition<component.getNextComponentNumber()) {
                boolean skipSdt = false;
                ComponentContext childContext = component;
                ComponentContext parentContext = component.getParentContext();
                while(!(childContext instanceof ParagraphComponent)) {
                    if(parentContext instanceof SdtParagraphContext) {
                        if(!childContext.getNode().isFirst()) {
                            // moving end-part of the sdt into the next paragraph
                            final IndexedNodeList<Object> sourceContent = (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent();
                            sourceContent.moveNodes(childContext.getNode(), -1, destParagraph.getContent(), destParagraph.getContent().getLastNode(), false, destParagraph);
                            skipSdt = true;
                        }
                    }
                    else if(parentContext instanceof ParagraphComponent) {
                        final IndexedNodeList<Object> parentContent = (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent();
                        IndexedNode<Object> sourceRefNode = childContext.getNode();
                        if(skipSdt) {
                            sourceRefNode = parentContent.getNextNode(sourceRefNode);
                        }
                        if(sourceRefNode!=null) {
                            parentContent.moveNodes(sourceRefNode, -1, destParagraph.getContent(), destParagraph.getContent().getLastNode(), false, destParagraph);
                        }
                    }
                    childContext = parentContext;
                    parentContext = childContext.getParentContext();
                }
                break;
            } else if (component instanceof TextRun_Base) {
                lastRPr = ((TextRun_Base)component).getTextRun().getRPr();
            }
            component = component.getNextComponent();
        }

        // taking care of paragraph attributes
        if(lastRPr!=null) {

            // if available, we have to get the character attributes from the last textrun
            final PPr paraProps = destParagraph.getPPr(true);

            // 30793 ... only setting character attributes if no listStyle is used at the paragraph
            if(paraProps.getNumPr(false)==null||paraProps.getNumPr(false).getNumId()==null) {
                ParaRPr paraRPr = TextUtils.cloneRPrToParaRPr(lastRPr, getOperationDocument().getPackage());
                if(paraRPr!=null) {
                    paraProps.setRPr(paraRPr);
                }
            }
        }
    }

    @Override
    public void mergeParagraph() {
    	final Component nextParagraphComponent = getNextComponent();
    	if(nextParagraphComponent instanceof ParagraphComponent) {
        	final P paragraph = (P)getObject();
        	final IndexedNodeList<Object> sourceContent = ((P)nextParagraphComponent.getObject()).getContent();
            final IndexedNodeList<Object> destContent = paragraph.getContent();
            if(sourceContent.size()>0) {
                sourceContent.moveNodes(destContent, paragraph);
            }
            IndexedNode<Object> parentContextNode = nextParagraphComponent.getParentContext().getNode();
        	Utils.getContent(parentContextNode.getData()).removeNodes(nextParagraphComponent.getNode(), null);
        	if(parentContextNode.getData() instanceof SdtBlock) {
        	    final ComponentContext sdtRootContext = nextParagraphComponent.getParentContext();
        	    final ComponentContext parentOfSdtRootContext = sdtRootContext.getParentContext();
                if(((ContentAccessor)sdtRootContext.getNode().getData()).getContent().isEmpty()) {
                    Utils.getContent(parentOfSdtRootContext.getNode().getData()).removeNodes(sdtRootContext.getNode(), null);
                }
        	}
    	}
    }
}
