/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ods.dom;

import java.util.Iterator;
import java.util.Map.Entry;
import org.apache.xerces.dom.ElementNSImpl;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.pkg.NamespaceName;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.IContentDom;
import com.openexchange.office.filter.odf.INodeAccessor;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawObject;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.styles.AutomaticStyles;
import com.openexchange.office.filter.odf.styles.FontFaceDecls;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.ods.dom.chart.ChartContent;
import com.openexchange.office.filter.ods.dom.components.RootComponent;

/**
 * The DOM representation of the ODS content.xml file of an ODF document.
 */
@SuppressWarnings("serial")
public class SpreadsheetContent extends OdfFileDom implements INodeAccessor, IContentDom {

	private int maxColumnCount;
	private DLList<Sheet> sheets;
	private CalculationSettings calculationSettings;
	private NamedExpressions namedExpressions;
	private DatabaseRanges databaseRanges;
	private ContentValidations contentValidations;
	private ElementNSImpl officeSpreadsheetElement;

	/**
	 * Creates the DOM representation of an XML file of an Odf document.
	 *
	 * @param odfDocument   the document the XML files belongs to
	 * @param packagePath   the internal package path to the XML file
	 */
	public SpreadsheetContent(OdfSpreadsheetDocument odfDocument, String packagePath) throws SAXException {
		super(odfDocument, packagePath);
	}

	@Override
	protected void initialize() throws SAXException {
		for (NamespaceName name : OdfDocumentNamespace.values()) {
			mUriByPrefix.put(name.getPrefix(), name.getUri());
			mPrefixByUri.put(name.getUri(), name.getPrefix());
		}
		final StyleManager styleManager = getDocument().getStyleManager();
		styleManager.setFontFaceDecls(new FontFaceDecls(this), true);
		styleManager.setAutomaticStyles(new AutomaticStyles(this), true);

		final XMLReader xmlReader = mPackage.getXMLReader();
		super.initialize(new SpreadsheetContentHandler(this, xmlReader), xmlReader);

        final Node root = getRootElement();
        root.insertBefore(styleManager.getAutomaticStyles(true), root.getFirstChild());
        root.insertBefore(styleManager.getFontFaceDecls(true), root.getFirstChild());
	}

	/**
	 * Retrieves the Odf Document
	 * 
	 * @return The <code>OdfDocument</code>
	 */
	@Override
	public OdfSpreadsheetDocument getDocument() {
		return (OdfSpreadsheetDocument)mPackageDocument;
	}

    @Override
    public RootComponent getRootComponent(String target) {
        INodeAccessor targetAccessor = this;
        if(target!=null&&!target.isEmpty()) {
            targetAccessor = getDocument().getTargetNodes().get(target);
        }
        return new RootComponent(targetAccessor);
    }

    @Override
    public  DLList<Object> getContent() {
        return (DLList)getSheets();
    }

	/* returns a live list to the sheets */
	public DLList<Sheet> getSheets() {
        if(sheets==null) {
            sheets = new DLList<Sheet>();
        }
		return sheets;
	}

	// returns the sheetindex for the given name or -1 if this sheet was not found
	public int getSheetIndex(String sheetName) {
		if(sheetName!=null) {
			sheetName = sheetName.replace("'", "");
			for(int i=0; i<getSheets().size(); i++) {
				final String name = getSheets().get(i).getSheetName();
				if(sheetName.equals(name)) {
					return i;
				}
			}
		}
		return -1;
	}

	public void setSpreadsheet(ElementNSImpl element) {
		officeSpreadsheetElement = element;
	}

	public void setCalculationSettings(CalculationSettings calculationSettings) {
	    this.calculationSettings = calculationSettings;
	}

	public CalculationSettings getCalculationSettings() {
	    return calculationSettings;
	}

	public void setNamedExpressions(NamedExpressions namedExpressions) {
		this.namedExpressions = namedExpressions;
	}

	public NamedExpressions getNamedExpressions(boolean forceCreate) {
		if(namedExpressions==null&&forceCreate) {
			namedExpressions = new NamedExpressions(this);

			// try to insert this node behind the last sheet---
			if(!sheets.isEmpty()) {
				final Node refNode = getSheets().get(sheets.size()-1).getNextSibling();
				getSheets().get(0).getParentNode().insertBefore(namedExpressions, refNode);
			}
			else {
				officeSpreadsheetElement.appendChild(namedExpressions);
			}
		}
		return namedExpressions;
	}

	public void setDatabaseRanges(DatabaseRanges databaseRanges) {
		this.databaseRanges = databaseRanges;
	}

	public DatabaseRanges getDatabaseRanges(boolean forceCreate) {
		if(databaseRanges==null&&forceCreate) {
			databaseRanges = new DatabaseRanges(this);

			if(namedExpressions!=null) {
				// inserting database ranges behind named expressions, otherwise the document won't be opened by Excel
				namedExpressions.getParentNode().insertBefore(databaseRanges, namedExpressions.getNextSibling());
			}
			else if(!sheets.isEmpty()) {
				final Node refNode = getSheets().get(sheets.size()-1).getNextSibling();
				getSheets().get(0).getParentNode().insertBefore(databaseRanges, refNode);
			}
			else {
				officeSpreadsheetElement.appendChild(databaseRanges);
			}
		}
		return databaseRanges;
	}

    public void setContentValidations(ContentValidations contentValidations) {
        this.contentValidations = contentValidations;
    }

	public ContentValidations getContentValidations(boolean forceCreate) {
		if(contentValidations==null&&forceCreate) {
			contentValidations = new ContentValidations(this);

			final CalculationSettings settingsNode = getCalculationSettings();
			if(settingsNode!=null) {
	            // office:content-validations needs to come behind calculation-settings, otherwise Excel won't load this document
			    settingsNode.getParentNode().insertBefore(contentValidations, settingsNode.getNextSibling());
			}
			else {
			    // if there are no calculation-settings, we will insert this as first child
			    officeSpreadsheetElement.insertBefore(contentValidations, officeSpreadsheetElement.getFirstChild());
			}
		}
		return contentValidations;
	}

	public int getMaxColumnCount() {
		return maxColumnCount==0?1024:maxColumnCount;
	}

	public void setMaxColumnCount(int value) {
		maxColumnCount = value;
	}

	public void insertSheet(int sheetIndex, String newName) {

		final Sheet newSheet = new Sheet(this, null);
		insertSheetNode(newSheet, sheetIndex);
		newSheet.setSheetName(newName);

		// update database ranges:
		if(databaseRanges!=null) {
			final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
			while(databaseRangeIter.hasNext()) {
				final Range range = databaseRangeIter.next().getValue().getRange();
				if(range.getSheetIndex()>=sheetIndex) {
					range.setSheetIndex(range.getSheetIndex()+1);
				}
			}
		}
	}

	private void insertSheetNode(Sheet sheet, int sheetIndex) {
		final boolean append = sheetIndex>(sheets.size()-1);
		if(append) {
			final Node refNode = getSheets().get(sheetIndex-1).getNextSibling();
			getSheets().get(0).getParentNode().insertBefore(sheet, refNode);
		}
		else {
			final Sheet refSheet = getSheets().get(sheetIndex);
			refSheet.getParentNode().insertBefore(sheet, refSheet);
		}
		getSheets().add(sheetIndex, sheet);
	}

	public void moveSheet(int sheetIndex, int to) {
		final Sheet sheet = getSheets().get(sheetIndex);
		sheet.getParentNode().removeChild(sheet);
		getSheets().remove(sheetIndex);
        insertSheetNode(sheet, to);

        // update database ranges:
		if(databaseRanges!=null) {
			final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
			while(databaseRangeIter.hasNext()) {
				final Range range = databaseRangeIter.next().getValue().getRange();
				if(range.getSheetIndex()==sheetIndex) {
					range.setSheetIndex(to);
					continue;
				}
				if(range.getSheetIndex()>sheetIndex) {
					range.setSheetIndex(range.getSheetIndex()-1);
				}
				if(range.getSheetIndex()>=to) {
					range.setSheetIndex(range.getSheetIndex()+1);
				}
			}
		}
	}

	public Sheet deleteSheet(int sheetIndex) {
		final Sheet sheet = getSheets().get(sheetIndex);
		final Drawings drawings = sheet.getDrawings();
		if(drawings!=null) {
		    for(int i=0; i<drawings.getCount(); i++) {
		        final Drawing drawing = drawings.getDrawing(i);
		        if(drawing.getType()==DrawingType.CHART) {
		            final ChartContent chart = ((DrawObject)((DrawFrame)drawing.getShape()).getDrawing()).getChart();
		            if (chart!=null) {
		                Drawings.deleteChartPart(getDocument().getPackage(), chart);
		            }
		        }
		    }
		}
		sheet.getParentNode().removeChild(sheet);
		getSheets().remove(sheetIndex);

		// update database ranges:
		if(databaseRanges!=null) {
			final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
			while(databaseRangeIter.hasNext()) {
				final Range range = databaseRangeIter.next().getValue().getRange();
				if(range.getSheetIndex()==sheetIndex) {
					databaseRangeIter.remove();
				}
				else if(range.getSheetIndex()>sheetIndex) {
					range.setSheetIndex(range.getSheetIndex()-1);
				}
			}
		}
		return sheet;
	}
}
