/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.hazelcast.osgi;

import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.osgi.framework.BundleException;
import org.slf4j.Logger;

import com.hazelcast.config.Config;
import com.hazelcast.config.MapConfig;
import com.hazelcast.core.HazelcastInstance;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.hazelcast.serialization.CustomPortableFactory;
import com.openexchange.management.ManagementService;
import com.openexchange.office.hazelcast.access.HazelcastAccess;
import com.openexchange.office.hazelcast.cleanup.CleanupMembershipListener;
import com.openexchange.office.hazelcast.doc.DocSaveStateDirectory;
import com.openexchange.office.hazelcast.doc.DocumentDirectory;
import com.openexchange.office.hazelcast.doc.DocumentResourcesDirectory;
import com.openexchange.office.hazelcast.doc.HazelcastDocSaveStateDirectory;
import com.openexchange.office.hazelcast.doc.HazelcastDocumentCleanupLock;
import com.openexchange.office.hazelcast.doc.HazelcastDocumentDirectory;
import com.openexchange.office.hazelcast.doc.HazelcastDocumentResourcesDirectory;
import com.openexchange.office.hazelcast.management.ManagementHouseKeeper;
import com.openexchange.office.hazelcast.serialization.document.PortableCleanupStatusFactory;
import com.openexchange.office.hazelcast.serialization.document.PortableDocSaveStateFactory;
import com.openexchange.office.hazelcast.serialization.document.PortableDocumentFactory;
import com.openexchange.office.hazelcast.serialization.document.PortableDocumentResourcesFactory;
import com.openexchange.office.hazelcast.serialization.document.PortableIDFactory;
import com.openexchange.office.hazelcast.serialization.document.PortableMemberPredicateFactory;
import com.openexchange.office.hazelcast.serialization.document.PortableResourceIDFactory;
import com.openexchange.office.hazelcast.serialization.document.PortableRestoreIDFactory;
import com.openexchange.office.tools.logging.LogFactory;
import com.openexchange.office.tools.osgi.ActivatorBase;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.threadpool.ThreadPoolService;
import com.openexchange.timer.TimerService;

//=============================================================================
public class HazelcastOfficeActivator extends ActivatorBase
{
	//-------------------------------------------------------------------------   
    private static final String SERVICE_NAME = "com.openexchange.office.hazelcast";

    //-------------------------------------------------------------------------   
    private static final Logger LOG = LogFactory.getSlfLog(HazelcastOfficeActivator.class);

	//-------------------------------------------------------------------------   
    public HazelcastOfficeActivator ()
        throws Exception
    {
    	super (SERVICE_NAME);
    }

    //-------------------------------------------------------------------------   
    @Override
    protected void listMandatoryServiceImports(final List< Class< ? >> lServices)
    	throws Exception
    {
        lServices.add(HazelcastInstance    .class);
		lServices.add(ManagementService    .class);
		lServices.add(TimerService         .class);
		lServices.add(ThreadPoolService    .class);
		lServices.add(ManagedFileManagement.class);
    }

    //-------------------------------------------------------------------------   
    @Override
    protected void listOptionalServiceImports(final List< Class< ? >> lServices)
    	throws Exception
    {
    	// no optional services
    }

    //-------------------------------------------------------------------------   
    @Override
    protected void activate()
    	throws Exception
    {
    	final ManagementHouseKeeper managementHouseKeeper = ManagementHouseKeeper.getInstance();
    	managementHouseKeeper.initialize(ServiceLookupRegistry.get());

    	final HazelcastInstance hazelcastInstance = getService(HazelcastInstance.class);
    	HazelcastAccess.setHazelcastInstance(hazelcastInstance);
    	final Config config = hazelcastInstance.getConfig();

    	final String docStateResource_map = discoverMapName(config, "officeDocumentDirectory-");
    	if(StringUtils.isEmpty(docStateResource_map)) {
    	    final String msg = "Distributed documents persistent state directory map couldn't be found in hazelcast configuration";
    	    throw new IllegalStateException(msg, new BundleException(msg, BundleException.ACTIVATOR_ERROR));
    	}
    	LOG.debug("Registered officeDocumentDirectory to Hazelcast");

    	final String docSaveStateResource_map = discoverMapName(config, "officeDocSaveStateDirectory-");
    	if(StringUtils.isEmpty(docSaveStateResource_map)) {
    	    final String msg = "Distributed document save state directory map couldn't be found in hazelcast configuration";
    	    throw new IllegalStateException(msg, new BundleException(msg, BundleException.ACTIVATOR_ERROR));
    	}
    	LOG.debug("Registered officeDocSaveStateDirectory to Hazelcast");

    	final String docResources_map = discoverMapName(config, "officeDocumentResources-");
    	if(StringUtils.isEmpty(docResources_map)) {
    	    final String msg = "Distributed document resources directory map couldn't be found in hazelcast configuration";
    	    throw new IllegalStateException(msg, new BundleException(msg, BundleException.ACTIVATOR_ERROR));
    	}
    	LOG.debug("Registered officeDocumentResources to Hazelcast");

		String lock_map = discoverMapName(config, "officeCleanupLock-");
		
		docStateDir = new HazelcastDocumentDirectory(docStateResource_map);
		docSaveStateDir = new HazelcastDocSaveStateDirectory(docSaveStateResource_map);
		docResourcesDir = new HazelcastDocumentResourcesDirectory(docResources_map);
		HazelcastDocumentCleanupLock lockDir = new HazelcastDocumentCleanupLock(lock_map);
		
		managementHouseKeeper.addManagementObject(docSaveStateDir.getManagementObject());
		registerService(DocSaveStateDirectory.class, docSaveStateDir, null);
		
		managementHouseKeeper.addManagementObject(docStateDir.getManagementObject());
		registerService(DocumentDirectory.class, docStateDir, null);
		
		managementHouseKeeper.addManagementObject(docResourcesDir.getManagementObject());
		registerService(DocumentResourcesDirectory.class, docResourcesDir, null);
		
		managementHouseKeeper.addManagementObject(lockDir.getManagementObject());
		
		registerService(CustomPortableFactory.class, new PortableIDFactory());
		registerService(CustomPortableFactory.class, new PortableDocumentFactory());
		registerService(CustomPortableFactory.class, new PortableDocSaveStateFactory());
		registerService(CustomPortableFactory.class, new PortableDocumentResourcesFactory());
		registerService(CustomPortableFactory.class, new PortableResourceIDFactory());
		registerService(CustomPortableFactory.class, new PortableRestoreIDFactory());
		registerService(CustomPortableFactory.class, new PortableCleanupStatusFactory());
		registerService(CustomPortableFactory.class, new PortableMemberPredicateFactory());

		// add cleanup listener to remove entries controlled by a cluster member that is shutdown
		final CleanupMembershipListener cleanupListener = new CleanupMembershipListener(lock_map, docStateDir, docResourcesDir);
		hazelcastInstance.getCluster().addMembershipListener(cleanupListener);

		managementHouseKeeper.exposeManagementObjects();
    }

    //-------------------------------------------------------------------------
    @Override
    protected void deactivate()
    	throws Exception
    {
        ManagementHouseKeeper.getInstance().cleanup();
    }

    //-------------------------------------------------------------------------
	/**
	 * Discovers map names in the supplied hazelcast configuration based on the map prefix.
	 *
	 * @param config The config object
	 * @return The prefix of the map name
	 */
	private String discoverMapName(Config config, String mapPrefix){
	    Map<String, MapConfig> mapConfigs = config.getMapConfigs();
	    if (null != mapConfigs && 0 < mapConfigs.size()) {
	        for (String mapName : mapConfigs.keySet()) {
	            if (mapName.startsWith(mapPrefix)) {
	                return mapName;
	            }
	        }
	    }
	    return null;
	}

	//-------------------------------------------------------------------------   
	private volatile HazelcastDocumentDirectory docStateDir = null;
	
	//-------------------------------------------------------------------------   
	private volatile HazelcastDocSaveStateDirectory docSaveStateDir = null;
	
	//-------------------------------------------------------------------------   
	private volatile HazelcastDocumentResourcesDirectory docResourcesDir = null;
}
