/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.core.chart;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.api.OCValue;

public class Chart extends AttributeSet {

    private final List<ChartSeries> series = new ArrayList<>();
    private final AttributeSet legend = new AttributeSet();
    private final Map<Integer, ChartAxis> axes = new HashMap<>();

    private final AttributeSet mainTitle = new AttributeSet();
    private boolean hasErrorBars = false;

    public Chart() {

    }

    public void createAttributes(JSONObject target) {
        JSONObject attrs = getAttributes();
        mergeAttributes(target, attrs);
    }

    public void createJSONOperations(List<Integer> position, JSONArray operationQueue) throws JSONException {

        if (hasErrorBars) {
            JSONObject chartOperation = operationQueue.getJSONObject(operationQueue.length() -1);
            JSONObject attrs;
            if (!chartOperation.has(OCKey.ATTRS.value())) {
                chartOperation.put(OCKey.ATTRS.value(), new JSONObject());
            }
            attrs =  chartOperation.getJSONObject(OCKey.ATTRS.value());

            if (!attrs.has(OCKey.CHART.value())) {
                attrs.put(OCKey.CHART.value(), new JSONObject());
            }
            
            attrs.getJSONObject(OCKey.CHART.value()).putSafe(OCKey.ERROR_BAR.value(), true);
        }
        
        int seriesIndex = 0;
        for (ChartSeries serie : series) {
            serie.createInsertOperation(position, operationQueue, seriesIndex);
            seriesIndex++;
        }

        if (!mainTitle.getAttributes().isEmpty()) {
            JSONObject op = new JSONObject(3);
            op.put(OCKey.START.value(), position);
            op.putSafe(OCKey.ATTRS.value(), mainTitle.getAttributes());
            op.putSafe(OCKey.NAME.value(), OCValue.SET_CHART_TITLE_ATTRIBUTES.value());
            operationQueue.put(op);
        }

        if (!legend.getAttributes().isEmpty()) {
            JSONObject op = new JSONObject(3);
            op.put(OCKey.START.value(), position);
            op.putSafe(OCKey.ATTRS.value(), legend.getAttributes());
            op.putSafe(OCKey.NAME.value(), OCValue.SET_CHART_LEGEND_ATTRIBUTES.value());
            operationQueue.put(op);
        }

        for (ChartAxis axis : axes.values()) {
            axis.createInsertOperation(position, operationQueue);
        }
    }

    public void setChartStyleId(int chartStyleId) {
        JSONObject chart = getAttributes().optJSONObject(OCKey.CHART.value());
        if (chart == null) {
            chart = new JSONObject();
            getAttributes().putSafe(OCKey.CHART.value(), chart);
        }
        chart.putSafe(OCKey.CHART_STYLE_ID.value(), chartStyleId);
    }

    public ChartSeries addDataSeries(ChartSeries serie) {
        series.add(serie);
        return serie;
    }

    public ChartSeries insertDataSeries(int seriesIndex, ChartSeries serie) {
        series.add(seriesIndex, serie);
        return serie;
    }

    public ChartSeries setDataSeriesAttributes(int seriesIndex, JSONObject attrs) {
        ChartSeries serie = series.get(seriesIndex);
        serie.setAttributes(attrs);
        return serie;
    }

    public ChartSeries deleteDataSeries(int seriesIndex) {
        return series.remove(seriesIndex);
    }

    public AttributeSet getLegend() {
        return legend;
    }

    public void setAxis(ChartAxis axis) {
        axes.put(axis.getAxis(), axis);
    }

    public ChartAxis getAxis(Integer axisId) {
        ChartAxis res = axes.get(axisId);
        if (res == null) {
            res = new ChartAxis();
            res.setAxis(axisId);
            setAxis(res);
        }
        return res;
    }
    
    public void removeAxes() {
        axes.clear();
        
    }
    
    public Collection<ChartAxis>getAxes () {
        return axes.values();
    }
    
    public boolean containsAxis(Integer axisId) {
        return axes.containsKey(axisId);
    }

    public AttributeSet getTitle(Integer axisId) {
        if (axisId == null) {
            return mainTitle;
        }
        return getAxis(axisId).getTitle();
    }

    public List<ChartSeries> getSeries() {
        return Collections.unmodifiableList(series);
    }

    public boolean isScatter() {
        if (series.isEmpty()) {
            return false;
        }
        return series.get(0).isScatter();
    }

    public boolean isBubble() {
        if (series.isEmpty()) {
            return false;
        }
        return series.get(0).isBubble();
    }

    public boolean isPieOrDonut() {
        if (series.isEmpty()) {
            return false;
        }
        return series.get(0).isPie() || series.get(0).isDonut();
    }
    
    public boolean isBar() {
        if (series.isEmpty()) {
            return false;
        }
        return series.get(0).isBar();
    }
    

    public void setHasErrorBar() {
        hasErrorBars = true;
    }
    
    public boolean hasErrorBars() {
        return hasErrorBars;
    }
}
