/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator'
], function (Utils, Iterator) {

    'use strict';

    // static class Iterator ==================================================

    describe('Toolkit class Iterator', function () {

        it('should exist', function () {
            expect(Iterator).to.be.a('function');
        });

        // constants ----------------------------------------------------------

        describe('constant "EMPTY"', function () {
            it('should exist', function () {
                expect(Iterator).to.have.a.property('EMPTY').that.is.an.instanceof(Iterator);
                expect(Iterator.EMPTY.next()).to.deep.equal({ done: true });
            });
        });

        // class SingleIterator -----------------------------------------------

        describe('sub class "SingleIterator"', function () {
            var SingleIterator = Iterator.SingleIterator;
            it('should exist', function () {
                expect(SingleIterator).to.be.a('function');
                expect(SingleIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should visit a single result', function () {
                var it = new SingleIterator({ value: 42, other: 'abc' });
                expect(it.next()).to.deep.equal({ done: false, value: 42, other: 'abc' });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        // class IndexIterator ------------------------------------------------

        describe('sub class "IndexIterator"', function () {
            var IndexIterator = Iterator.IndexIterator;
            it('should exist', function () {
                expect(IndexIterator).to.be.a('function');
                expect(IndexIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should visit all indexes', function () {
                var it = new IndexIterator(3);
                expect(it.next()).to.deep.equal({ done: false, value: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes in reversed order', function () {
                var it = new IndexIterator(3, { reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 0 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes with an offset', function () {
                var it = new IndexIterator(3, { offset: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes in reversed order with an offset', function () {
                var it = new IndexIterator(3, { reverse: true, offset: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        // class IntervalIterator ---------------------------------------------

        describe('sub clas "IntervalIterator"', function () {
            var IntervalIterator = Iterator.IntervalIterator;
            it('should exist', function () {
                expect(IntervalIterator).to.be.a('function');
                expect(IntervalIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should visit all indexes', function () {
                var it = new IntervalIterator(3, 5);
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: false, value: 5 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes in reversed order', function () {
                var it = new IntervalIterator(3, 5, { reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: 5 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        // class ArrayIterator ------------------------------------------------

        describe('sub class "ArrayIterator"', function () {
            var ArrayIterator = Iterator.ArrayIterator;
            it('should exist', function () {
                expect(ArrayIterator).to.be.a('function');
                expect(ArrayIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should visit all elements of an array', function () {
                var it = new ArrayIterator([1, 2, 3]);
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array from a specific start index', function () {
                var it = new ArrayIterator([1, 2, 3], { begin: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array to a specific end index', function () {
                var it = new ArrayIterator([1, 2, 3], { end: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in a specific interval', function () {
                var it = new ArrayIterator([1, 2, 3], { begin: 1, end: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order', function () {
                var it = new ArrayIterator([1, 2, 3], { reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order from a specific start index', function () {
                var it = new ArrayIterator([1, 2, 3], { reverse: true, begin: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order to a specific end index', function () {
                var it = new ArrayIterator([1, 2, 3], { reverse: true, end: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order in a specific interval', function () {
                var it = new ArrayIterator([1, 2, 3], { reverse: true, begin: 1, end: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array-like object', function () {
                var it = new ArrayIterator('abc');
                expect(it.next()).to.deep.equal({ done: false, value: 'a', index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 'b', index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 'c', index: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        // class ObjectIterator -----------------------------------------------

        describe('sub class "ObjectIterator"', function () {
            var ObjectIterator = Iterator.ObjectIterator;
            it('should exist', function () {
                expect(ObjectIterator).to.be.a('function');
                expect(ObjectIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should visit all properties of an object', function () {
                var it = new ObjectIterator({ a: 42, b: true, c: 'abc' });
                var results = [it.next(), it.next(), it.next()];
                results = _.sortBy(results, function (result) { return result.key; });
                expect(results[0]).to.deep.equal({ done: false, value: 42, key: 'a' });
                expect(results[1]).to.deep.equal({ done: false, value: true, key: 'b' });
                expect(results[2]).to.deep.equal({ done: false, value: 'abc', key: 'c' });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        // class GeneratorIterator --------------------------------------------

        describe('sub class "GeneratorIterator"', function () {
            var GeneratorIterator = Iterator.GeneratorIterator;
            it('should exist', function () {
                expect(GeneratorIterator).to.be.a('function');
                expect(GeneratorIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create an iterator', function () {
                var v = 0;
                function generate() { v += 1; return (v % 2) ? null : (v === 8) ? { done: true } : { value: v }; }
                var spy = sinon.spy(generate), context = {};
                var genIt = new GeneratorIterator(spy, context);
                expect(genIt.next()).to.deep.equal({ done: false, value: 2 });
                expect(genIt.next()).to.deep.equal({ done: false, value: 4 });
                expect(genIt.next()).to.deep.equal({ done: false, value: 6 });
                expect(genIt.next()).to.deep.equal({ done: true });
                sinon.assert.callCount(spy, 8);
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        // class TransformIterator --------------------------------------------

        describe('sub class "TransformIterator"', function () {
            var TransformIterator = Iterator.TransformIterator;
            it('should exist', function () {
                expect(TransformIterator).to.be.a('function');
                expect(TransformIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create a transformation iterator', function () {
                function sqrt(v) { return (v < 0) ? null : (v === 0) ? { done: true } : { value: Math.sqrt(v) }; }
                var spy = sinon.spy(sqrt), context = {};
                var transformIt = new TransformIterator([36, -4, 4, -36, 0, 1], spy, context);
                expect(transformIt.next()).to.deep.equal({ done: false, value: 6 });
                expect(transformIt.next()).to.deep.equal({ done: false, value: 2 });
                expect(transformIt.next()).to.deep.equal({ done: true });
                sinon.assert.callCount(spy, 5);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 36, { done: false, value: 36, index: 0 });
                sinon.assert.calledWithExactly(spy.getCall(1), -4, { done: false, value: -4, index: 1 });
                sinon.assert.calledWithExactly(spy.getCall(2), 4, { done: false, value: 4, index: 2 });
            });
            it('should create a filter iterator returning an object property', function () {
                var arrayIt = new Iterator.ArrayIterator([{ a: 1 }, { a: 0 }, { a: 'abc' }]);
                var transformIt = new TransformIterator(arrayIt, 'a');
                expect(transformIt.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(transformIt.next()).to.deep.equal({ done: false, value: 0, index: 1 });
                expect(transformIt.next()).to.deep.equal({ done: false, value: 'abc', index: 2 });
                expect(transformIt.next()).to.deep.equal({ done: true });
            });
        });

        // class FilterIterator -----------------------------------------------

        describe('sub class "FilterIterator"', function () {
            var FilterIterator = Iterator.FilterIterator;
            it('should exist', function () {
                expect(FilterIterator).to.be.a('function');
                expect(FilterIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create a filter iterator', function () {
                function even(v) { return (v % 2) === 0; }
                var spy = sinon.spy(even), context = {};
                var filterIt = new FilterIterator([6, 3, 2, 7, 0, 4, 1, 5], spy, context);
                expect(filterIt.next()).to.deep.equal({ done: false, value: 6, index: 0 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: 2, index: 2 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: 0, index: 4 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: 4, index: 5 });
                expect(filterIt.next()).to.deep.equal({ done: true });
                sinon.assert.callCount(spy, 8);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 6, { done: false, value: 6, index: 0 });
                sinon.assert.calledWithExactly(spy.getCall(1), 3, { done: false, value: 3, index: 1 });
                sinon.assert.calledWithExactly(spy.getCall(2), 2, { done: false, value: 2, index: 2 });
            });
            it('should create a filter iterator checking an object property', function () {
                var filterIt = new FilterIterator([{ a: 1 }, { a: 0 }, {}, { a: 2 }], 'a');
                expect(filterIt.next()).to.deep.equal({ done: false, value: { a: 1 }, index: 0 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: { a: 2 }, index: 3 });
                expect(filterIt.next()).to.deep.equal({ done: true });
            });
        });

        // class ReduceIterator -----------------------------------------------

        describe('sub class "ReduceIterator"', function () {
            var ReduceIterator = Iterator.ReduceIterator;
            it('should exist', function () {
                expect(ReduceIterator).to.be.a('function');
                expect(ReduceIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create a reducing iterator', function () {
                function reduce(r1, r2) { return (r1.value === r2.value) ? r1 : null; }
                var spy = sinon.spy(reduce), context = {};
                var reduceIt = new ReduceIterator([1, 2, 3, 3, 3, 2, 1, 1], spy, context);
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 2, index: 5 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 1, index: 6 });
                expect(reduceIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        // class NestedIterator -----------------------------------------------

        describe('sub class "NestedIterator"', function () {
            var NestedIterator = Iterator.NestedIterator;
            it('should exist', function () {
                expect(NestedIterator).to.be.a('function');
                expect(NestedIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create a nested iterator', function () {
                function createIndexIterator(value) {
                    return (value > 0) ? new Iterator.IndexIterator(value) : null;
                }
                function combineResults(result1, result2) {
                    var v1 = result1.value, v2 = result2.value;
                    return (v1 === 42) ? null : (v1 > 9) ? { done: true } : { value: v1 + ',' + v2 };
                }
                var spy1 = sinon.spy(createIndexIterator), spy2 = sinon.spy(combineResults), context = {};
                var nestedIt = new NestedIterator([3, 1, 42, 0, 2, 999, 4], spy1, spy2, context);
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '3,0' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '3,1' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '3,2' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '1,0' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '2,0' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '2,1' });
                expect(nestedIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy1, context);
                sinon.assert.calledWithExactly(spy1.getCall(0), 3, { done: false, value: 3, index: 0 });
                sinon.assert.calledWithExactly(spy1.getCall(1), 1, { done: false, value: 1, index: 1 });
                sinon.assert.alwaysCalledOn(spy2, context);
            });
        });

        // class SerialIterator -----------------------------------------------

        describe('sub class "SerialIterator"', function () {
            var SerialIterator = Iterator.SerialIterator;
            it('should exist', function () {
                expect(SerialIterator).to.be.a('function');
                expect(SerialIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create a serial iterator', function () {
                var it1 = new Iterator.ArrayIterator([2, 3]);
                var it2 = Iterator.EMPTY;
                var it3 = [4, 5];
                var serialIt = new SerialIterator(it1, it2, null, it3);
                expect(serialIt.next()).to.deep.equal({ done: false, value: 2, index: 0 });
                expect(serialIt.next()).to.deep.equal({ done: false, value: 3, index: 1 });
                expect(serialIt.next()).to.deep.equal({ done: false, value: 4, index: 0 });
                expect(serialIt.next()).to.deep.equal({ done: false, value: 5, index: 1 });
                expect(serialIt.next()).to.deep.equal({ done: true });
            });
            it('should create an empty iterator', function () {
                var serialIt = new SerialIterator(null);
                expect(serialIt.next()).to.deep.equal({ done: true });
            });
        });

        // class OrderedIterator ----------------------------------------------

        describe('sub class "OrderedIterator"', function () {
            var OrderedIterator = Iterator.OrderedIterator;
            it('should exist', function () {
                expect(OrderedIterator).to.be.a('function');
                expect(OrderedIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create an ordered iterator', function () {
                var arr1 = [2, 3, 4, 5, 6, 7, 8];
                var arr2 = [0, 3, 6];
                var arr3 = [-1, -9];
                var it1 = new Iterator.ArrayIterator(arr1);
                var spy1 = sinon.spy(_.identity);
                var spy2 = sinon.spy(_.identity);
                var spy3 = sinon.spy(function (n) { return -n; });
                var context = {};
                var orderIt = new OrderedIterator([it1, arr2, arr3], [spy1, spy2, spy3], context);
                expect(orderIt.next()).to.deep.equal({ done: false, value: 0, index: 0, offset: 0, srcIndex: 1 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: -1, index: 0, offset: 1, srcIndex: 2 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 2, index: 0, offset: 2, srcIndex: 0 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 3, index: 1, offset: 3, srcIndex: 0 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 3, index: 1, offset: 3, srcIndex: 1 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 4, index: 2, offset: 4, srcIndex: 0 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 5, index: 3, offset: 5, srcIndex: 0 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 6, index: 4, offset: 6, srcIndex: 0 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 6, index: 2, offset: 6, srcIndex: 1 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 7, index: 5, offset: 7, srcIndex: 0 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: 8, index: 6, offset: 8, srcIndex: 0 });
                expect(orderIt.next()).to.deep.equal({ done: false, value: -9, index: 1, offset: 9, srcIndex: 2 });
                expect(orderIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy1, context);
                sinon.assert.alwaysCalledOn(spy2, context);
                sinon.assert.alwaysCalledOn(spy3, context);
                sinon.assert.calledWithExactly(spy1.getCall(0), 2, { done: false, value: 2, index: 0 }, 0);
                sinon.assert.calledWithExactly(spy2.getCall(0), 0, { done: false, value: 0, index: 0 }, 1);
                sinon.assert.calledWithExactly(spy3.getCall(0), -1, { done: false, value: -1, index: 0 }, 2);
            });
        });

        // class ParallelIterator ---------------------------------------------

        describe('sub class "ParallelIterator"', function () {
            var ParallelIterator = Iterator.ParallelIterator;
            it('should exist', function () {
                expect(ParallelIterator).to.be.a('function');
                expect(ParallelIterator.prototype).to.be.an.instanceof(Iterator);
            });
            it('should create a parallel iterator', function () {
                var arr1 = [0, 2, 6, 8, 10,     12];
                var arr2 = [0, 1, 3,     5,      6, 7];
                var arr3 = [1, 3, 7, 9, 11, 12, 13];
                var it1 = new Iterator.ArrayIterator(arr1);
                var spy1 = sinon.spy(_.identity);
                var spy2 = sinon.spy(function (n) { return 2 * n; });
                var spy3 = sinon.spy(function (n) { return n - 1; });
                var context = {};
                var parallelIt = new ParallelIterator([it1, arr2, arr3], [spy1, spy2, spy3], context);
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [0, 0, 1], results: [{ done: false, value: 0, index: 0 }, { done: false, value: 0, index: 0 }, { done: false, value: 1, index: 0 }], offset: 0, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [2, 1, 3], results: [{ done: false, value: 2, index: 1 }, { done: false, value: 1, index: 1 }, { done: false, value: 3, index: 1 }], offset: 2, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [6, 3, 7], results: [{ done: false, value: 6, index: 2 }, { done: false, value: 3, index: 2 }, { done: false, value: 7, index: 2 }], offset: 6, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [8, null, 9], results: [{ done: false, value: 8, index: 3 }, null, { done: false, value: 9, index: 3 }], offset: 8, complete: false });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [10, 5, 11], results: [{ done: false, value: 10, index: 4 }, { done: false, value: 5, index: 3 }, { done: false, value: 11, index: 4 }], offset: 10, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [null, null, 12], results: [null, null, { done: false, value: 12, index: 5 }], offset: 11, complete: false });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [12, 6, 13], results: [{ done: false, value: 12, index: 5 }, { done: false, value: 6, index: 4 }, { done: false, value: 13, index: 6 }], offset: 12, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [null, 7, null], results: [null, { done: false, value: 7, index: 5 }, null], offset: 14, complete: false });
                expect(parallelIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy1, context);
                sinon.assert.alwaysCalledOn(spy2, context);
                sinon.assert.alwaysCalledOn(spy3, context);
                sinon.assert.calledWithExactly(spy1.getCall(0), 0, { done: false, value: 0, index: 0 }, 0);
                sinon.assert.calledWithExactly(spy2.getCall(0), 0, { done: false, value: 0, index: 0 }, 1);
                sinon.assert.calledWithExactly(spy3.getCall(0), 1, { done: false, value: 1, index: 0 }, 2);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "from"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('from');
            });
            it('should convert different types to iterators', function () {
                var EMPTY = Iterator.EMPTY;
                var it1 = new Iterator.SingleIterator({ value: 1 });
                expect(Iterator.from(null)).to.equal(EMPTY);
                expect(Iterator.from(it1)).to.equal(it1);
                expect(Iterator.from([1, 2, 3])).to.be.an.instanceof(Iterator.ArrayIterator);
                expect(Iterator.from(_.noop)).to.be.an.instanceof(Iterator.GeneratorIterator);
                expect(Iterator.from({ iterator: _.constant(it1) })).to.equal(it1);
                expect(Iterator.from({})).to.equal(EMPTY);
                expect(Iterator.from(42)).to.equal(EMPTY);
            });
        });

        describe('method "toArray"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('toArray');
            });
            it('should collect all values of an array iterator', function () {
                var it = new Iterator.ArrayIterator([1, 2, 3]);
                expect(Iterator.toArray(it)).to.deep.equal([1, 2, 3]);
            });
            it('should collect all values of a generator function', function () {
                var i = 0;
                function generator() { var result = (i < 5) ? { value: i } : { done: true }; i += 1; return result; }
                expect(Iterator.toArray(generator)).to.deep.equal([0, 1, 2, 3, 4]);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('forEach');
            });
            it('should visit all steps of an iterator', function () {
                var it = new Iterator.ArrayIterator([1, 2, 3]), spy = sinon.spy(), context = {};
                expect(Iterator.forEach(it, spy, context)).to.equal(undefined);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, { value: 1, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), 2, { value: 2, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 3, { value: 3, index: 2, done: false });
            });
            it('should break if callback returns Utils.BREAK', function () {
                var it = new Iterator.ArrayIterator([1, 2, 3], { reverse: true });
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                expect(Iterator.forEach(it, stub)).to.equal(Utils.BREAK);
                sinon.assert.callCount(stub, 2);
                sinon.assert.calledWithExactly(stub.getCall(0), 3, { value: 3, index: 2, done: false });
                sinon.assert.calledWithExactly(stub.getCall(1), 2, { value: 2, index: 1, done: false });
            });
        });

        describe('method "some"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('some');
            });
            it('should visit an iterator until it finds a truthy value', function () {
                var it = new Iterator.ArrayIterator([0, false, 2, '', true]), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.some(it, spy, context)).to.equal(true);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 0, { value: 0, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), false, { value: false, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 2, { value: 2, index: 2, done: false });
            });
            it('should visit all falsy values', function () {
                var it = new Iterator.ArrayIterator([0, false, '']), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.some(it, spy, context)).to.equal(false);
                sinon.assert.callCount(spy, 3);
            });
            it('should return false for an empty iterator', function () {
                expect(Iterator.some(Iterator.EMPTY, _.identity)).to.equal(false);
            });
        });

        describe('method "every"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('every');
            });
            it('should visit an iterator until it finds a falsy value', function () {
                var it = new Iterator.ArrayIterator([1, true, 0, 'abc', false]), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.every(it, spy, context)).to.equal(false);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, { value: 1, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), true, { value: true, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 0, { value: 0, index: 2, done: false });
            });
            it('should visit all truthy values', function () {
                var it = new Iterator.ArrayIterator([1, true, 'abc']), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.every(it, spy, context)).to.equal(true);
                sinon.assert.callCount(spy, 3);
            });
            it('should return true for an empty iterator', function () {
                expect(Iterator.every(Iterator.EMPTY, _.identity)).to.equal(true);
            });
        });

        describe('method "map"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('map');
            });
            it('should return all results of the callback function', function () {
                var it = new Iterator.ArrayIterator([2, 3, 4]), spy = sinon.spy(function (a) { return a * a; }), context = {};
                expect(Iterator.map(it, spy, context)).to.deep.equal([4, 9, 16]);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 2, { value: 2, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), 3, { value: 3, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 4, { value: 4, index: 2, done: false });
            });
        });

        describe('method "reduce"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('reduce');
            });
            it('should visit all steps of an iterator', function () {
                var it = new Iterator.ArrayIterator([2, 3, 4]), spy = sinon.spy(function (a, b) { return a + b; }), context = {};
                expect(Iterator.reduce(1, it, spy, context)).to.equal(10);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, 2, { value: 2, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), 3, 3, { value: 3, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 6, 4, { value: 4, index: 2, done: false });
            });
        });

        describe('method "find"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('find');
            });
            it('should visit an iterator until it finds a truthy value', function () {
                var it = new Iterator.ArrayIterator([0, false, 42, '', true, null]), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.find(it, spy, context)).to.deep.equal({ value: 42, index: 2, done: false });
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 0, { value: 0, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), false, { value: false, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 42, { value: 42, index: 2, done: false });
            });
            it('should visit all falsy values', function () {
                var it = new Iterator.ArrayIterator([0, false, '', null]), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.find(it, spy, context)).to.equal(null);
                sinon.assert.callCount(spy, 4);
            });
            it('should return false for an empty iterator', function () {
                expect(Iterator.find(Iterator.EMPTY, _.identity)).to.equal(null);
            });
        });

        describe('method "findLast"', function () {
            it('should exist', function () {
                expect(Iterator).itself.to.respondTo('findLast');
            });
            it('should visit an iterator completely', function () {
                var it = new Iterator.ArrayIterator([0, false, 42, '', true, null]), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.findLast(it, spy, context)).to.deep.equal({ value: true, index: 4, done: false });
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, context);
            });
            it('should visit all falsy values', function () {
                var it = new Iterator.ArrayIterator([0, false, '', null]), spy = sinon.spy(_.identity), context = {};
                expect(Iterator.findLast(it, spy, context)).to.equal(null);
                sinon.assert.callCount(spy, 4);
            });
            it('should return false for an empty iterator', function () {
                expect(Iterator.findLast(Iterator.EMPTY, _.identity)).to.equal(null);
            });
        });
    });

    // ========================================================================
});
