/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils/enum'
], function (Enum) {

    'use strict';

    // class Enum =============================================================

    describe('Toolkit class Enum', function () {

        it('should exist', function () {
            expect(Enum).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var Weekday = null;
        var Month = null;

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(Enum).itself.to.respondTo('create');
            });
            it('should create new enumerations', function () {
                Weekday = Enum.create('Monday Tuesday Wednesday Thursday Friday Saturday Sunday');
                expect(Weekday).to.be.a('function');
                expect(Weekday).to.have.a.property('MONDAY').that.is.an.instanceof(Weekday);
                expect(Weekday.MONDAY).to.have.a.property('name', 'MONDAY');
                expect(Weekday.MONDAY).to.have.a.property('json', 'Monday');
                expect(Weekday.MONDAY).to.have.a.property('index', 0);
                expect(Weekday).to.have.a.property('SUNDAY').that.is.an.instanceof(Weekday);
                expect(Weekday.SUNDAY).to.have.a.property('name', 'SUNDAY');
                expect(Weekday.SUNDAY).to.have.a.property('json', 'Sunday');
                expect(Weekday.SUNDAY).to.have.a.property('index', 6);
                Month = Enum.create('January February March April May June July August September October November December');
                expect(Month).to.be.a('function');
                expect(Month).to.have.a.property('JANUARY').that.is.an.instanceof(Month);
                expect(Weekday).to.not.equal(Month);
            });
            it('should convert camel-case to upper-case', function () {
                var MyEnum = Enum.create('abcd BCDE cdEf deFG EFgh fg42hi gh42Ij');
                expect(MyEnum).to.have.a.property('ABCD');
                expect(MyEnum).to.have.a.property('BCDE');
                expect(MyEnum).to.have.a.property('CD_EF');
                expect(MyEnum).to.have.a.property('DE_FG');
                expect(MyEnum).to.have.a.property('EFGH');
                expect(MyEnum).to.have.a.property('FG_42HI');
                expect(MyEnum).to.have.a.property('GH_42_IJ');
            });
            it('should detect invalid names', function () {
                expect(Enum.create.bind(Enum, 'abc 1')).to.throw(Error);
                expect(Enum.create.bind(Enum, 'abC AB_C')).to.throw(Error);
            });
        });

        describe('method "parse"', function () {
            it('should exist in subclasses', function () {
                expect(Enum).itself.to.not.respondTo('parse');
                expect(Weekday).itself.to.respondTo('parse');
            });
            it('should parse enumeration names', function () {
                expect(Weekday.parse('Monday')).to.equal(Weekday.MONDAY);
                expect(Weekday.parse('January')).to.equal(null);
                expect(Weekday.parse('January', Weekday.TUESDAY)).to.equal(Weekday.TUESDAY);
                expect(Month.parse('January')).to.equal(Month.JANUARY);
                expect(Month.parse('Monday')).to.equal(null);
                expect(Month.parse('Monday', Month.FEBRUARY)).to.equal(Month.FEBRUARY);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Weekday).to.respondTo('toString');
            });
            it('should convert enumeration values to strings', function () {
                expect(Weekday.MONDAY.toString()).to.equal('MONDAY');
                expect(String(Weekday.MONDAY)).to.equal('MONDAY');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Weekday).to.respondTo('toJSON');
            });
            it('should convert enumeration values to strings', function () {
                expect(Weekday.MONDAY.toJSON()).to.equal('Monday');
                expect(JSON.stringify(Weekday.MONDAY)).to.equal('"Monday"');
                expect(JSON.stringify({ day: Weekday.MONDAY })).to.equal('{"day":"Monday"}');
            });
        });

        describe('method "valueOf"', function () {
            it('should exist', function () {
                expect(Weekday).to.respondTo('valueOf');
            });
            it('should convert enumeration values to indexes', function () {
                expect(Weekday.MONDAY.valueOf()).to.equal(0);
                expect(Weekday.SUNDAY.valueOf()).to.equal(6);
                expect(Weekday.MONDAY < Weekday.SUNDAY).to.equal(true);
                expect(Weekday.MONDAY > Weekday.SUNDAY).to.equal(false);
                expect(Weekday.SUNDAY - 1).to.equal(5);
            });
        });
    });

    // ========================================================================
});
